/*
 *
 *      Copyright (c) 1995  Claus-Justus Heine 
 *
 *      This file defines a volume table as defined in the QIC-80 development 
 *      standards defined.
 *
 *      This is a minimal implementation, just allowing ordinary DOS :(
 *      prgrams to identify the cartridge as used.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2, or (at
 * your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; see the file COPYING.  If not, write to
 * the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139,
 * USA.
 *
 * claus
 *
 * 1.14
 * 1995/11/16 22:34:54
 * Exp
 *
 */
static char RCSid[] =
"qic80-vtbl.c,v 1.14 1995/11/16 22:34:54 claus Exp";

#include "ftape.h"
#include <linux/errno.h>
#include <sys/types.h>

#include "qic80-compress.h"
#include "qic80-vtbl.h" 
#include "ftape-rw.h"
#include "ftape-write.h"
#include "ftape-ctl.h"
#include "ftape-bsm.h"
#include "ftape-read.h"

/*
 *  global variables 
 */
int ftape_qic80_mode = 1; /* use the vtbl */
int ftape_old_ftape  = 0; /* prevents old ftaped tapes to be overwritten */
int ftape_volume_table_changed = 0; /* for write_header_segments() */

/*
 *  private variables (only exported for inline functions)
 */
#ifdef DYN_ALLOC
struct qic80_internal_vtbl *__qic80_vtbl= NULL;
#else
struct qic80_internal_vtbl __qic80_vtbl[MAX_VOLUMES] = { { 0, 0, 0, 0, 0, 0 },  };
#endif
struct qic80_internal_vtbl *__qic80_last_vtbl = NULL;
unsigned int __qic80_num_of_volumes = 0;
unsigned __qic80_file_no = 0;

/*
 *  hack to prevent vtbl to be created on an old ftape, which would destroy the
 *  first data-segment. Also used in raw mode, faking one single volume for
 *  the entire tape
 */

static unsigned faked_vtbl = 0; /* flag */

/*
 *  check_volume_label().
 *
 *  We store the block-size of the volume in the volume-label, using the 
 *  keyword "blocksize". The blocksize written to the volume-label is in bytes.
 *
 */
static int
check_volume_label( char *label, unsigned short *block_size )
{ 
TRACE_FUN( 8, "check_volume_label");
int valid_format;
char *blocksize;

  if ( strncmp( label, VOL_NAME, strlen(VOL_NAME)) != 0 ) {
    *block_size = 1; /* smallest block size that we allow */
    valid_format = 0;
  } else {
    /* get the default blocksize */
    /* use the kernel strstr()   */
    blocksize= strstr( label, " blocksize " );
    if ( blocksize ) {
      blocksize += strlen(" blocksize ");
      for( *block_size= 0; *blocksize >= '0' && *blocksize <= '9'; blocksize++ ) {
        *block_size *= 10;
        *block_size += *blocksize - '0';
      }
      if ( *block_size > MAX_BLOCK_SIZE ) {
        *block_size= 1;
        valid_format= 0;
      } else {
        valid_format = 1;
      }
    } else {
      *block_size= 1;
      valid_format= 0;
    }
  }
  TRACE_EXIT;
  return valid_format;
}

/*
 *  initialize vtbl, called by ftape_new_cartridge()
 */
void qic80_init_vtbl( void )
{ 
TRACE_FUN( 5, "init_vtbl");
int i;

  for (i = 0; i < MAX_VOLUMES; i++ ) {
    __qic80_vtbl[i].start_seg = 0;
    __qic80_vtbl[i].end_seg   = 0;
    __qic80_vtbl[i].size      = 0;
  }
  __qic80_last_vtbl = &__qic80_vtbl[0];
  __qic80_num_of_volumes = 0;
  TRACE_EXIT;
} 

/*
 *  this one creates the volume headers for each volume. 
 *  It is assumed that buffer already contains the old volume-table,
 *  so that vtbl entries without the zftape_volume flag set can savely be
 *  ignored.
 *
 *  We also clear the compression map for all segments after the last
 *  volume.
 *
 */
void qic80_create_volume_headers( byte *buffer )
{   
TRACE_FUN( 5, "create_volume_headers");
int i;
struct qic80_vtbl *entry;
char number[6];

  entry = (struct qic80_vtbl *) buffer;
  for ( i=0; i < __qic80_num_of_volumes; i++ ) {
    if ( __qic80_vtbl[i].zftape_volume ) {
      sprintf( number, " %d", i);
      entry->signature= *((unsigned long *)"VTBL");
      entry->first_segment = __qic80_vtbl[i].start_seg;
      entry->last_segment = __qic80_vtbl[i].end_seg;
      strcpy( entry->description, VOL_NAME);
      strcat( entry->description, number );
      strcat( entry->description, ", blocksize ");
      sprintf( entry->description + strlen(entry->description),"%d",__qic80_vtbl[i].block_size );
      entry->date = 0UL; /* should be initialized */
      entry->flags.dummy = 0;
      entry->flags.flags.not_verified     =
      entry->flags.flags.segment_spanning = 1;
      entry->multi_cartridge_count = 1;
      memset( entry->vendor_extension, '\0', 26 );
      memset( entry->password, '\0', 8 );
      entry->dir_section_size = 0;
      entry->data_section_size = __qic80_vtbl[i].size;
      entry->OS_version_major = 1;
      entry->OS_version_minor = 2;
      memset( entry->source_volume, '\0', 16 );
      entry->logical_dev_file_set = 0;
      entry->physical_dev_file_set = 0;
      entry->QIC_compression_method = 0x3f; /* not registered */
      if ( __qic80_vtbl[i].use_compression ) {
        entry->QIC_compression_method |= 0x80;  /* use compression */
        TRACEx1(3,"volume %d uses compression", i);
      }
      entry->OS_flags.dummy = 0x0000;
      entry->OS_flag_dummy = 0x0000;
      entry->OS_flags.flags.OS_UNIX = 1;
      entry->ISO_compression_method = 0;
      memset( entry->reserved, '\0', 4 );
    } else if ( entry->signature != *((unsigned long *)"VTBL") ) {
      /*
       *  check for a valid signature. If none is found, create an 
       *  empty vtbl entry that contains nothing but the signature
       */                         
      memset( entry, '\0', sizeof(struct qic80_vtbl) );
      entry->signature= *((unsigned long *)"VTBL");
    }
    entry++;
  }        
  TRACEi( 9, "sizeof(struct qic80_vtbl): ",sizeof(struct qic80_vtbl));
  TRACEi( 9, "sizeof(struct qic80_vtbl->OS_flags): ",sizeof(entry->OS_flags));
  TRACEi( 9, "sizeof(struct qic80_vtbl->OS_flags): ",sizeof(entry->flags));
  memset( entry, '\0', (SECTORS_PER_SEGMENT - 3) *SECTOR_SIZE - __qic80_num_of_volumes * sizeof(struct qic80_vtbl) );
  entry->signature = 0UL;
  if ( __qic80_num_of_volumes == 0 ) {
    memset( &ftape_compression_map[0], '\0', COMPRESSION_MAP_SIZE );
  } else {
    TRACEx2( 4, "last_vtbl: %08lx, __qic80_last_vtbl->end_seg: %d", (unsigned long)__qic80_last_vtbl, __qic80_last_vtbl->end_seg);
    if ( COMPRESSION_MAP_SIZE/sizeof(unsigned long) - 1 < __qic80_last_vtbl->end_seg ) {
      memset( &ftape_compression_map[ __qic80_last_vtbl->end_seg + 1],
	     '\0',
	     COMPRESSION_MAP_SIZE - sizeof(unsigned long) * (__qic80_last_vtbl->end_seg + 1) );
    }
  }
  TRACE_EXIT;
}

/*
 *  extract the volume table from buffer. "buffer" must already contain
 *  the vtbl-segment
 */

void 
qic80_extract_volume_headers( byte *buffer )
{                            
TRACE_FUN( 5, "extract_volume_headers");
struct qic80_vtbl *entry;

  entry = (struct qic80_vtbl *) buffer;
  __qic80_last_vtbl = &__qic80_vtbl[0];
  __qic80_num_of_volumes = 0;
  /*
   *  the end of the vtbl is indicated by entry->signature == 0UL
   */
  while( entry->signature == *((unsigned long *)"VTBL") ) {
    __qic80_last_vtbl->start_seg = entry->first_segment;
    __qic80_last_vtbl->end_seg   = entry->last_segment;     
    /*
     *  check if we created this volume and get the block_size
     */
    __qic80_last_vtbl->zftape_volume = check_volume_label(  entry->description,
                                                   &__qic80_last_vtbl->block_size );
    if ( ! __qic80_last_vtbl->zftape_volume ) {
      /*
       *  non zftape volumes are handled in raw mode. Thus we need to 
       *  calculate the raw amount of data contained in those segments.
       *
       */
      __qic80_last_vtbl->size = ftape_calc_data_pos( __qic80_last_vtbl->end_seg + 1, 0 );
      __qic80_last_vtbl->size -= ftape_calc_data_pos( __qic80_last_vtbl->start_seg, 0 );
      TRACEx2( 4, "Fake alien volume's size from %d to %d", (int)entry->data_section_size, (int)__qic80_last_vtbl->size );
    } else {
      __qic80_last_vtbl->size = entry->data_section_size;
      __qic80_last_vtbl->use_compression = entry->QIC_compression_method & 0x80 != 0; 
      TRACEi( 4, "use_compression: ", __qic80_last_vtbl->use_compression != 0 );
      TRACEx1( 4, "QIC compression: %x", entry->QIC_compression_method );
    }
    TRACEx3( 4, "volume start at %d, volume end at %d, size %d", __qic80_last_vtbl->start_seg, __qic80_last_vtbl->end_seg, (int)__qic80_last_vtbl->size );
    TRACEx1( 4, "Volume description: %s",entry->description);
    __qic80_num_of_volumes ++;
    __qic80_last_vtbl++;
    entry++;
  }
  __qic80_last_vtbl--;  /* should be &__qic80_vtbl[__qic80_num_of_volumes - 1 ] */
  faked_vtbl = 0;
  TRACE_EXIT;
}     


/*
 *  write volume table to tape. Calls qic80_create_volume_headers()
 */
int
qic80_update_volume_table( unsigned segment, byte *buffer )
{
  TRACE_FUN( 5, "ftape_update_volume_table");
  int result = 0;

  /*
   *  I wonder why I inserted that loop_until_writes_done() ?
   */
  if (ftape_state == writing) {
    result = loop_until_writes_done();
  }
  if (result >= 0) {
    result = ftape_abort_operation();
    if (result >= 0) {
      TRACE( 4, "Reading volume table ... ");
      ftape_deblock_segment = -1;
      result = read_segment( segment, buffer, 0);
      if (result < 0) {
        TRACE_EXIT;
        return result;
      }
      qic80_create_volume_headers( buffer );
      result = ftape_verify_write_segment( segment, buffer );
    }
  }
  TRACE_EXIT;
  return result;
}

/*
 *  update the internal volume table
 *
 *  if before start of last volume: erase all following volumes
 *  if inside a volume: set end of volume to infinity
 *
 *  this function is intended to be called every time _ftape_write()
 *  is called
 *
 *   return: 0 if no new volume was created, 1 if a new volume was created
 *
 *  NOTE: we don't need to check for qic80_mode as ftape_write() does that
 *  already. This function gets never called without accessing zftape via the
 *  *qft* devices
 */

int 
qic80_maybe_write_beof( unsigned seg_pos, unsigned no_new_one )
{ 
TRACE_FUN( 5, "maybe_write_beof");
int result;

  ftape_volume_table_changed = 1;
  if ( __qic80_num_of_volumes == 0 ) { 
    /*
     *  always create a volume if empty tape
     */
    TRACE( 7, "__qic80_num_of_volumes == 0");
    __qic80_last_vtbl = &__qic80_vtbl[0];
    __qic80_last_vtbl->start_seg = seg_pos;
    __qic80_num_of_volumes = 1;
    result = 1;
  } else if ( seg_pos < __qic80_last_vtbl->start_seg ) {
    TRACEi( 7, "start_seg:", __qic80_last_vtbl->start_seg );
    TRACEi( 7, "end_seg:", __qic80_last_vtbl->end_seg );
    TRACEi( 7, "seg:", seg_pos);
    TRACE( 7, "before last volume");
    /*
     *  we allow nothing but to apppend files to a tape. Thus if one really
     *  wants to write data to tape before the last registered volume,
     *  then the volume-entry for this volume is deleted
     */
    while ( __qic80_last_vtbl->start_seg > seg_pos ) {
      __qic80_last_vtbl->start_seg =
      __qic80_last_vtbl->end_seg   = 0;
      __qic80_last_vtbl->size      = 0UL;
      __qic80_last_vtbl--;
      __qic80_num_of_volumes--;
    }
    result = 0;
  } else if ( no_new_one ||
              (seg_pos >= __qic80_last_vtbl->start_seg && 
               seg_pos <= __qic80_last_vtbl->end_seg      )
            ) 
  {
    /*
     *  easy: just set end of volume to infinity, done below at end of function
     */
    result = 0;
  } else if ( seg_pos == __qic80_last_vtbl->end_seg + 1 ) {
    /*
     *  create a new volume. That's save, because we set the end of volume 
     *  to infinity when we start adding data to it.
     */
    TRACE( 4, "create new volume");
    __qic80_last_vtbl++;
    __qic80_num_of_volumes++;
    __qic80_last_vtbl->start_seg = seg_pos;
    result = 1;
  } else {
    TRACEx2( 1, "Error: seg_pos: %d, __qic80_last_vtbl->end_seg: %d", seg_pos, __qic80_last_vtbl->end_seg );
    result = -EIO;
  }                            
  if ( result >= 0 ) {
    if ( result == 1 ) {
      ftape_uncmpr_pos = ftape_seg_data_pos   = 0;
    }
    if ( ftape_uncmpr_pos == 0 ) {
      __qic80_last_vtbl->block_size      = ftape_block_size;
      __qic80_last_vtbl->zftape_volume   = 1;
      __qic80_last_vtbl->use_compression = ftape_use_compression;
      __qic80_last_vtbl->size = 0UL;
    }
    __qic80_last_vtbl->end_seg = ftape_last_data_segment;
    ftape_volume_table_changed = 1;
  }
  TRACE_EXIT;
  return result;
}

/*
 *   This one checks for EOF.
 *   return 0 if not EOF, return 1 in case of EOF
 */

int 
qic80_check_for_eof( unsigned seg_pos, unsigned long data_pos, long *distance )
{     
  TRACE_FUN( 8,"check_for_eof ");
  int result;

  if ( ftape_qic80_mode ) {
    /*
     *  find the right file
     */       
    result = qic80_find_file_no( seg_pos );
    if ( result < __qic80_num_of_volumes ) {
      *distance = __qic80_vtbl[ result ].size - data_pos;
      if ( *distance < 0 ) {
	TRACEx1( 4, "end of file reached. Excess: %ld", *distance );
	result = 1;
      } else {
	result = 0;
      }
    } else { /* at EOM */
      *distance = 0;
      result = 1;
    }
  } else if ( seg_pos <= ftape_last_data_segment ) {
    *distance = ftape_total_data_amount - data_pos;
    if ( *distance < 0 ) {
      TRACEx1( 4, "end of tape reached. Excess: %ld", *distance );
      result = 1;
    } else {
      result = 0;
    }
  } else { /* at EOT, should not happen */
    *distance = 0;
    result = 1;
  }
  TRACE_EXIT;
  return result;
}

/*
 *   perform mtfsf, mtbsf, not allowed without ftape_qic80_mode
 */

int
qic80_skip_files( unsigned segment, int count )
{ 
TRACE_FUN( 8, "ftape_skip_files");
int result = 0;
int vol_no;

  TRACEi(5,"count: ", count);
  vol_no  = qic80_find_file_no( segment );
  vol_no += count;
  ftape_resid = vol_no;
  if ( vol_no < 0 ) {
    result = -EIO; 
    TRACE(-1,"Error: try to seek to negativ file number");
    vol_no -= count; /* for ftape_resid */
    ftape_seg_pos = __qic80_vtbl[ 0 ].start_seg;
  } else if ( vol_no >= __qic80_num_of_volumes ) {
    if ( vol_no > __qic80_num_of_volumes ) {
      result = -EIO;
      TRACEi( 1, "Error: not so many files. Number of files:", __qic80_num_of_volumes);
      vol_no = __qic80_num_of_volumes;
    }
    if ( vol_no > 0 ) {
      ftape_seg_pos = __qic80_vtbl[ __qic80_num_of_volumes - 1 ].end_seg + 1;
    } else {
      ftape_seg_pos = ftape_first_user_segment;
    }
  } else {
    ftape_seg_pos = __qic80_vtbl[ vol_no ].start_seg;
  }
  ftape_seg_data_pos = 0;
  ftape_data_pos = ftape_calc_data_pos( ftape_seg_pos, ftape_seg_data_pos );
  ftape_uncmpr_pos = 0;
  ftape_reset_cmpr_locals();
  TRACE( 4, "repositioning");
  TRACEi( 4, "ftape_seg_pos        : ", ftape_seg_pos);
  TRACEi( 4, "ftape_seg_data_pos   : ", ftape_seg_data_pos);
  TRACEi( 4, "ftape_data_pos       : ", ftape_data_pos);
  TRACEi( 4, "ftape_uncmpr_pos     : ", ftape_uncmpr_pos);
  TRACEi( 4, "file number          : ", vol_no);
  ftape_resid -= vol_no;
  if ( ftape_resid < 0 ) {
    ftape_resid = -ftape_resid;
  }
  TRACE_EXIT;
  return result;
}


/*
 *  skip to eom, used for MTEOM
 *
 * NOTE: without ftape_seg_data_pos, that is in raw mode, this results in
 *       skipping to end of tape
 */
void
qic80_skip_to_eom( void )
{ 
  TRACE_FUN( 8, "qic80_skip_to_eom");

  if ( ftape_qic80_mode ) {
    if ( __qic80_num_of_volumes > 0 ) {
      ftape_seg_pos = __qic80_vtbl[ __qic80_num_of_volumes - 1 ].end_seg + 1;
      ftape_seg_data_pos = ftape_uncmpr_pos = just_before_eof = 0;
      ftape_data_pos = ftape_calc_data_pos( ftape_seg_pos, ftape_seg_data_pos );
    } else {
      ftape_reset_position();
    }
  } else {
    ftape_seg_pos = ftape_last_data_segment + 1;
    ftape_data_pos = ftape_total_data_amount;
    ftape_seg_data_pos = ftape_uncmpr_pos = just_before_eof = 0;
  }
  TRACEx2( 4, "ftape positioned to segment %d, data pos %d", ftape_seg_pos, ftape_data_pos );
  TRACE_EXIT;
}

/*
 *   this functions translates the failed_sector_log, misused as EOF-marker
 *   list, into a virtual volume table. The table mustn't be written to 
 *   tape, because this would occupy the first data segment, which should be
 *   the volume table, but is actualy the first segment that is filled with
 *   data (when using standard ftape).
 *   We assume, that we get a non-empty failed_sector_log.
 */

void qic80_fake_volume_headers( eof_mark_union *eof_map, int num_failed_sectors )
{
TRACE_FUN ( 8, "qic80_fake_volume_headers");

  __qic80_last_vtbl = &__qic80_vtbl[0];
  __qic80_last_vtbl->start_seg = first_data_segment;
  for(__qic80_num_of_volumes= 0; __qic80_num_of_volumes < num_failed_sectors; __qic80_num_of_volumes++) {
    __qic80_last_vtbl->end_seg = eof_map[__qic80_num_of_volumes].mark.segment;
    __qic80_last_vtbl->zftape_volume = 0;
    __qic80_last_vtbl->block_size = 1024;
    __qic80_last_vtbl->size = ftape_calc_data_pos( __qic80_last_vtbl->end_seg, eof_map[__qic80_num_of_volumes].mark.sector*SECTOR_SIZE );
    __qic80_last_vtbl->size -= ftape_calc_data_pos( __qic80_last_vtbl->start_seg, 0 );
    __qic80_last_vtbl++;
    __qic80_last_vtbl->start_seg = eof_map[__qic80_num_of_volumes].mark.segment + 1;
  }      
  __qic80_last_vtbl --;
  faked_vtbl = 1; /* hack to prevent hurting old ftape-tapes */
  TRACE_EXIT;
}

/*
 *  write count file-marks at current position. 
 *  NOTE: this command is always destructive as 
 *  volumes end on segment boundary, but files not. Also we 
 *  only allow appending of files to a tape. That means, that 
 *  this command deletes all volume-entries after the current position.
 *
 *  The tape is positioned after the eof-marker, that is at byte 0
 *  of the segment following the eof-marker
 *
 *  this  function is only allowed in ftape_qic80_mode
 */                                   

int
qic80_weof( unsigned count )
{
TRACE_FUN( 5, "qic80_weof");
int file_no;
int result = 0;

  /*
   *   hack to prevent writing volume-tables on old ftape-tapes,
   *   which would destroy the first_data_segment
   */                                            
  if ( faked_vtbl ) {
    TRACE( 1, "Error: Sorry, can't write EOF-marks on old ftape-tapes. Try \"mt erase\"");
    TRACE( 1, "       But this will definitely damage the data on your tape.");
    TRACE_EXIT;
    return -EPERM;
  }
  if ( count > 0 ) {
    ftape_volume_table_changed = 1;
    if ( (file_no = qic80_find_file_no( ftape_seg_pos )) < __qic80_num_of_volumes ) {
      /*
       *  this implies also __qic80_num_of_volumes > 0, that is, we are inside
       *  a valid file. Just terminate volume at the current segpos
       */
      __qic80_last_vtbl = &__qic80_vtbl[ file_no ];
      TRACEx2( 4, "Truncating file %d. Original size: %d", (int)file_no, (int)__qic80_last_vtbl->size );
      if ( ftape_seg_data_pos == 0 ) {
        if ( ftape_seg_pos == __qic80_last_vtbl->start_seg ) {
          /*
           *  empty volume
           */
          TRACEx1( 4, "File %d is empty now.", file_no);
          ftape_data_pos += ftape_get_seg_size( ftape_seg_pos );
          __qic80_last_vtbl->end_seg = ftape_seg_pos;
          __qic80_last_vtbl->size = 0;
          __qic80_last_vtbl->zftape_volume = 1;
          __qic80_last_vtbl->use_compression = ftape_use_compression;
          __qic80_last_vtbl->block_size = ftape_block_size;
          ftape_seg_pos ++;
        } else {
          /*
           *  at 0th byte of segment, volume not empty
           */                                         
          __qic80_last_vtbl->end_seg = ftape_seg_pos - 1;
          __qic80_last_vtbl->size    = ftape_uncmpr_pos;
        }
      } else {
        /*
         * inside a segment inside a volume
         */
        __qic80_last_vtbl->end_seg = ftape_seg_pos;
        __qic80_last_vtbl->size    = ftape_uncmpr_pos;
        /*
         *   need to position at start of next segment. Volumes always
         *   start at segment boundary
         */
        ftape_advance_to_next_segment();
      }
      TRACEx2( 4, "File %d's new size is %d", (int)file_no, (int)__qic80_last_vtbl->size );
      __qic80_num_of_volumes = file_no + 1;
      count --;
    }
    ftape_uncmpr_pos = just_before_eof = 0;
    /*
     *  now it's easy, just append eof-marks, that is empty volumes,
     *  to the end of the already recorded media.
     */
    while ( count > 0 && 
            ftape_seg_pos <= ftape_last_data_segment && 
            __qic80_num_of_volumes < MAX_VOLUMES ) {
      /*
       *  write the remaining eof-marks.
       */
      TRACEi( 4, "Writing zero sized file at segment ", ftape_seg_pos );
      __qic80_last_vtbl ++;
      __qic80_last_vtbl->start_seg     =
      __qic80_last_vtbl->end_seg       = ftape_seg_pos;
      __qic80_last_vtbl->size          = 0;
      __qic80_last_vtbl->block_size    = ftape_block_size;
      __qic80_last_vtbl->zftape_volume = 1;
      __qic80_last_vtbl->use_compression = ftape_use_compression;
      __qic80_num_of_volumes ++;
      ftape_data_pos += ftape_get_seg_size( ftape_seg_pos );
      ftape_seg_pos ++;
      count --;
    } 
    if ( count > 0 ) {
      /* there are two possibilities: end of tape, or the maximum
       * number of files is exhausted. I return -ENOSPC in both cases.
       * Maybe it should be EINVAL for the latter case ...
       */
#ifndef NO_TRACE_AT_ALL
      if ( __qic80_num_of_volumes == MAX_VOLUMES ) {
	TRACEi( 1, "Error: maximum allowed number of files exhausted:", MAX_VOLUMES );
      } else {
	TRACE( 1, "Error: reached end of tape");
      }
#endif
      result = -ENOSPC;
    }
  }
  ftape_resid = count;
  TRACEi( 4, "Number of marks NOT written: ", ftape_resid );
  TRACE_EXIT;
  return result;
}












