/*
 *      Copyright (C) 1994, 1995  Claus-Justus Heine
 *
 * This file implements a "generic" interface between the ftape-driver and a
 * compression-algorithm. The compression-algorithm currently used is a
 * LZ77. I use the implementation lzrw3 by Ross N. Williams
 * (Renaissance Software). The compression program itself is in the file
 * lzrw3.c and compress.h.
 * To adopt another compression algorithm the functions 
 * ftape_compress_info(), ftape_compress() and ftape_uncompress()
 * must be changed appropriately. See below.


 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2, or (at
 * your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; see the file COPYING.  If not, write to
 * the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139,
 * USA.
 *
 claus
 *
 1.5
 1995/11/16 13:35:00
 Exp
 *
 */
static char RCSid[] =
"qic80-compress.c,v 1.5 1995/11/16 13:35:00 claus Exp";


#include "ftape.h"
#include <linux/string.h>
#include <linux/errno.h>
#include <asm/segment.h>

#include "ftape-write.h"
#include "ftape-read.h"
#include "qic117.h"
#include "ftape-io.h"
#include "ftape-rw.h"
#include "ftape-eof.h"
#include "ftape-ctl.h"
#include "ftape-bsm.h"
#include "ftape-write.h"
#include "ftape-read.h"
#include "ecc.h"
#include "qic80-compress.h"
#include "compress/compress.h"

/*
 *   global variables
 */

int ftape_total_data_amount        = 0;
int ftape_data_pos                 = 0;
int ftape_uncmpr_pos               = 0;
int ftape_seg_data_pos             = 0;
int ftape_use_compression          = 0;
int ftape_compression_map_changed  = 0;
int ftape_compression_map_location = 1;
int ftape_compression_map_length   = 0;
int ftape_first_user_segment       = -1;

#ifdef DYN_ALLOC

unsigned long *ftape_compression_map = NULL;
byte *ftape_compression_wrk_mem      = NULL;
byte *ftape_compression_buffer       = NULL;
int ftape_cmpr_wrk_mem_size = 0;

#else

unsigned long ftape_compression_map[ COMPRESSION_MAP_SIZE/sizeof(unsigned long) ] = { 0UL, };
byte ftape_compression_wrk_mem [ CMPR_WRK_MEM_SIZE ] = { '\0', };
byte ftape_compression_buffer  [ CMPR_BUFFER_SIZE ]  = { '\0', };

#endif /* DYN_ALLOC */

/*
 *   local variables
 */

static int compressed_size = 0; /* how much data is left in the 
                                 * ftape_compression_buffer
                                 */
static int compos          = 0; /* position inside ftape_compression_buffer */


/*
 *  this function is needed for ftape_reset_position in ftape-io.c
 */

void
ftape_reset_cmpr_locals( void )
{
  compressed_size = 
  compos          = 0;
}


/******************************************************************************
 *
 *  The following three functions "ftape_compress_info()", "ftape_compress()"
 *  and "ftape_uncompress()" are the interface to the actual compression
 *  algorithm (i.e. they are calling the "compress()" function from the 
 *  lzrw3 package for now). These routines could quite easily be changed to 
 *  adopt another compression algorithm instead of lzrw3, which currently is
 *  used.
 *
 *****************************************************************************/

/*
 *  the purpose of this function is to return the amount of memory that
 *  is required by the compression-algorithm, because this memory
 *  is allocated dynamicaly (vmalloc) if DYN_ALLOC is defined. If 
 *  DYN_ALLOC is not defined this function must return a value that
 *  is smaller than or equal to CMPR_WRK_MEM_SIZE, defined in 
 *  "ftape-compress.h".
 *  If it returns a value greater than CMPR_WRK_MEM_SIZE, the driver failes to 
 *  initialize.
 */

int
ftape_compress_info( void )
{        
TRACE_FUN( 5, "ftape_compress_info");
struct compress_identity *id;

  compress(COMPRESS_ACTION_IDENTITY, NULL, NULL, 0, NULL, (unsigned long *)&id);
  if ( id )
  {
    printk("Id              : %08lx\n"
           "Required memory : %ld bytes\n"
           "Name            : %s\n"
           "Version         : %s\n"
           "Date            : %s\n"
           "Copyright       : %s\n"
           "Author          : %s\n"
           "Affiliation     : %s\n"
           "Vendor          : %s\n\n"
           , id->id, id->memory, id->name, id->version, id->date, 
           id->copyright, id->author, id->affiliation, id->vendor);
  }        
  TRACE_EXIT;
  return id->memory;
}   

/*
 *  called by ftape_compress_write() to perform the compression. Must return
 *  the size of the compressed data.
 *
 *  NOTE: The size of the compressed data should not exceed the size of
 *        the uncompressed data. Most compression Algorithms have means to
 *        store data unchanged if the "compressed" data amount would exceed
 *        the original one. Mostly this is done by storing some flag-bytes
 *        in front of the compressed data to indicate if it is compressed or
 *        not. Thus the worst compression result length is the original length
 *        plus those flag-bytes. 
 *
 *        We don't want that, as the QIC-80 standard provides a means of 
 *        marking uncomrpessed blocks by simply setting bit 15 of the 
 *        compressed block's length. Thus a compessed block can have at most a 
 *        length of 2^15-1 bytes. The QIC-80 standard restricts the 
 *        block-length even further, allowing only 29k - 6 bytes.
 *
 *        Currently, the maximum blocksize used by zftape is 28k.
 *
 *        In short: don't exceed the length of the input-package, set bit 15 
 *        of the compressed size to 1 if you have copied data instead of 
 *        compressing it.
 *
 */
static unsigned short
ftape_compress( byte *in_buffer, int in_size, byte *out_buffer )
{ 
TRACE_FUN( 5, "ftape_compress");
unsigned long compressed_size;

  compress( COMPRESS_ACTION_COMPRESS,
            ftape_compression_wrk_mem, in_buffer, in_size, out_buffer, 
                                                      &compressed_size);

#ifndef NO_TRACE_AT_ALL
  history.wr_uncompressed += in_size;
  history.wr_compressed += compressed_size & ~0x8000;
#endif

  TRACEx2( 7, "before compression: %d bytes, after compresison: %ld", 
                                  in_size,        compressed_size & ~0x8000 );
  TRACE_EXIT;
  return (unsigned short)compressed_size;
}

/*
 * called by ftape_compress_read() to decompress the data. Must return the size
 * of the decompressed data for sanity checks (compared with ftape_block_size)
 *
 *  NOTE: Read the note for ftape_compress() above!
 *        If bit 15 of the parameter in_size is set, then the data in in_buffer
 *        isn't compressed, which must be handled by the un-compression 
 *        algorithm. (I changed lzrw3 to handle this.)
 */
static unsigned short
ftape_uncompress( byte *in_buffer, unsigned short in_size, byte *out_buffer )
{ 
TRACE_FUN( 5, "ftape_uncompress");
unsigned long uncompressed_size;

  compress( COMPRESS_ACTION_DECOMPRESS,
            ftape_compression_wrk_mem, in_buffer, (int)in_size, out_buffer,
                                                    &uncompressed_size);

#ifndef NO_TRACE_AT_ALL
  history.rd_compressed += in_size & ~0x8000;
  history.rd_uncompressed += uncompressed_size;
#endif       

  TRACEx2( 7, "before decompression: %d bytes, after decompression: %ld", 
                                    in_size & ~0x8000, uncompressed_size );
  TRACE_EXIT;
  return (unsigned short)uncompressed_size;
}



/*****************************************************************************
 *
 * the remainder need not be changed to simply implement another compression
 * algorithm.
 *
 *****************************************************************************/


#if 0
void
qic80_decode_compression_header( byte * buffer, int segment )
{
TRACE_FUN( -1,"qic80_decode_compression_header");
  int first_cluster_offset; 
  int cluster_length;
  int file_set_offset;                                     
  int buffer_offset = 0;
  int this_seg_size;
  int len_left;
  int buf_offset;

  this_seg_size = ftape_get_seg_size( segment );
  TRACEi( -1, "Segment: ", segment );
  TRACEi( -1, "  size           : ", this_seg_size );
  first_cluster_offset = GET2( buffer, 0 );
  TRACEi( -1, "  first cluster  : ", first_cluster_offset );

  buf_offset = first_cluster_offset;
  if ( first_cluster_offset > 0 && this_seg_size > (18 + buf_offset) ) {
    file_set_offset = GET4( buffer, buf_offset );
    TRACEi( -1, "  file set offset: ", file_set_offset );
    buf_offset = first_cluster_offset + 4;
  } else {
    buf_offset = this_seg_size;
  }
  while( first_cluster_offset > 0 && (buf_offset + 18 < this_seg_size) ) {
    first_cluster_offset = GET2( buffer, buf_offset );
    if ( first_cluster_offset & 0x8000 ) {
      TRACE(-1,"compressed cluster");
      first_cluster_offset &= ~0x8000;
    }
    TRACEi( -1, "  cluster length : ", first_cluster_offset );
    buf_offset += first_cluster_offset + 2;

  }
  TRACE_EXIT;
}
#endif



/*
 * return-value: the number of bytes removed from the user-buffer
 *
 * out: int *write_cnt: how much actually has been moved to the deblock_buffer
 *      int *req_len  : MUST NOT BE CHANGED, except at EOT, in which case it
 *                      MUST BE SET TO 0.
 *
 * in : char *buff          : the user buffer
 *      int buf_pos_write   : copy of buf_pos_wr from "_ftape_write()"
 *      int this_segs_size  : the size in bytes of the actual segment
 *      char *deblock_buffer: the deblock_buffer
 *
 */                         

int
ftape_compress_write ( int *write_cnt, int *req_len, const char *buff, 
                       int buf_pos_write, int this_seg_size, 
                       char *deblock_buffer)
{
TRACE_FUN( 5, "ftape_compress_write");
int req_len_left = *req_len;
int prev_cnt; /* what remains in ftape_compression_buffer from previous call */
int result = 0;
int len_left;
int size;

  if (  ftape_data_pos >
       (ftape_total_data_amount - (ftape_block_size + 8))
     ) {
/*
 * compressed segment:
 *
 *   Offset 0   : first-block-offset (2 bytes)
 *   Offset 2   : remainder from previous segment (n bytes)
 *   Offset n+2 : file-set-offset of first cluster (4 bytes)
 *   Offset n+6 : length of cluster (2 bytes)
 *   Offset n+8 : cluster (at most ftape_block_size bytes)
 *
 * Thus a cluster consumes in the worst case ftape_block_size + 8 bytes 
 * provided that the compressed data takes at most the size of the 
 * uncompressed data (which is assured). Note that ftape_data_pos 
 * already covers the remainder from the previous segment
 *
 */
    TRACEx1( 7, "clipping req_len from %d to 0", *req_len );
    *req_len     =
    req_len_left = 0;
  }    

  if ( buf_pos_write == 0 ) { /* fill a new segment */
    /*
     * compressed_size holds the amount of compressed data still inside the 
     * ftape_compression_buffer
     */ 
    TRACEx1( 7, "Remaining in compression-buffer from previous block: %d",
               compressed_size );
    prev_cnt = this_seg_size - 2 - compressed_size;
    TRACEi( 7, "this_seg_size: ",this_seg_size );
    TRACEi( 7, "compressed_size: ", compressed_size );
    TRACEi( 7, "prev_cnt: ", prev_cnt );

    if ( prev_cnt > 18 ) {
      prev_cnt = compressed_size;
      if ( compressed_size != 0 && compos  != 0 ) {
        /* increment the write count, because we
         * are to finish the last block in this
         * segment
         */
        result = ftape_block_size;
      }
      TRACEx2( 4, "Adding segment %d to compression-map, "
                  "uncompressed data position: %d",
                   ftape_seg_pos, ftape_uncmpr_pos + result );
      /*
       *  compression-map holds the file-set offset of the first block 
       *  starting in this segment
       */
      ftape_compression_map[ ftape_seg_pos ] = ftape_uncmpr_pos + result;
      /*
       *  start of first cluster in this segment
       */
      PUT2( deblock_buffer, 0                  , 2 + prev_cnt );
      /*
       *  same as compression map entry, offset from start of volume
       */
      PUT4( deblock_buffer, 2 + prev_cnt, ftape_uncmpr_pos + result);
      /*
       *  increment position in deblock_buffer
       *  wrote   6 bytes  (header) + prev_cnt (data)
       *  bytes to deblock_buffer
       */
      buf_pos_write = 6 + prev_cnt;
    } else if ( prev_cnt >= 0 ) {
      /*
       *  old data fills the entire new segment, but entire old data amount
       *  fits into this segment.
       */                        
      TRACE( 7, "need entire segment for old data, finish old data ");
      /*
       *  erase unused part of segment.
       */
      memset( deblock_buffer + 2 + compressed_size, '\0',  prev_cnt );
      prev_cnt = compressed_size;
      if ( compressed_size != 0 && compos  != 0 ) {
        /* 
         * increment the write count, because we
         * are to finish the last block in this
         * segment
         */
        result += ftape_block_size;
      }
      /*
       *  first cluster offset >= segment-size - 18 means no first cluster
       *  in this segment
       * 
       */
      PUT2( deblock_buffer, 0, 2 + prev_cnt );        /* first cluster   */
      /*
       *  no new block starts in this segment
       */
      ftape_compression_map[ ftape_seg_pos ] = 0L;
      /*
       *  segment finished
       */
      buf_pos_write = this_seg_size;
    } else {
      /*
       *  old data occupies entire segment and continues in next segment
       */
      TRACE( 7, "need entire segment for old data, continue in next segment");
      prev_cnt = this_seg_size - 2;
      if ( prev_cnt < 0 ) {
        prev_cnt = 0;
      }
      /*
       *  indicate that we need entire segment and more space
       */
      PUT2( deblock_buffer, 0, 0 );        /* first cluster   */
      ftape_compression_map[ ftape_seg_pos ] = 0L;
      buf_pos_write = this_seg_size;
    }
    /*
     * copy remaining data from previous segment to deblock_buffer
     */
    memcpy( deblock_buffer + 2, ftape_compression_buffer + compos, prev_cnt );
    compressed_size -= prev_cnt; /* how much is left in the compr.-buffer */
    if ( compressed_size == 0 ) {
      if ( compos  != 0 ) {
        req_len_left -= ftape_block_size;
        /*
         * compos == compressed_size == 0 denotes an empty
         * ftape_compression_buffer
         */
        compos  = 0;
      }
    } else {
      compos  += prev_cnt; /* increment position in ftape_compression_buffer */
    }
    *write_cnt = buf_pos_write;
  } else {
    *write_cnt = 0;
  }

  len_left = this_seg_size - buf_pos_write;
  while ( (req_len_left > 0) && (len_left > 18) ) {
  /*
   * now we have some size left for a new compressed block. We know, that the
   * compression buffer is empty (else there wouldn't be any space left).
   *
   */

    /*
     *   copy to scratch_buffer which is the compression-input-buffer
     *   we've already called verify_write() from within the calling function
     *   _ftape_write()
     *
     *   buff is the user-buffer, result is how much we've already taken from
     *   the user-buffer.
     */
    memcpy_fromfs( scratch_buffer, buff + result, ftape_block_size);
    /*
     *  req_len_left holds the amount of data left in the user-buffer
     *  but we do not increment result, which is done when we have finished
     *  copying the compressed block to the deblock_buffer
     */
    req_len_left -= ftape_block_size;
    TRACEx2( 7,"Compressing block %d, data-position (uncompressed) %d bytes.",
               ftape_uncmpr_pos/ftape_block_size,ftape_uncmpr_pos);
    /*
     *  compress it. ftape_compression_buffer is the compression-output-buffer
     */
    compressed_size= ftape_compress( scratch_buffer,
                                     ftape_block_size    , 
                                     ftape_compression_buffer );
    /*
     *  write the length of the next cluster to the deblock buffer
     */
    PUT2( deblock_buffer, buf_pos_write, compressed_size );
    /*
     *  maybe data has not been compressed, but just copied
     */
    compressed_size &= ~0x8000;
    buf_pos_write += 2;
    *write_cnt    += 2;
    /*
     *  copy as much as fits in the deblock_buffer
     *
     *  compos  is the position inside the compression-buffer
     */                                      
    len_left -= 2;
    /*
     *  increment "result" iff we copied the entire compressed block to the
     *  deblock_buffer
     */
    if ( len_left >= compressed_size ) {
      size      = compressed_size;
      len_left -= compressed_size;
      compos          =
      compressed_size = 0;
      result += ftape_block_size;
    } else {
      size             =
      compos           = len_left;
      compressed_size -= len_left;
      len_left = 0;
    }
    memcpy( deblock_buffer + buf_pos_write, ftape_compression_buffer, size );
    *write_cnt      += size;
    buf_pos_write   += size;
  }
  /*
   *  increment global data offset in current volume
   */
  if ( result > 0 ) {
    ftape_uncmpr_pos    += result;
    TRACEi( 7, "ftape_uncmr_pos", ftape_uncmpr_pos );
  }
  /*   
   *   erase the remainder of the segment if less than 18 bytes left
   *   (18 bytes is due to the QIC-80 standard)
   */
  if ( len_left <= 18 ) {
    memset( deblock_buffer + buf_pos_write, '\0', len_left );
    (*write_cnt) += len_left;
  }
  TRACEx2( 7, "compressed_size: %d, compos : %d", compressed_size, compos );
  TRACE_EXIT;
  return result;
}   

/*
 * out:
 *
 * int *read_cnt: the number of bytes we removed from the deblock_buffer
 *                (result)
 * int *to_do   : the remaining size of the read-request.
 *
 * in:
 *
 * char *buff   : buff is the address of the upper part of the user buffer,
 *                that hasn't been filled with data yet.
 * int buf_pos_read:  copy of from _ftape_read()
 * int buf_len_read:  copy of buf_len_rd from _ftape_read()
 * char *deblock_buffer: deblock_buffer
 * unsigned short block_size: the block size valid for this volume, may differ
 *                            from ftape_block_size.
 *
 * returns the amount of data actually copied to the user-buffer
 *
 * *to_do MUST NOT SHRINK except to indicate an EOF. In this case *to_do has to
 * be set to 0
 *
 */
int
ftape_compress_read ( int *read_cnt, int *to_do, char *buff, 
                      int buf_pos_read, int buf_len_read, 
                      char *deblock_buffer, unsigned short block_size )
{          
TRACE_FUN( 5, "ftape_compress_read");
int byte_pos;
int uncompressed_size;         
int to_do_left = *to_do;
int result = 0;
unsigned first_block_offset;
static unsigned short stored_size; /* size with 15th bit set if compressed */

  if ( buf_pos_read == 0 ) {
    /*
     *  new segment just read
     */
    TRACEx1( 7, "carry from previous block: %d", compressed_size );
    /*
     *  get the byte offset of the start of the first compressed block
     */
    first_block_offset = GET2( deblock_buffer, 0 );
    if ( first_block_offset == 0 ) {
      /*
       *  block occupies entire segment and continues in next, so just copy the
       *  entire segment. Note: this is a convention from the QIC-80 standard:
       *  if the first two bytes of a segment are zero, then the entire segment
       *  is used up with continuing the last block of the previous segment.
       */                                                    
      memcpy( ftape_compression_buffer + compos,
              deblock_buffer + 2, buf_len_read - 2 );
      *read_cnt = buf_pos_read = buf_len_read; /* just skip remainder of segment */
      buf_len_read = 0;
      compos          += buf_len_read - 2;
      compressed_size -= buf_len_read - 2;
    } else if ( buf_len_read - first_block_offset > 18 ) {
    /*
     *  block ends in this segment, space left for new block
     *  Another convention from the QIC-80 standard: if the difference
     *  is less than 18 bytes, then no knew block is starts in this
     *  segment.
     */

      /*
       *  first check whether the remainder from the previous segment is 
       *  consistent with the stored values in this segment
       *
       *  hopefully the layout is:
       *      2 bytes:  offset of compression info of first new block in segment
       *                (variable: first_block_offset = n+2 )
       *      n bytes:  remainder of last block of last segment
       *                (variable compressed_size = n)
       *      4 bytes:  file set offset of first block starting in this segment
       *                (the offset relative to the uncompressed data)
       *      2 bytes:  size of the compressed block, the amount the compressed block
       *                occupies on the tape. 
       *
       */
      if ( first_block_offset - 2 != compressed_size ) {
        TRACEx2( 1, "Error: First compressed block should start at byte %d,"
                    " stored is %d.", compressed_size + 8, first_block_offset + 6 );
        *to_do    = *read_cnt = 0;
        TRACE_EXIT;
        return -EIO;
      }
      /*
       *   now get the offset the first block should have in the 
       *   the uncompressed data stream. This is just a 
       *   consistency check, the information is already stored
       *   in the compression map.
       */
      byte_pos = GET4( deblock_buffer, first_block_offset );
      if ( byte_pos != ftape_compression_map[ ftape_seg_pos ] ) {
        TRACEx3( 1, "Error: compression_map (%ld)"
                    " != "
                    "uncmpr. pos (%d) stored in segment %d",
                  ftape_compression_map[ ftape_seg_pos ], 
                  byte_pos, ftape_seg_pos );
        *to_do = *read_cnt = 0;
        TRACE_EXIT;
        return -EIO;
      }
      /* 
       *  blocks must start at block_size boundary, just another
       *  paranoia check
       */
      if ( (byte_pos - ftape_uncmpr_pos) % block_size != 0 ) {
        TRACEx2( 1, "Error: computed uncompressed pos. (%d)"
                    " != "
                    "recorded uncmpr. pos %d",ftape_uncmpr_pos, byte_pos );
        *to_do  = *read_cnt = 0;
        TRACE_EXIT;
        return -EIO;
      } 
      /*
       *  now everything is fine, fill the compression buffer with the
       *  remainder from the previous segment
       */
      memcpy( ftape_compression_buffer + compos ,
              deblock_buffer + 2, first_block_offset - 2 );
      *read_cnt = buf_pos_read  = first_block_offset + 4;
      buf_len_read -= buf_pos_read;
      compos       += compressed_size;
      compressed_size  = 0;
    } else {
    /*
     *   block occupies entire segment and ends in this segment
     *   because there are less than 18 bytes left in this segment
     *   after the end of this block
     */

      /*
       *  first check whether the remainder from the previous segment is 
       *  consistent with the stored values in this segment
       */
      if ( first_block_offset - 2 != compressed_size ) {
        TRACEx2(-1,"Error: Compressed block should end before byte %d,"
                  " stored is %d.", compressed_size + 8, first_block_offset + 6 );
        *to_do = *read_cnt = 0;
        TRACE_EXIT;
        return -EIO;
      }
      memcpy( ftape_compression_buffer + compos ,
              deblock_buffer + 2, first_block_offset - 2 );
      *read_cnt = buf_pos_read  = buf_len_read;
      compos          += compressed_size;
      compressed_size = buf_len_read = 0;
    }
  } else {
    *read_cnt = 0; /* inside same segment as at last time function was called */
  }
  
  /*
   *  loop and uncompress until user buffer full or deblock-buffer empty
   */
  do {
    TRACEx4( 7, "compressed_size: %d, compos : %d, buf_len_read: %d, buf_pos_read: %d",
                compressed_size, compos , buf_len_read, buf_pos_read);
    if ( compressed_size == 0 ) {
      /*
       *  compos == 0: compression_buffer empty, need to get more 
       *  data from deblock_buffer
       */
      if ( compos  != 0 ) { 
        /*
         * compression_buffer not empty
         * we got an compressed block in the ftape_compression_buffer
         * Note that we need to pass stored_size to ftape_uncompress()
         * because it still has bit 15 set if we have to handle uncompressed
         * (i.e. just copied) data. compressed_size has bit 15 stripped
         */
        uncompressed_size = ftape_uncompress( ftape_compression_buffer, 
                                              stored_size,
                                              scratch_buffer );
        if ( uncompressed_size != block_size ) {
          TRACEx2(-1,"Error: Uncompressed block (%d) != block-size (%d)",
                                   uncompressed_size, block_size);
          *read_cnt = *to_do = 0;
          TRACE_EXIT;
          return -EIO;
        }       
        /*
         * verify_read() was already called by ftape_read()
         */
        memcpy_tofs( buff + result, scratch_buffer, uncompressed_size );
        to_do_left -= uncompressed_size;
        result     += uncompressed_size;
        /*
         *  compos == 0 == compressed_size means an empty 
         *  ftape_compression_buffer and also leads to finishing this loop
         */
        compos  = 0;
      }                                              
      if ( (to_do_left > 0) && (buf_len_read > 18) ) {
      /*
       *  some space left in the user-buffer and in the deblock_buffer
       */
        /*
         *  get size of compressed block with high bit set if the data is
         *  stored uncompressed.
         */
        stored_size = GET2( deblock_buffer, buf_pos_read );
        /*
         *  strip uncompress flag
         */
        compressed_size = stored_size & ~0x8000;
        TRACEi( 7, "read compressed_size:", compressed_size );
        if ( compressed_size == 0 ) {
          /*
           *  This should be the end of a volume.
           *  But this shouldn't happen, as we store the size
           *  of a volume and check the amount of data requested
           *  before we call this function.
           */    
	  TRACE( 2, "Warning: unexpected end of volume" );
          buf_len_read =
          *to_do       =
          to_do_left   =
          compos       = 0;                        
        } else {
          buf_pos_read += 2;
          *read_cnt    += 2;
          buf_len_read -= 2;
          compos  = buf_len_read < compressed_size ?
                    buf_len_read : compressed_size;
          memcpy( ftape_compression_buffer , 
                  deblock_buffer + buf_pos_read, compos );
          compressed_size  -= compos ;
          *read_cnt        += compos ;
          buf_len_read     -= compos ;
          buf_pos_read     += compos ;
        }
      }
    } /* compressed_size != 0 */
  } while ( (to_do_left != 0) && (compos  != 0) && (compressed_size == 0) );
  /* 
   * skip unused bytes at end of segment
   */
  if ( buf_len_read <= 18 ) {
    (*read_cnt) += buf_len_read;
  }
  TRACEx2( 7, "compressed_size: %d, compos : %d", compressed_size, compos );
  ftape_uncmpr_pos += result;
  TRACEi( 7, "ftape_uncmpr_pos: ",ftape_uncmpr_pos );
  TRACE_EXIT;
  return result;
}                

/*
 *  seeks to the new data-position. Reads sometimes a segment.
 *  This function uses the compression map.
 *  
 *  start_seg and end_seg give the boundaries of the current volume
 *  block_size is the block_size of the current volume as stored in the
 *  volume label
 *
 */
int
ftape_compress_seek( int new_block_pos, 
                     unsigned start_seg, 
                     unsigned end_seg, 
                     unsigned block_size )
{ 
TRACE_FUN( 5, "ftape_compress_seek");
int result = 0;
int new_uncmpr_pos;
int stored_pos;
int biggest;              
int new_byte_pos;
int rec_len;
int new_seg_pos;
int this_segs_size;

  compressed_size = compos = 0;
  TRACEx1( 4, "desired new block positon: %d", new_block_pos );
  if ( new_block_pos <= 0 ) { /* handle it fast, nothing to do */
    ftape_seg_pos      = start_seg;
    ftape_seg_data_pos = 
    ftape_uncmpr_pos   = 0;
  } else {
    new_uncmpr_pos = new_block_pos * block_size;
    TRACEx1( 7, "desired new position: %d bytes", new_uncmpr_pos);
    /*
     *   get the biggest segment-id with compression map entry != 0
     */
    for ( biggest= end_seg;
          biggest > start_seg && ftape_compression_map[ biggest ] == 0;
          biggest -- );
    TRACEi( 7, "biggest:", biggest);
    /*
     *  now get the biggest segment id, that has an compression-map entry != 0
     *  which is >= the desired position
     */
    for ( new_seg_pos = start_seg;
             new_seg_pos < biggest
          && new_uncmpr_pos > ftape_compression_map[new_seg_pos];
          new_seg_pos++ );
    /* 
     *  We may have skipped some segments with zero compression map entry.
     *  Now go back until we find a segment with an compression map entry
     *  that is <= new_umcmpr_pos.
     */
    while ( (new_seg_pos > start_seg) && 
            (ftape_compression_map[ new_seg_pos ] == 0 ) ) {
      new_seg_pos --;
    } 
    TRACEx2( 7, "last entry in compression_map: %d/%ld",
                 biggest, ftape_compression_map[biggest]);
    TRACEx2( 7, "New segment:                   %d/%ld",
                 new_seg_pos, ftape_compression_map[new_seg_pos]);
    TRACEi( 7, "desired new position:", new_uncmpr_pos);
    /*
     * now biggest is the biggest entry != 0
     * and new_seg_pos is the segment, that new_uncmpr_pos belongs to
     */     
    this_segs_size = ftape_get_seg_size( new_seg_pos );
    /*
     *  read segment if necessary
     *  NOTE: ftape_deblock_segment stores the segment id of the segment
     *        that is held in the deblock_buffer, if any. Else 
     *        ftape_deblock_segment is set to -1
     */
    if ( new_seg_pos != ftape_deblock_segment ) {
      if ( read_segment( new_seg_pos, deblock_buffer, 0) != this_segs_size ) {
        TRACEx2( 1,"Error: Can't seek to byte %d, segment %d is unreachable.",
                    new_uncmpr_pos, new_seg_pos );
        ftape_deblock_segment = -1;
        TRACE_EXIT;
        return -EIO;
      }
      ftape_deblock_segment = new_seg_pos;
    }
    TRACEi( 7, "new_seg_pos:", new_seg_pos);
    /* 
     *  Get the offset of the first compression header in segment
     */
    new_byte_pos = GET2( deblock_buffer, 0 );
    /*
     *  get the data offset (counted from the beginning of the volume)
     *  that is stored in the segment (not in the map)
     */
    stored_pos = GET4( deblock_buffer, new_byte_pos );
    TRACEi(4,"store_pos:", stored_pos);                    
    /*
     *   check it against offset in map
     */
    if ( stored_pos != ftape_compression_map[new_seg_pos] ) {
      TRACEx1( 1, "Error: inconsitent compression-map at segment %d.", new_seg_pos);
      TRACE_EXIT;
      return -EIO;
    } 
    /*
     *  get offset of first compressed block in segment
     */                                                
    new_byte_pos += 4;
    /*
     *   new_byte_pos now points to two bytes where the length of the first 
     *   cluster of data is stored.
     */
    TRACEi( 7, "new_byte_pos:", new_byte_pos);                               
    /*
     *  loop while current position is valid, i.e. inside current segment
     *  and
     *  position stored in segment < desired position
     */
    while ( new_byte_pos < this_segs_size && stored_pos < new_uncmpr_pos ) {
      /*
       *  get length of compressed cluster
       */
      rec_len = GET2(deblock_buffer, new_byte_pos );
      TRACEi( 7, "rec_len:", rec_len);
      if ( rec_len == 0 ) { /* end of recorded media */
        new_uncmpr_pos = stored_pos; /* where we are */
        TRACEi( 7, "Positioned at byte ", stored_pos);
        if ( new_seg_pos == ftape_last_data_segment ) {
          result = -ENOSPC; /* after end of tape */
        } else {
          result = -EIO; /* after end of volume */
        }
      } else {
        new_byte_pos += rec_len + 2; /* actual position in segment */
        TRACEi( 7, "new_byte_pos:",new_byte_pos);
        stored_pos += block_size; /* uncompressed position */
        TRACEi( 7, "store_pos:", stored_pos);
      }
    }
    if ( new_byte_pos >= this_segs_size && stored_pos != new_uncmpr_pos ) {
    /*
     *  should only happen, when end of compressed block is
     *  incidentally aligned at segment-boundary
     *
     *  But then we should have started in the next segment,
     *  and not in this segment. So this MUST BE EOF, the end of a volume.
     */
      TRACEi( 7, "Positioned at byte ", stored_pos);
      new_uncmpr_pos = stored_pos;
      new_byte_pos   = 0;
      new_seg_pos ++;
      result = -EIO;
    }
    ftape_seg_pos      = new_seg_pos;
    ftape_seg_data_pos = new_byte_pos;
    ftape_uncmpr_pos   = new_uncmpr_pos;
    TRACEi( 4, "ftape_uncmpr_pos:", ftape_uncmpr_pos );
    TRACEi( 4, "ftape_seg_pos:", ftape_seg_pos);
    TRACEi( 4, "ftape_seg_data_pos:", ftape_seg_data_pos);
  }
  ftape_data_pos = ftape_calc_data_pos(ftape_seg_pos, ftape_seg_data_pos);
  TRACEi( 4, "ftape_data_pos:", ftape_data_pos );
  TRACE_EXIT;
  return result;
}            


/*
 *  extract the compression map from the compression-map segments, starting
 *  at "location". Support for 1100 ft. tapes is added, but untested, as I 
 *  don't have one.
 * 
 */
int
ftape_extract_compression_map( byte *address, unsigned location )
{
  TRACE_FUN( 5, "extract_compression_map");
  int remaining;
  int result= 0;

  /*
   *  initialize variables in case we can't extract the compression map.
   *  first_data_segment holds the volume-table
   */
  ftape_compression_map_length= 0;     
  ftape_first_user_segment = first_data_segment + 1;
  if( location > 0 ) {
    /*
     *  location == 0 means no compression map.
     */
    TRACEi( 3, "compression_map_location: ", location );
    /*
     *  we read ahead in case the DCMS fills more than one segment
     */
    result = read_segment( location, address, 1);
    if( result < 0 ) {
      TRACEi( 1, "Error: Couldn't read compression map segment at", location );
      TRACE_EXIT;
      return -EIO ;
    }
    if( *((long *)"DCMS") != *((long *)address) ) {
      TRACEi( 1, "Error: Wrong signature of compression map segment at", location );
      TRACE_EXIT;
      return -EIO;
    }
    remaining = ftape_compression_map_length = GET2( address, 4);
    ftape_compression_map_length /= 4;
#ifndef NO_TRACE_AT_ALL
/*
 *  just a little more noise
 */
    switch( ftape_compression_map_length )
    {
      case MAP_205_FEET  : TRACEx1( 4, "Found compression map segment at %d for 205 feet tape", location );
                           break;
      case MAP_307_5_FEET: TRACEx1( 4, "Found compression map segment at %d for 307,5 feet tape", location );
                           break;
      case MAP_770_FEET: TRACEx1( 4, "Found compression map segment at %d for 307,5 feet tape", location );
                           break;
      case MAP_1100_FEET : TRACEx1( 4, "Found compression map segment at %d for 1100 feet tape.", location );
                           break;
      default   : TRACEx1( 4, "Got compression map with %d entries, tape is supposed to have that many segments", ftape_compression_map_length );
                  break;
    } 
#endif
    /*
     *  copy data from buffer to compression map
     *  read in segments until we got the entire compression map
     */
    do {
      if ( result > remaining ) {
        result = remaining;
      }
      memcpy( ftape_compression_map, address, result );
      remaining -= result;
      if ( remaining > 0 ) {
        result = read_segment( ++location, address, 1);
        if( result < 0 ) {
          TRACEi( 1, "Error: Couldn't read compression map segment at", location );
          TRACE_EXIT;
          return -EIO ;
        }
      }
    } while ( remaining > 0 );
    /*
     *  first segment after compression map segments
     */
    ftape_first_user_segment = location + 1;
  }
  TRACE_EXIT;
  return 0;
}

/*
 *  write one or more compression map segments starting at "segment"
 *  using buffer. Adjusts ftape_first_user_segment
 *
 */

int
ftape_update_compression_segments( unsigned segment, byte* buffer )
{
  TRACE_FUN( 5, "update_compression_segments");
  int result = 0;
  int remaining;
  int written;
  int rel_seg = 0;

  if(segment == 0 ||                              
    segment == header_segment_1 || segment == header_segment_2 ) {
    result = -EINVAL;
  } else {
    TRACEx1( 4, "Creating or updating compression-map-segment at %d", segment);
    /*
     * ftape_compression_map_length holds the number of segments on tape,
     * that is tracks_per_tape * segments_per_track
     */
    remaining = ftape_compression_map_length*sizeof(unsigned long) - DEBLOCK_SIZE;
    if ( remaining >= 0 ) {
      memcpy( buffer, ftape_compression_map, DEBLOCK_SIZE );
    } else {
      memcpy( buffer, ftape_compression_map,
              sizeof(unsigned long)*ftape_compression_map_length );
      (void)memset(buffer+sizeof(unsigned long)*ftape_compression_map_length,
                   '0', -remaining);
      remaining = 0;
    }
    PUT4( buffer, 0, *((long *)"DCMS"));
    PUT2( buffer, 4, sizeof(unsigned long)*ftape_compression_map_length );
    /*
     *  now write one or more compression map segments
     *
     */
    result = ftape_verify_write_segment( segment, buffer );
    written = result;
    while ( result >= 0 && written < (ftape_compression_map_length*4) ) {
      if ( result < DEBLOCK_SIZE ) {
        memmove( buffer, buffer + result, DEBLOCK_SIZE - result );
      }
      if ( remaining < result ) {
        memcpy( 
          buffer + DEBLOCK_SIZE - result,
            ftape_compression_map 
          + ftape_compression_map_length 
          - (remaining/4), /* NOTE: pointer arithmetic */
          remaining );
        memset( buffer + DEBLOCK_SIZE - result + remaining,
                '\0',
                result - remaining );
        remaining = 0;
      } else {
        memcpy( 
          buffer + DEBLOCK_SIZE - result,
          ftape_compression_map + ftape_compression_map_length - (remaining/4),
          result );
        remaining -= result;
      }
      result = ftape_verify_write_segment( ++rel_seg + segment, buffer );
      written += result;
    }
    /*
     *   this is the first segment usable for data:
     */
    ftape_first_user_segment = segment + rel_seg + 1;
  }
  TRACE_EXIT;
  return result;
}


