/*
 * 	NET3	Protocol independent device support routines.
 *
 *		This program is free software; you can redistribute it and/or
 *		modify it under the terms of the GNU General Public License
 *		as published by the Free Software Foundation; either version
 *		2 of the License, or (at your option) any later version.
 *
 *	Derived from the non IP parts of dev.c 1.0.19
 * 		Authors:	Ross Biro, <bir7@leland.Stanford.Edu>
 *				Fred N. van Kempen, <waltje@uWalt.NL.Mugnet.ORG>
 *				Mark Evans, <evansmp@uhura.aston.ac.uk>
 *
 *	Additional Authors:
 *		Florian la Roche <rzsfl@rz.uni-sb.de>
 *		Alan Cox <gw4pts@gw4pts.ampr.org>
 *		David Hinds <dhinds@allegro.stanford.edu>
 *
 *	Changes:
 *		Alan Cox	:	device private ioctl copies fields back.
 *		Alan Cox	:	Transmit queue code does relevant stunts to
 *					keep the queue safe.
 *		Adam Caldwell	:	Loopback output packets as ETH_P_ALL.
 *		Alan Cox	:	Promiscuous mode NULL pointer trap.
 *
 *	Cleaned up and recommented by Alan Cox 2nd April 1994. I hope to have
 *	the rest as well commented in the end.
 */

/*
 *	A lot of these includes will be going walkies very soon 
 */
 
#include <asm/segment.h>
#include <asm/system.h>
#include <asm/bitops.h>
#include <linux/config.h>
#include <linux/types.h>
#include <linux/kernel.h>
#include <linux/sched.h>
#include <linux/string.h>
#include <linux/mm.h>
#include <linux/socket.h>
#include <linux/sockios.h>
#include <linux/in.h>
#include <linux/errno.h>
#include <linux/interrupt.h>
#include <linux/if_ether.h>
#include <linux/inet.h>
#include <linux/skbuff.h>
#include <linux/netprotocol.h>
#include <linux/netdevice.h>
#include <linux/etherdevice.h>
#include "ip.h"
#include "route.h"
#include "sock.h"
#include "arp.h"

#ifdef CONFIG_NIT
#include <linux/if_nit.h>
extern struct protocol transmit_protocol;
#endif


/*
 *	Return the lesser of the two values. 
 */
 
static __inline__ unsigned long min(unsigned long a, unsigned long b)
{
  return (a < b)? a : b;
}



/*****************************************************************************************

			    Device Interface Subroutines

******************************************************************************************/

/* 
 *	Find an interface by name.
 */
 
struct device *dev_get(char *name)
{
	struct device *dev;

	for (dev = dev_base; dev != NULL; dev = dev->next) 
	{
		if (strcmp(dev->name, name) == 0)
			return(dev);
	}
	return(NULL);
}


/*
 *	Prepare an interface for use. 
 */
 
int dev_open(struct device *dev)
{
	int ret = 0;

	/*
	 *	Call device private open method
	 */
	if (dev->open) 
  		ret = dev->open(dev);
  		
  	/*
	 *	If it went open OK then set the flags
	 */
	 
	if (ret != 0) 
		return ret;
		
	dev->flags |= (IFF_UP | IFF_RUNNING);
  
  	/*
  	 *	Build the protocol stack. If this is NULL the device built its own for us.
  	 */
  	 
  	if(dev->mac_protocol==NULL)
  	{
  		dev->mac_protocol=protocol_clone_keep(dev->default_protocol);
  		dev->base_protocol=dev->mac_protocol;
  	}
  	if(dev->mac_protocol==NULL)
  	{
		dev_close(dev);
		return -ENOMEM;
	}
	dev->mac_protocol->device=dev;
	
	return(ret);
}


/*
 *	Completely shutdown an interface.
 */
 
int dev_close(struct device *dev)
{
	/*
	 *	Only close a device if it is up.
	 */
	 
	if (dev->flags != 0) 
	{
  		int ct=0;
		dev->flags = 0;
		/*
		 *	Call the device specific close. This cannot fail.
		 */
		if (dev->stop) 
			dev->stop(dev);
		/*
		 *	Delete the route to the device.
		 */
#ifdef CONFIG_INET		 
		ip_rt_flush(dev);
#ifdef CONFIG_IP_MULTICAST		
		ip_mc_dropdevice(dev);
#endif		
		arp_device_down(dev);
#endif		
#ifdef CONFIG_IPX
		ipxrtr_device_down(dev);
#endif	
		/*
		 *	Blank the IP addresses
		 */
		dev->pa_addr = 0;
		dev->pa_dstaddr = 0;
		dev->pa_brdaddr = 0;
		dev->pa_mask = 0;
		/*
		 *	Purge any queued packets when we down the link 
		 */
		while(ct<DEV_NUMBUFFS)
		{
			sk_buff *skb;
			while((skb=skb_dequeue(&dev->buffs[ct]))!=NULL)
				if(skb->free)
					kfree_skb(skb,FREE_WRITE);
			ct++;
		}
		
		if(dev->base_protocol!=NULL)
		{
			dev->base_protocol->device=NULL;
			protocol_delete(dev->base_protocol);
			dev->base_protocol=NULL;
			dev->mac_protocol=NULL;
		}
	}
	return(0);
}


/*
 *	Send (or queue for sending) a packet. 
 *
 *	IMPORTANT: When this is called to resend frames. The caller MUST
 *	already have locked the sk_buff. Apart from that we do the
 *	rest of the magic.
 */

void dev_queue_xmit(sk_buff *skb, struct device *dev, int pri)
{
	unsigned long flags;
	int where = 0;		/* used to say if the packet should go	*/
				/* at the front or the back of the	*/
				/* queue - front is a retranmsit try	*/
	if (dev == NULL) 
	{
		printk("dev.c: dev_queue_xmit: dev = NULL\n");
		return;
	}
	
	if(pri>=0 && !skb_device_locked(skb))
		skb_device_lock(skb);	/* Shove a lock on the frame */
#ifdef CONFIG_SLAVE_BALANCING
	save_flags(flags);
	cli();
	if(dev->slave!=NULL && dev->slave->pkt_queue < dev->pkt_queue &&
				(dev->slave->flags & IFF_UP))
		dev=dev->slave;
	restore_flags(flags);
#endif		
 
	IS_SKB(skb);
    
	skb->dev = dev;

	/*
	 *	This just eliminates some race conditions, but not all... 
	 */

	if (skb->next != NULL) 
	{
		/*
		 *	Make sure we haven't missed an interrupt. 
		 */
		printk("dev_queue_xmit: worked around a missed interrupt\n");
		dev->hard_start_xmit(NULL, dev);
		return;
  	}

	/*
	 *	Negative priority is used to flag a frame that is being pulled from the
	 *	queue front as a retransmit attempt. It therefore goes back on the queue
	 *	start on a failure.
	 */
	 
  	if (pri < 0) 
  	{
		pri = -pri-1;
		where = 1;
  	}

	if (pri >= DEV_NUMBUFFS) 
	{
		printk("bad priority in dev_queue_xmit.\n");
		pri = 1;
	}

	save_flags(flags);
	cli();	
	if (!where) {
#ifdef CONFIG_SLAVE_BALANCING	
		skb->in_dev_queue=1;
#endif		
		skb_queue_tail(dev->buffs + pri,skb);
		skb_device_unlock(skb);		/* Buffer is on the device queue and can be freed safely */
		skb = skb_dequeue(dev->buffs + pri);
		skb_device_lock(skb);		/* New buffer needs locking down */
#ifdef CONFIG_SLAVE_BALANCING		
		skb->in_dev_queue=0;
#endif		
	}
	restore_flags(flags);

/*
 *	Queue the packet on all our NIT inputs.
 */

#ifdef CONFIG_NIT
	
	if(!where && tx_nit_count)
	{
		sk_buff *tmp=skb_clone(skb, GFP_ATOMIC);
		if(tmp)
			transmit_protocol.input(&transmit_protocol, NULL ,tmp,NULL,dev->name);
	}
	
#endif
	
	if (dev->hard_start_xmit(skb, dev) == 0) {
		/*
		 *	Packet is now solely the responsibility of the driver
		 */
		return;
	}

	/*
	 *	Transmission failed, put skb back into a list. Once on the list its safe and
	 *	no longer device locked (it can be freed safely from the device queue)
	 */

	cli();
#ifdef CONFIG_SLAVE_BALANCING
	skb->in_dev_queue=1;
	dev->pkt_queue++;
#endif		
	skb_device_unlock(skb);
	skb_queue_head(dev->buffs + pri,skb);
	restore_flags(flags);
}

/*
 *	Receive a packet from a device driver and queue it for the upper
 *	(protocol) levels.  It always succeeds. This is the recommended 
 *	interface to use.
 */

void netif_rx(sk_buff *skb)
{
	/*
	 *	Any received buffers are un-owned and should be discarded
	 *	when freed. These will be updated later as the frames get
	 *	owners.
	 */
	skb->sk = NULL;
	skb->free = 1;
	if(skb->stamp.tv_sec==0)
		skb->stamp = xtime;
	
	/*
	 *	Base protocol is the protocol at the bottom of the device stack. It may not be the
	 *	one offered for binding.
	 */	
	 
	skb->protocol=skb->dev->base_protocol;

#ifdef CONFIG_NIT
	
	if(skb->protnum==0)		/* Ethernet */
	{
		skb->protnum=skb_data(skb)[12]<<8|skb_data(skb)[13];
		if(skb->protnum<1500)
			skb->protnum=htons(ETH_P_802_3);
	}
	
	if(rx_nit_count)
	{
		sk_buff *tmp=skb_clone(skb, GFP_ATOMIC);
		if(tmp)
			receive_protocol.input(&receive_protocol,NULL, tmp,NULL,skb->dev->name);
	}
	if(pk_nit_count)
	{
		sk_buff *tmp=skb_clone(skb, GFP_ATOMIC);
		if(tmp)
		{
			if(!protocol_pass_demultiplex(&packet_protocol, &skb->protnum, skb, NULL, skb->dev->name))
				kfree_skb(tmp, FREE_READ);
		}
	}
#endif
	
	skb->protocol->input(skb->protocol,NULL,skb, NULL, NULL);
}
	

/*
 *	The old interface to fetch a packet from a device driver.
 *	This function is the base level entry point for all drivers that
 *	want to send a packet to the upper (protocol) levels.  It takes
 *	care of de-multiplexing the packet to the various modules based
 *	on their protocol ID.
 *
 *	Return values:	1 <- exit I can't do any more
 *			0 <- feed me more (i.e. "done", "OK"). 
 *
 *	This function is OBSOLETE and should not be used by any new
 *	device.
 */

int dev_rint(unsigned char *buff, long len, int flags, struct device *dev, int proto)
{
	sk_buff *skb = NULL;
	unsigned char *to;
	int amount, left;
	int len2;

	if (dev == NULL || buff == NULL || len <= 0) 
		return(1);

	if (flags & IN_SKBUFF) 
	{
		skb = (sk_buff *) buff;
	}
	else
	{
		skb = dev_alloc_skb(len);
		if (skb == NULL) 
		{
			printk("dev_rint: packet dropped on %s (no memory) !\n",
			       dev->name);
			return(1);
		}
		
		skb->protnum=htons(proto);

		/* 
		 *	First we copy the packet into a buffer, and save it for later. We
		 *	in effect handle the incoming data as if it were from a circular buffer
		 */

		to = skb_put(skb,len);
		if(to==NULL)
		{
			printk("dev_rint: brown and sticky time!\n");
			return 1;
		}
		left = len;

		len2 = len;
		while (len2 > 0) 
		{
			amount = min(len2, (unsigned long) dev->rmem_end -
						(unsigned long) buff);
			memcpy(to, buff, amount);
			len2 -= amount;
			left -= amount;
			buff += amount;
			to += amount;
			if ((unsigned long) buff == dev->rmem_end)
				buff = (unsigned char *) dev->rmem_start;
		}
	}

	/*
	 *	Tag the frame and kick it to the proper receive routine
	 */
	 
	skb->dev = dev;
	skb->free = 1;

	netif_rx(skb);
	/*
	 *	OK, all done. 
	 */
	return(0);
}


/*
 *	This routine causes all interfaces to try to send some data. 
 */
 
void dev_transmit(void)
{
	struct device *dev;

	for (dev = dev_base; dev != NULL; dev = dev->next) 
	{
		if (dev->flags != 0 && !dev->tbusy) {
			/*
			 *	Kick the device
			 */
			dev_tint(dev);
		}
	}
}



/*
 *	This routine is called when an device driver (i.e. an
 *	interface) is ready to transmit a packet.
 */
 
void dev_tint(struct device *dev)
{
	int i;
	sk_buff *skb;
	unsigned long flags;
	
	save_flags(flags);	
	/*
	 *	Work the queues in priority order
	 */
	 
	for(i = 0;i < DEV_NUMBUFFS; i++) 
	{
		/*
		 *	Pull packets from the queue
		 */
		 

		cli();
		while((skb=skb_dequeue(&dev->buffs[i]))!=NULL)
		{
			/*
			 *	Stop anyone freeing the buffer while we retransmit it
			 */
			skb_device_lock(skb);
			restore_flags(flags);
			/*
			 *	Feed them to the output stage and if it fails
			 *	indicate they re-queue at the front.
			 */
			dev_queue_xmit(skb,dev,-i - 1);
			/*
			 *	If we can take no more then stop here.
			 */
			if (dev->tbusy)
				return;
			cli();
		}
	}
	restore_flags(flags);
}


/*
 *	Perform a SIOCGIFCONF call. This structure will change
 *	size shortly, and there is nothing I can do about it.
 *	Thus we will need a 'compatibility mode'.
 */

static int dev_ifconf(char *arg)
{
	struct ifconf ifc;
	struct ifreq ifr;
	struct device *dev;
	char *pos;
	int len;
	int err;

	/*
	 *	Fetch the caller's info block. 
	 */
	 
	err=verify_area(VERIFY_WRITE, arg, sizeof(struct ifconf));
	if(err)
	  	return err;
	memcpy_fromfs(&ifc, arg, sizeof(struct ifconf));
	len = ifc.ifc_len;
	pos = ifc.ifc_buf;

	/*
	 *	We now walk the device list filling each active device
	 *	into the array.
	 */
	 
	err=verify_area(VERIFY_WRITE,pos,len);
	if(err)
	  	return err;
  	
	/*
	 *	Loop over the interfaces, and write an info block for each. 
	 */

	for (dev = dev_base; dev != NULL; dev = dev->next) 
	{
        	if(!(dev->flags & IFF_UP))	/* Downed devices don't count */
	        	continue;
		memset(&ifr, 0, sizeof(struct ifreq));
		strcpy(ifr.ifr_name, dev->name);
		(*(struct sockaddr_in *) &ifr.ifr_addr).sin_family = dev->family;
		(*(struct sockaddr_in *) &ifr.ifr_addr).sin_addr.s_addr = dev->pa_addr;

		/*
		 *	Write this block to the caller's space. 
		 */
		 
		memcpy_tofs(pos, &ifr, sizeof(struct ifreq));
		pos += sizeof(struct ifreq);
		len -= sizeof(struct ifreq);
		
		/*
		 *	Have we run out of space here ?
		 */
	
		if (len < sizeof(struct ifreq)) 
			break;
  	}

	/*
	 *	All done.  Write the updated control block back to the caller. 
	 */
	 
	ifc.ifc_len = (pos - ifc.ifc_buf);
	ifc.ifc_req = (struct ifreq *) ifc.ifc_buf;
	memcpy_tofs(arg, &ifc, sizeof(struct ifconf));
	
	/*
	 *	Report how much was filled in
	 */
	 
	return(pos - arg);
}


/*
 *	This is invoked by the /proc filesystem handler to display a device
 *	in detail.
 */

static int sprintf_stats(char *buffer, struct device *dev)
{
	struct enet_statistics *stats = (dev->get_stats ? dev->get_stats(dev): NULL);
	int size;
	
	if (stats)
		size = sprintf(buffer, "%6s:%7d %4d %4d %4d %4d %8d %4d %4d %4d %5d %4d\n",
		   dev->name,
		   stats->rx_packets, stats->rx_errors,
		   stats->rx_dropped + stats->rx_missed_errors,
		   stats->rx_fifo_errors,
		   stats->rx_length_errors + stats->rx_over_errors
		   + stats->rx_crc_errors + stats->rx_frame_errors,
		   stats->tx_packets, stats->tx_errors, stats->tx_dropped,
		   stats->tx_fifo_errors, stats->collisions,
		   stats->tx_carrier_errors + stats->tx_aborted_errors
		   + stats->tx_window_errors + stats->tx_heartbeat_errors);
	else
		size = sprintf(buffer, "%6s: No statistics available.\n", dev->name);

	return size;
}

/*
 *	Called from the PROCfs module. This now uses the new arbitrary sized /proc/net interface
 *	to create /proc/net/dev
 */
 
int dev_get_info(char *buffer, char **start, off_t offset, int length)
{
	int len=0;
	off_t begin=0;
	off_t pos=0;
	int size;
	
	struct device *dev;


	size = sprintf(buffer, "Inter-|   Receive                  |  Transmit\n"
			    " face |packets errs drop fifo frame|packets errs drop fifo colls carrier\n");
	
	pos+=size;
	len+=size;
	

	for (dev = dev_base; dev != NULL; dev = dev->next) 
	{
		size = sprintf_stats(buffer+len, dev);
		len+=size;
		pos=begin+len;
				
		if(pos<offset)
		{
			len=0;
			begin=pos;
		}
		if(pos>offset+length)
			break;
	}
	
	*start=buffer+(offset-begin);	/* Start of wanted data */
	len-=(offset-begin);		/* Start slop */
	if(len>length)
		len=length;		/* Ending slop */
	return len;
}


/*
 *	This checks bitmasks for the ioctl calls for devices.
 */
 
static inline int bad_mask(unsigned long mask, unsigned long addr)
{
	if (addr & (mask = ~mask))
		return 1;
	mask = ntohl(mask);
	if (mask & (mask+1))
		return 1;
	return 0;
}

/*
 *	Perform the SIOCxIFxxx calls. 
 *
 *	The socket layer has seen an ioctl the address family thinks is
 *	for the device. At this point we get invoked to make a decision
 */
 
static int dev_ifsioc(void *arg, unsigned int getset)
{
	struct ifreq ifr;
	struct device *dev;
	int ret = 0;

	/*
	 *	Fetch the caller's info block into kernel space
	 */

	int err=verify_area(VERIFY_WRITE, arg, sizeof(struct ifreq));
	if(err)
		return err;
	
	memcpy_fromfs(&ifr, arg, sizeof(struct ifreq));

	/*
	 *	See which interface the caller is talking about. 
	 */
	 
	if ((dev = dev_get(ifr.ifr_name)) == NULL) 
		return(-ENODEV);

	switch(getset) 
	{
		case SIOCGIFFLAGS:	/* Get interface flags */
			ifr.ifr_flags = dev->flags;
			break;
		case SIOCSIFFLAGS:	/* Set interface flags */
			{
				int old_flags = dev->flags;
#ifdef CONFIG_SLAVE_BALANCING				
				if(dev->flags&IFF_SLAVE)
					return -EBUSY;
#endif					
				dev->flags = ifr.ifr_flags & (
					IFF_UP | IFF_BROADCAST | IFF_DEBUG | IFF_LOOPBACK |
					IFF_POINTOPOINT | IFF_NOTRAILERS | IFF_RUNNING |
					IFF_NOARP | IFF_PROMISC | IFF_ALLMULTI | IFF_SLAVE | IFF_MASTER
#ifdef CONFIG_IP_MULTICAST					
					| IFF_MULTICAST | IFF_FILTMULTI
#endif					
					);
#ifdef CONFIG_SLAVE_BALANCING				
				if(!(dev->flags&IFF_MASTER) && dev->slave)
				{
					dev->slave->flags&=~IFF_SLAVE;
					dev->slave=NULL;
				}
#endif				
				
				if (dev->set_multicast_list != NULL)
				{
					/*
					 *	Has promiscuous mode been turned off
					 */	
					if ( (old_flags & IFF_PROMISC) && ((dev->flags & IFF_PROMISC) == 0))
			 			dev->set_multicast_list(dev,0,NULL);
	
			 		/*
			 		 *	Has it been turned on
			 		 */
	
					if ( (dev->flags & IFF_PROMISC) && ((old_flags & IFF_PROMISC) == 0))
				  		dev->set_multicast_list(dev,-1,NULL);
				}
							  		
			  	/*
			  	 *	Have we downed the interface
			  	 */
		
				if ((old_flags & IFF_UP) && ((dev->flags & IFF_UP) == 0)) 
				{
					ret = dev_close(dev);
				}
				else
				{
					/*
					 *	Have we upped the interface 
					 */
					 
			      		ret = (! (old_flags & IFF_UP) && (dev->flags & IFF_UP))
						? dev_open(dev) : 0;
					/* 
					 *	Check the flags.
					 */
					if(ret<0)
						dev->flags&=~IFF_UP;	/* Didn't open so down the if */
			  	}
	        	}
			return ret;
		
		case SIOCGIFADDR:	/* Get interface address (and family) */
			(*(struct sockaddr_in *)
				  &ifr.ifr_addr).sin_addr.s_addr = dev->pa_addr;
			(*(struct sockaddr_in *)
				  &ifr.ifr_addr).sin_family = dev->family;
			(*(struct sockaddr_in *)
				  &ifr.ifr_addr).sin_port = 0;
			break;
	
		case SIOCSIFADDR:	/* Set interface address (and family) */
			dev->pa_addr = (*(struct sockaddr_in *)
				 &ifr.ifr_addr).sin_addr.s_addr;
			dev->family = ifr.ifr_addr.sa_family;
#ifdef CONFIG_INET	
			/* This is naughty. When net-tools-1.3 comes out It wants moving into the net032
			   code not the kernel. Till then it can sit here (SIGH) */		
			dev->pa_mask = ip_get_mask(dev->pa_addr);
#endif			
			dev->pa_brdaddr = dev->pa_addr | ~dev->pa_mask;
			return 0;
			
		case SIOCGIFBRDADDR:	/* Get the broadcast address */
			(*(struct sockaddr_in *)
				&ifr.ifr_broadaddr).sin_addr.s_addr = dev->pa_brdaddr;
			(*(struct sockaddr_in *)
				&ifr.ifr_broadaddr).sin_family = dev->family;
			(*(struct sockaddr_in *)
				&ifr.ifr_broadaddr).sin_port = 0;
			break;

		case SIOCSIFBRDADDR:	/* Set the broadcast address */
			dev->pa_brdaddr = (*(struct sockaddr_in *)
				&ifr.ifr_broadaddr).sin_addr.s_addr;
			return 0;
			
		case SIOCGIFDSTADDR:	/* Get the destination address (for point-to-point links) */
			(*(struct sockaddr_in *)
				&ifr.ifr_dstaddr).sin_addr.s_addr = dev->pa_dstaddr;
			(*(struct sockaddr_in *)
				&ifr.ifr_broadaddr).sin_family = dev->family;
			(*(struct sockaddr_in *)
				&ifr.ifr_broadaddr).sin_port = 0;
			break;
	
		case SIOCSIFDSTADDR:	/* Set the destination address (for point-to-point links) */
			dev->pa_dstaddr = (*(struct sockaddr_in *)
				&ifr.ifr_dstaddr).sin_addr.s_addr;
			return 0;
			
		case SIOCGIFNETMASK:	/* Get the netmask for the interface */
			(*(struct sockaddr_in *)
				&ifr.ifr_netmask).sin_addr.s_addr = dev->pa_mask;
			(*(struct sockaddr_in *)
				&ifr.ifr_netmask).sin_family = dev->family;
			(*(struct sockaddr_in *)
				&ifr.ifr_netmask).sin_port = 0;
			break;

		case SIOCSIFNETMASK: 	/* Set the netmask for the interface */
			{
				unsigned long mask = (*(struct sockaddr_in *)
					&ifr.ifr_netmask).sin_addr.s_addr;
				/*
				 *	The mask we set must be legal.
				 */
				if (bad_mask(mask,0))
					return -EINVAL;
				dev->pa_mask = mask;
				return 0;
			}
			
		case SIOCGIFMETRIC:	/* Get the metric on the interface (currently unused) */
			
			ifr.ifr_metric = dev->metric;
			break;
			
		case SIOCSIFMETRIC:	/* Set the metric on the interface (currently unused) */
			dev->metric = ifr.ifr_metric;
			return 0;
	
		case SIOCGIFMTU:	/* Get the MTU of a device */
			ifr.ifr_mtu = dev->mtu;
			break;
	
		case SIOCSIFMTU:	/* Set the MTU of a device */
		
			/*
			 *	MTU must be positive
			 */
			 
			if(ifr.ifr_mtu<1)
				return -EINVAL;
			dev->mtu = ifr.ifr_mtu;
			return 0;
	
		case SIOCGIFMEM:	/* Get the per device memory space. We can add this but currently
					   do not support it */
			ret = -EINVAL;
			break;
		
		case SIOCSIFMEM:	/* Set the per device memory buffer space. Not applicable in our case */
			printk("NET: ioctl(SIOCSIFMEM, 0x%08X)\n", (int)arg);
			return -EINVAL;

		case OLD_SIOCGIFHWADDR:	/* Get the hardware address. This will change and SIFHWADDR will be added */
			printk("Obsolete call to SIOCGIFHWADDR\n");
			memcpy(ifr.old_ifr_hwaddr,dev->dev_addr, MAX_ADDR_LEN);
			break;

		case SIOCGIFHWADDR:
			memcpy(ifr.ifr_hwaddr.sa_data,dev->dev_addr, MAX_ADDR_LEN);
			ifr.ifr_hwaddr.sa_family=dev->type;			
			break;
			
		case SIOCSIFHWADDR:
			if(dev->set_mac_address==NULL)
				return -EOPNOTSUPP;
			if(ifr.ifr_hwaddr.sa_family!=dev->type)
				return -EINVAL;
			return dev->set_mac_address(dev,ifr.ifr_hwaddr.sa_data);
		
		case SIOCDEVPRIVATE:
			if(dev->do_ioctl==NULL)
				return -EOPNOTSUPP;
			ret=dev->do_ioctl(dev, &ifr);
			break;
			
		case SIOCGIFMAP:
			ifr.ifr_map.mem_start=dev->mem_start;
			ifr.ifr_map.mem_end=dev->mem_end;
			ifr.ifr_map.base_addr=dev->base_addr;
			ifr.ifr_map.irq=dev->irq;
			ifr.ifr_map.dma=dev->dma;
			ifr.ifr_map.port=dev->if_port;
			break;
			
		case SIOCSIFMAP:
			if(dev->set_config==NULL)
				return -EOPNOTSUPP;
			return dev->set_config(dev,&ifr.ifr_map);
			
		case SIOCGIFSLAVE:
#ifdef CONFIG_SLAVE_BALANCING		
			if(dev->slave==NULL)
				return -ENOENT;
			strncpy(ifr.ifr_name,dev->name,sizeof(ifr.ifr_name));
#else
			return -ENOENT;
#endif			
			break;
#ifdef CONFIG_SLAVE_BALANCING			
		case SIOCSIFSLAVE:
		{
		
		/*
		 *	Fun game. Get the device up and the flags right without
		 *	letting some scummy user confuse us.
		 */
			unsigned long flags;
			struct device *slave=dev_get(ifr.ifr_slave);
			save_flags(flags);
			if(slave==NULL)
			{
				return -ENODEV;
			}
			cli();
			if((slave->flags&(IFF_UP|IFF_RUNNING))!=(IFF_UP|IFF_RUNNING))
			{
				restore_flags(flags);
				return -EINVAL;
			}
			if(dev->flags&IFF_SLAVE)
			{
				restore_flags(flags);
				return -EBUSY;
			}
			if(dev->slave!=NULL)
			{
				restore_flags(flags);
				return -EBUSY;
			}
			if(slave->flags&IFF_SLAVE)
			{
				restore_flags(flags);
				return -EBUSY;
			}
			dev->slave=slave;
			slave->flags|=IFF_SLAVE;
			dev->flags|=IFF_MASTER;
			restore_flags(flags);
			return 0;
		}
		break;
#endif			
		/*
		 *	Unknown ioctl
		 */

		default:
			return -EINVAL;
	}

	memcpy_tofs(arg,&ifr,sizeof(struct ifreq));
	return(ret);
}


/*
 *	This function handles all "interface"-type I/O control requests. The actual
 *	'doing' part of this is dev_ifsioc above.
 */

int dev_ioctl(unsigned int cmd, void *arg)
{
	switch(cmd) 
	{
		case SIOCGIFCONF:
			(void) dev_ifconf((char *) arg);
			return 0;

		/*
		 *	Ioctl calls that can be done by all.
		 */
		 
		case SIOCGIFFLAGS:
		case SIOCGIFADDR:
		case SIOCGIFDSTADDR:
		case SIOCGIFBRDADDR:
		case SIOCGIFNETMASK:
		case SIOCGIFMETRIC:
		case SIOCGIFMTU:
		case SIOCGIFMEM:
		case SIOCGIFHWADDR:
		case SIOCSIFHWADDR:
		case OLD_SIOCGIFHWADDR:
		case SIOCGIFSLAVE:
		case SIOCGIFMAP:
			return dev_ifsioc(arg, cmd);

		/*
		 *	Ioctl calls requiring the power of a superuser
		 */
		 
		case SIOCSIFFLAGS:
		case SIOCSIFADDR:
		case SIOCSIFDSTADDR:
		case SIOCSIFBRDADDR:
		case SIOCSIFNETMASK:
		case SIOCSIFMETRIC:
		case SIOCSIFMTU:
		case SIOCSIFMEM:
		case SIOCSIFMAP:
		case SIOCSIFSLAVE:
		case SIOCDEVPRIVATE:
			if (!suser())
				return -EPERM;
			return dev_ifsioc(arg, cmd);
	
		case SIOCSIFLINK:
			return -EINVAL;

		/*
		 *	Unknown ioctl.
		 */	
		 
		default:
			return -EINVAL;
	}
}


/*
 *	Initialize the DEV module. At boot time this walks the device list and
 *	unhooks any devices that fail to initialise (normally hardware not 
 *	present) and leaves us with a valid list of present and active devices.
 *
 *	The PCMCIA code may need to change this a little, and add a pair
 *	of register_inet_device() unregister_inet_device() calls. This will be
 *	needed for ethernet as modules support.
 */
 
void dev_init(void)
{
	struct device *dev, *dev2;

	/*
	 *	Add the devices.
	 *	If the call to dev->init fails, the dev is removed
	 *	from the chain disconnecting the device until the
	 *	next reboot.
	 */
	 
	dev2 = NULL;
	for (dev = dev_base; dev != NULL; dev=dev->next) 
	{
		if (dev->init && dev->init(dev)) 
		{
			/*
			 *	It failed to come up. Unhook it.
			 */
			 
			if (dev2 == NULL) 
				dev_base = dev->next;
			else 
				dev2->next = dev->next;
		} 
		else
		{
			dev2 = dev;
		}
	}
}

