/* i386-opcode.h -- Intel 80386 opcode table
   Copyright 1989, 1991, 1992 Free Software Foundation.

This file is part of GAS, the GNU Assembler, and GDB, the GNU Debugger.

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.  */

#ifdef CHECKER
#define C(a) ,a
#else
#define C(a)
#endif

static const template i386_optab[] = {

#define _ None
/* move instructions */
{ "mov", 2, 0xa0, _, DW|NoModrm, { Disp32, Acc, 0 } C(ReadMem)},
{ "mov", 2, 0x88, _, DW|Modrm, { Reg, Reg|Mem, 0 } C(WriteMem)},
{ "mov", 2, 0xb0, _, ShortFormW, { Imm, Reg, 0 } C(NoMemAcc)},
{ "mov", 2, 0xc6, _,  W|Modrm,  { Imm, Reg|Mem, 0 } C(WriteMem)},
{ "mov", 2, 0x8c, _, D|Modrm,  { SReg3|SReg2, Reg16|Mem, 0 } C(WriteMem)},
/* move to/from control debug registers */
{ "mov", 2, 0x0f20, _, D|Modrm, { Control, Reg32, 0} C(NoMemAcc)},
{ "mov", 2, 0x0f21, _, D|Modrm, { Debug, Reg32, 0} C(NoMemAcc)},
{ "mov", 2, 0x0f24, _, D|Modrm, { Test, Reg32, 0} C(NoMemAcc)},

/* move with sign extend */
/* "movsbl" & "movsbw" must not be unified into "movsb" to avoid
   conflict with the "movs" string move instruction.  Thus,
   {"movsb", 2, 0x0fbe, _, ReverseRegRegmem|Modrm, { Reg8|Mem,  Reg16|Reg32, 0} },
   is not kosher; we must seperate the two instructions. */
{"movsbl", 2, 0x0fbe, _, ReverseRegRegmem|Modrm, { Reg8|Mem,  Reg32, 0} C(CvOp|ReadMem)},
{"movsbw", 2, 0x660fbe, _, ReverseRegRegmem|Modrm, { Reg8|Mem,  Reg16, 0} C(CvOp|ReadMem)},
{"movswl", 2, 0x0fbf, _, ReverseRegRegmem|Modrm, { Reg16|Mem, Reg32, 0} C(CvOp|ReadMem)},

/* move with zero extend */
{"movzb", 2, 0x0fb6, _, ReverseRegRegmem|Modrm, { Reg8|Mem, Reg16|Reg32, 0} C(CvOp|ReadMem)},
{"movzwl", 2, 0x0fb7, _, ReverseRegRegmem|Modrm, { Reg16|Mem, Reg32, 0} C(CvOp|ReadMem)},

/* push instructions */
{"push", 1, 0x50, _, ShortForm, { WordReg,0,0 } C(PushOp)},
{"push", 1, 0xff, 0x6,  Modrm, { WordReg|WordMem, 0, 0 } C(PushOp)},
{"push", 1, 0x6a, _, NoModrm, { Imm8S, 0, 0} C(PushOp)},
#if 1
{"push", 1, 0x68, _, NoModrm, {Imm16|Imm32, 0, 0} C(PushOp)},
#else
{"push", 1, 0x68, _, NoModrm, { Imm32, 0, 0} C(PushOp)},
#endif
{"push", 1, 0x06, _,  Seg2ShortForm, { SReg2,0,0 } C(PushOp)},
{"push", 1, 0x0fa0, _, Seg3ShortForm, { SReg3,0,0 } C(PushOp)},
/* push all */
{"pusha", 0, 0x60, _, NoModrm, { 0, 0, 0 } C(PushOp)},

/* pop instructions */
{"pop", 1, 0x58, _, ShortForm, { WordReg,0,0 } C(PopOp)},
{"pop", 1, 0x8f, 0x0,  Modrm, { WordReg|WordMem, 0, 0 } C(PopOp)},
#define POP_SEG_SHORT 0x7
{"pop", 1, 0x07, _,  Seg2ShortForm, { SReg2,0,0 } C(PopOp)},
{"pop", 1, 0x0fa1, _, Seg3ShortForm, { SReg3,0,0 } C(PopOp)},
/* pop all */
{"popa", 0, 0x61, _, NoModrm, { 0, 0, 0 } C(PopOp)},

/* xchg exchange instructions
   xchg commutes:  we allow both operand orders */
{"xchg", 2, 0x90, _, ShortForm, { WordReg, Acc, 0 } C(NoMemAcc)},
{"xchg", 2, 0x90, _, ShortForm, { Acc, WordReg, 0 } C(NoMemAcc)},
{"xchg", 2, 0x86, _, W|Modrm, { Reg, Reg|Mem, 0 } C(RAndWMem)},
{"xchg", 2, 0x86, _, W|Modrm, { Reg|Mem, Reg, 0 } C(RAndWMem)},

/* in/out from ports */
{"in", 2, 0xe4, _, W|NoModrm, { Imm8, Acc, 0 } C(NoMemAcc)},
{"in", 2, 0xec, _, W|NoModrm, { InOutPortReg, Acc, 0 } C(NoMemAcc)},
{"in", 1, 0xe4, _, W|NoModrm, { Imm8, 0, 0 } C(NoMemAcc)},
{"in", 1, 0xec, _, W|NoModrm, { InOutPortReg, 0, 0 } C(NoMemAcc)},
{"out", 2, 0xe6, _, W|NoModrm, { Acc, Imm8, 0 } C(NoMemAcc)},
{"out", 2, 0xee, _, W|NoModrm, { Acc, InOutPortReg, 0 } C(NoMemAcc)},
{"out", 1, 0xe6, _, W|NoModrm, { Imm8, 0, 0 } C(NoMemAcc)},
{"out", 1, 0xee, _, W|NoModrm, { InOutPortReg, 0, 0 } C(NoMemAcc)},

/* load effective address */
{"lea", 2, 0x8d, _, Modrm, { WordMem, WordReg, 0 } C(NoMemAcc)},

/* load segment registers from memory */
{"lds", 2, 0xc5, _, Modrm, { Mem, Reg32, 0} C(ReadMem)}, /*TG: FIXME */
{"les", 2, 0xc4, _, Modrm, { Mem, Reg32, 0} C(ReadMem)},
{"lfs", 2, 0x0fb4, _, Modrm, { Mem, Reg32, 0} C(ReadMem)},
{"lgs", 2, 0x0fb5, _, Modrm, { Mem, Reg32, 0} C(ReadMem)},
{"lss", 2, 0x0fb2, _, Modrm, { Mem, Reg32, 0} C(ReadMem)},

/* flags register instructions */
{"clc", 0, 0xf8, _, NoModrm, { 0, 0, 0} C(ReadMem)},
{"cld", 0, 0xfc, _, NoModrm, { 0, 0, 0} C(ReadMem)},
{"cli", 0, 0xfa, _, NoModrm, { 0, 0, 0} C(ReadMem)},
{"clts", 0, 0x0f06, _, NoModrm, { 0, 0, 0} C(ReadMem)},
{"cmc", 0, 0xf5, _, NoModrm, { 0, 0, 0} C(ReadMem)},
{"lahf", 0, 0x9f, _, NoModrm, { 0, 0, 0} C(ReadMem)},
{"sahf", 0, 0x9e, _, NoModrm, { 0, 0, 0} C(ReadMem)},
{"pushf", 0, 0x9c, _, NoModrm, { 0, 0, 0} C(PushOp)},
{"popf", 0, 0x9d, _, NoModrm, { 0, 0, 0} C(PopOp)},
{"stc", 0, 0xf9, _, NoModrm, { 0, 0, 0} C(ReadMem)},
{"std", 0, 0xfd, _, NoModrm, { 0, 0, 0} C(ReadMem)},
{"sti", 0, 0xfb, _, NoModrm, { 0, 0, 0} C(ReadMem)},

{"add", 2, 0x0,  _, DW|Modrm, { Reg, Reg|Mem, 0} C(RAndWMem)},
{"add", 2, 0x83, 0,  Modrm, { Imm8S, WordReg|WordMem, 0} C(RAndWMem)},
{"add", 2, 0x4,  _,  W|NoModrm, { Imm,  Acc,    0} C(NoMemAcc)},
{"add", 2, 0x80, 0, W|Modrm, { Imm, Reg|Mem, 0} C(StOp|RAndWMem)},

{"inc", 1, 0x40, _, ShortForm, { WordReg, 0, 0} C(NoMemAcc)},
{"inc", 1, 0xfe, 0, W|Modrm, { Reg|Mem, 0, 0} C(RAndWMem)},

{"sub", 2, 0x28,  _, DW|Modrm, { Reg, Reg|Mem, 0} C(RAndWMem)},
{"sub", 2, 0x83, 5,  Modrm, { Imm8S, WordReg|WordMem, 0} C(RAndWMem)},
{"sub", 2, 0x2c,  _,  W|NoModrm, { Imm,  Acc,    0} C(NoMemAcc)},
{"sub", 2, 0x80, 5,  W|Modrm, { Imm, Reg|Mem, 0} C(RAndWMem)},

{"dec", 1, 0x48, _, ShortForm, { WordReg, 0, 0} C(NoMemAcc)},
{"dec", 1, 0xfe, 1, W|Modrm, { Reg|Mem, 0, 0} C(RAndWMem)},

{"sbb", 2, 0x18,  _, DW|Modrm, { Reg, Reg|Mem, 0} C(RAndWMem)},
{"sbb", 2, 0x83, 3,  Modrm, { Imm8S, WordReg|WordMem, 0} C(RAndWMem)},
{"sbb", 2, 0x1c,  _,  W|NoModrm, { Imm,  Acc,    0} C(NoMemAcc)},
{"sbb", 2, 0x80, 3,  W|Modrm, { Imm, Reg|Mem, 0} C(RAndWMem)},

{"cmp", 2, 0x38,  _, DW|Modrm, { Reg, Reg|Mem, 0} C(ReadMem)},
{"cmp", 2, 0x83, 7,  Modrm, { Imm8S, WordReg|WordMem, 0} C(ReadMem)},
{"cmp", 2, 0x3c,  _,  W|NoModrm, { Imm,  Acc,    0} C(NoMemAcc)},
{"cmp", 2, 0x80, 7,  W|Modrm, { Imm, Reg|Mem, 0} C(ReadMem)},

{"test", 2, 0x84, _, W|Modrm, { Reg|Mem, Reg, 0} C(ReadMem)},
{"test", 2, 0x84, _, W|Modrm, { Reg, Reg|Mem, 0} C(ReadMem)},
{"test", 2, 0xa8, _, W|NoModrm, { Imm, Acc, 0} C(NoMemAcc)},
{"test", 2, 0xf6, 0, W|Modrm, { Imm, Reg|Mem, 0} C(ReadMem)},

{"and", 2, 0x20,  _, DW|Modrm, { Reg, Reg|Mem, 0} C(WriteMem)},
{"and", 2, 0x83, 4,  Modrm, { Imm8S, WordReg|WordMem, 0} C(WriteMem)},
{"and", 2, 0x24,  _,  W|NoModrm, { Imm,  Acc,    0} C(NoMemAcc)},
{"and", 2, 0x80, 4,  W|Modrm, { Imm, Reg|Mem, 0} C(WriteMem)},

{"or", 2, 0x08,  _, DW|Modrm, { Reg, Reg|Mem, 0} C(WriteMem)},
{"or", 2, 0x83, 1,  Modrm, { Imm8S, WordReg|WordMem, 0} C(WriteMem)},
{"or", 2, 0x0c,  _,  W|NoModrm, { Imm,  Acc,    0} C(NoMemAcc)},
{"or", 2, 0x80, 1,  W|Modrm, { Imm, Reg|Mem, 0} C(WriteMem)},

{"xor", 2, 0x30,  _, DW|Modrm, { Reg, Reg|Mem, 0} C(RAndWMem)},
{"xor", 2, 0x83, 6,  Modrm, { Imm8S, WordReg|WordMem, 0} C(RAndWMem)},
{"xor", 2, 0x34,  _,  W|NoModrm, { Imm,  Acc,    0} C(NoMemAcc)},
{"xor", 2, 0x80, 6,  W|Modrm, { Imm, Reg|Mem, 0} C(RAndWMem)},

{"adc", 2, 0x10,  _, DW|Modrm, { Reg, Reg|Mem, 0} C(RAndWMem)},
{"adc", 2, 0x83, 2,  Modrm, { Imm8S, WordReg|WordMem, 0} C(RAndWMem)},
{"adc", 2, 0x14,  _,  W|NoModrm, { Imm,  Acc,    0} C(NoMemAcc)},
{"adc", 2, 0x80, 2,  W|Modrm, { Imm, Reg|Mem, 0} C(RAndWMem)},

{"neg", 1, 0xf6, 3, W|Modrm, { Reg|Mem, 0, 0} C(RAndWMem)},
{"not", 1, 0xf6, 2, W|Modrm, { Reg|Mem, 0, 0} C(RAndWMem)},

{"aaa", 0, 0x37, _, NoModrm, { 0, 0, 0} C(NoMemAcc)},
{"aas", 0, 0x3f, _, NoModrm, { 0, 0, 0} C(NoMemAcc)},
{"daa", 0, 0x27, _, NoModrm, { 0, 0, 0} C(NoMemAcc)},
{"das", 0, 0x2f, _, NoModrm, { 0, 0, 0} C(NoMemAcc)},
{"aad", 0, 0xd50a, _, NoModrm, { 0, 0, 0} C(NoMemAcc)},
{"aam", 0, 0xd40a, _, NoModrm, { 0, 0, 0} C(NoMemAcc)},

/* conversion insns */
/* conversion:  intel naming */
{"cbw", 0, 0x6698, _, NoModrm, { 0, 0, 0} C(NoMemAcc)},
{"cwd", 0, 0x6699, _, NoModrm, { 0, 0, 0} C(NoMemAcc)},
{"cwde", 0, 0x98, _, NoModrm, { 0, 0, 0} C(NoMemAcc)},
{"cdq", 0, 0x99, _, NoModrm, { 0, 0, 0} C(NoMemAcc)},
/*  att naming */
{"cbtw", 0, 0x6698, _, NoModrm, { 0, 0, 0} C(NoMemAcc)},
{"cwtl", 0, 0x98, _, NoModrm, { 0, 0, 0} C(NoMemAcc)},
{"cwtd", 0, 0x6699, _, NoModrm, { 0, 0, 0} C(NoMemAcc)},
{"cltd", 0, 0x99, _, NoModrm, { 0, 0, 0} C(NoMemAcc)},

/* Warning! the mul/imul (opcode 0xf6) must only have 1 operand!  They are
   expanding 64-bit multiplies, and *cannot* be selected to accomplish
   'imul %ebx, %eax' (opcode 0x0faf must be used in this case)
   These multiplies can only be selected with single opearnd forms. */
{"mul",  1, 0xf6, 4, W|Modrm, { Reg|Mem, 0, 0} C(ReadMem)},
{"imul", 1, 0xf6, 5, W|Modrm, { Reg|Mem, 0, 0} C(ReadMem)},




/* imulKludge here is needed to reverse the i.rm.reg & i.rm.regmem fields.
   These instructions are exceptions:  'imul $2, %eax, %ecx' would put
   '%eax' in the reg field and '%ecx' in the regmem field if we did not
   switch them. */
{"imul", 2, 0x0faf, _, Modrm|ReverseRegRegmem, { WordReg|Mem, WordReg, 0} C(ReadMem)},
{"imul", 3, 0x6b, _, Modrm|ReverseRegRegmem, { Imm8S, WordReg|Mem, WordReg} C(ReadMem)},
{"imul", 3, 0x69, _, Modrm|ReverseRegRegmem, { Imm16|Imm32, WordReg|Mem, WordReg} C(ReadMem)},
/*
  imul with 2 operands mimicks imul with 3 by puting register both
  in i.rm.reg & i.rm.regmem fields
*/
{"imul", 2, 0x6b, _, Modrm|imulKludge, { Imm8S, WordReg, 0} C(NoMemAcc)},
{"imul", 2, 0x69, _, Modrm|imulKludge, { Imm16|Imm32, WordReg, 0} C(NoMemAcc)},
{"div", 1, 0xf6, 6, W|Modrm, { Reg|Mem, 0, 0} C(ReadMem)},
{"div", 2, 0xf6, 6, W|Modrm, { Reg|Mem, Acc, 0} C(ReadMem)},
{"idiv", 1, 0xf6, 7, W|Modrm, { Reg|Mem, 0, 0} C(ReadMem)},
{"idiv", 2, 0xf6, 7, W|Modrm, { Reg|Mem, Acc, 0} C(ReadMem)},

{"rol", 2, 0xd0, 0, W|Modrm, { Imm1, Reg|Mem, 0} C(RAndWMem)},
{"rol", 2, 0xc0, 0, W|Modrm, { Imm8, Reg|Mem, 0} C(RAndWMem)},
{"rol", 2, 0xd2, 0, W|Modrm, { ShiftCount, Reg|Mem, 0} C(RAndWMem)},
{"rol", 1, 0xd0, 0, W|Modrm, { Reg|Mem, 0, 0} C(RAndWMem)},

{"ror", 2, 0xd0, 1, W|Modrm, { Imm1, Reg|Mem, 0} C(RAndWMem)},
{"ror", 2, 0xc0, 1, W|Modrm, { Imm8, Reg|Mem, 0} C(RAndWMem)},
{"ror", 2, 0xd2, 1, W|Modrm, { ShiftCount, Reg|Mem, 0} C(RAndWMem)},
{"ror", 1, 0xd0, 1, W|Modrm, { Reg|Mem, 0, 0} C(RAndWMem)},

{"rcl", 2, 0xd0, 2, W|Modrm, { Imm1, Reg|Mem, 0} C(RAndWMem)},
{"rcl", 2, 0xc0, 2, W|Modrm, { Imm8, Reg|Mem, 0} C(RAndWMem)},
{"rcl", 2, 0xd2, 2, W|Modrm, { ShiftCount, Reg|Mem, 0} C(RAndWMem)},
{"rcl", 1, 0xd0, 2, W|Modrm, { Reg|Mem, 0, 0} C(RAndWMem)},

{"rcr", 2, 0xd0, 3, W|Modrm, { Imm1, Reg|Mem, 0} C(RAndWMem)},
{"rcr", 2, 0xc0, 3, W|Modrm, { Imm8, Reg|Mem, 0} C(RAndWMem)},
{"rcr", 2, 0xd2, 3, W|Modrm, { ShiftCount, Reg|Mem, 0} C(RAndWMem)},
{"rcr", 1, 0xd0, 3, W|Modrm, { Reg|Mem, 0, 0} C(RAndWMem)},

{"sal", 2, 0xd0, 4, W|Modrm, { Imm1, Reg|Mem, 0} C(RAndWMem)},
{"sal", 2, 0xc0, 4, W|Modrm, { Imm8, Reg|Mem, 0} C(RAndWMem)},
{"sal", 2, 0xd2, 4, W|Modrm, { ShiftCount, Reg|Mem, 0} C(RAndWMem)},
{"sal", 1, 0xd0, 4, W|Modrm, { Reg|Mem, 0, 0} C(RAndWMem)},
{"shl", 2, 0xd0, 4, W|Modrm, { Imm1, Reg|Mem, 0} C(RAndWMem)},
{"shl", 2, 0xc0, 4, W|Modrm, { Imm8, Reg|Mem, 0} C(RAndWMem)},
{"shl", 2, 0xd2, 4, W|Modrm, { ShiftCount, Reg|Mem, 0} C(RAndWMem)},
{"shl", 1, 0xd0, 4, W|Modrm, { Reg|Mem, 0, 0} C(RAndWMem)},

{"shld", 3, 0x0fa4, _, Modrm, { Imm8, WordReg, WordReg|Mem} C(RAndWMem)},
{"shld", 3, 0x0fa5, _, Modrm, { ShiftCount, WordReg, WordReg|Mem} C(RAndWMem)},
{"shld", 2, 0x0fa5, _, Modrm, { WordReg, WordReg|Mem, 0} },

{"shr", 2, 0xd0, 5, W|Modrm, { Imm1, Reg|Mem, 0} C(RAndWMem)},
{"shr", 2, 0xc0, 5, W|Modrm, { Imm8, Reg|Mem, 0} C(RAndWMem)},
{"shr", 2, 0xd2, 5, W|Modrm, { ShiftCount, Reg|Mem, 0} C(RAndWMem)},
{"shr", 1, 0xd0, 5, W|Modrm, { Reg|Mem, 0, 0} C(RAndWMem)},

{"shrd", 3, 0x0fac, _, Modrm, { Imm8, WordReg, WordReg|Mem} C(RAndWMem)},
{"shrd", 3, 0x0fad, _, Modrm, { ShiftCount, WordReg, WordReg|Mem} C(RAndWMem)},
{"shrd", 2, 0x0fad, _, Modrm, { WordReg, WordReg|Mem, 0} C(RAndWMem)},

{"sar", 2, 0xd0, 7, W|Modrm, { Imm1, Reg|Mem, 0} C(RAndWMem)},
{"sar", 2, 0xc0, 7, W|Modrm, { Imm8, Reg|Mem, 0} C(RAndWMem)},
{"sar", 2, 0xd2, 7, W|Modrm, { ShiftCount, Reg|Mem, 0} C(RAndWMem)},
{"sar", 1, 0xd0, 7, W|Modrm, { Reg|Mem, 0, 0} C(RAndWMem)},

/* control transfer instructions */
#define CALL_PC_RELATIVE 0xe8
{"call", 1, 0xe8, _, JumpDword, { Disp32, 0, 0} C(StackOp)},
{"call", 1, 0xff, 2, Modrm, { Reg|Mem|JumpAbsolute, 0, 0} C(ReadMem)},
#define CALL_FAR_IMMEDIATE 0x9a
{"lcall", 2, 0x9a, _, JumpInterSegment, { Imm16, Abs32|Imm32, 0} C(StackOp)},
{"lcall", 1, 0xff, 3, Modrm, { Mem, 0, 0} C(ReadMem)},

#define JUMP_PC_RELATIVE 0xeb
{"jmp", 1, 0xeb, _, Jump, { Disp, 0, 0} C(NoMemAcc)},
{"jmp", 1, 0xff, 4, Modrm, { Reg32|Mem|JumpAbsolute, 0, 0} C(ReadMem)},
#define JUMP_FAR_IMMEDIATE 0xea
{"ljmp", 2, 0xea, _, JumpInterSegment, { Imm16, Imm32, 0} C(NoMemAcc)},
{"ljmp", 1, 0xff, 5, Modrm, { Mem, 0, 0} C(ReadMem)},

{"ret", 0, 0xc3, _, NoModrm, { 0, 0, 0} C(StackOp)},
{"ret", 1, 0xc2, _, NoModrm, { Imm16, 0, 0} C(StackOp)},
{"lret", 0, 0xcb, _, NoModrm, { 0, 0, 0} C(StackOp)},
{"lret", 1, 0xca, _, NoModrm, { Imm16, 0, 0} C(StackOp)},
{"enter", 2, 0xc8, _, NoModrm, { Imm16, Imm8, 0} C(StackOp)},
{"leave", 0, 0xc9, _, NoModrm, { 0, 0, 0} C(StOp|StackOp)},

/* conditional jumps */
{"jo", 1, 0x70, _, Jump, { Disp, 0, 0} C(NoMemAcc)},

{"jno", 1, 0x71, _, Jump, { Disp, 0, 0} C(NoMemAcc)},

{"jb", 1, 0x72, _, Jump, { Disp, 0, 0} C(NoMemAcc)},
{"jc", 1, 0x72, _, Jump, { Disp, 0, 0} C(NoMemAcc)},
{"jnae", 1, 0x72, _, Jump, { Disp, 0, 0} C(NoMemAcc)},

{"jnb", 1, 0x73, _, Jump, { Disp, 0, 0} C(NoMemAcc)},
{"jnc", 1, 0x73, _, Jump, { Disp, 0, 0} C(NoMemAcc)},
{"jae", 1, 0x73, _, Jump, { Disp, 0, 0} C(NoMemAcc)},

{"je", 1, 0x74, _, Jump, { Disp, 0, 0} C(NoMemAcc)},
{"jz", 1, 0x74, _, Jump, { Disp, 0, 0} C(NoMemAcc)},

{"jne", 1, 0x75, _, Jump, { Disp, 0, 0} C(NoMemAcc)},
{"jnz", 1, 0x75, _, Jump, { Disp, 0, 0} C(NoMemAcc)},

{"jbe", 1, 0x76, _, Jump, { Disp, 0, 0} C(NoMemAcc)},
{"jna", 1, 0x76, _, Jump, { Disp, 0, 0} C(NoMemAcc)},

{"jnbe", 1, 0x77, _, Jump, { Disp, 0, 0} C(NoMemAcc)},
{"ja", 1, 0x77, _, Jump, { Disp, 0, 0} C(NoMemAcc)},

{"js", 1, 0x78, _, Jump, { Disp, 0, 0} C(NoMemAcc)},

{"jns", 1, 0x79, _, Jump, { Disp, 0, 0} C(NoMemAcc)},

{"jp", 1, 0x7a, _, Jump, { Disp, 0, 0} C(NoMemAcc)},
{"jpe", 1, 0x7a, _, Jump, { Disp, 0, 0} C(NoMemAcc)},

{"jnp", 1, 0x7b, _, Jump, { Disp, 0, 0} C(NoMemAcc)},
{"jpo", 1, 0x7b, _, Jump, { Disp, 0, 0} C(NoMemAcc)},

{"jl", 1, 0x7c, _, Jump, { Disp, 0, 0} C(NoMemAcc)},
{"jnge", 1, 0x7c, _, Jump, { Disp, 0, 0} C(NoMemAcc)},

{"jnl", 1, 0x7d, _, Jump, { Disp, 0, 0} C(NoMemAcc)},
{"jge", 1, 0x7d, _, Jump, { Disp, 0, 0} C(NoMemAcc)},

{"jle", 1, 0x7e, _, Jump, { Disp, 0, 0} C(NoMemAcc)},
{"jng", 1, 0x7e, _, Jump, { Disp, 0, 0} C(NoMemAcc)},

{"jnle", 1, 0x7f, _, Jump, { Disp, 0, 0} C(NoMemAcc)},
{"jg", 1, 0x7f, _, Jump, { Disp, 0, 0} C(NoMemAcc)},

/* these turn into pseudo operations when disp is larger than 8 bits */
#define IS_JUMP_ON_CX_ZERO(o) \
  (o == 0x66e3)
#define IS_JUMP_ON_ECX_ZERO(o) \
  (o == 0xe3)

{"jcxz", 1, 0x66e3, _, JumpByte, { Disp, 0, 0} C(NoMemAcc)},
{"jecxz", 1, 0xe3, _, JumpByte, { Disp, 0, 0} C(NoMemAcc)},

#define IS_LOOP_ECX_TIMES(o) \
  (o == 0xe2 || o == 0xe1 || o == 0xe0)

{"loop", 1, 0xe2, _, JumpByte, { Disp, 0, 0} C(NoMemAcc)},

{"loopz", 1, 0xe1, _, JumpByte, { Disp, 0, 0} C(NoMemAcc)},
{"loope", 1, 0xe1, _, JumpByte, { Disp, 0, 0} C(NoMemAcc)},

{"loopnz", 1, 0xe0, _, JumpByte, { Disp, 0, 0} C(NoMemAcc)},
{"loopne", 1, 0xe0, _, JumpByte, { Disp, 0, 0} C(NoMemAcc)},

/* set byte on flag instructions */ /*TG: FIXME */
{"seto", 1, 0x0f90, 0, Modrm, { Reg8|Mem, 0, 0} C(WriteMem)},

{"setno", 1, 0x0f91, 0, Modrm, { Reg8|Mem, 0, 0} C(WriteMem)},

{"setb", 1, 0x0f92, 0, Modrm, { Reg8|Mem, 0, 0} C(WriteMem)},
{"setc", 1, 0x0f92, 0, Modrm, { Reg8|Mem, 0, 0} C(WriteMem)},
{"setnae", 1, 0x0f92, 0, Modrm, { Reg8|Mem, 0, 0} C(WriteMem)},

{"setnb", 1, 0x0f93, 0, Modrm, { Reg8|Mem, 0, 0} C(WriteMem)},
{"setnc", 1, 0x0f93, 0, Modrm, { Reg8|Mem, 0, 0} C(WriteMem)},
{"setae", 1, 0x0f93, 0, Modrm, { Reg8|Mem, 0, 0} C(WriteMem)},

{"sete", 1, 0x0f94, 0, Modrm, { Reg8|Mem, 0, 0} C(WriteMem)},
{"setz", 1, 0x0f94, 0, Modrm, { Reg8|Mem, 0, 0} C(WriteMem)},

{"setne", 1, 0x0f95, 0, Modrm, { Reg8|Mem, 0, 0} C(WriteMem)},
{"setnz", 1, 0x0f95, 0, Modrm, { Reg8|Mem, 0, 0} C(WriteMem)},

{"setbe", 1, 0x0f96, 0, Modrm, { Reg8|Mem, 0, 0} C(WriteMem)},
{"setna", 1, 0x0f96, 0, Modrm, { Reg8|Mem, 0, 0} C(WriteMem)},

{"setnbe", 1, 0x0f97, 0, Modrm, { Reg8|Mem, 0, 0} C(WriteMem)},
{"seta", 1, 0x0f97, 0, Modrm, { Reg8|Mem, 0, 0} C(WriteMem)},

{"sets", 1, 0x0f98, 0, Modrm, { Reg8|Mem, 0, 0} C(WriteMem)},

{"setns", 1, 0x0f99, 0, Modrm, { Reg8|Mem, 0, 0} C(WriteMem)},

{"setp", 1, 0x0f9a, 0, Modrm, { Reg8|Mem, 0, 0} C(WriteMem)},
{"setpe", 1, 0x0f9a, 0, Modrm, { Reg8|Mem, 0, 0} C(WriteMem)},

{"setnp", 1, 0x0f9b, 0, Modrm, { Reg8|Mem, 0, 0} C(WriteMem)},
{"setpo", 1, 0x0f9b, 0, Modrm, { Reg8|Mem, 0, 0} C(WriteMem)},

{"setl", 1, 0x0f9c, 0, Modrm, { Reg8|Mem, 0, 0} C(WriteMem)},
{"setnge", 1, 0x0f9c, 0, Modrm, { Reg8|Mem, 0, 0} C(WriteMem)},

{"setnl", 1, 0x0f9d, 0, Modrm, { Reg8|Mem, 0, 0} C(WriteMem)},
{"setge", 1, 0x0f9d, 0, Modrm, { Reg8|Mem, 0, 0} C(WriteMem)},

{"setle", 1, 0x0f9e, 0, Modrm, { Reg8|Mem, 0, 0} C(WriteMem)},
{"setng", 1, 0x0f9e, 0, Modrm, { Reg8|Mem, 0, 0} C(WriteMem)},

{"setnle", 1, 0x0f9f, 0, Modrm, { Reg8|Mem, 0, 0} C(WriteMem)},
{"setg", 1, 0x0f9f, 0, Modrm, { Reg8|Mem, 0, 0} C(WriteMem)},

#define IS_STRING_INSTRUCTION(o) \
  ((o) == 0xa6 || (o) == 0x6c || (o) == 0x6e || (o) == 0x6e || \
   (o) == 0xac || (o) == 0xa4 || (o) == 0xae || (o) == 0xaa || \
   (o) == 0xd7)

/* string manipulation */
{"cmps", 0, 0xa6, _, W|NoModrm, { 0, 0, 0} C(StrOp)},
{"scmp", 0, 0xa6, _, W|NoModrm, { 0, 0, 0} C(StrOp)},
{"ins", 0, 0x6c, _, W|NoModrm, { 0, 0, 0} C(StrOp)},
{"outs", 0, 0x6e, _, W|NoModrm, { 0, 0, 0} C(StrOp)},
{"lods", 0, 0xac, _, W|NoModrm, { 0, 0, 0} C(StrOp)},
{"slod", 0, 0xac, _, W|NoModrm, { 0, 0, 0} C(StrOp)},
{"movs", 0, 0xa4, _, W|NoModrm, { 0, 0, 0} C(StrOp)},
{"smov", 0, 0xa4, _, W|NoModrm, { 0, 0, 0} C(StrOp)},
{"scas", 0, 0xae, _, W|NoModrm, { 0, 0, 0} C(StrOp)},
{"ssca", 0, 0xae, _, W|NoModrm, { 0, 0, 0} C(StrOp)},
{"stos", 0, 0xaa, _, W|NoModrm, { 0, 0, 0} C(StrOp)},
{"ssto", 0, 0xaa, _, W|NoModrm, { 0, 0, 0} C(StrOp)},
{"xlat", 0, 0xd7, _, NoModrm, { 0, 0, 0} C(StrOp)},

/* bit manipulation */
{"bsf", 2, 0x0fbc, _, Modrm|ReverseRegRegmem, { Reg|Mem, Reg, 0} C(RAndWMem)},
{"bsr", 2, 0x0fbd, _, Modrm|ReverseRegRegmem, { Reg|Mem, Reg, 0} C(RAndWMem)},
{"bt", 2, 0x0fa3, _, Modrm, { Reg, Reg|Mem, 0} C(RAndWMem)},
{"bt", 2, 0x0fba, 4, Modrm, { Imm8, Reg|Mem, 0} C(RAndWMem)},
{"btc", 2, 0x0fbb, _, Modrm, { Reg, Reg|Mem, 0} C(RAndWMem)},
{"btc", 2, 0x0fba, 7, Modrm, { Imm8, Reg|Mem, 0} C(RAndWMem)},
{"btr", 2, 0x0fb3, _, Modrm, { Reg, Reg|Mem, 0} C(RAndWMem)},
{"btr", 2, 0x0fba, 6, Modrm, { Imm8, Reg|Mem, 0} C(RAndWMem)},
{"bts", 2, 0x0fab, _, Modrm, { Reg, Reg|Mem, 0} C(RAndWMem)},
{"bts", 2, 0x0fba, 5, Modrm, { Imm8, Reg|Mem, 0} C(RAndWMem)},

/* interrupts & op. sys insns */
/* See gas/config/tc-i386.c for conversion of 'int $3' into the special
   int 3 insn. */
#define INT_OPCODE 0xcd
#define INT3_OPCODE 0xcc
{"int", 1, 0xcd, _, NoModrm, { Imm8, 0, 0} C(IntOp)},
{"int3", 0, 0xcc, _, NoModrm, { 0, 0, 0} C(IntOp)},
{"into", 0, 0xce, _, NoModrm, { 0, 0, 0} C(IntOp)},
{"iret", 0, 0xcf, _, NoModrm, { 0, 0, 0} C(NoMemAcc)},
/* i386sl (and i486sl?) only */ /*TG: FIXME */
{"rsm", 0, 0x0faa, _, NoModrm,{ 0, 0, 0} C(NoMemAcc)},

{"boundl", 2, 0x62, _, Modrm, { Reg32, Mem, 0} C(RF64bOp)},
{"boundw", 2, 0x6662, _, Modrm, { Reg16, Mem, 0} C(RF32bOp)},

{"hlt", 0, 0xf4, _, NoModrm, { 0, 0, 0} C(NoMemAcc)},
{"wait", 0, 0x9b, _, NoModrm, { 0, 0, 0} C(NoMemAcc)},
/* nop is actually 'xchgl %eax, %eax' */
{"nop", 0, 0x90, _, NoModrm, { 0, 0, 0} C(NoMemAcc)},

/* protection control */
{"arpl", 2, 0x63, _, Modrm, { Reg16, Reg16|Mem, 0} C(ReadMem)},
{"lar", 2, 0x0f02, _, Modrm|ReverseRegRegmem, { WordReg|Mem, WordReg, 0} C(ReadMem)},
{"lgdt", 1, 0x0f01, 2, Modrm, { Mem, 0, 0} C(RProtOp)},
{"lidt", 1, 0x0f01, 3, Modrm, { Mem, 0, 0} C(RProtOp)},
{"lldt", 1, 0x0f00, 2, Modrm, { WordReg|Mem, 0, 0} C(ReadMem)},
{"lmsw", 1, 0x0f01, 6, Modrm, { WordReg|Mem, 0, 0} C(ReadMem)},
{"lsl", 2, 0x0f03, _, Modrm|ReverseRegRegmem, { WordReg|Mem, WordReg, 0} C(ReadMem)},
{"ltr", 1, 0x0f00, 3, Modrm, { WordReg|Mem, 0, 0} C(ReadMem)},

{"sgdt", 1, 0x0f01, 0, Modrm, { Mem, 0, 0} C(WProtOp)},
{"sidt", 1, 0x0f01, 1, Modrm, { Mem, 0, 0} C(WProtOp)},
{"sldt", 1, 0x0f00, 0, Modrm, { WordReg|Mem, 0, 0} C(WriteMem)},
{"smsw", 1, 0x0f01, 4, Modrm, { WordReg|Mem, 0, 0} C(WriteMem)},
{"str", 1, 0x0f00, 1, Modrm, { Reg16|Mem, 0, 0} C(WriteMem)},

{"verr", 1, 0x0f00, 4, Modrm, { WordReg|Mem, 0, 0} C(NoMemAcc)},
{"verw", 1, 0x0f00, 5, Modrm, { WordReg|Mem, 0, 0} C(NoMemAcc)},

/* floating point instructions */

/* load */
{"fld", 1, 0xd9c0, _, ShortForm, { FloatReg, 0, 0} C(NoMemAcc)}, /* register */
{"flds", 1, 0xd9, 0, Modrm, { Mem, 0, 0} C(RF32bOp)},		/* %st0 <-- mem float */
{"fldl", 1, 0xdd, 0, Modrm, { Mem, 0, 0} C(RF64bOp)},		/* %st0 <-- mem double */
{"fldl", 1, 0xd9c0, _, ShortForm, { FloatReg, 0, 0} C(NoMemAcc)}, /* register */
{"fild", 1, 0xdf, 0, Modrm, { Mem, 0, 0} C(RF16bOp)},		/* %st0 <-- mem word (16) */
{"fildl", 1, 0xdb, 0, Modrm, { Mem, 0, 0} C(RF32bOp)},	/* %st0 <-- mem dword (32) */
{"fildq",1, 0xdf, 5, Modrm, { Mem, 0, 0} C(RF64bOp)},		/* %st0 <-- mem qword (64) */
{"fildll",1, 0xdf, 5, Modrm, { Mem, 0, 0} C(RF64bOp)},	/* %st0 <-- mem qword (64) */
{"fldt", 1, 0xdb, 5, Modrm, { Mem, 0, 0} C(RF80bOp)},		/* %st0 <-- mem efloat */
{"fbld", 1, 0xdf, 4, Modrm, { Mem, 0, 0} C(RF80bOp)},		/* %st0 <-- mem bcd */

/* store (no pop) */
{"fst", 1, 0xddd0, _, ShortForm, { FloatReg, 0, 0} C(NoMemAcc)}, /* register */
{"fsts", 1, 0xd9, 2, Modrm, { Mem, 0, 0} C(WF32bOp)},		/* %st0 --> mem float */
{"fstl", 1, 0xdd, 2, Modrm, { Mem, 0, 0} C(WF64bOp)},		/* %st0 --> mem double */
{"fstl", 1, 0xddd0, _, ShortForm, { FloatReg, 0, 0} C(NoMemAcc)}, /* register */
{"fist", 1, 0xdf, 2, Modrm, { Mem, 0, 0} C(WF16bOp)},		/* %st0 --> mem word (16) */
{"fistl", 1, 0xdb, 2, Modrm, { Mem, 0, 0} C(WF32bOp)},	/* %st0 --> mem dword (32) */

/* store (with pop) */
{"fstp", 1, 0xddd8, _, ShortForm, { FloatReg, 0, 0} C(NoMemAcc)}, /* register */
{"fstps", 1, 0xd9, 3, Modrm, { Mem, 0, 0} C(WF32bOp)},	/* %st0 --> mem float */
{"fstpl", 1, 0xdd, 3, Modrm, { Mem, 0, 0} C(WF64bOp)},	/* %st0 --> mem double */
{"fstpl", 1, 0xddd8, _, ShortForm, { FloatReg, 0, 0} C(NoMemAcc)}, /* register */
{"fistp", 1, 0xdf, 3, Modrm, { Mem, 0, 0} C(WF16bOp)},	/* %st0 --> mem word (16) */
{"fistpl",1, 0xdb, 3, Modrm, { Mem, 0, 0} C(WF32bOp)},	/* %st0 --> mem dword (32) */
{"fistpq",1, 0xdf, 7, Modrm, { Mem, 0, 0} C(WF64bOp)},	/* %st0 --> mem qword (64) */
{"fistpll",1,0xdf, 7, Modrm, { Mem, 0, 0} C(WF64bOp)},	/* %st0 --> mem qword (64) */
{"fstpt", 1, 0xdb, 7, Modrm, { Mem, 0, 0} C(WF80bOp)},	/* %st0 --> mem efloat */
{"fbstp", 1, 0xdf, 6, Modrm, { Mem, 0, 0} C(WF80bOp)},	/* %st0 --> mem bcd */

/* exchange %st<n> with %st0 */
{"fxch", 1, 0xd9c8, _, ShortForm, { FloatReg, 0, 0} C(NoMemAcc)},
{"fxch", 0, 0xd9c9, _, NoModrm, { 0, 0, 0} C(NoMemAcc)}, /* alias for fxch %st, %st(1) */

/* comparison (without pop) */
{"fcom", 1, 0xd8d0, _, ShortForm, { FloatReg, 0, 0} C(NoMemAcc)},
{"fcoms", 1, 0xd8, 2, Modrm, { Mem, 0, 0} C(RF32bOp)},	/* compare %st0, mem float  */
{"ficoml", 1, 0xda, 2, Modrm, { Mem, 0, 0} C(RF16bOp)},	/* compare %st0, mem word  */ 
{"fcoml", 1, 0xdc, 2, Modrm, { Mem, 0, 0} C(RF64bOp)},	/* compare %st0, mem double  */
{"fcoml", 1, 0xd8d0, _, ShortForm, { FloatReg, 0, 0} C(NoMemAcc)},
{"ficoms", 1, 0xde, 2, Modrm, { Mem, 0, 0} C(RF32bOp)},	/* compare %st0, mem dword */

/* comparison (with pop) */
{"fcomp", 1, 0xd8d8, _, ShortForm, { FloatReg, 0, 0} C(NoMemAcc)},
{"fcomps", 1, 0xd8, 3, Modrm, { Mem, 0, 0} C(RF32bOp)},	/* compare %st0, mem float  */
{"ficompl", 1, 0xda, 3, Modrm, { Mem, 0, 0} C(RF16bOp)},	/* compare %st0, mem word  */ 
{"fcompl", 1, 0xdc, 3, Modrm, { Mem, 0, 0} C(RF64bOp)},	/* compare %st0, mem double  */
{"fcompl", 1, 0xd8d8, _, ShortForm, { FloatReg, 0, 0} C(NoMemAcc)},
{"ficomps", 1, 0xde, 3, Modrm, { Mem, 0, 0} C(RF32bOp)},	/* compare %st0, mem dword */
{"fcompp", 0, 0xded9, _, NoModrm, { 0, 0, 0} C(NoMemAcc)},	/* compare %st0, %st1 & pop 2 */

/* unordered comparison (with pop) */
{"fucom", 1, 0xdde0, _, ShortForm, { FloatReg, 0, 0} C(NoMemAcc)},
{"fucomp", 1, 0xdde8, _, ShortForm, { FloatReg, 0, 0} C(NoMemAcc)},
{"fucompp", 0, 0xdae9, _, NoModrm, { 0, 0, 0} C(NoMemAcc)}, /* ucompare %st0, %st1 & pop twice */

{"ftst", 0, 0xd9e4, _, NoModrm, { 0, 0, 0} C(NoMemAcc)},	/* test %st0 */
{"fxam", 0, 0xd9e5, _, NoModrm, { 0, 0, 0} C(NoMemAcc)},	/* examine %st0 */

/* load constants into %st0 */
{"fld1", 0, 0xd9e8, _, NoModrm, { 0, 0, 0} C(NoMemAcc)},	/* %st0 <-- 1.0 */
{"fldl2t", 0, 0xd9e9, _, NoModrm, { 0, 0, 0} C(NoMemAcc)},	/* %st0 <-- log2(10) */
{"fldl2e", 0, 0xd9ea, _, NoModrm, { 0, 0, 0} C(NoMemAcc)},	/* %st0 <-- log2(e) */
{"fldpi", 0, 0xd9eb, _, NoModrm, { 0, 0, 0} C(NoMemAcc)},	/* %st0 <-- pi */
{"fldlg2", 0, 0xd9ec, _, NoModrm, { 0, 0, 0} C(NoMemAcc)},	/* %st0 <-- log10(2) */
{"fldln2", 0, 0xd9ed, _, NoModrm, { 0, 0, 0} C(NoMemAcc)},	/* %st0 <-- ln(2) */
{"fldz", 0, 0xd9ee, _, NoModrm, { 0, 0, 0} C(NoMemAcc)},	/* %st0 <-- 0.0 */

/* arithmetic */

/* add */
{"fadd", 1, 0xd8c0, _, ShortForm, { FloatReg, 0, 0} C(NoMemAcc)},
{"fadd", 2, 0xd8c0, _, ShortForm|FloatD, { FloatReg, FloatAcc, 0} C(NoMemAcc)},
{"fadd", 0, 0xdcc1, _, NoModrm, { 0, 0, 0} C(NoMemAcc)}, /* alias for fadd %st, %st(1) */
{"faddp", 1, 0xdac0, _, ShortForm, { FloatReg, 0, 0} C(NoMemAcc)},
{"faddp", 2, 0xdac0, _, ShortForm|FloatD, { FloatReg, FloatAcc, 0} C(NoMemAcc)},
{"faddp", 0, 0xdec1, _, NoModrm, { 0, 0, 0} C(NoMemAcc)}, /* alias for faddp %st, %st(1) */
{"fadds", 1, 0xd8, 0, Modrm, { Mem, 0, 0} C(RF32bOp)},
{"fiaddl", 1, 0xda, 0, Modrm, { Mem, 0, 0} C(RF32bOp)},
{"faddl", 1, 0xdc, 0, Modrm, { Mem, 0, 0} C(RF64bOp)},
{"fiadds", 1, 0xde, 0, Modrm, { Mem, 0, 0} C(RF16bOp)},

/* sub */
/* Note:  intel has decided that certain of these operations are reversed
   in assembler syntax. */
{"fsub", 1, 0xd8e0, _, ShortForm, { FloatReg, 0, 0} C(NoMemAcc)},
{"fsub", 2, 0xd8e0, _, ShortForm, { FloatReg, FloatAcc, 0} C(NoMemAcc)},
#ifdef NON_BROKEN_OPCODES
{"fsub", 2, 0xdce8, _, ShortForm, { FloatAcc, FloatReg, 0} C(NoMemAcc)},
#else
{"fsub", 2, 0xdce0, _, ShortForm, { FloatAcc, FloatReg, 0} C(NoMemAcc)},
#endif
{"fsub", 0, 0xdce1, _, NoModrm, { 0, 0, 0} C(NoMemAcc)},
{"fsubp", 1, 0xdae0, _, ShortForm, { FloatReg, 0, 0} C(NoMemAcc)},
{"fsubp", 2, 0xdae0, _, ShortForm, { FloatReg, FloatAcc, 0} C(NoMemAcc)},
#ifdef NON_BROKEN_OPCODES
{"fsubp", 2, 0xdee8, _, ShortForm, { FloatAcc, FloatReg, 0} C(NoMemAcc)},
#else
{"fsubp", 2, 0xdee0, _, ShortForm, { FloatAcc, FloatReg, 0} C(NoMemAcc)},
#endif
{"fsubp", 0, 0xdee1, _, NoModrm, { 0, 0, 0} C(NoMemAcc)},
{"fsubs", 1, 0xd8, 4, Modrm, { Mem, 0, 0} C(RF32bOp)},
{"fisubl", 1, 0xda, 4, Modrm, { Mem, 0, 0} C(RF32bOp)},
{"fsubl", 1, 0xdc, 4, Modrm, { Mem, 0, 0} C(RF64bOp)},
{"fisubs", 1, 0xde, 4, Modrm, { Mem, 0, 0} C(RF16bOp)},

/* sub reverse */
{"fsubr", 1, 0xd8e8, _, ShortForm, { FloatReg, 0, 0} C(NoMemAcc)},
{"fsubr", 2, 0xd8e8, _, ShortForm, { FloatReg, FloatAcc, 0} C(NoMemAcc)},
#ifdef NON_BROKEN_OPCODES
{"fsubr", 2, 0xdce0, _, ShortForm, { FloatAcc, FloatReg, 0} C(NoMemAcc)},
#else
{"fsubr", 2, 0xdce8, _, ShortForm, { FloatAcc, FloatReg, 0} C(NoMemAcc)},
#endif
{"fsubr", 0, 0xdce9, _, NoModrm, { 0, 0, 0} C(NoMemAcc)},
{"fsubrp", 1, 0xdae8, _, ShortForm, { FloatReg, 0, 0} C(NoMemAcc)},
{"fsubrp", 2, 0xdae8, _, ShortForm, { FloatReg, FloatAcc, 0} C(NoMemAcc)},
#ifdef NON_BROKEN_OPCODES
{"fsubrp", 2, 0xdee0, _, ShortForm, { FloatAcc, FloatReg, 0} C(NoMemAcc)},
#else
{"fsubrp", 2, 0xdee8, _, ShortForm, { FloatAcc, FloatReg, 0} C(NoMemAcc)},
#endif
{"fsubrp", 0, 0xdee9, _, NoModrm, { 0, 0, 0} C(NoMemAcc)},
{"fsubrs", 1, 0xd8, 5, Modrm, { Mem, 0, 0} C(RF32bOp)},
{"fisubrl", 1, 0xda, 5, Modrm, { Mem, 0, 0} C(RF32bOp)},
{"fsubrl", 1, 0xdc, 5, Modrm, { Mem, 0, 0} C(RF64bOp)},
{"fisubrs", 1, 0xde, 5, Modrm, { Mem, 0, 0} C(RF16bOp)},

/* mul */
{"fmul", 1, 0xd8c8, _, ShortForm, { FloatReg, 0, 0} C(NoMemAcc)},
{"fmul", 2, 0xd8c8, _, ShortForm|FloatD, { FloatReg, FloatAcc, 0} C(NoMemAcc)},
{"fmul", 0, 0xdcc9, _, NoModrm, { 0, 0, 0} C(NoMemAcc)},
{"fmulp", 1, 0xdac8, _, ShortForm, { FloatReg, 0, 0} C(NoMemAcc)},
{"fmulp", 2, 0xdac8, _, ShortForm|FloatD, { FloatReg, FloatAcc, 0} C(NoMemAcc)},
{"fmulp", 0, 0xdec9, _, NoModrm, { 0, 0, 0} C(NoMemAcc)},
{"fmuls", 1, 0xd8, 1, Modrm, { Mem, 0, 0} C(RF32bOp)},
{"fimull", 1, 0xda, 1, Modrm, { Mem, 0, 0} C(RF32bOp)},
{"fmull", 1, 0xdc, 1, Modrm, { Mem, 0, 0} C(RF64bOp)},
{"fimuls", 1, 0xde, 1, Modrm, { Mem, 0, 0} C(RF16bOp)},

/* div */
/* Note:  intel has decided that certain of these operations are reversed
   in assembler syntax. */
{"fdiv", 1, 0xd8f0, _, ShortForm, { FloatReg, 0, 0} C(NoMemAcc)},
{"fdiv", 2, 0xd8f0, _, ShortForm, { FloatReg, FloatAcc, 0} C(NoMemAcc)},
#ifdef NON_BROKEN_OPCODES
{"fdiv", 2, 0xdcf8, _, ShortForm, { FloatAcc, FloatReg, 0} C(NoMemAcc)},
#else
{"fdiv", 2, 0xdcf0, _, ShortForm, { FloatAcc, FloatReg, 0} C(NoMemAcc)},
#endif
{"fdiv", 0, 0xdcf1, _, NoModrm, { 0, 0, 0} C(NoMemAcc)},
{"fdivp", 1, 0xdaf0, _, ShortForm, { FloatReg, 0, 0} C(NoMemAcc)},
{"fdivp", 2, 0xdaf0, _, ShortForm, { FloatReg, FloatAcc, 0} C(NoMemAcc)},
#ifdef NON_BROKEN_OPCODES
{"fdivp", 2, 0xdef8, _, ShortForm, { FloatAcc, FloatReg, 0} C(NoMemAcc)},
#else
{"fdivp", 2, 0xdef0, _, ShortForm, { FloatAcc, FloatReg, 0} C(NoMemAcc)},
#endif
{"fdivp", 0, 0xdef1, _, NoModrm, { 0, 0, 0} C(NoMemAcc)},
{"fdivs", 1, 0xd8, 6, Modrm, { Mem, 0, 0} C(RF32bOp)},
{"fidivl", 1, 0xda, 6, Modrm, { Mem, 0, 0} C(RF32bOp)},
{"fdivl", 1, 0xdc, 6, Modrm, { Mem, 0, 0} C(RF64bOp)},
{"fidivs", 1, 0xde, 6, Modrm, { Mem, 0, 0} C(RF16bOp)},

/* div reverse */
{"fdivr", 1, 0xd8f8, _, ShortForm, { FloatReg, 0, 0} C(NoMemAcc)},
{"fdivr", 2, 0xd8f8, _, ShortForm, { FloatReg, FloatAcc, 0} C(NoMemAcc)},
#ifdef NON_BROKEN_OPCODES
{"fdivr", 2, 0xdcf0, _, ShortForm, { FloatAcc, FloatReg, 0} C(NoMemAcc)},
#else
{"fdivr", 2, 0xdcf8, _, ShortForm, { FloatAcc, FloatReg, 0} C(NoMemAcc)},
#endif
{"fdivr", 0, 0xdcf9, _, NoModrm, { 0, 0, 0} C(NoMemAcc)},
{"fdivrp", 1, 0xdaf8, _, ShortForm, { FloatReg, 0, 0} C(NoMemAcc)},
{"fdivrp", 2, 0xdaf8, _, ShortForm, { FloatReg, FloatAcc, 0} C(NoMemAcc)},
#ifdef NON_BROKEN_OPCODES
{"fdivrp", 2, 0xdef0, _, ShortForm, { FloatAcc, FloatReg, 0} C(NoMemAcc)},
#else
{"fdivrp", 2, 0xdef8, _, ShortForm, { FloatAcc, FloatReg, 0} C(NoMemAcc)},
#endif
{"fdivrp", 0, 0xdef9, _, NoModrm, { 0, 0, 0} C(NoMemAcc)},
{"fdivrs", 1, 0xd8, 7, Modrm, { Mem, 0, 0} C(RF32bOp)},
{"fidivrl", 1, 0xda, 7, Modrm, { Mem, 0, 0} C(RF32bOp)},
{"fdivrl", 1, 0xdc, 7, Modrm, { Mem, 0, 0} C(RF64bOp)},
{"fidivrs", 1, 0xde, 7, Modrm, { Mem, 0, 0} C(RF16bOp)},

{"f2xm1", 0,   0xd9f0, _, NoModrm, { 0, 0, 0} C(NoMemAcc)},
{"fyl2x", 0,   0xd9f1, _, NoModrm, { 0, 0, 0} C(NoMemAcc)},
{"fptan", 0,   0xd9f2, _, NoModrm, { 0, 0, 0} C(NoMemAcc)},
{"fpatan", 0,  0xd9f3, _, NoModrm, { 0, 0, 0} C(NoMemAcc)},
{"fxtract", 0, 0xd9f4, _, NoModrm, { 0, 0, 0} C(NoMemAcc)},
{"fprem1", 0,  0xd9f5, _, NoModrm, { 0, 0, 0} C(NoMemAcc)},
{"fdecstp", 0,  0xd9f6, _, NoModrm, { 0, 0, 0} C(NoMemAcc)},
{"fincstp", 0,  0xd9f7, _, NoModrm, { 0, 0, 0} C(NoMemAcc)},
{"fprem", 0,   0xd9f8, _, NoModrm, { 0, 0, 0} C(NoMemAcc)},
{"fyl2xp1", 0, 0xd9f9, _, NoModrm, { 0, 0, 0} C(NoMemAcc)},
{"fsqrt", 0,   0xd9fa, _, NoModrm, { 0, 0, 0} C(NoMemAcc)},
{"fsincos", 0, 0xd9fb, _, NoModrm, { 0, 0, 0} C(NoMemAcc)},
{"frndint", 0, 0xd9fc, _, NoModrm, { 0, 0, 0} C(NoMemAcc)},
{"fscale", 0,  0xd9fd, _, NoModrm, { 0, 0, 0} C(NoMemAcc)},
{"fsin", 0,    0xd9fe, _, NoModrm, { 0, 0, 0} C(NoMemAcc)},
{"fcos", 0,    0xd9ff, _, NoModrm, { 0, 0, 0} C(NoMemAcc)},

{"fchs", 0, 0xd9e0, _, NoModrm, { 0, 0, 0} C(NoMemAcc)},
{"fabs", 0, 0xd9e1, _, NoModrm, { 0, 0, 0} C(NoMemAcc)},

/* processor control */
{"fninit", 0, 0xdbe3, _, NoModrm, { 0, 0, 0} C(NoMemAcc)},
{"finit", 0, 0xdbe3, _, NoModrm, { 0, 0, 0} C(NoMemAcc)},
{"fldcw", 1, 0xd9, 5, Modrm, { Mem, 0, 0} C(WF32bOp)},
{"fnstcw", 1, 0xd9, 7, Modrm, { Mem, 0, 0} C(WF32bOp)},
{"fstcw", 1, 0xd9, 7, Modrm, { Mem, 0, 0} C(WF32bOp)},
{"fnstsw", 1, 0xdfe0, _, NoModrm, { Acc, 0, 0} C(NoMemAcc)},
{"fnstsw", 1, 0xdd, 7, Modrm, { Mem, 0, 0} C(WF32bOp)},
{"fnstsw", 0, 0xdfe0, _, NoModrm, { 0, 0, 0} C(NoMemAcc)},
{"fstsw", 1, 0xdfe0, _, NoModrm, { Acc, 0, 0} C(NoMemAcc)},
{"fstsw", 1, 0xdd, 7, Modrm, { Mem, 0, 0} C(WF32bOp)},
{"fstsw", 0, 0xdfe0, _, NoModrm, { 0, 0, 0} C(NoMemAcc)},
{"fnclex", 0, 0xdbe2, _, NoModrm, { 0, 0, 0} C(NoMemAcc)},
{"fclex", 0, 0xdbe2, _, NoModrm, { 0, 0, 0} C(NoMemAcc)},
/*
 We ignore the short format (287) versions of fstenv/fldenv & fsave/frstor
 instructions;  i'm not sure how to add them or how they are different.
 My 386/387 book offers no details about this.
*/ /*TG: FIXME */
{"fnstenv", 1, 0xd9, 6, Modrm, { Mem, 0, 0} C(WF80bOp)},
{"fstenv", 1, 0xd9, 6, Modrm, { Mem, 0, 0} C(WF80bOp)},
{"fldenv", 1, 0xd9, 4, Modrm, { Mem, 0, 0} C(WF80bOp)},
{"fnsave", 1, 0xdd, 6, Modrm, { Mem, 0, 0} C(WF80bOp)},
{"fsave", 1, 0xdd, 6, Modrm, { Mem, 0, 0} C(WF80bOp)},
{"frstor", 1, 0xdd, 4, Modrm, { Mem, 0, 0} C(WF80bOp)},

{"ffree", 1, 0xddc0, _, ShortForm, { FloatReg, 0, 0} C(NoMemAcc)},
{"fnop", 0, 0xd9d0, _, NoModrm, { 0, 0, 0} C(NoMemAcc)},
{"fwait", 0, 0x9b, _, NoModrm, { 0, 0, 0} C(NoMemAcc)},

/*
  opcode prefixes; we allow them as seperate insns too
  (see prefix table below)
*/
{"aword", 0, 0x67, _, NoModrm, { 0, 0, 0} C(NoMemAcc)},
{"addr16", 0, 0x67, _, NoModrm, { 0, 0, 0} C(NoMemAcc)},
{"word", 0, 0x66, _, NoModrm, { 0, 0, 0} C(NoMemAcc)},
{"data16", 0, 0x66, _, NoModrm, { 0, 0, 0} C(NoMemAcc)},
{"lock", 0, 0xf0, _, NoModrm, { 0, 0, 0} C(NoMemAcc)},
{"cs", 0, 0x2e, _, NoModrm, { 0, 0, 0} C(NoMemAcc)},
{"ds", 0, 0x3e, _, NoModrm, { 0, 0, 0} C(NoMemAcc)},
{"es", 0, 0x26, _, NoModrm, { 0, 0, 0} C(NoMemAcc)},
{"fs", 0, 0x64, _, NoModrm, { 0, 0, 0} C(NoMemAcc)},
{"gs", 0, 0x65, _, NoModrm, { 0, 0, 0} C(NoMemAcc)},
{"ss", 0, 0x36, _, NoModrm, { 0, 0, 0} C(NoMemAcc)},
{"rep", 0, 0xf3, _, NoModrm, { 0, 0, 0} C(StrOp)},
{"repe", 0, 0xf3, _, NoModrm, { 0, 0, 0} C(StrOp)},
{"repz", 0, 0xf3, _, NoModrm, { 0, 0, 0} C(StrOp)},
{"repne", 0, 0xf2, _, NoModrm, { 0, 0, 0} C(StrOp)},
{"repnz", 0, 0xf2, _, NoModrm, { 0, 0, 0} C(StrOp)},

/* 486 extensions */

{"bswap", 1, 0x0fc8, _, ShortForm, { Reg32,0,0 } C(NoMemAcc)},
{"xadd", 2, 0x0fc0, _, DW|Modrm, { Reg, Reg|Mem, 0 } C(RAndWMem)},
{"cmpxchg", 2, 0x0fb0, _, DW|Modrm, { Reg, Reg|Mem, 0 } C(RAndWMem)},
{"invd", 0, 0x0f08, _, NoModrm, { 0, 0, 0} C(NoMemAcc)},
{"wbinvd", 0, 0x0f09, _, NoModrm, { 0, 0, 0} C(NoMemAcc)},
{"invlpg", 1, 0x0f01, 7, Modrm, { Mem, 0, 0} C(NoMemAcc)},

{"", 0, 0, 0, 0, { 0, 0, 0} C(NoMemAcc)}	/* sentinal */
};
#undef _

static const template *const i386_optab_end
  = i386_optab + sizeof (i386_optab)/sizeof(i386_optab[0]);

/* 386 register table */

static const reg_entry i386_regtab[] = {
  /* 8 bit regs */
  {"al", Reg8|Acc, 0}, {"cl", Reg8|ShiftCount, 1}, {"dl", Reg8, 2},
  {"bl", Reg8, 3},
  {"ah", Reg8, 4}, {"ch", Reg8, 5}, {"dh", Reg8, 6}, {"bh", Reg8, 7},
  /* 16 bit regs */
  {"ax", Reg16|Acc, 0}, {"cx", Reg16, 1}, {"dx", Reg16|InOutPortReg, 2}, {"bx", Reg16, 3},
  {"sp", Reg16, 4}, {"bp", Reg16, 5}, {"si", Reg16, 6}, {"di", Reg16, 7},
  /* 32 bit regs */
  {"eax", Reg32|Acc, 0}, {"ecx", Reg32, 1}, {"edx", Reg32, 2}, {"ebx", Reg32, 3},
  {"esp", Reg32, 4}, {"ebp", Reg32, 5}, {"esi", Reg32, 6}, {"edi", Reg32, 7},
  /* segment registers */
  {"es", SReg2, 0}, {"cs", SReg2, 1}, {"ss", SReg2, 2},
  {"ds", SReg2, 3}, {"fs", SReg3, 4}, {"gs", SReg3, 5},
  /* control registers */
  {"cr0", Control, 0},   {"cr2", Control, 2},   {"cr3", Control, 3},
  /* debug registers */
  {"db0", Debug, 0},   {"db1", Debug, 1},   {"db2", Debug, 2},
  {"db3", Debug, 3},   {"db6", Debug, 6},   {"db7", Debug, 7},
  /* test registers */
  {"tr6", Test, 6}, {"tr7", Test, 7},
  /* float registers */
  {"st(0)", FloatReg|FloatAcc, 0},
  {"st", FloatReg|FloatAcc, 0},
  {"st(1)", FloatReg, 1}, {"st(2)", FloatReg, 2}, 
  {"st(3)", FloatReg, 3}, {"st(4)", FloatReg, 4}, {"st(5)", FloatReg, 5}, 
  {"st(6)", FloatReg, 6}, {"st(7)", FloatReg, 7}
};

#define MAX_REG_NAME_SIZE 8	/* for parsing register names from input */

static const reg_entry *const i386_regtab_end
  = i386_regtab + sizeof(i386_regtab)/sizeof(i386_regtab[0]);

/* segment stuff */
static const seg_entry cs = { "cs", 0x2e };
static const seg_entry ds = { "ds", 0x3e };
static const seg_entry ss = { "ss", 0x36 };
static const seg_entry es = { "es", 0x26 };
static const seg_entry fs = { "fs", 0x64 };
static const seg_entry gs = { "gs", 0x65 };
static const seg_entry null = { "", 0x0 };

/*
  This table is used to store the default segment register implied by all
  possible memory addressing modes.
  It is indexed by the mode & modrm entries of the modrm byte as follows:
      index = (mode<<3) | modrm;
*/
static const seg_entry *const one_byte_segment_defaults[] = {
  /* mode 0 */
  &ds, &ds, &ds, &ds, &null, &ds, &ds, &ds,
  /* mode 1 */
  &ds, &ds, &ds, &ds, &null, &ss, &ds, &ds,
  /* mode 2 */
  &ds, &ds, &ds, &ds, &null, &ss, &ds, &ds,
  /* mode 3 --- not a memory reference; never referenced */
};

static const seg_entry *const two_byte_segment_defaults[] = {
  /* mode 0 */
  &ds, &ds, &ds, &ds, &ss, &ds, &ds, &ds,
  /* mode 1 */
  &ds, &ds, &ds, &ds, &ss, &ds, &ds, &ds,
  /* mode 2 */
  &ds, &ds, &ds, &ds, &ss, &ds, &ds, &ds,
  /* mode 3 --- not a memory reference; never referenced */
};

static const prefix_entry i386_prefixtab[] = {
  { "addr16", 0x67 },		/* address size prefix ==> 16bit addressing
				 * (How is this useful?) */
#define WORD_PREFIX_OPCODE 0x66
  { "data16", 0x66 },		/* operand size prefix */
  { "lock", 0xf0 },		/* bus lock prefix */
  { "wait", 0x9b },		/* wait for coprocessor */
  { "cs", 0x2e }, { "ds", 0x3e }, /* segment overrides ... */
  { "es", 0x26 }, { "fs", 0x64 },
  { "gs", 0x65 }, { "ss", 0x36 },
/* REPE & REPNE used to detect rep/repne with a non-string instruction */
#define REPNE 0xf2
#define REPE  0xf3
  { "rep", 0xf3 }, 		/* repeat string instructions */
  { "repe", 0xf3 },  { "repz", 0xf3 },
  { "repne", 0xf2 }, { "repnz", 0xf2 }
};

static const prefix_entry *const i386_prefixtab_end
  = i386_prefixtab + sizeof(i386_prefixtab)/sizeof(i386_prefixtab[0]);

/* end of i386-opcode.h */
