/*********************************************************************
 * PLP test suite: libplp test program.
 */

/* these are not likely to cause trouble:
 *   nobody is in its own group.
 *   here's hoping nobody in whitehouse.gov tries this... ;)
 */
#define INVALID_USER	"nobody"
#define INVALID_HOST	"whitehouse.gov"

/*********************************************************************/
#include "lp.h"

static char HOST[BUFSIZ], DOMAIN[BUFSIZ], MIXED_CASE_HOST[BUFSIZ],
	HOST_IPADDR[BUFSIZ];

plp_signal_t cleanup (int sig) { ; }

extern char *find_executable (char *);
extern int cmd_split (char *, char **);
extern char *prepend_testdir (char *);

char *append_domain (char *str) {
    char *ret;
    if (strstr (str, DOMAIN) != NULL) {
	ret = str;
    } else {
	malloc_or_die (ret, BUFSIZ);
	(void) sprintf (ret, "%s.%s", str, DOMAIN);
    }
    return ret;
}

char *join_list (char **list, char *joinstr) {
    static char joined[BUFSIZ];
    char *j, *ej;

    j = joined; ej = j + sizeof (joined);

    while (*list != NULL) {
	j = estrcp (j, *list, ej);
	j = estrcp (j, joinstr, ej);
	list++;
    }
    if (j > joined) {
	*(j-1) = '\0';
    }
    return (joined);
}

/* Test general library functions (where possible) */

void do_test_general (void) {
    int argno; char *args[BUFSIZ];
    char *if_path;

    {
        do_test ("find_executable with full path",
                !strcmp (find_executable ("/bin/cat"), "/bin/cat"));

        if_path = prepend_testdir ("src/filt/if");
        do_test ("find_executable with filter-path",
                !strcmp (find_executable ("if"), if_path));

        do_test ("find_executable (inverse)",
                !strcmp (find_executable ("_not_there_"), "_not_there_"));
    }

    {
	char from[] = "TEST", to1[BUFSIZ], to2[2];
	do_test ("estrcp 1",
		estrcp (to1, from, to1 + sizeof(to1)) == to1 + strlen (from));
	do_test ("estrcp 2",
		estrcp (to2, from, to2 + sizeof(to2)) == NULL);
    }

    {
	argno = cmd_split ("this is 'a test'", args);
	do_test ("cmd_split 1",
		!strcmp (join_list (args, "#"), "this#is#a test"));

	argno = cmd_split ("foo bar 'baz \"go\"o' baz", args);
	do_test ("cmd_split 2",
		!strcmp (join_list (args, "#"), "foo#bar#baz \"go\"o#baz"));

	argno = cmd_split ("this' is' a test", args);
	do_test ("cmd_split 3",
		!strcmp (join_list (args, "#"), "this is#a#test"));

	argno = cmd_split ("this \"is 'a\" test", args);
	do_test ("cmd_split 4",
		!strcmp (join_list (args, "#"), "this#is 'a#test"));

	argno = cmd_split ("this 'is \\'a' test", args);
	do_test ("cmd_split 5",
		!strcmp (join_list (args, "#"), "this#is 'a#test"));

	{
	    int cpy;
	    cpy = dup (2); (void) close (2); (void) open ("/dev/null", O_WRONLY);
	    /* 2>& /dev/null */

	    argno = cmd_split ("this is a; test", args);
	    do_test ("cmd_split inverse 1", argno < 0);

	    argno = cmd_split ("this is a test > /etc/crack", args);
	    do_test ("cmd_split inverse 2", argno < 0);

	    (void) close (2); (void) dup (cpy);		/* 2>& stderr */
	}
    }

    {
	do_test ("hostcmp basic", !hostcmp (HOST, HOST));
	do_test ("hostcmp fqdn", !hostcmp (HOST, append_domain (HOST)));
	do_test ("hostcmp case", !hostcmp (MIXED_CASE_HOST, HOST));
	do_test ("hostcmp (inverse)", hostcmp (HOST, INVALID_HOST));
    }

    {
	do_test ("host_in_domain local", host_in_domain (HOST, DOMAIN));
	do_test ("host_in_domain (inverse)",
		(host_in_domain (INVALID_HOST, DOMAIN)) == NULL);
    }
}

void test_printer (char *pr) {
    char label[BUFSIZ], correct[BUFSIZ];
    int retval;

    retval = Get_pc_entry (pr, All_pc_vars, All_pc_len);
    (void) sprintf (label, "printcap lookup (%s)", pr);
    (void) sprintf (correct, "/%s", pr);
    do_test (label, retval && !strcmp (LP, correct));
}

void do_test_printcap (void) {
    int retval;
    char *allp;

    allp = join_list (All_printers(), ",");
    do_test ("All_printers", !strcmp (allp, 
"test1,test2,test3,test4,sub_5,test5,test6,test7,test8,test9,test10,test11"));

    do_test ("First_printer", !strcmp (First_printer(), "test1"));

    test_printer ("test1");

    {
	retval = Get_pc_entry ("test2", All_pc_vars, All_pc_len);
	do_test ("printcap lookup (test2)", retval && !strcmp (LP, "/test2") &&
		!strcmp (All_names, "test2|test_aliases|foo|bar|squidge"));
    }

    test_printer ("test3"); test_printer ("test4");
    test_printer ("test5"); test_printer ("test6");
    test_printer ("test7"); test_printer ("test8");

    {
	char *correct;
	correct = prepend_testdir ("spool/OUT");
	retval = Get_pc_entry ("test9", All_pc_vars, All_pc_len);
	do_test ("printcap lookup (test9)", retval && !strcmp (LP, correct));
    }

    test_printer ("test10"); test_printer ("test11");

    {
	retval = Get_multi_pc_entry ("test10", "test11", All_pc_vars, All_pc_len);

	do_test ("double-printcap lookup (test10/test11)", retval &&
		 !strcmp (LP, "/test10") && !strcmp (OF, "/usr/lib/ps/psof"));
    }

    retval = Get_pc_entry ("squidge", All_pc_vars, All_pc_len);
    do_test ("printcap aliases (left)", retval && !strcmp (LP, "/test2"));

    retval = Get_pc_entry ("foo", All_pc_vars, All_pc_len);
    do_test ("printcap aliases (right)", retval && !strcmp (LP, "/test2"));

    retval = Get_pc_entry ("test8", All_pc_vars, All_pc_len);
    do_test ("hash in printcap file", retval && (MX == 1234));

    retval = Get_pc_entry ("test10", All_pc_vars, All_pc_len);
    do_test ("escaped hash in default-printcap", retval && (PW == 80));
}

void do_test_perms (void) {
    int perms = 'R';

    /* check hostname matching with domain */
    do_test ("perms (basic)",
    Checkperm (append_domain (HOST), Daemon_user, "t_perms_1", &perms, 0, 0) > 0);

    do_test ("perms (basic) (inverse)",
    Checkperm (INVALID_HOST, Daemon_user, "t_perms_1", &perms, 0, 0) <= 0);

    /* check hostname matching without domain */
    do_test ("perms without domain",
    Checkperm (HOST, Daemon_user, "t_perms_1", &perms, 0, 0) > 0);

    do_test ("perms without domain (inverse)",
    Checkperm ("not_a_real_host", Daemon_user, "t_perms_1", &perms, 0, 0) <= 0);

    /* check case insensitivity */
    do_test ("perms with funny case",
    Checkperm (MIXED_CASE_HOST, Daemon_user, "t_perms_1", &perms, 0, 0) > 0);

    /* check matching on username */
    do_test ("perms with username",
    Checkperm (append_domain (HOST), Daemon_user, "t_perms_1", &perms, 0, 0) > 0);

    do_test ("perms with username (inverse)",
    Checkperm (append_domain (HOST), INVALID_USER, "t_perms_1", &perms, 0, 0) <= 0);

    /* check matching on groupname */
    do_test ("perms with group",
    Checkperm (append_domain (HOST), Daemon_user, "t_perms_2", &perms, 0, 0) > 0);

    do_test ("perms with group (inverse)",
    Checkperm (append_domain (HOST), INVALID_USER, "t_perms_2", &perms, 0, 0) <= 0);

    /* check wildcard matching */
    do_test ("perms wildcards (user)",
    Checkperm (append_domain (HOST), Daemon_user, "t_perms_3_u", &perms, 0, 0) > 0);

    do_test ("perms wildcards (user) (inverse)",
    Checkperm (append_domain (HOST), INVALID_USER, "t_perms_3_u", &perms, 0, 0) <= 0);

    do_test ("perms wildcards (group)",
    Checkperm (append_domain (HOST), Daemon_user, "t_perms_3_g", &perms, 0, 0) > 0);

    do_test ("perms wildcards (group) (inverse)",
    Checkperm (append_domain (HOST), INVALID_USER, "t_perms_3_g", &perms, 0, 0) <= 0);

    do_test ("perms wildcards (queue)",
    Checkperm (append_domain (HOST), Daemon_user, "t_perms_3_q_foobar", &perms, 0, 0) > 0);

    do_test ("perms wildcards (queue) (inverse)",
    Checkperm (append_domain (HOST), INVALID_USER, "t_perms_3_q_foobar", &perms, 0, 0) <= 0);

    do_test ("perms without host",
    Checkperm (NULL, Daemon_user, "t_perms_1", &perms, 0, 0) > 0);

    do_test ("perms without username",
    Checkperm (HOST, NULL, "t_perms_1", &perms, 0, 0) > 0);

    do_test ("perms with ip-address (in args)",
    Checkperm (HOST_IPADDR, Daemon_user, "t_perms_1", &perms, 0, 0) > 0);

    do_test ("perms with ip-address (in permfile)",
    Checkperm (HOST, Daemon_user, "t_perms_4", &perms, 0, 0) > 0);

    do_test ("perms with ip-address (in both)",
    Checkperm (HOST_IPADDR, Daemon_user, "t_perms_4", &perms, 0, 0) > 0);

    do_test ("perms with ip-address (inverse)",
    Checkperm (INVALID_HOST, Daemon_user, "t_perms_4", &perms, 0, 0) <= 0);

    do_test ("perms with ip-address (null arg)",
    Checkperm (NULL, Daemon_user, "t_perms_4", &perms, 0, 0) > 0);
}

void
do_test_waitpid (void) {
    int pid, pid2, result, result2;
    plp_status_t stb;

    {
	if ((pid = fork ()) != 0) {
	    result = plp_waitpid (pid, &stb, WUNTRACED);
	} else {
	    sleep (1);
	    exit (0);
	}

	do_test ("plp_waitpid(pid,WUNTRACED)", (result >= 0) && 
	    PLP_WIFEXITED(stb) && (PLP_WEXITSTATUS(stb) == 0));
    }

    {
	if ((pid = fork ()) != 0) {
	    result = plp_waitpid (-1, &stb, WUNTRACED);
	} else {
	    sleep (1);
	    exit (0);
	}

	do_test ("plp_waitpid(-1,WUNTRACED)", (result >= 0) && 
	    PLP_WIFEXITED(stb) && (PLP_WEXITSTATUS(stb) == 0));
    }

    {
	if ((pid = fork ()) != 0) {
	    result = plp_waitpid (pid, &stb, WNOHANG);
	    sleep (2);
	    result2 = plp_waitpid (pid, &stb, WNOHANG);
	} else {
	    sleep (1);
	    exit (0);
	}

	do_test ("plp_waitpid(pid,WNOHANG)",
	    (result == 0) && (result2 >= 0) &&
	    PLP_WIFEXITED(stb) && (PLP_WEXITSTATUS(stb) == 0));
    }

    {
	if ((pid = fork ()) != 0) {
	    result = plp_waitpid (-1, &stb, WNOHANG);
	    sleep (2);
	    result2 = plp_waitpid (-1, &stb, WNOHANG);
	} else {
	    sleep (1);
	    exit (0);
	}

	do_test ("plp_waitpid(-1,WNOHANG)",
	    (result == 0) && (result2 >= 0) &&
	    PLP_WIFEXITED(stb) && (PLP_WEXITSTATUS(stb) == 0));
    }

    {
	if ((pid = fork ()) != 0) {
	    sleep (2);
	    result = plp_waitpid (pid + 1, &stb, WNOHANG);
	    result2 = plp_waitpid (pid, &stb, WNOHANG);
	} else {
	    sleep (1);
	    exit (0);
	}

	do_test ("plp_waitpid saved buffer",
	    (result <= 0) && (result2 >= 0) &&
	    PLP_WIFEXITED(stb) && (PLP_WEXITSTATUS(stb) == 0));
    }

    {
	if ((pid = fork ()) != 0) {
	    sleep (2);
	    if ((pid2 = fork ()) == 0) {
		exit (1);
	    }
	} else {
	    sleep (1);
	    exit (2);
	}

	result2 = plp_waitpid (pid2, &stb, WNOHANG);
	result = plp_waitpid (pid, &stb, WNOHANG);

	do_test ("plp_waitpid reverse order",
	    (result == pid) && (result2 == pid2) &&
	    PLP_WIFEXITED(stb) && (PLP_WEXITSTATUS(stb) == 2));
    }
}

void fix_conf (void) {
    Readconf();
    free (Permfile_path); Permfile_path = prepend_testdir ("perms.tst");
    free (Printcap_path); Printcap_path = prepend_testdir ("pcap.tst");
    free (Printcap_include_path); Printcap_include_path = prepend_testdir ("src");
    create_pc_cache ();
    create_perm_cache ();
    free (Filter_path); Filter_path = prepend_testdir ("src/filt");
}

void Get_args (int argc, char **argv) {
    int i;

    if (argc == 6) {
	Parse_debug (argv[1]);
	i = 2;
    } else if (argc != 5) {
	printf
("usage: testlib [debuglevel] {host} {domain} {mixedcasehost} {hostipaddr}\n");
	exit (1);
    } else {
	i = 1;
    }

    (void) strcpy (HOST, argv[i]); i++;
    (void) strcpy (DOMAIN, argv[i]); i++;
    (void) strcpy (MIXED_CASE_HOST, argv[i]); i++;
    (void) strcpy (HOST_IPADDR, argv[i]);
}

int
main (int argc, char **argv, char **envp) {
    Get_args (argc, argv);

    fix_conf ();
    Std_environ (argc, argv, envp);

    printf ("[testing general library functions]\n");
    do_test_general ();

    printf ("[printcap: cache off]\n");	Use_printcap_cache = 0; do_test_printcap ();
    printf ("[perms: cache off]\n");	Use_printcap_cache = 0; do_test_perms ();

    printf ("[printcap: cache on]\n");	Use_printcap_cache = 1; do_test_printcap ();
    printf ("[perms: cache on]\n");	Use_printcap_cache = 1; do_test_perms ();

    fix_conf ();

    printf ("[printcap: reloaded]\n");	Use_printcap_cache = 1; do_test_printcap ();
    printf ("[perms: reloaded]\n");	Use_printcap_cache = 1; do_test_perms ();

    printf ("[testing plp_waitpid]\n");
    do_test_waitpid ();

    return 0;
}
