/*
 * readconf.c: read the config file.
 * 
 * The tokenizing code is ripped off xntpd 3.3, which is copyright (c) David L. Mills 1992,
 * 1993, 1994. I am so lazy. ;)
 * 
 * - Justin Mason <jmason@iona.ie> July '94.
 */

#include "lp.h"
#include "library/defaults.h"
#include "library/errormsg.h"
#include "library/filestack.h"
#include "library/printcap_file.h"
#include "library/checkperm.h"
#include "library/readconf.h"
#include "library/utils.h"

/*
 * Translation table - keywords to function index
 */
struct keyword {
    char *text;
    int keytype;
};

#include "library/confkeys.h"

/*
 * Limits on things
 */
#define MAXTOKENS       20	/* 20 tokens on line */
#define MAXLINE         1024	/* maximum length of line */
#define MAXFILENAME     511	/* maximum length of a file name (alloca()?) */

/*
 * Miscellaneous macros
 */
#define ISEOL(c)                ((c) == '#' || (c) == '\n' || (c) == '\0')
#define ISSPACE(c)              ((c) == ' ' || (c) == '\t')

static int gettokens (FILE *, char *, char **, int *);
static int matchkey (char *, struct keyword *);
static void Expand_paths (void);
  
static void
delete_confs (void)
{
    while (num_confs > 0) {
	num_confs--;
	if (conf[num_confs].domain) {
	    (void) free (conf[num_confs].domain);
	    conf[num_confs].domain = NULL;
	}
    }
}

static char *
recombine_tokens (int starti, int endi, char **toks) {
    int i, len;
    size_t sofar, cursize;
    char *str;

    sofar = 0;
    cursize = 1024;
    str = (char *) malloc (cursize);

    for (i = starti; i <= endi; i++) {
	if (DbgConf > 8)
	    log (XLOG_DEBUG, "recombining: adding '%s' at %d", toks[i], sofar);
	while ((len = strlen (toks[i])) + sofar > cursize) {
	    cursize *= 2;
	    str = (char *) realloc (str, cursize);
	}
	(void) strncpy (&(str[sofar]), toks[i], len);
	sofar += len + 1;
	str[sofar - 1] = ' ';
	str[sofar] = '\0';
    }
    str[sofar - 1] = '\0';	/* trim last space */
    if (DbgConf > 7)
	log (XLOG_DEBUG, "recombined: '%s'", str);
    return (str);
}

/*
 * add_domains: eg. add_domains(hesiod, 3, {engineering.iona.ie, iona.ie, ie) makes...
 * 
 * old: files hesiod nis
 * 
 * new: files hesiod:engineering.iona.ie hesiod:iona.ie hesiod:ie nis
 */

#if defined(HESIOD) || defined(NIS)
static void
add_domains (char type, int ndomains, char **domains) {
    int i, j, found, new_num_confs;

    ndomains--;			/* we should already have one of this type. */
    found = 0;

    for (i = num_confs - 1; i >= 0; i--) {
	if (conf[i].type == type)
	    found = i;
    }

    if (found) {
	new_num_confs = num_confs + ndomains;

	/* move the trailing entries down */
	i = num_confs - 1;
	j = new_num_confs - 1;
	while (i > found) {
	    conf[j].type = conf[i].type;
	    conf[j].domain = conf[i].domain;
	    i--;
	    j--;
	}

	/* add the new entries */
	i = found;
	j = 0;
	while (j <= ndomains) {
	    conf[i].domain = (char *) NULL;
	    conf[i].type = type;
	    allocandcpy (conf[i].domain, domains[j]);
	    i++;
	    j++;
	}
    } else {
	fprintf (stderr, "%s: add_domains: type %d is not used by 'order' keyword.\n",
		 ConfigFile, type);
    }
    num_confs = new_num_confs;
}

#endif

static void
Pre_readconf (void) {
    delete_confs ();
    Defaults ();
}

static void
Post_readconf (void) {
    Get_Daemon ();
    Get_Local_fqdn ();
    Expand_paths ();
    Setuid_debug ("Post_readconf");
#ifdef LOCAL
    create_pc_cache ();		/* read the printcap file */
    create_perm_cache ();	/* read the main perms file */
#endif
}

void
Readconf (void) {
    FILE *fp;
    int ntokens, tok, i;
    char line[MAXLINE];
    char *(tokens[MAXTOKENS]);

    Pre_readconf ();
    if (ConfigFile == NULL) {
	ConfigFile = CONFIG_FILE;
    }
    if (DbgConf > 3)
	log (XLOG_DEBUG, "using plp.conf file '%s'", ConfigFile);
    if ((fp = fopen (ConfigFile, "r")) == NULL) {
	logerr_die (XLOG_INFO, "failed to open configuration file '%s'\n", ConfigFile);
    }
    while ((tok = gettokens (fp, line, tokens, &ntokens)) != CF_UNKNOWN) {

	if (tok == CF_PRIVPORTS) {
	    for (i = 1; i < ntokens; i++) {
		if (!strcmp (tokens[i], "allow")) {
		    AllowNonPrivPorts = 1;
		} else if (!strcmp (tokens[i], "deny")) {
		    AllowNonPrivPorts = 0;
		} else if (!strcmp (tokens[i], "quiet")) {
		    LogNonPrivPorts = 0;
		} else if (!strcmp (tokens[i], "log")) {
		    LogNonPrivPorts = 1;
		} else {
		    fprintf (stderr, "%s: bad usage for '%s', arg '%s' ignored.\n",
			     ConfigFile, tokens[0], tokens[i]);
		}
	    }
	    continue;
	}
	if (tok == CF_LOGREQUESTS) {
	    Log_no_filenames = Log_LPRMs = Log_LPQs = Log_LPCs = Log_LPRs = 0;
	    for (i = 1; i < ntokens; i++) {
		if (!strcmp (tokens[i], "lprm")) {
		    Log_LPRMs = 1;
		} else if (!strcmp (tokens[i], "lpq")) {
		    Log_LPQs = 1;
		} else if (!strcmp (tokens[i], "lpc")) {
		    Log_LPCs = 1;
		} else if (!strcmp (tokens[i], "lpr")) {
		    Log_LPRs = 1;
		} else if (!strcmp (tokens[i], "private")) {
		    Log_no_filenames = 1;
		} else {
		    fprintf (stderr, "%s: bad usage for '%s', arg '%s' ignored.\n",
			     ConfigFile, tokens[0], tokens[i]);
		}
	    }
	    continue;
	}
	if (tok == CF_PCAPORDER) {
	    delete_confs ();
	    for (i = 1; i < ntokens; i++) {
#ifdef LOCAL
		if (!strcmp (tokens[i], "files")) {
		    conf[num_confs].type = LOCAL_TYPE;
		    num_confs++;
		    continue;
		}
#endif
#ifdef NIS
		if (!strcmp (tokens[i], "nis")) {
		    conf[num_confs].type = NIS_TYPE;
		    num_confs++;
		    continue;
		}
#endif
#ifdef HESIOD
		if (!strcmp (tokens[i], "hesiod")) {
		    conf[num_confs].type = HESIOD_TYPE;
		    num_confs++;
		    continue;
		}
#endif
		fprintf (stderr, "%s: bad order keyword: '%s', ignored.\n",
			 ConfigFile, tokens[i]);
	    }
	    continue;
	}
	if (tok == CF_HESDOMAINS) {
#ifdef HESIOD
	    add_domains (HESIOD_TYPE, ntokens - 1, &(tokens[1]));
#endif
	    continue;
	}
	if (tok == CF_NISDOMAINS) {
#ifdef NIS
	    add_domains (NIS_TYPE, ntokens - 1, &(tokens[1]));
#endif
	    continue;
	}
	if (tok == CF_ERRORSFILE) {
	    if (tokens[1][0] == 'y') {
		if (ntokens > 2) {
		    allocandcpy (Errors_file, tokens[2]);
		}		/* else stick with the default */
	    } else
		*Errors_file = 0;
	    continue;
	}
	if (tok == CF_FORMSFILE) {
#ifdef EUCS_ZOPTIONS
	    if (tokens[1][0] == 'y') {
		if (ntokens > 2) {
		    allocandcpy (Forms_file, tokens[2]);
		}		/* else stick with the default */
	    } else
		*Forms_file = 0;
#endif
	    continue;
	}
	if (tok == CF_SENDMAIL) {
	    if (tokens[1][0] == 'y') {
		if (ntokens > 2) {
		    char *s = recombine_tokens (2, ntokens - 1, tokens);
		    allocandcpy (Mail_command, s);
		    free (s);
		}		/* else stick with the default */
	    } else
		*Mail_command = 0;
	    continue;
	}
	if (tok == CF_CTRLLOGDIR) {
	    if (tokens[1][0] == 'y') {
		if (ntokens > 2) {
		    allocandcpy (Ctrlfile_logdir, tokens[2]);
		}		/* else stick with the default */
	    } else
		*Ctrlfile_logdir = 0;
	    continue;
	}
	switch (ntokens) {

	    /*************************************************************
	     * TWO ARGUMENTS:
	     */
	case 2:
	    switch (tok) {
	    case CF_LPDLOG:
		allocandcpy (Lpdlogf, tokens[1]);
		break;

	    case CF_LPDLOCK:
		allocandcpy (Masterlock, tokens[1]);
		break;

	    case CF_USER:
		allocandcpy (Daemon_user, tokens[1]);
		break;

	    case CF_DEFAULTPRINTCAP:
		allocandcpy (DefaultPrintcap, tokens[1]);
		break;

	    case CF_PRINTCAP:
#ifdef LOCAL
		allocandcpy (Printcap_path, tokens[1]);
		break;
#endif

	    case CF_PPERMS:
#ifdef LOCAL
		allocandcpy (Permfile_path, tokens[1]);
		break;
#endif

	    case CF_ZOPTIONS:
#ifdef EUCS_ZOPTIONS
		allocandcpy (Zoptions_path, tokens[1]);
		break;
#endif

	    case CF_COSTCODES:
#ifdef EUCS_ZOPTIONS
		allocandcpy (Costcode_path, tokens[1]);
		break;
#endif

	    case CF_HESPCAPKEY:
#ifdef HESIOD
		allocandcpy (Hesiod_printcap_key, tokens[1]);
		break;
#endif

	    case CF_HESCOSTCODEKEY:
#ifdef HESIOD
		allocandcpy (Hesiod_costcode_key, tokens[1]);
		break;
#endif

	    case CF_PRINTERPORT:
		allocandcpy (Printer_port, tokens[1]);
		break;

	    case CF_MAILFORMAT:
		allocandcpy (Mail_addressing, tokens[1]);
		break;

	    case CF_FROMADDRESS:
		allocandcpy (Mail_from_address, tokens[1]);
		break;

	    case CF_HOSTDOMAIN:
		allocandcpy (Host_domain, tokens[1]);
		break;

	    case CF_USEPCAPCACHE:
		if (tokens[1][0] == 'y') {
		    Use_printcap_cache = 1;
		} else {
		    Use_printcap_cache = 0;
		}
		break;

	    case CF_USEPERMCACHE:
		if (tokens[1][0] == 'y') {
		    Use_perm_cache = 1;
		} else {
		    Use_perm_cache = 0;
		}
		break;

	    case CF_CHECKFORBINARY:
		if (tokens[1][0] == 'y') {
		    Check_for_nonprintable = 1;
		} else {
		    Check_for_nonprintable = 0;
		}
		break;

	    case CF_MAILONERR:
		if (tokens[1][0] == 'y') {
		    Mail_on_errors = 1;
		} else {
		    Mail_on_errors = 0;
		}
		break;

	    case CF_RESTRICTSYMLINKS:
		if (tokens[1][0] == 'y') {
		    Restrict_symlinks = 1;
		} else {
		    Restrict_symlinks = 0;
		}
		break;

	    case CF_SHOWEMPTYQUEUES:
		if (tokens[1][0] == 'y') {
		    Show_empty_queues = 1;
		} else {
		    Show_empty_queues = 0;
		}
		break;

	    case CF_BANNERFMT:
#ifdef EUCS_BANNER
		if (tokens[1][0] == 'y') {
		    EUCS_banner_format = 1;
		} else {
		    EUCS_banner_format = 0;
		}
#endif
		break;

	    case CF_FILTERPATH:
		allocandcpy (Filter_path, tokens[1]);
		break;

	    case CF_FILTERLDPATH:
		allocandcpy (Filter_LD_path, tokens[1]);
		break;

	    case CF_ARCHITECTURE:
		allocandcpy (Architecture, tokens[1]);
		break;

	    case CF_PRINTCAPINCLUDEPATH:
		allocandcpy (Printcap_include_path, tokens[1]);
		break;

	    case CF_CONFIGINCLUDEPATH:
		allocandcpy (Config_include_path, tokens[1]);
		break;

	    case CF_PERMFILEINCLUDEPATH:
		allocandcpy (Permfile_include_path, tokens[1]);
		break;

	    default:
		fprintf (stderr, "%s: bad usage for '%s', line ignored.\n",
			 ConfigFile, tokens[0]);
	    }
	    break;

	    /************************************************************
	     * THREE ARGUMENTS:
	     */
	case 3:
	    switch (tok) {
	    case CF_NISPRINTCAP:
#ifdef NIS
		allocandcpy (NIS_printcap_bykey, tokens[1]);
		allocandcpy (NIS_printcap_byname, tokens[2]);
		break;
#endif

	    case CF_NISCOSTCODE:
#ifdef NIS
		allocandcpy (NIS_costcode_bykey, tokens[1]);
		allocandcpy (NIS_costcode_byname, tokens[2]);
		break;
#endif

	    default:
		fprintf (stderr, "%s: bad usage for '%s', line ignored.\n",
			 ConfigFile, tokens[0]);
	    }
	    break;

	    /************************************************************/
	default:
	    fprintf (stderr, "%s: bad usage for '%s', line ignored.\n",
		     ConfigFile, tokens[0]);
	}
    }
    (void) fclose (fp);
    Post_readconf ();
}

/*
 * gettokens - read a line and return tokens
 */
static int
gettokens (FILE *fp, char *line, char **tokenlist, int *ntokens) {
    register char *cp;
    register int eol;
    register int ntok;
    register int quoted = 0;

    /*
     * Find start of first token
     */
again:
    while ((cp = Gets (line, MAXLINE, &fp, Config_include_path)) != NULL) {
	cp = line;
	while (ISSPACE (*cp))
	    cp++;
	if (!ISEOL (*cp))
	    break;
    }
    if (cp == NULL) {
	*ntokens = 0;
	return CF_UNKNOWN;	/* hack.  Is recognized as EOF */
    }
    /*
     * Now separate out the tokens
     */
    eol = 0;
    ntok = 0;
    while (!eol) {
	tokenlist[ntok++] = cp;

	/* special-case '\#' for default-printcap (kludge!) */
	while (!(ISEOL (*cp) && !(*cp == '#' && *(cp - 1) == '\\'))
	       && (!ISSPACE (*cp) || quoted))
	    quoted ^= (*cp++ == '"');

	if (ISEOL (*cp)) {
	    *cp = '\0';
	    eol = 1;
	} else {		/* must be space */
	    *cp++ = '\0';
	    while (ISSPACE (*cp))
		cp++;
	    if (ISEOL (*cp))
		eol = 1;
	}
	if (ntok == MAXTOKENS)
	    eol = 1;
    }
    /*
     * Return the match
     */
    *ntokens = ntok;
    ntok = matchkey (tokenlist[0], keywords);
    if (ntok == CF_UNKNOWN)
	goto again;
    return ntok;
}

/*
 * matchkey - match a keyword to a list
 */
static int
matchkey (char *word, struct keyword *keys) {
    for (;;) {
	if (keys->keytype == CF_UNKNOWN) {
	    fprintf (stderr, "%s: keyword \"%s\" unknown, line ignored.\n",
		     ConfigFile, word);
	    return CF_UNKNOWN;
	}
	if (strsame (word, keys->text))
	    return keys->keytype;
	keys++;
    }
}

/***************************************************************************
 * char *fname_reallocstrcpy (to, from)
 * strcpy with dynamic allocation and some %-expansions.
 * The "to" buffer is returned, as its address may have changed.
 *
 * KLUDGE ALERT: since this uses the fqdn, but we need `domain-name'
 * before we get the fqdn with get_fqdn(), we have to leave it to
 * be expanded AFTER we expand everything else and finish reading
 * the config file. ARGH ARGH ARGH.
 *
 * anyway, use as follows: to = fname_reallocstrcpy(to, "hello");
 **************************************************************************/

static char *fmtbuf;
static int fmtsiz;

#define expand_confstr(str)	(str = expand_confstr_backend(str))

static char *
expand_confstr_backend (char *str) {
    /* this copies the string into a static buffer, then uses
     * this copy to produce the original string in the original
     * buffer, increasing the buffer's size if necessary.
     */
    char *inp;
    int jindex = 0, arysize, len;
    static int shorthostlen, hostlen, archlen;

    len = strlen (str) + 1;
    if (fmtsiz == 0) {
	/* allocate the buffer */
	fmtsiz = len + HOSTLEN; fmtbuf = (char *) malloc (fmtsiz);

    } else {
	/* check the buffer is big enough */
	if (fmtsiz < len) {
	    /* it isn't, so reallocate it */
	    fmtsiz = len + HOSTLEN; fmtbuf = (char *) realloc (fmtbuf, fmtsiz);
	}
    }
    (void) strcpy (fmtbuf, str);

    /* now, interpret the format. */

    if (DbgConf > 8)
	log (XLOG_DEBUG, "expanding %s", fmtbuf);

    inp = fmtbuf;
    if (str && *str) {
	arysize = strlen (str);
    } else {
	arysize = 128;
    }
    while (*inp) {
	chkandrealloc_str (str, jindex, arysize, 32);
	switch (*inp) {
	case '%':
	    inp++;
	    switch (*inp) {
	    case 'h':
		if (shorthostlen == 0)
		    shorthostlen = strlen (ShortHost);
		jindex += shorthostlen;
		chkandrealloc_str (str, jindex, arysize, 32);
		(void) strcpy (&(str[jindex - shorthostlen]), ShortHost);
		break;

	    case 'H':
		if (DomainNameUnset) {
		    /* oops -- user wants fqdn and we can't work ours out. */
		    log (XLOG_INFO,
		    "can't fully-qualify hostname; set `domain-name' in plp.conf");
		}
		if (hostlen == 0)
		    hostlen = strlen (Host);
		jindex += hostlen;
		chkandrealloc_str (str, jindex, arysize, 32);
		(void) strcpy (&(str[jindex - hostlen]), Host);
		break;

 	    case 'a':
		if (Architecture) {		/* ensure it's non-null. */
		    if (archlen == 0)
			archlen = strlen (Architecture);
		    jindex += archlen;
		    chkandrealloc_str (str, jindex, arysize, 32);
		    (void) strcpy (&(str[jindex - archlen]), Architecture);
		}
		break;

	    default:
		str[jindex++] = '%';
		if (*inp == 's') {
		    fatal (XLOG_INFO, "illegal sequence in plp.conf: %s");
		} else if (*inp == '\0') {
		    goto read_null;
		} else {
		    str[jindex++] = *(inp++);
		}
	    }
	    inp++;
	    break;

	default:
	    str[jindex++] = *(inp++);
	}
    }

read_null:
    str[jindex] = '\0';
    if (DbgConf > 8)
	log (XLOG_DEBUG, "expanded to %s", str);
    return (str);
}

char *
Expand_str (char *str) {
    fmtsiz = 0; fmtbuf = NULL;
    expand_confstr (str);
    fmtsiz = 0; free (fmtbuf);
    return str;
}

static void Expand_paths (void) {
    fmtsiz = 0; fmtbuf = NULL;		/* these all share this buffer */
    expand_confstr (Lpdlogf);
    expand_confstr (Masterlock);
#ifdef LOCAL
    expand_confstr (Permfile_path);
    expand_confstr (Permfile_include_path);
    expand_confstr (Printcap_path);
    expand_confstr (Printcap_include_path);
#endif
    expand_confstr (Filter_path);
    expand_confstr (Filter_LD_path);
    expand_confstr (Errors_file);
    expand_confstr (Attach_file);
#ifdef EUCS_ZOPTIONS
    expand_confstr (Zoptions_path);
    expand_confstr (Costcode_path);
    expand_confstr (Forms_file);
#endif
    fmtsiz = 0; free (fmtbuf);
}
