/* hpnp-lp: HP Net Peripheral LP-pipe.
 *
 * Uses the hpnp software, available from ftp-boi.external.hp.com.
 * Get it, it's worth it!!
 *
 * urgh: we can't use a logfile as the status file, 'cos hpnpf bombs
 * out if we do. So we emulate "tail -f". urgh.
 * 
 * urgh 2: hpnpf will NOT work if you send it the data before
 * it's connected to the printer, so we have to wait until it tells
 * us it's ready (via the logfile, of all things). urgh urgh.
 * 
 * strange... it also seems to output the string "-12345X", among
 * others, to stderr when printing. Is this what the printer is
 * saying? What does it all mean?
 *
 * by the way: in case you're wondering how hpnpf can restart
 * print jobs from where they failed last time they were printed,
 * it does it by putting the current pagenumber in a file in
 * /tmp. What a hack! ;)
 *                                      -- Justin Mason Aug 94.
 */

/* You may need to change this:
 * the location of the `hpnpf' executable.
 */
#define HPNPF	"/usr/lib/hpnp/hpnpf"

#include "lp-pipe.h"

char logpath[BUFSIZ];
char *Host;
char *NetIFType;
pid_t hpnpf_pid;
int pipefd = -1;
int logfd = -1;
int hpnpf_is_connected = 0;

void make_logfile ();
void open_hpnpf (), check_hpnpf (), close_hpnpf ();
void handle_logfile ();

void
usage () {
    fprintf (stderr, "usage: hpnp-lp host [... args]\n");
    exit (LP_FAILED);
}

void hpnp_pipe ()
{
    int still_reading = 1;
    fd_set rfds;
    struct timeval timeout;

    while (1) {
	int sel;

	check_hpnpf ();		/* if hpnpf has exited, we exit here too. */

	if (still_reading && hpnpf_is_connected) {
	    /* We use select to poll stdin. We can't use it on the
	     * logfile, because that's _always_ ready to be read,
	     * whether there's data there or not.
	     */
	    FD_ZERO (&rfds); FD_SET (0, &rfds);
	    timeout.tv_sec = 1; timeout.tv_usec = 0;

	    if ((sel = select (1, &rfds, NULL, NULL, &timeout)) < 0) {
		perror ("hpnp-lp: select");
		close_hpnpf ();
		if (logfd) { (void) close (logfd); (void) unlink (logpath); }
		exit (LP_RETRY);
	    }

	    /* something to be read on stdin. */
	    if (FD_ISSET (0, &rfds)) {
		int in, out;
		char buf[BUFSIZ];

		if ((in = read (0, buf, BUFSIZ)) > 0) {
		    out = write (pipefd, buf, in);
		    if (in != out) {
			perror ("hpnp-lp: write error");
			close_hpnpf ();
			if (logfd) { (void) close (logfd); (void) unlink (logpath); }
			exit (LP_RETRY);
		    }
		}
		if (in < 0) {
		    perror ("hpnp-lp: read");
		    close_hpnpf ();
		    if (logfd) { (void) close (logfd); (void) unlink (logpath); }
		    exit (LP_RETRY);

		} else if (in == 0) {
		    /* we've got EOF on stdin, close the pipe.
		     * The only reason we stay in this select is
		     * so we can keep an eye on the logfile.
		     */

		    out = write (pipefd, "\004", 1);
		    if (close (pipefd) < 0) {
			perror ("hpnp-lp: close");
		    }
		    still_reading = 0;
		}
	    }
	} else {
	    sleep (1);
	}

	handle_logfile ();
    }
    /*NOTREACHED*/
}

plp_signal_t
kill_job ()             /* the current job has been lprm'ed */
{
    if (pipefd > 0) {
        /* send an interrupt character to stop the current job;
	 * this is a PS-ism, but it'll probably work here.
	 * lpd will already have sent a SIGINT to the hpnpf
	 * process, as it's in the process group, so that
	 * should do the trick also.
	 */
        (void) write (pipefd, "\003", 1);
	close_hpnpf ();
    }
    /* exit with a success code; we don't want the queue to stop */
    if (logfd) { (void) close (logfd); (void) unlink (logpath); }
    exit (LP_SUCCESS);
}

int main (argc, argv)
    int argc;
    char **argv;
{
    int i;
    /* first arg is the printer device's hostname. */

    if (argc < 3) usage ();
    Host = argv[1];
    NetIFType = argv[2];

    for (i = 3; i < argc; i++) {
	/* ignore the options. */
        if (*argv[i] != '-') {
            /* it's the status filename. */
            status_fname = argv[i];
        }
    }

    (void) signal (SIGINT, kill_job);

    make_logfile ();
    open_hpnpf ();
    hpnp_pipe ();	/* main loop -- read from stdin, handle status file */
    /*NOTREACHED*/
}

void make_logfile ()
{
    (void) sprintf (logpath, "/tmp/hpnp-lp.%d", (int) getpid());
    close (open (logpath, O_WRONLY | O_CREAT));	   /* touch it */
    logfd = open (logpath, O_RDONLY);
    if (logfd < 0) {
	perror ("hpnp-lp: open logfile failed");
	exit (LP_FAILED);
    }
}

void
open_hpnpf ()
{
    int fds[2];

    if (pipe (fds) < 0) {
	perror("pipe");
	if (logfd) { (void) close (logfd); (void) unlink (logpath); }
	exit (LP_RETRY);
    }
    if ((hpnpf_pid = fork ()) < 0) {
	perror ("fork");
	if (logfd) { (void) close (logfd); (void) unlink (logpath); }
	exit (LP_RETRY);

    } else if (hpnpf_pid == 0) {	/* we're the daughter */
	execl (HPNPF, HPNPF, "-a", NetIFType, "-w", "-s", logpath,
		"-x", Host, NULL);
	perror ("can't run hpnpf");
	if (logfd) { (void) close (logfd); (void) unlink (logpath); }
	exit (LP_FAILED);
    }

    pipefd = fds[0];
}

void
check_hpnpf ()
{
    int exited;
    pid_t pid;
    plp_status_t status;

    if ((pid = plp_wait_kid (&status, WNOHANG)) > 0) {
	/* hpnpf has exited -- we should too. */

	exited = PLP_WEXITSTATUS (status);
	fprintf (stderr, "pid %d exit status: %d\n", (int) pid, exited);
	(void) close (logfd);
	(void) unlink (logpath);
	exit (exited);
    }
}

void close_hpnpf () { close (pipefd); check_hpnpf(); }

void
handle_logfile ()
{
    char *s, *date, *status, *last_nonspc;
    char buf[BUFSIZ];
    int retval;

    retval = read (logfd, buf, sizeof (buf));
    if (retval == 0) {
	return;		/* nothing to read; ignore. */
    }
    if (retval < 0) {
	perror ("hpnp-lp: read from logfile");
	if (logfd) { (void) close (logfd); (void) unlink (logpath); }
	exit (1);
    }
    /* else we read something. */

    s = buf;
    status = buf;
    date = NULL;

    while (*s) {
	while (*s == ' ' || *s == '\n') s++;

	date = s;
	while (*s != '%') {		/* find the end of date str */
		if (!*s)
	    	    goto hit_null;
		s++;
	}
	*s = '\0';

	while (*s != ' ') s++;		/* skip "%%[ " */
	while (*s == ' ') s++;		/* skip whitespace */

	if (!*s) goto hit_null;

	if (!strncmp (s, "status:  ", 9)) {
	    /* a generic prefix for HPNP verbosity, skip it */
	    s += 9;
	    while (*s == ' ') s++;
	}
	status = last_nonspc = s;

	/* find " ]%%" (or "]%%") -- end of message marker */
	while (*s != ']') {
	    if (!*s)
	        goto hit_null;
	    if (*s != ' ') {
		last_nonspc = s;
	    }
	    s++;
	}

	*(last_nonspc + 1) = '\0';
	while (*s != '\n') s++;		/* ffwd to newline */

	/* date now has the format: "Aug 29 15:18:30".
	 * status now has the format: "transmitting data" or
	 * "printer: paper out". Put them in the status file.
	 */

	if (!strncmp (status, "Job: ", 5)) {
	    /* we don't need this -- lpd handles that sort of thing. */
	    return;
	}
	/* uncomment for debugging.
		fprintf (stderr, "---- %s (%s)\n", status, date);
	*/
    }

    if (!strncmp (status, "connected to ", 13)) {
	hpnpf_is_connected = 1;
    }
hit_null:
    if (date == NULL) {
	setstatus ("%s", status);
    } else {
	setstatus ("%s since %s", status, date);
    }
    return;
}
