/*
 * annex-lp: a PLP LP-pipe command for communicating with printers
 *           connected to Annex terminal servers.
 *
 * You are safe using the first few arguments to communicate LP-pipe-
 * specific configuration (such as the remote hostname and remote
 * port); PLP will append it's arguments to those specified in the
 * printcap entry.
 *                              -- Justin Mason 26 Aug 94.
 *
 * (from original source file:)
 * >
 * > Open a socket to an Annex terminal server. This code has been excerpted
 * > from modifications made to the Berkeley LPD code, by ENCORE Computers Inc.,
 * > and has been used with their permission.
 * >
 * > It appears that the terminal server accepts a port connection on the LPD
 * > port. I suppose that it then treats the selected printer nicely. Not bad,
 * > as far as hacks go, but it does expose a severe security loophole.
 * >
 * > Fri Sep 30 09:59:59 CDT 1988 Patrick Powell
 *
 * lppipename annexhost line \	(these two are in the printcap entry)
 *
 * -PPrinter -wwidth -llength -xwidth -ylength [-c] [-iindent] \
 * [-Zoptions] [-Cclass] [-Jjob] [-Raccntname] -nlogin -hhost  \
 * -Fformat -Ddebug [affile]
 *				(all added by lpd)
 */

#include "lp-pipe.h"

#define OPEN_RETRY_DELAY         15      /* delay between retries */
int sockfd = -1;
char *Annex;
unsigned short Port;
int Line;

usage () {
    fprintf (stderr, "usage: annex-lp annexhost line [... args]\n");
    exit (LP_FAILED);
}

void open_sock ()
{
    struct hostent *host;       /* host entry pointer */
    struct sockaddr_in sin;

    if ((host = gethostbyname (Annex)) == NULL) {
        fprintf (stderr, "unknown host %s: %s\n", Annex, ERRSTR);
    }
    bzero ((char *) &sin, sizeof (sin));
    sin.sin_family = AF_INET;
    bcopy (host->h_addr, (caddr_t) & sin.sin_addr, host->h_length);
    sin.sin_family = host->h_addrtype;
    sin.sin_port = htons (Port);

    do {
        /* open the socket. */

        if ((sockfd = socket (AF_INET, SOCK_STREAM, 0)) < 0) {
            perror("can't open socket");
        }

        if (connect (sockfd, (struct sockaddr *) & sin, sizeof (sin)) == 0) {
            break;

        } else {
            setstatus ("Retrying connection - offline? (%s)", ERRSTR);
            close (sockfd);
            sleep (OPEN_RETRY_DELAY);
        }
    } while (1);                /* block forever */
}

plp_signal_t
kill_job ()		/* the current job has been lprm'ed */
{
    if (sockfd > 0) {
	(void) close (sockfd);
    }
    /* exit with a success code; we don't want the queue to stop */
    exit (LP_SUCCESS);
}

void annex_pipe ();

main (argc, argv)
    int argc;
    char **argv;
{
    int i;

    /* first two args are the host/port pair.
     * 
     */
    if (argc < 3) usage ();
    Annex = argv[1];

    if ((Line = atoi (argv[2])) <= 0)
        usage ();

    for (i = 3; i < argc; i++) {
	if (*argv[i] == '-') {
	    /* annex-lp doesn't need to use any of the filter options,
	     * so just ignore them.
	     */

	} else {
	    /* it's the status filename. */
	    status_fname = argv[i];
	}
    }

    (void) signal (SIGINT, kill_job);

    annex_pipe ();
    exit (LP_SUCCESS);
}

#include <setjmp.h>

static jmp_buf ack_jmp_buf;
/*
 * SIGALRM catcher for Annex ACK listener. If a SIGALRM comes, then we didn't get an
 * initial ACK, meaning that this Annex is probably using the old Annex LPD protocol.
 */
plp_signal_t
noannexack ()
{
    longjmp (ack_jmp_buf, 1);
}

/*
 * Open a socket to an Annex terminal server. This code has been excerpted from
 * modifications made to the Berkeley LPD code, by ENCORE Computers Inc., and has been
 * used with their permission.
 * 
 * It appears that the terminal server accepts a port connection on the LPD port.  I suppose
 * that it then treats the selected printer nicely. Not bad,  as far as hacks go,  but it
 * does expose a severe security loophole.
 * 
 * Fri Sep 30 09:59:59 CDT 1988 Patrick Powell
 */
#define SERIAL  1
#define PAR     2
#define ANNEX_SPOOL   '\011'
#define ACK_WAIT 10             /* seconds to wait for annex to ACK */

int
Open_annex ()
{
    int i;                      /* ACME Screw and Integers, Inc. even used by ENCORE */
    plp_signal_t (*func) ();    /* what SIGALRM was set to */
    char line[BUFSIZ];          /* temporary buffer */
    int n;                      /* number chars read/written */
    int err;                    /* save value of errno */
    struct servent *srv;

    setstatus ("Trying to open Annex %s port %d (offline?)", Annex, Line);
    (void) alarm (0);           /* disarm */
    func = signal (SIGALRM, noannexack);        /* set up time out function */

    for (i = 1;; i = i < 512 ? i << 1 : i) {
	if ((srv = getservbyname ("printer", "tcp")) == NULL) {
	    perror ("get port (printer)");
	    Port = 515;
	} else {
	    Port = srv->s_port;
	}

	open_sock ();

        if (sockfd >= 0) {
            /*
             * you are trying to connect to the ANNEX PRINTER port, and set up a simple
             * transfer protocol.
             */
            (void) sprintf (line, "%c%d;%d\n",
                            ANNEX_SPOOL,
                            Line <= 0 ? PAR : SERIAL,
                            Line <= 0 ? 1 : Line);

            n = strlen (line);

            /*
             * make sure that it is set correctly, sleep in some implementations will
             * clobber the SIGALRM value and not restore it
             */
            (void) signal (SIGALRM, noannexack);

            if (setjmp (ack_jmp_buf)) {
                fprintf (stderr, "%s port %d response timed out, old Annex?\n",
                        Annex, Line);
            } else {
                (void) alarm (ACK_WAIT);

                if (write (sockfd, line, n) == n) {
                    n = read (sockfd, line, 1);
                    err = errno;
                    (void) alarm (0);   /* disarm */
                    errno = err;
                    if (n != 1) {
                        fprintf (stderr,
                                "annex open: read error from %s port %d: %s\n",
				Annex, Line, ERRSTR);
			perror ("read");

                    } else if (line[0] != '\0') {
                        fprintf (stderr,
                             "annex open: '%d' read from %s port %d: %s\n",
                             line[0], Annex, Line, ERRSTR);

                    } else {
                        /* success */
                        /* this should exit from the loop */
                        break;
                    }
                } else {
                    (void) alarm (0);   /* disarm */
                }
            }
            (void) close (sockfd);
        }
        /*
         * note that sleep may clobber SIGALRM entry
         */
        sleep ((unsigned) i);
    }
    (void) alarm (0);           /* disarm */
    (void) signal (SIGALRM, func);
    setstatus ("Sending to %s (line %d)", Annex, Line);
}

/***************************************************************************
 * Annex_ack()
 * (called on close)
 * 1. send an OOB ACK to indicate that you really have finished (EOF)
 * 2. Write a 0 byte to flush the output message (4.2 Bugger)
 * 3. Wait for a response that it has been done
 ***************************************************************************/
void
Annex_ack ()
{
    int n;
    char nullstr[2];
#define ACK_CL_WAIT (10*60)

    (void) signal (SIGALRM, noannexack);
    if (setjmp (ack_jmp_buf)) {
        fprintf (stderr, "%s port %d closing ack timed out\n", Annex, Line);

    } else {
        nullstr[0] = 0;
        (void) alarm (ACK_CL_WAIT);
        if (send (sockfd, nullstr, 1, MSG_OOB) != 1) {
            (void) alarm (0);   /* disarm */
            perror ("closing ack: send-oob failed");
            return;
        }
        if (send (sockfd, nullstr, 1, 0) != 1) {
            (void) alarm (0);   /* disarm */
            perror ("closing ack: send failed");
            return;
        }
        n = read (sockfd, nullstr, 1);
        (void) alarm (0);       /* disarm */

        if (n != 1) {
            fprintf (stderr, "closing ack: read error from %s port %d", Annex, Line);
	    perror (":");

        } else if (nullstr[0] != '\0') {
            fprintf (stderr,
                 "Annex_ack: '%d' read from %s port %d\n", nullstr[0], Annex,
                 Line);
        }
    }
}

void annex_pipe ()
{
    int in, out;
    char buf[BUFSIZ];

    Open_annex ();
    while ((in = read (0, buf, sizeof (buf))) > 0) {
	out = write (sockfd, buf, in);
	if (in != out) {
	    perror ("write error");
	    close (sockfd);
	    exit (LP_RETRY);
	}
    }
    if (in < 0) {
	perror ("read error");
	close (sockfd);
	exit (LP_RETRY);
    }
    Annex_ack ();
    (void) close (sockfd);
}
