#include "lp.h"
#include "library/errormsg.h"
#include "library/filestack.h"
#include "library/filecache.h"
#include "library/printcap.h"
#include "library/utils.h"

#ifdef LOCAL
static int getpcent_file (char *tb, int size);
static int openprent (void);
static int nextprent (void);
static void closeprent (void);
static int pc_entry_file (char *buf, int size);

static FILE *pfp = NULL;	/* printcap data base file pointer */

static struct file_cache pc_cache;

/**************************************************************************
 *     finds the next entry in the printcap file;
 *     it places it in the tb buffer, and returns
 *     1 if entry found, 0 otherwise
 * Also associated functions: {open,next,close}prent().
 *************************************************************************/

static char *current_pcfile = NULL;
static char *next_pcfile = NULL;
static char *tokpath = NULL;
static char read_from_cache;

static int
openprent (void) {

    if (Use_printcap_cache) {
	rewind_cache (&pc_cache);

	if (pc_cache.entries[0]) {
	    /* we have a cache, use it. */
	    read_from_cache = 1;
	    if (DbgPcap > 2)
		log (XLOG_DEBUG, "openprent: reading printcaps from cache");
	    return (0);
	}
	/* otherwise, we are being called by build_pc_cache (). */
	read_from_cache = 0;
	if (DbgPcap > 2)
	    log (XLOG_DEBUG, "openprent: reading printcaps from file");
    }

    if (tokpath)
	free (tokpath);

    if (Printcap_path == NULL || !*Printcap_path) {
	logerr_die (XLOG_INFO, "printcap-path is unset!");
    }

    tokpath = (char *) strdup (Printcap_path);		/* just copy the path. */

    if (DbgPcap > 2) {
	log (XLOG_DEBUG, "openprent: path='%s'", tokpath);
    }

    /* init the strtok(). */
    next_pcfile = (char *) strtok (tokpath, ":");

    if (next_pcfile == NULL)
	fatal (XLOG_NOTICE, "Empty printcap path");
    return (nextprent ());
}

static int
nextprent (void) {
    do {
	if (next_pcfile == NULL) {
	    if (DbgPcap > 6)
		log (XLOG_DEBUG, "nextprent: end of path");
	    return (1);
	}
	if (pfp != NULL) {
	    (void) fclose (pfp);
	    pfp = NULL;
	}
	if ((pfp = fopen (next_pcfile, "r")) == NULL) {
	    if (DbgPcap > 6)
		logerr (XLOG_DEBUG,
			"nextprent: failed to open %s, skipping", next_pcfile);
	    next_pcfile = (char *) strtok (NULL, ":");

	} else {
	    current_pcfile = next_pcfile;
	    if (DbgPcap > 6)
		log (XLOG_DEBUG,
		     "nextprent: opened %s", current_pcfile);

	    next_pcfile = (char *) strtok (NULL, ":");
	    return (0);
	}
    } while (pfp == NULL);
    return (1);			/* the last entr{y,ies} were invalid, end of path */
}

static void
closeprent (void) {
    if (DbgPcap > 5)
	log (XLOG_DEBUG, "closeprent");

    if (Use_printcap_cache) {
	rewind_cache (&pc_cache);
	if (read_from_cache) {
	    return;
	}
    }

    if (pfp != NULL) {
	(void) fclose (pfp);
	pfp = NULL;
    }
    if (tokpath != NULL) {
	free (tokpath);
	tokpath = NULL;
    }
    next_pcfile = current_pcfile = NULL;
}

static int
getpcent_file (char *tb, int size) {
    char buf[BUFSIZ];		/* read buffer */
    char *bp, *cp, *ep;		/* next read position */
    int l;			/* line length */
    int last_eof;		/* run out of files */

    if (current_pcfile == NULL) {
	openprent ();
    }
    last_eof = 0;

    /*
     * We read a single printcap entry into the bp buffer. We scan lines until we hit one
     * that does not start with a # or is a blank line.
     */
    while (!last_eof) {
	bp = Gets (buf, sizeof (buf), &pfp, Printcap_include_path);
	if (bp == 0) {
	    last_eof = nextprent ();
	    continue;
	}
	if (buf[0] != '#' && buf[0] != '\n') {
	    break;
	}
    }
    if (last_eof) {
	return (0);		/* ran out of files */
    }
    /*
     * read the file in, stripping off the \<newlines>
     */
    ep = tb + size;
    bp = tb;
    while (!last_eof) {
	cp = strchr (buf, '\n');
	if (cp == NULL) {
	    fatal (XLOG_INFO, "bad line in printcap file %s: '%s'",
		   current_pcfile, buf);
	}
	*cp = 0;
	if (cp != buf) {
	    l = (cp[-1] == '\\');
	    if (l) {
		cp[-1] = 0;
	    }
	} else {
	    l = 0;
	}
	bp = estrcp (bp, buf, ep);
	if (bp == 0) {
	    fatal (XLOG_INFO, "printcap entry too long in file %s: '%s'",
		   current_pcfile, bp);
	}
	if (l == 0) {
	    if (DbgPcap > 9)
		log (XLOG_DEBUG, "getpcent_file: %s", tb);
	    return (1);
	}
	while ((Gets (buf, sizeof (buf), &pfp, Printcap_include_path) == 0)
		&& (last_eof == 0))
	{
	    last_eof = nextprent ();
	}
    }

    log (XLOG_INFO, "bad termcap entry: %s", tb);
    return (0);
    /* let the calling code do closeprent(). */
}

/*
 * All_printers: reads the printcap file, and forms a list of all the printer names in
 * the file.
 */
void
All_printers_file (void) {
    char capbuf[BUFSIZ];

    openprent ();
    while (pc_entry_file (capbuf, BUFSIZ))
	extract_pc_entry (capbuf, DO_CONT);
    closeprent ();
    return;
}

/*
 * First_printer: reads the printcap file and finds the first printer name
 */
char *
First_printer_file (void) {
    char capbuf[BUFSIZ];
    char *bp;
    static char first[PRNAMELEN + 1];

    openprent ();
    while (pc_entry_file (capbuf, BUFSIZ)) {
	init_pc_entry (Status_pc_vars, Status_pc_len);
	getpcvars (Status_pc_vars, Status_pc_len, capbuf, DO_CONT);

	/* get the printcap variable values */
	if (DbgPcap > 6)
	    show_pc (Status_pc_vars, Status_pc_len);

	if (SD != NULL && *SD != '\0') {
	    if ((bp = strchr (capbuf, '|')) == NULL) {
		bp = strchr (capbuf, ':');
	    }
	    if (bp != NULL)
		*bp = '\0';
	    (void) strcpy (first, capbuf);
	    closeprent ();
	    return (first);
	}
    }
    closeprent ();
    return (NULL);
}

/* order of lookup: printername/hostname printername */

int
Set_pc_entry_file (char *name, char *capbuf) {
    int found, namelen;
    char *with_host, *hostptr;
    char tcapbuf[BUFSIZ];

    if (strchr (name, PNAME_SEPCH)) {
        fatal (XLOG_INFO, "printer name '%s' contains '%c'",
                    name, PNAME_SEPCH);
    }
    if (strchr (name, '@')) {
        fatal (XLOG_INFO, "printer name '%s' contains '@'", name);
    }

    /* make a copy so we can tag on the hostname */
    namelen = strlen (name);
    with_host = (char *) malloc (namelen + strlen (ShortHost) + 2);
    (void) strcpy (with_host, name);
    hostptr = with_host + namelen;
    *hostptr = PNAME_SEPCH;
    strcpy (hostptr + 1, ShortHost);

#if (defined(NIS) || defined(HESIOD)) && defined(CROSS_DOMAIN)
    if (change_domain)
	out_domain = NULL;
#endif

    if (DbgPcap > 5) {
	log (XLOG_DEBUG, "doing printcap lookup (file) for printers: %s %s",
		 with_host, name);
    }
    openprent ();
    found = 0;
    while (pc_entry_file (tcapbuf, BUFSIZ) > 0) {
	if (DbgPcap > 7)
	    log (XLOG_DEBUG, "Set_pc_entry: pc_entry %s", tcapbuf);
	
	if (!strncmp (tcapbuf, "include ", 8)) {
	    log (XLOG_INFO, "missing newline before '%s'", tcapbuf);

	} else if (Chk_name (tcapbuf, with_host) || (Chk_name (tcapbuf, name))) {
	    strcpy(capbuf, tcapbuf);
	    found = 1;
	    break;
	}
    }
    closeprent ();

    if (DbgPcap > 5)
	log (XLOG_DEBUG, "found printcap entry: %s", found ? "yes" : "no");
    free (with_host);
    return (found);
}

void
build_pc_cache (void) {
    char buf[BUFSIZ];
    struct file_cache *pc_cache_ref;

    if (!Use_printcap_cache) {
	if (DbgPcap > 6)
	    log (XLOG_DEBUG, "printcap caching disabled");
        return;
    }

    pc_cache_ref = &pc_cache; chkandrealloc_cache (&pc_cache_ref);
    flush_cache (&pc_cache);		/* force openprent() to reread files */

    openprent ();
    while (getpcent_file (buf, BUFSIZ)) {
	if (DbgPcap > 7)
	    log (XLOG_DEBUG, "caching pc entry %d [%s]", pc_cache.ind, buf);

	pc_cache_ref = &pc_cache; chkandrealloc_cache (&pc_cache_ref);
	cache_addline (&pc_cache, buf);
    }

    cache_addeof (&pc_cache);
    closeprent ();
}

void
create_pc_cache (void) {
    if (!Use_printcap_cache) {
	if (DbgPcap > 6)
	    log (XLOG_DEBUG, "printcap caching disabled");
        return;
    }
    pc_cache.size = 0;
    build_pc_cache ();
}

static int
pc_entry_file (char *buf, int size) {
    if (!Use_printcap_cache) {
	if (DbgPcap > 6)
	    log (XLOG_DEBUG, "printcap caching disabled");
        return (getpcent_file (buf, size));
    }

    if (pc_cache.size == 0) {
	fatal (XLOG_NOTICE, "printcap not cached!");
    }
    if (pc_cache.entries[pc_cache.ind] == NULL) {
	/* there's nothing more to read */
	rewind_cache (&pc_cache);
	return 0;
    }

    (void) strncpy (buf, pc_cache.entries[pc_cache.ind], size - 1);
    if (DbgPcap > 7) {
	log (XLOG_DEBUG, "in cache: got entry %d [%s]", pc_cache.ind, buf);
    }
    pc_cache.ind++;

    return 1;
}

void
dump_printcap_cache (void)		/* debugging. */
{
    int i;
    for (i = 0; pc_cache.entries[i]; i++) {
	printf ("CACHED (pcap): [%s]\n", &(pc_cache.entries[i][1]));
    }
}

void
flush_printcap_cache (void) {
    if (!Use_printcap_cache) {
	return;
    }
    if (DbgPcap > 2)
      log (XLOG_DEBUG, "flushing printcap cache");

    build_pc_cache ();
}
#endif				/* LOCAL */
