/***************************************************************************
 * U. Minnesota LPD Software * Copyright 1987, 1988, Patrick Powell
 * version 3.4.2 Justin Mason July 1994
 ***************************************************************************
 * MODULE: lock_device.c
 * Lock the printer device (if it's a device file)
 ***************************************************************************/

#include "lp.h"
#include "library/errormsg.h"
#include "LPD/term.h"

/****************************************************************************
 * try to get an exclusive open on the device. How to do this differs
 * widely from system to system, unfortunately.
 * Most of this code has been contributed by Michael Joosten
 * <joost@cadlab.de>.
 ***************************************************************************/

/*
 * the actual locking code -- #ifdef'ed based on the flag constants used by the lock
 * functions.
 */
#ifdef HAVE_FCNTL_LOCK
static int 
devlock_fcntl (int fd, int nowait) {
    struct flock file_lock;
    if (DbgLocal > 5)
	log (XLOG_DEBUG, "do_lock: using fcntl with SEEK_SET");

    file_lock.l_type = F_WRLCK;
    file_lock.l_whence = SEEK_SET;
    file_lock.l_start = 0;
    file_lock.l_len = 0;
    if (fcntl (fd, F_SETLK, &file_lock) < 0) {
	if (DbgLocal)
	    log (XLOG_DEBUG, "fcntl failed: %s", Errormsg (errno));
	return (0);
    }
    return (1);
}

#endif

#ifdef HAVE_FLOCK
static int 
devlock_flock (int fd, int nowait) {
    if (flock (fd, nowait ? (LOCK_EX | LOCK_NB) : LOCK_EX) < 0) {
	if (DbgLocal)
	    log (XLOG_DEBUG, "flock failed: %s", Errormsg (errno));
	if (errno == EWOULDBLOCK) {
	    return (0);
	}
    }
    return (1);
}

#endif

#ifdef HAVE_LOCKF
static int 
devlock_lockf (int fd, int nowait) {
    /*
     * want to try F_TLOCK
     */
    if (lockf (fd, nowait ? F_TLOCK : F_LOCK, 0L) < 0) {
	if (DbgLocal)
	    log (XLOG_DEBUG, "lockf failed: %s", Errormsg (errno));
	return (0);
    }
    return (1);
}

#endif

/* a wrapper func for the above. */

static int 
devlock (int fd, int nowait) {
    /*
     * put special cases here.
     */
#ifdef LOCK_DEVS_WITH_FLOCK
    /*
     * using lockf() or fcntl(..,F_SETLK,..) does not work on character devices in SunOS
     * 4.1.?. (mj)
     */
    return (devlock_flock (fd, nowait));
#else

    /*
     * general case -- try using the chosen lock function.
     */
#ifdef HAVE_FCNTL_LOCK
    return (devlock_fcntl (fd, nowait));
#else
#ifdef HAVE_LOCKF
    return (devlock_lockf (fd, nowait));
#else
#ifdef HAVE_FLOCK
    return (devlock_flock (fd, nowait));
#endif
#endif
#endif
#endif
}

/***************************************************************************
 * do_devlock( fd , char *filename, nowait)
 * does a lock on a device file; for reasons see comment below...
 * Returns: 1 if successful; 0 if locked; -1 otherwise
 * If nowait == 0, does a blocking lock, nonblocking otherwise
 ***************************************************************************/
int
do_devlock (int fd, char *filename, int nowait) {
    int code;

    if (DbgLocal > 5)
	log (XLOG_DEBUG, "locking device file %s before use", filename);
    code = -1;

#ifdef TIOCEXCL
    if (isatty (fd)) {
        /* use the TIOCEXCL ioctl. See termio(4). */
        if (ioctl (fd, TIOCEXCL, (char *) 0) < 0) {
	    logerr (XLOG_INFO, "TIOCEXCL failed");
	}
    }
#endif

    code = devlock (fd, nowait);

    if (DbgLocal > 5)
	log (XLOG_DEBUG, "device lock: status %d", code);
    return (code);
}
