/****************************************************************************
 *
 *	Program:	view.c
 *	Author:		Marc van Kempen
 *	desc:		Viewing routines. 
 *
 ****************************************************************************/

#include <stdlib.h>
#include <stdio.h>
#include <sys/param.h>
#include <string.h>
#include <unistd.h>

#include "mlist.h"
#include "config.h"
#include "scroll.h"
#include "curses.h"

extern 	int	_screenx, _screeny;

void
view_file(char *fname)
/*
 *	desc:	check the filetype of fname and 
 *		execute the appropriate viewer
 */
{
		switch(get_filetype(fname)) {
			case	ZIPFILE :
					view_zipfile(fname);
					break;
			case 	GZIPFILE :
					view_gzipfile(fname);
					break;
			case	BINARYFILE :
					view_binaryfile(fname);
					break;
			case	TEXTFILE :
					view_textfile(fname);
					break;
			case	GIFFILE :
					view_giffile(fname);
					break;
			case 	PSFILE :
					view_postscriptfile(fname);
					break;
			case	TARFILE:
					view_tarfile(fname);
					break;
			case	FALSE :	/* read error on file <fname> */
					break;
			default :
					view_textfile(fname);
					break;
		}
}/* view_file() */

void
view_postscriptfile(char *fname)
/*
 *	desc:	view a postscriptfile, using an external program
 */
{
	char    prog[MAXPATHLEN];
	char	*yesno[] = { "view as text file       ", 
			     "view as postscript file " },
		*answer;
	int	i;
	
	if (getenv("DISPLAY")) {
		i = 0;
		answer = ScrollList(yesno, 2, &i, 3, 30, 3, 6);
		if (answer) {
			if (answer == yesno[0]) {
				view_textfile(fname);
			} else {
				sprintf(prog, "%s %s", PSVIEWER, fname);
				system(prog);
			}
		}
	} else {
		/* No display variable, so look at it as text */
		MoveCursor(_screeny-1, 0);
		CleartoEOLN();
		MoveCursor(_screeny-1, 0);
		Write_to_screen("No DISPLAY variable set. Press any key", 0);
		ReadCh();
		view_textfile(fname);
	}

	return;
} /* view_postscriptfile() */
	

void
view_giffile(char *fname)
/*
 *	desc:	view a giffile, using an external program
 */
{
	char    prog[MAXPATHLEN];
	char	*yesno[] = { "view as binary file", 
			     "view as gif-file" },
		*answer;
	int	i;
	
	if (getenv("DISPLAY")) {
		i = 0;
		answer = ScrollList(yesno, 2, &i, 3, 25, 3, 6);
		if (answer) {
			if (answer == yesno[0]) {
				view_textfile(fname);
			} else {
				sprintf(prog, "%s %s", GIFVIEWER, fname);
				system(prog);
			}
		}
	} else {
		MoveCursor(_screeny-1, 0);
		CleartoEOLN();
		MoveCursor(_screeny-1, 0);
		Write_to_screen("No DISPLAY variable set. Press any key", 0);
		ReadCh();
		view_textfile(fname);
	}
	
	return;
} /* view_giffile() */
	


void
view_zipfile(char *fname)
/*
 *	desc:	view a zipfile, first show a list of the contents of the
 *		zipfile from which you can choose a file, which will then 
 *		be displayed.
 */
{
	char	prog[MAXPATHLEN], *tmp_file;
	char	**zipfiles, 
		zipfile[MAXPATHLEN];
	char	d[100], *extractname, 
		extractname_tmp[MAXPATHLEN];
	FILE	*f;
	int	i, iret, StartItem, maxlen, j;
	char	**size;
	int	MaxZipFiles = MAXZIPFILES;
	
	
	/* Get the zipfile info and store it into a temporary file */
	tmp_file = tempnam(NULL, "zip.");
	sprintf(prog, "%s -v %s > %s", UNZIP, fname, tmp_file);
	system(prog);
	
	f = fopen(tmp_file, "r");
	if (f == NULL) {
		printf("view_zipfile: Serious error, couldn't open %s\n", tmp_file);
		exit(-1);
	}
	
	/* Allocate memory for the zipnameslist */
	zipfiles = (char **) malloc ( sizeof(char **) * MaxZipFiles );
	size = (char **) malloc ( sizeof(char **) * MaxZipFiles );
	
	/* remove the leading garbage from the unzip -v output */
	for (i=0; i<11; i++) {
		fscanf(f, "%s", d); 
	}
	fscanf(f, "\n");
	for (i=0; i<9; i++) {
		fscanf(f, "%s", d); 
	}
		
	/* get the filenames from the unzip -v output */
	i = 0;
	iret = 8;
	maxlen = 0;
	while (iret == 8) {
		size[i] = (char *) malloc( 20 );
		iret = fscanf(f, "%s %s %s %s %s %s %s %s\n", 
			      size[i], d, d, d, d, d, d, zipfile);
		zipfiles[i] = (char *) malloc( MAXPATHLEN );
		memset(zipfiles[i], ' ', MAXPATHLEN - 1);
		zipfiles[i][MAXPATHLEN - 1] = 0;
		if (zipfile[0]=='^') {
			memcpy(zipfiles[i], zipfile+1, strlen(zipfile+1));
		} else {
			memcpy(zipfiles[i], zipfile, strlen(zipfile));
		}
		maxlen = max(maxlen, strlen(zipfile)+strlen(size[i])+2);
		i++;
		if (i == MaxZipFiles) {
			zipfiles = realloc(zipfiles, MaxZipFiles + MAXZIPFILES);
			if (zipfiles == NULL) {
				printf("view_zipfile: out of memory\n");
				exit(-1);
			}
			MaxZipFiles += MAXZIPFILES;
		}
	}
	for (j=0; j<i-2; j++) {
		if (zipfiles[j][0]=='^') {
			sprintf(zipfiles[j] + maxlen - strlen(size[j]), "%s", size[j]);
		} else {
			sprintf(zipfiles[j] + maxlen - strlen(size[j]), "%s", size[j]);
		}
	}
	
	fclose(f);
	unlink(tmp_file);
	
	tmp_file = tempnam(NULL, "zip.");
	StartItem = 0;
	do {
		extractname = ScrollList(zipfiles, i-2, &StartItem, 5, 5+maxlen+3, 5, _screeny-5);
		if (extractname) {
			sscanf(extractname, "%s ", extractname_tmp);
			sprintf(prog, "%s -c -qq %s %s > %s", 
				UNZIP, fname, extractname_tmp, tmp_file);
			system(prog);
			view_file(tmp_file);

			if (unlink(tmp_file)) {
				printf("view_zipfile: serious error, could not delete temporary file\n");
				/* exit(-1); */
			}
		}
	} while (extractname);

	fclose(f);
	
	/* free the space allocated to zipfiles */
	for (j=0; j<i; j++) {
		free(zipfiles[j]);
		free(size[j]);
	}
	free(zipfiles);
	free(size);
	
	return;
} /* view_zipfile() */

void
view_tarfile(char *fname)
/*
 *	desc:	view a tarfile, first show a list of the contents of the
 *		tarfile from which you can choose a file, which will then 
 *		be displayed.
 */
{
	char	prog[MAXPATHLEN], *tmp_file;
	char	**tarfiles, tarfile[MAXPATHLEN];
	char	**size, d[20], *extractname,
		extractname_tmp[MAXPATHLEN];
	char	line[MAXPATHLEN];
	FILE	*f;
	int	i, StartItem, maxlen, j;
	int	MaxTarFiles = MAXTARFILES;
	
	tmp_file = tempnam(NULL, "tar.");
	sprintf(prog, "%s tvf %s > %s", TAR,  fname, tmp_file);
	system(prog);
	
	f = fopen(tmp_file, "r");
	if (f == NULL) {
		printf("Serious error, couldn't open %s\n", tmp_file);
		exit(-1);
	}
	
	/* Allocate memory for the tarnameslist */
        tarfiles = (char **) malloc ( sizeof(char *) * MaxTarFiles );
        size = (char **) malloc ( sizeof(char *) * MaxTarFiles );
		
	/* get the filenames from the tar output */
	i = 0;
	maxlen = 0;
	while (!feof(f)) {                
		size[i] = (char *) malloc( 20 );
		fgets(line, MAXPATHLEN, f);
		sscanf(line, "%s %s %s %s %s %s %s %s\n", d, d, size[i], d, d, d, d, tarfile);
                tarfiles[i] = (char *) malloc( MAXPATHLEN );
                memset(tarfiles[i], ' ', MAXPATHLEN - 1);
                tarfiles[i][MAXPATHLEN - 1] = 0;
		memcpy(tarfiles[i], tarfile, strlen(tarfile));
		maxlen = max(maxlen, strlen(tarfile)+strlen(size[i])+2);
		i++;
                if (i == MaxTarFiles) {
                        tarfiles = realloc(tarfiles, MaxTarFiles + MAXTARFILES);
                        if (tarfiles == NULL) {
                                printf("view_tarfile: out of memory\n");
                                exit(-1);
                        }
                        MaxTarFiles += MAXTARFILES;
                }
	}
	for (j=0; j<i-1; j++) {
		sprintf(tarfiles[j] + maxlen - strlen(size[j]), "%s", size[j]);
	}
	
	fclose(f);
	unlink(tmp_file);

	tmp_file = tempnam(NULL, "tar.");
	StartItem = 0;
	do {
		extractname = ScrollList(tarfiles, i-1, &StartItem, 5, 5+maxlen+3, 5, _screeny-5);
		if (extractname) {
			sscanf(extractname, "%s ", extractname_tmp);
			sprintf(prog, "%s xOf %s %s > %s", 
				TAR, fname, extractname_tmp, tmp_file);
			system(prog);
			view_file(tmp_file);

			if (unlink(tmp_file)) {
				printf("view_zipfile: serious error, could not delete temporary file\n");
				exit(-1);
			}
		}
	} while (extractname);

	for (j=0; j<i; j++) {
		free(tarfiles[j]);
		free(size[j]);
	}
	free(tarfiles);
	free(size);
	
	return;
} /* view_tarfile() */


void
view_gzipfile(char *fname)
/*
 *	desc:	view a gzipped file
 */
{
	char	prog[255];
	char	*tmpname;
	
	tmpname = tempnam(NULL, "gzip.");
	sprintf(prog, "%s -c %s > %s", GUNZIP, fname, tmpname);
	system(prog);
	view_file(tmpname);
	unlink(tmpname);
	
	return;
} /* view_gzipfile() */
	
void 
view_binaryfile(char *fname)
/*
 *	desc:	view a binary file
 */
{
	/* obsolete when "most" is used as pager */
	view_textfile(fname);
	
	return;
} /* view_binaryfile() */

void 
view_textfile(char *fname)
/*
 *	desc:	display the file <fname>
 *	pre:	<fname> is the name of a file
 *	post:	file has been displayed
 */
{
	char	prog[255];
	char    pager[255];
	
	if (getenv("PAGER")) {
	  strcpy(pager, getenv("PAGER"));
	} else {
	  strcpy(pager, PAGER);
	}
	
	ClearScreen();				/* clear stdscr */	
	sprintf(prog, "%s %s", pager, fname);
	system(prog);

#ifdef PAGER_DOESNOT_WAIT
	MoveCursor(_screeny-1, 0);
	Write_to_screen("Press any key ...", 0);
	getkey();
#endif

	return;
} /* display_file */


		
	
	
	
	
	

