#include <stdio.h>
#include <fcntl.h>
#include <bsd/sgtty.h>
#include <bsd/signal.h>
#include <string.h>
#include <strings.h>
#include <pwd.h>
#include <bsd/utmp.h>
#include <errno.h>
#include <sys/ioctl.h>
#include <sys/types.h>
#include <sys/dir.h>
#include <sys/file.h>
#include <sys/time.h>

/* Definitions of system stuff */

#define LDECCTQ 0x4000   /* FROM /usr/include/rpcsvc/rex.h, */
#define LNOFLSH 0x8000   /*         are these right for Linux? */

#define FALSE 0
#define TRUE  1
#define READ  0
#define WRITE 1
#define ERROR 2
#define BUFFER_SIZE    255
#define HISTORY_SIZE   101
#define USER_NAME_SIZE   8

#define EOL (-2)

/* special characters used by ile */

#define del '\177'
#define CA '\1'
#define CB '\2'
#define CC '\3'
#define CD '\4'
#define CE '\5'
#define CF '\6'
#define bel '\7'
#define bs '\10'
#define CI '\11'
#define nl '\12'
#define CK '\13'
#define CL '\14'
#define cr '\15'
#define CN '\16'
#define CO '\17'
#define CP '\20'
#define CQ '\21'
#define CR '\22'
#define CS '\23'
#define CT '\24'
#define CU '\25'
#define CV '\26'
#define CW '\27'
#define CX '\30'
#define CY '\31'
#define CZ '\32'
#define esc '\33'

/* areas and variables used to get termcap information */

char *getenv();
char *tgetnum();
char *tgetflag();
char *tgetstr();
char termcap_entry[1024];         /* termcap entry for the users terminal */
char term_seqs[1024];             /* area to store control sequences in */
char *where = term_seqs;
char *cle;                        /* move cursor left one space */
char *cce;                        /* clear to end of line */
char *cbl;                        /* audible bell */
char *cnl;                        /* new line character */
char *ccr;                        /* carriage return */

char *homedir = NULL;
char currentdir[MAXNAMLEN + 1] = "";

/* The original tty status flags are stored so that they can be restored when ile exits. */

struct sgttyb tty_sgttyb;
struct tchars tty_tchars;
struct ltchars tty_ltchars;
struct winsize tty_winsize;
int windowchanged;
int tty_ldisc;
int tty_mode;

/* tty, pty global variable declarations */

int master_pty;                 /* file descriptors */
int slave_tty;
char ttydev[] = "/dev/ttyxx";   /* the names of the tty and pty opened by getpty */
char ptydev[] = "/dev/ptyxx";
char lock[] = "/tmp/ile.lock";  /* path and name of the lock file */

/* getpty opens a pty, storing file descriptors in pty and tty.
   It trys pairs in order until it finds a pair that is not in use. */

int getpty(int* pty, int* tty) {
    int devindex, letter;
    static char ptychar1[] = "p";
    static char ptychar2[] = "0123456789abcdef";
    letter = 0;
    while (letter < 1) {
        ttydev[strlen(ttydev) - 2] = ptychar1[letter];
        ptydev[strlen(ptydev) - 2] = ptychar1[letter];
        letter++;
        devindex = 0;
        while (devindex < 16) {
            ttydev[strlen(ttydev) - 1] = ptychar2[devindex];
            ptydev[strlen(ptydev) - 1] = ptychar2[devindex];
            devindex++;
            if ((*pty = open(ptydev, O_RDWR)) >= 0) {
                if ((*tty = open(ttydev, O_RDWR)) >= 0)
                    return;
                else
                {
/*                  fprintf(stderr,"\"%s\" error on open %s\n",strerror(errno),ttydev);
*/                  (void) close(*pty);
                }
            } /* else
                fprintf(stderr,"\"%s\" error on open %s\n",strerror(errno),ptydev);
              */
        }
    }

    (void) fprintf(stderr, "ile: unable to allocate pty/tty pair\n");
    exit(1);
}

/* Termcap entries may have a sequences of digits optionally followed
   by a '*' in front of the actual sequence. This routine increments
   the pointer past this information. */

void strip(char **ptr) {
    while (('0' <= **ptr) && (**ptr <= '9'))
        (*ptr)++;
    if (**ptr == '*')
        (*ptr)++;
}

/* Set up everything needed to use the control sequences from the
   termcap entry for the terminal. */

void get_termcap(void) {
    char *terminal_type;        /* type of terminal */
    terminal_type = getenv("TERM");    /* get the terminal name */
/* get termcap entry */
    if (tgetent(termcap_entry, terminal_type) < 1) {
        (void) fprintf(stderr, "ile: can't find %s\n", terminal_type);
        exit(1);
    }
/* get the control sequences ile needs */
    if ((cbl = tgetstr("bl", &where)) == NULL) cbl = "\7";
    if ((cnl = tgetstr("nl", &where)) == NULL) cnl = "\n";
    if ((ccr = tgetstr("cr", &where)) == NULL) ccr = "\r";
    if ((cle = tgetstr("le", &where)) == NULL) if (tgetflag("bs")) cle = "\b";

    if ((cle == NULL) || ((cce = tgetstr("ce", &where)) == NULL)) {
        (void) fprintf(stderr, "ile: can't run on %s (need capabilities \"le\" and \"ce\")\n", terminal_type);
        exit(1);
    }
    strip(&cle);    /* strip timing info from strings */
    strip(&cce);
    strip(&cbl);
    strip(&cnl);
    strip(&ccr);
}

/* If the window changes size, tell the slave_tty about it. */

void change_window(void) {
    int pgrp;
    (void) ioctl(READ, TIOCGWINSZ, &tty_winsize);
    (void) ioctl(slave_tty, TIOCSWINSZ, &tty_winsize);
    (void) ioctl(slave_tty, TIOCGPGRP, (char *) &pgrp);
    (void) killpg(pgrp, SIGWINCH);
    windowchanged = TRUE;    /* note the change so that we don't die after select */
}
/* set/clear the utmp slot for the pty */
int setutmp(int fd, int set) {
    /* not needed for Linux */
    return (0);
}

/*
clean up and leave.

This function is bound to the SIGCHLD signal so that when the
child process exits, so does ile. It is also called when an exception
is detected by select() in ile().
*/
void clean_up(void) {
    int pgrp;
    (void) ioctl(slave_tty, TIOCGPGRP, (char *) &pgrp);    /* kill off the child process */
    (void) killpg(pgrp, SIGTERM);
    (void) ioctl(READ, TIOCSETP, &tty_sgttyb);             /* restore terminal status */
    (void) ioctl(READ, TIOCSETC, &tty_tchars);
    (void) ioctl(READ, TIOCSLTC, &tty_ltchars);
    (void) ioctl(READ, TIOCLSET, &tty_mode);
    (void) setutmp(slave_tty, FALSE);                      /* "logout" the user */
    (void) close(master_pty);                              /* clean up the tty/pty pair */
    (void) close(slave_tty);
    exit(0);
}

/*
Write a line to the slave_tty.
Get the slave_tty parameters, turn off echo, send the line to the
slave_tty, restore the slave_tty paramters to the way they were
before. If echo was already off, this will have no effect.
*/
write_line( char *line, int length) {
    struct sgttyb params;
    struct sgttyb new_params;

    (void) ioctl(slave_tty, TIOCGETP, &params);    /* get the current parameters */
    new_params = params;
    new_params.sg_flags &= ~ECHO;

    /* turn off echo so we don't see the characters twice */

    (void) ioctl(slave_tty, TIOCSETP, &new_params);
    (void) write(master_pty, line, length);

    /* set the parameters back the way they were */

    (void) ioctl(slave_tty, TIOCSETN, &params);
}

/*
The editing routines are called through the edit variable. This allows
the quote and escape commands to be implemented as a straight forward
state machine instead of requiring state flags and complex switch
statements.
*/

static char line[BUFFER_SIZE];   /* line edit buffer */
static int point;                /* insertion point */
static int length;               /* total chars in buffer */

void (*edit) ();                 /* procedure to edit next character */

struct {                         /* history buffer */
    int length;
    char *line;
} hist[HISTORY_SIZE];

int head;                        /* insertion point */
int here;                        /* current displayed line */

/*
The delimiter vector is used by the forward, backward, and delete
word operations to decide that a character is a delimiter.
*/
#define CHAR_SET_SIZE 127
#define CHAR_MASK 0177
char delimit[CHAR_SET_SIZE];

/* The action_table is used to bind sequences of keys to operations or strings. */

typedef enum { is_action, is_string } action_type;

struct {
    action_type flag;
    union {
        void (*action) ();
        char *string;
    } aors;
} action_table[4][CHAR_SET_SIZE];


void echo(char);
void echoline(char*, int);
void cleartoend(void);
void clearline(char);
void backspace(char);
void quote_edit(char);
void bell(char);
void insert(char);
void edit_0(char);
void edit_1(char);
void edit_2(char);
void edit_3(char);

/*
The following routines are action routines that are executed by the
editor to carry out commands. Each routine has a single character
argument. Each routine is invoked with the character that caused it
to be invoked as its argument.

The argument isn't always useful, but it is included to provide a
consistent interface for the routines.
*/

/*
Given a specific directory and the starting string of a file name,
find the longest partial file name that starts with the substring.
*/
void complete_file_name(char *dir, char *name) {
    DIR *dirp;
    struct direct *dp;
    int len, maxlen, oldlen;
    char oldname[MAXNAMLEN + 1], newname[MAXNAMLEN + 1];

    if ((dir != NULL) && (name != NULL) && ((oldlen = strlen(name)) > 0) && ((dirp = opendir(dir)) != NULL)) {
        maxlen = oldlen;
        (void) strcpy(oldname, name);
        (void) strcpy(newname, name);
/* find the longest name starting with name */
        for (dp = readdir(dirp); dp != NULL; dp = readdir(dirp))
            if (dp->d_name != NULL) {
                len = strlen(dp->d_name);
                if ((maxlen < len) && (strncmp(oldname, dp->d_name, oldlen) == 0)) {
                    maxlen = len;
                    (void) strcpy(newname, dp->d_name);
                }
            }
        rewinddir(dirp);
/* find the longest common sub string */
        for (dp = readdir(dirp); dp != NULL; dp = readdir(dirp))
            if (dp->d_name != NULL) {
                len = strlen(dp->d_name);
                if ((len <= maxlen) && (strncmp(oldname, dp->d_name, oldlen) == 0)) {
                    for (; (oldlen < len) && (strncmp(newname, dp->d_name, len) != 0); len--);
                    maxlen = len;
                    newname[maxlen] = '\0';
                }
            }
        if (strlen(name) != strlen(newname))
            (void) strcpy(name, newname);            /* return the extended name */
        else
            bell('\0');            /* no difference so beep */
        (void) closedir(dirp);
    }
}
/*
Hidden parameters to dirselect. They must be hidden because dirselect
is passed as an argument to scandir.
*/

static char *namep;
static int namelen;

/* Passed to scandir. It is used to decide which files to display. */

static dirselect(struct direct *dp) {
    return (strncmp(dp->d_name, namep, namelen) == 0);
}

/*
List all the files in a given directory that start with the string
passed as name.
*/
void list_file_names(char *dir, char *name) {
    struct direct **dlist;
    int i, nfiles, colwidth, cols, ncols, nlines;
    int alphasort();

    if (dir == NULL || name == NULL)
        return;
    cols = tty_winsize.ws_col;
    if (cols <= 0) cols = 80;
    namelen = strlen(name);
    namep = name;
    nfiles = scandir(dir, &dlist, dirselect, alphasort);

    /* determine longest file name length */
    colwidth = 3;                /* minimum width */
    for (i = 0; i < nfiles; i++) {
        struct direct *dp;
        dp = dlist[i];
        if (dp->d_namlen > colwidth)
            colwidth = dp->d_namlen;
    }
    colwidth++;                        /* at least 1 space between them */

    /* print the names, sorted vertically per column */
    ncols = cols / colwidth;
    if (ncols == 0) ncols = 1;        /* longest filename is wider than the screen */
    nlines = (nfiles + ncols - 1) / ncols;
    if (nfiles > 0) {
        for (i = 0; i < nlines; i++) {
            int j, l=0;
            for (j = 0; j < ncols; j++) {
                int m;
                struct direct *dp;
                m = l + i;
                if (m >= nfiles)
                    break;
                dp = dlist[m];
                fputs(dp->d_name, stdout);
                if (j < (ncols - 1)) {
                    int k;
                    for (k = dp->d_namlen; k < colwidth; k++)
                        (void) fputc(' ', stdout);
                    l += nlines;
                }
            }
            fputs(ccr, stdout);
            fputs(cnl, stdout);
        }
    }
    free((char *) dlist);
}
/* Assuming a file name under the cursor, return a path and a file name. If no name return "." */
int get_dir_and_name(char*dir, char*name, char*username, int*userend, int*start, int*middle, int*tail) {
    int dirlen, newstart, punlen, i;
    char pun[USER_NAME_SIZE + 1];
    struct passwd *userpwd;
    dir[0] = '\0';    /* set the default path and file name */
    name[0] = '\0';
    username[0] = '\0';
/* search for the start of the file name; start will be left pointing to the first char of the path */
    for ((*start) = point; ((0 < (*start)) && (line[(*start) - 1] != ' ')); (*start)--);
/* search for the end of the file name; tail will be left pointing at the last char of the path */
    for ((*tail) = point - 1; (((*tail) < (length - 1)) && (line[(*tail) + 1] != ' ')); (*tail)++);
/* search for the middle of the file name */
/* middle will be left pointing at the first character of the last element of the path */
    for ((*middle) = (*tail) + 1; ((0 < (*middle)) && (line[(*middle) - 1] != '/') && (line[(*middle) - 1] != ' ')); (*middle)--);
    /* copy path from line to dir */
    /* what base path */
    newstart = (*start);
    if ((line[newstart] == '~') && ((newstart + 1) < length) && (line[newstart + 1] == '/')) {
        newstart++;        /* "~/" means use the value of HOME */
        (void) strcpy(dir, homedir);
    }
    else if (line[newstart] == '~') {
        /* "~username" means use the users login directory */
        /* search for the end of the user name */
        for ((*userend) = newstart,
            punlen = 0;
            (((*userend) < (length - 1)) &&
                (line[(*userend) + 1] != ' ') &&
                (line[(*userend) + 1] != '/'));
            (*userend)++,
            punlen++);
        /* make middle point to middle */
        if ((*start) == (*middle))
            (*middle) = (*start) + punlen + 1;
        /* extract partial user name from line */
        (void) strncpy(pun, &line[newstart + 1], punlen);
        pun[punlen] = '\0';
        /* search passwd file for partial match */
        for (userpwd = getpwent(); userpwd != NULL; userpwd = getpwent()) {
            if ((punlen <= strlen(userpwd->pw_name)) && (strncmp(pun, userpwd->pw_name, punlen) == 0)) {
                if (strlen(dir) == 0) {                /* we have a partial match, record it */
                    newstart = (*userend) + 1;
                    (void) strcpy(dir, userpwd->pw_dir);
                    (void) strcpy(username, userpwd->pw_name);
                } else {                    /* second partial match, forget the first one. */
                    newstart = (*start);
                    dir[0] = '\0';
                    username[0] = '\0';
                    return (FALSE);
                }
            }
        }
        (void) setpwent();
    }
    else if ((line[newstart] == '.') && ((newstart + 1) < length) && (line[newstart + 1] == '/')) {
        newstart++;        /* if it's "./" use current dir */
        (void) strcpy(dir, currentdir);
    }
    else if ((line[newstart] == '.') && ((newstart + 1) < length) &&
             (line[newstart + 1] == '.') && ((newstart + 2) < length) &&
             (line[newstart + 2] == '/')) {
        newstart += 2;        /* if it's "../" strip off one name from currentdir and use that */
        (void) strcpy(dir, currentdir);
        for (i = strlen(dir); (i > 0) && (dir[i] != '/'); i--);
        dir[i] = '\0';
    } else if (line[newstart] != '/') {
        /* doesn't start with a "/"? use currentdir */
        (void) strcpy(dir, currentdir);
        (void) strcat(dir, "/");
    }
    dirlen = strlen(dir);    /* add on the rest of the path */
    for (i = 0; i < ((*middle) - newstart); i++)
        dir[dirlen + i] = line[newstart + i];
    dir[dirlen + i] = '\0';
    for (i = 0; i < ((*tail) - (*middle) + 1); i++)    /* copy file name from line to name */
        name[i] = line[(*middle) + i];
    name[i] = '\0';
    return (TRUE);
}
/* Perform file name completion. Put the full path and file name in the line. */
void complete_file_full(char ch) {
    char dir[10*(MAXNAMLEN+1)],name[MAXNAMLEN+1],username[USER_NAME_SIZE+1],newline[BUFFER_SIZE];
    int newlength, newpoint, userend, start, middle, tail, i;
    /* get the path and file name in the line */
    if (get_dir_and_name(dir, name, username, &userend, &start, &middle, &tail)) {
        complete_file_name(dir, name);        /* complete the file name if possible */
        (void) strncpy(newline, line, start); /* create a new line */
        newline[start] = '\0';                /* start with the line prefix */
        (void) strcat(newline, dir);          /* add in the new path */
        (void) strcat(newline, name);         /* stick in the new file name */
        newpoint = strlen(newline);           /* finish with the line postfix */
        (void) strncat(newline, &line[tail + 1], (length - tail - 1));
        newlength = strlen(newline);
        clearline('\0');                      /* display the new line */
        point = newpoint;
        length = newlength;
        (void) strncpy(line, newline, newlength);
        echoline(line, length);
        for (i = point; i < length; i++)
            backspace(line[i]);
    }
    else
        bell('\0');
}
/* Perform file name completion in much the same style as csh. */
void complete_file(char ch) {
    char dir[10*(MAXNAMLEN+1)], name[MAXNAMLEN+1], username[USER_NAME_SIZE+1], newline[BUFFER_SIZE];
    int userend, start, middle, tail, newlength, newpoint, userlen, len, i;
    /* get the path and file name in the line */
    if (strrchr(line,'~') != NULL)       /* use full completion if ~'s around */
        complete_file_full(ch);
    else if (get_dir_and_name(dir,name,username,&userend,&start,&middle,&tail)){
        complete_file_name(dir, name);        /* complete the file name if possible */
        (void) strncpy(newline, line, start); /* create a new line */
        newline[start] = '\0';                /* start with the line prefix */
        len = strlen(newline);
        (void) strncat(newline, &line[start], middle - start);
        newline[len + (middle - start)] = '\0';
        (void) strcat(newline, name);         /* stick in the new file name */
        newpoint = strlen(newline);           /* finish with the line postfix */
        (void) strncat(newline, &line[tail + 1], (length - tail - 1));
        newlength = strlen(newline);
        clearline('\0');                      /* display the new line */
        point = newpoint;
        length = newlength;
        (void) strncpy(line, newline, newlength);
        echoline(line, length);
        for (i = point; i < length; i++)
            backspace(line[i]);
    } else
        bell('\0');
}
/*
List the names of files that start with the directory path and
file name under the cursor.
*/
void show_files(char ch) {
    static char divider[] = "----------";
    void retype_line();
    char dir[10 * (MAXNAMLEN + 1)];
    char name[MAXNAMLEN + 1];
    char username[USER_NAME_SIZE + 1];
    int userend;
    int start;
    int middle;
    int tail;
    if (get_dir_and_name(dir, name, username, &userend, &start, &middle, &tail)) {
        fputs(ccr, stdout);
        fputs(cnl, stdout);
        fputs(divider, stdout);
        fputs(ccr, stdout);
        fputs(cnl, stdout);
        list_file_names(dir, name);
        fputs(divider, stdout);
        fputs(ccr, stdout);
        fputs(cnl, stdout);
        retype_line('\0');
    } else
        bell('\0');
}
/* Ring the bell on the terminal. */
void bell(char ch) {
    fputs(cbl, stdout);
}
/* Pass characters to the slave. Don't mess with them at all. */
void pass(char ch) {
    (void) write(master_pty, &ch, 1);
}
/*
Insert a character at point in the line buffer. While we are at it
update the display to show the insertion.
*/
void insert(char ch) {
    int i;
    if (length < (BUFFER_SIZE - 2)) {
        echo(ch);        /* display the character */
        echoline(&line[point], (length - point));        /* redisplay the rest of the line */
        /* move the characters in the line buffer and put the cursor back at point */
        for (i = length; i > point; i--) {
            line[i] = line[i - 1];
            backspace(line[i]);
        }
        /* add the character to the line buffer and increment point and length */
        line[point] = ch;
        length++;
        point++;
    }
    else
        bell('\0');
}
/*
Transpose the letter under the cursor and the letter immediately to
the left of the cursor.
*/
void transpose_chars(char ch) {
    char tch;
    if ((0 < point) && (point < length)) {        /* first, update the display */
        backspace(line[point]);
        echo(line[point]);
        echo(line[point - 1]);
        tch = line[point];        /* now swap the chars in the line buffer */
        line[point] = line[point - 1];
        line[point - 1] = tch;
        point++;        /* point moved forward one char */
    }
}
/*
Delete a character at point in the line buffer. While we are at it
update the display to reflect the deletion.
*/
void delete_char_under(char ch) {
    int i;
    if (point < length) {        /* clear to the end of the line */
        cleartoend();        /* retype the rest of the line */
        echoline(&line[point + 1], (length - point - 1));        /* build the new line */
        for (i = point + 1; i < length; i++) {
            line[i - 1] = line[i];
            backspace(line[i]);
        }
        length--;
        if (point > length) point = length;
    }
}
/*
Delete the character to the left of point in the line buffer. While we
are at it update the display to reflect the deletion.
*/
void delete_char(char ch) {
    int i;
    if (point > 0) {                                /* move the cursor left one character */
        backspace(line[point - 1]);                 /* clear to the end of the line */
        cleartoend();
        echoline(&line[point], (length - point));   /* retype the rest of the line */
        for (i = point; i < length; i++) {          /* build the new line */
            line[i - 1] = line[i];
            backspace(line[i]);
        }
        length--;
        point--;
    }
}
/* Bind edit vector to quote_edit so that next character will be placed in line buffer. */
void quote(char ch) {
    edit = quote_edit;
}
/* The next character will select an action from action_table[1] */
void escape_1(char ch) {
    edit = edit_1;
}
/* The next character will select an action from action_table[2] */
void escape_2(char ch) {
    edit = edit_2;
}
/* The next character will select an action from action_table[3] */
void escape_3(char ch) {
    edit = edit_3;
}
/* Delete the word to the left of the cursor. */
void delete_word(char ch) {
    int i;
    int old;
    if (length > 0) {
        old = point;        /* find the new deletion point */
        for (; (point > 0) && (delimit[line[point - 1]]); point--)  /* skip over delimiters */
            backspace(line[point - 1]);
        for (; (point > 0) && (!delimit[line[point - 1]]); point--) /* delete until a delimiter */
            backspace(line[point - 1]);
        cleartoend();        /* clear to the end of the line */
        echoline(&line[old], (length - old));        /* retype the rest of the line */
        for (i = 0; i < (length - old); i++) {        /* construct the new line */
            line[point + i] = line[old + i];
            backspace(line[point + i]);
        }
        length = length - (old - point);        /* update the length */
    }
}
/* Go forward one word. */
void forward_word(char ch) {
    if (length > 0) {
        for (; (point < length) && (delimit[line[point]]); point++)
            echo(line[point]);
        for (; (point < length) && (!delimit[line[point]]); point++)
            echo(line[point]);
    }
}
/* Lower case the word. */
void lower_word(char ch) {
    if (length > 0) {
        for (; (point < length) && (delimit[line[point]]); point++)
            echo(line[point]);
        for (; (point < length) && (!delimit[line[point]]); point++) {
            if ((line[point] >= 'A') && (line[point] <= 'Z')) {
                line[point] = line[point] - 'A' + 'a';
                echo(line[point]);
            } else
                echo(line[point]);
        }
    }

}
/* Upper case the word. */
void upper_word(char ch) {
    if (length > 0) {                                                  /* first skip any delimiters */
        for (; (point < length) && (delimit[line[point]]); point++)
            echo(line[point]);
        for (; (point < length) && (!delimit[line[point]]); point++) { /* skip until a delimiter */
            if ((line[point] >= 'a') && (line[point] <= 'z')) {
                line[point] = line[point] - 'a' + 'A';
                echo(line[point]);
            } else
                echo(line[point]);
        }
    }
}
/* Capitalize the word. */
void capitalize_word(char ch) {
    if (length > 0) {                                           /* first skip any delimiters */
        for (; (point < length) && (delimit[line[point]]); point++)
            echo(line[point]);
        if ((point < length) && (!delimit[line[point]]))        /* now skip until a delimiter */
            if ((line[point] >= 'a') && (line[point] <= 'z')) {
                line[point] = line[point] - 'a' + 'A';
                echo(line[point]);
            } else
                echo(line[point]);
        point++;
        for (; (point < length) && (!delimit[line[point]]); point++) {
            if ((line[point] >= 'A') && (line[point] <= 'Z')) {
                line[point] = line[point] - 'A' + 'a';
                echo(line[point]);
            } else
                echo(line[point]);
        }
    }
}
/* Go backward one word. */
void backward_word(char ch) {
    if (length > 0) {
/* first backspace over any delimiters */
        for (; (point > 0) && (delimit[line[point - 1]]); point--)
            backspace(line[point - 1]);
/* now backspace until we find a delimiter */
        for (; (point > 0) && (!delimit[line[point - 1]]); point--)
            backspace(line[point - 1]);
    }
}
/* Move the cursor to the start of the line. */
void start_of_line(char ch) {
    int i;
    if (length > 0) {
        for (i = 0; i < point; i++)
            backspace(line[i]);
        point = 0;
    }
}
/* Move the cursor one character to the left. */
void backward_char(char ch) {
    if ((length > 0) && (point > 0)) {
        backspace(line[point - 1]);
        point--;
    }
}
/* Move the cursor to the right of the last character on the line. */
void end_of_line(char ch) {
    if ((length > 0) && (point < length)) {
        echoline(&line[point], (length - point));
        point = length;
    }
}
/* Move the cursor one character to the right. */
void forward_char(char ch) {
    if ((length > 0) && (point < length)) {
        echo(line[point]);
        point++;
    }
}
/* Add a line to the history buffer and pass it to the child process as input. */
void add_to_history(char ch) {
    /* Put the line in the history buffer. Make here point to the current line. And increment head to point to the next history slot. */
    /* If the current line is identical to the current history line, don't add it. */
    /* don't save blank lines */

    int prev;
    if ((head - 1) < 0)
        prev = HISTORY_SIZE - 1;
    else
        prev = head - 1;

    if ((length != 0) && ((length != hist[prev].length) || (strncmp(hist[prev].line, line, length) != 0))) {
        hist[head].length = length;        /* set the length of the entry */

        /* make sure there is enough storage for the new line */
        if (hist[head].line == NULL) {
            if ((hist[head].line = (char *) malloc((unsigned) length)) == NULL)
                perror("ile");
        } else if ((hist[head].line = (char *) realloc(hist[head].line, (unsigned) length)) == NULL)
            perror("ile");
        (void) strncpy(hist[head].line, line, length);
        head = (head + 1) % HISTORY_SIZE;
        if (hist[head].line != NULL) {
            free(hist[head].line);
            hist[head].length = 0;
            hist[head].line = NULL;
        }
    }
    here = head;               /* reset here */
    fputs(ccr, stdout);        /* echo carriage return */
    fputs(cnl, stdout);        /* echo newline */
    line[length] = nl;
    length++;
    write_line(line, length);  /* send line to child */
    point = 0;                 /* clear buffer for re-use */
    length = 0;

}
/* Erase the entire line. */
void erase_line(char ch) {
    clearline(ch);    /* remove any text from the display */
    point = 0;    /* nothing in the line buffer */
    length = 0;
    here = head;    /* reset here */
}
/* Erase from the current cursor position to the end of the line. */
void erase_to_end_of_line(char ch) {
    if ((length > 0) && (point < length)) {
        cleartoend();
        length = point;
    }
}
/* Retype the current contents of the edit buffer. */
void retype_line(char ch) {
    int i;
    fputs(ccr, stdout);
    fputs(cnl, stdout);
    echoline(line, length);
    for (i = point; i < length; i++)
        backspace(line[i]);
}
/*
Go to the the next entry in the history buffer and display it.
If we are past the last history entry, then beep.
*/
void forward_history(char ch) {
    if (here != head) {
        clearline(ch);
        here = (here + 1) % HISTORY_SIZE;
        length = hist[here].length;
        point = length;
        (void) strncpy(line, hist[here].line, length);
        echoline(line, length);
    }
    else
        bell('\0');
}
/*
Search backward in the history list for a line that starts with
the characters left of the cursor. If it is found make it the
current line.
*/
void search_backward_history(char ch) {
    int prev;
    int i;
    prev = here;    /* search backward in the history */

    do {
        prev--;
        if (prev < 0)
            prev = HISTORY_SIZE - 1;
    }
    while ((hist[prev].line != NULL) && (strncmp(line, hist[prev].line, point) != 0));
    if (hist[prev].line != NULL) {         /* if something was found, make it the current line */
        here = prev;                       /* remember the position in the history */
        length = hist[here].length;        /* set the length and point correctly */
        if (point > length)
            point = length;
        clearline(ch);                     /* redraw the line */
        (void) strncpy(line, hist[here].line, hist[here].length);
        echoline(line, length);
        for (i = point; i < length; i++)
            backspace(line[i]);
    }
    else
        bell('\0');
}
/*
Search forward in the history list for a line that starts with
the characters left of the cursor. If it is found make it the
current line.
*/
void search_forward_history(char ch) {
    int next;
    int i;
    next = here;    /* search forward in the history */

    do {
        next++;
        if (next > HISTORY_SIZE - 1)
            next = 0;
    }
    while ((hist[next].line != NULL) && (strncmp(line, hist[next].line, point) != 0));
    if (hist[next].line != NULL) {         /* if something was found, make it the current line */
        here = next;                       /* remember the position in the history */
        length = hist[here].length;        /* set the length and point correctly */
        if (point > length)
            point = length;
        clearline(ch);                     /* redraw the line */
        (void) strncpy(line, hist[here].line, hist[here].length);
        echoline(line, length);
        for (i = point; i < length; i++)
            backspace(line[i]);
    }
    else
        bell('\0');
}
/*
Go back one entry in the history buffer and display it. If we are
already at the last entry, then beep.
*/
void backward_history(char ch) {
    int prev;
    prev = here - 1;

    if (prev < 0) prev = HISTORY_SIZE - 1;

    if (hist[prev].line != NULL) {
        clearline(ch);
        here = prev;
        length = hist[here].length;
        point = length;
        (void) strncpy(line, hist[here].line, length);
        echoline(line, length);
    }
    else
        bell('\0');
}

/* The following routines are utility routines used by the editing routines. */

/* Clear to the end of the current input line. */
void cleartoend(void) {
    fputs(cce, stdout);         /* send the clear character */
    fputs("\0\0\0\0", stdout);  /* send somes nulls for padding */
}
/*
Clear the input line. Backspace to the start of the line. Then clear
to the end of the line.
*/
void clearline(char ch) {
    int i;
    for (i = 0; i < point; i++)
        backspace(line[i]);
    cleartoend();
}
/*
Echo a character. Not all characters are created equal. Control characters
are echoed in ^X form. So they take up two character positions instead of
the normal 1 character position.
*/
void echo(char ch) {
    if (ch < ' ') {                     /* how should we echo the char? */
        (void) fputc('^', stdout);
        (void) fputc('@' + ch, stdout);
    }
    else
        (void) fputc(ch, stdout);
}
/* Echo a line. Print a whole line with control characters printed in ^X form. */
void echoline(char *line, int length) {
    int i;
    for (i = 0; i < length; i++)
        echo(*line++);
}
/*
Backspace over a character. Generate enough bs characters to backspace
over any character.
*/
void backspace(char ch) {
    if (ch < ' ') {
        fputs(cle, stdout);
        fputs(cle, stdout);
    } else
        fputs(cle, stdout);
}
/* Add any character to the line buffer. */
void quote_edit(char ch) {
    insert(ch);
    edit = edit_0;
}
/*
Given a character and an action table number either execute the
action or pass the string to (*edit)(ch)
*/
void dispatch(int table, char ch) {
    char *cptr;
    switch (action_table[table][ch].flag) {
    case is_action:
        (*(action_table[table][ch].aors.action)) (ch);
        break;
    case is_string:
        cptr = action_table[table][ch].aors.string;
        while ((*cptr) != '\0') {
            (*edit) (*cptr);
            cptr++;
        }
        break;
    }
}

/* Select an action from action_table[3] and execute it. */

void edit_3(char ch) {
    /* reset so that next input is handled by edit_0 unless over ridden by
     * the action. */
    edit = edit_0;
    dispatch(3, ch);
    (void) fflush(stdout);
}

/* Select an action from action_table[2] and execute it. */

void edit_2(char ch) {
    /* reset so that next input is handled by edit_0 unless over ridden by
     * the action. */
    edit = edit_0;
    dispatch(2, ch);
    (void) fflush(stdout);
}

/* Select an action from action_table[1] and execute it. */

void edit_1(char ch) {
    /* reset so that next input is handled by edit_0 unless over ridden by the action. */
    edit = edit_0;
    dispatch(1, ch);
    (void) fflush(stdout);
}

/* Select an action from action_table[0] and execute it. */

void edit_0(char ch) {
    dispatch(0, ch);
    (void) fflush(stdout);
}
/*
Input line editor.

Initialize the world. Then loop forever using select to wait for
characters to be available from either stdin or from master_pty.
When characters are available, pass them on after doing any needed
editing.
*/
void ile() {
    int i;
    char buffer[BUFFER_SIZE];   /* arguments for read and write calls */
    int cc;
    struct sgttyb slave_params; /* current slave_tty parameters */
    int nfds;                   /* Arguments for select call */
    int width;
    int readfds;

    (void) signal(SIGCHLD, clean_up);    /* what to do if the child or parent dies */
    (void) signal(SIGSEGV, clean_up);
    (void) signal(SIGBUS, clean_up);
    (void) signal(SIGTERM, clean_up);
    (void) signal(SIGHUP, clean_up);
    (void) signal(SIGINT, clean_up);
    (void) signal(SIGQUIT, clean_up);

    (void) signal(SIGWINCH, change_window);    /* what to do it the window changes size */

    /* copy the current ttys' state to the slave_tty. */

    (void) ioctl(READ, TIOCGETP, &tty_sgttyb);         /* tty sgttyb */
    (void) ioctl(slave_tty, TIOCSETP, &tty_sgttyb);
    (void) ioctl(READ, TIOCGETD, &tty_ldisc);          /* tty line discipline */
    (void) ioctl(slave_tty, TIOCSETD, &tty_ldisc);
    (void) ioctl(READ, TIOCGETC, &tty_tchars);         /* tty tchars */
    (void) ioctl(slave_tty, TIOCSETC, &tty_tchars);
    (void) ioctl(READ, TIOCLGET, &tty_mode);           /* tty mode */
    (void) ioctl(slave_tty, TIOCLSET, &tty_mode);
    (void) ioctl(READ, TIOCGLTC, &tty_ltchars);        /* tty ltchars */
    (void) ioctl(slave_tty, TIOCSLTC, &tty_ltchars);
    (void) ioctl(READ, TIOCGWINSZ, &tty_winsize);      /* tty windsize */
    (void) ioctl(slave_tty, TIOCSWINSZ, &tty_winsize);
    windowchanged = FALSE;

    (void) setutmp(slave_tty, TRUE);    /* "login" the user */

    {   struct sgttyb params;           /* set raw mode on tty */
        struct tchars tparams;
        struct ltchars ltparams;

/* Simulate RAW but allow original parity to work.  Thus use CBREAK with all options turned off. */
        params = tty_sgttyb;
        params.sg_flags = CBREAK;
        (void) ioctl(READ, TIOCSETP, &params);
        tparams = tty_tchars;
        tparams.t_intrc = -1;
        tparams.t_quitc = -1;
        tparams.t_startc = -1;
        tparams.t_stopc = -1;
        tparams.t_eofc = -1;
        tparams.t_brkc = -1;
        (void) ioctl(READ, TIOCSETC, &tparams);
        ltparams = tty_ltchars;
        ltparams.t_suspc = -1;
        ltparams.t_dsuspc = -1;
        ltparams.t_rprntc = -1;
        ltparams.t_flushc = -1;
        ltparams.t_lnextc = -1;
        (void) ioctl(READ, TIOCSLTC, &ltparams);
    }

    {   int mode;                           /* set new mode on tty */
        mode = LNOFLSH | LDECCTQ | LLITOUT;
        (void) ioctl(READ, TIOCLSET, &mode);
    }

    width = getdtablesize();                /* get descriptor table size */
    if (width > 32) width = 32;
    edit = edit_0;    /* set initial edit function */
    point = 0;    /* initialize line buffer */
    length = 0;
    head = 0;    /* initialize history buffer */
    here = 0;

    for (i = 0; i < HISTORY_SIZE; i++) {
        hist[i].length = 0;
        hist[i].line = NULL;
    }

    for (;;) {
        readfds = (1 << READ) | (1 << master_pty);
        /* wait for input from stdin or master_pty */
        nfds=select(width,(fd_set*)&readfds,(fd_set*)NULL,(fd_set*)NULL,(struct timeval*)NULL);

        if (nfds == -1) {                  /* an exception has occured */
            if (windowchanged)
                windowchanged = FALSE;     /* nothing serious, the window changed size */
            else {
                perror("ile");
                clean_up();
            }
        }
        else if ((nfds > 0) && (readfds != 0)) {      /* something to read */
            if ((readfds & (1 << master_pty)) != 0) { /* read the pending characters. */
                cc = read(master_pty, buffer, BUFFER_SIZE); /* display the characters. */
                (void) write(WRITE, buffer, cc);
            }
            if ((readfds & (1 << READ)) != 0) { /* read the pending characters. */
                cc = read(READ, buffer, BUFFER_SIZE);
/* if slave in RAW or CBREAK mode, or has turned off ECHO we shouldn't mess with its input characters */
                (void) ioctl(slave_tty, TIOCGETP, &slave_params);

                if (((slave_params.sg_flags&(RAW|CBREAK))!=0)||(slave_params.sg_flags&ECHO)==0)
                    edit = pass;
                else if (edit == (void (*) ()) pass)
                    edit = edit_0;
                for (i = 0; i < cc; i++)                /* decide what to do with the characters. */
                    (*edit) (CHAR_MASK & buffer[i]);
            }

        }
    }

}
/*
The child process.

Make the pty the processes controling terminal. Bind the pty to
stdin, stdout, and stderr. Then exec the users program.
*/
void child(char *argv[]) {
    char *shellname;                    /* shell name pointers */
    char *shellpath;
    char *dashshellname;

    (void) close(READ);                /* close all file descriptors */
    (void) close(WRITE);
    (void) close(ERROR);
    (void) close(slave_tty);

    {    /* get rid of controlling terminal */
        int tty;
        if (tty=open("/dev/tty", O_RDWR)==-1||ioctl(0, TIOCNOTTY, 0)==-1||close(tty)==-1)
            perror("ile");
    }

    /* open the tty again, this makes the pty the controlling terminal */

    if ((slave_tty = open(ttydev, O_RDWR)) == -1)
        perror("ile");

    (void) dup2(slave_tty, WRITE);    /* slave_tty is now stdin, bind slave_tty to stdout */
    (void) dup2(slave_tty, ERROR);    /* bind slave_tty to stderr */
    (void) close(master_pty);         /* close master_pty descriptor */

    /* Fire up application program. If no program is given fire up $SHELL */
    /* get the name of the users shell. default to rc */

    if ((shellpath = getenv("SHELL")) == NULL || (*shellpath == '\0'))
        shellpath = "rc";
    if ((shellname = strrchr(shellpath, '/')) != NULL)  /* strip leading path */
        shellname += sizeof(char);

    /* if the current argv[0] starts with -, then the new argv[0] should start with - */

    if (*(argv[0]) == '-') {
        dashshellname = (char *) malloc((unsigned) strlen(shellname) + 2);
        (void) strcpy(dashshellname, "-");
        (void) strcat(dashshellname, shellpath);
    }
    else
        dashshellname = shellname;

    if (argv[1] == NULL)
        execlp(shellpath, dashshellname, 0);        /* execute default shell */
    else if (*argv[1] == '-') {
        if (argv[2] == NULL)        /* there is an initialization file */
            execlp(shellpath, dashshellname, 0);    /* execute default shell */
        else
            execvp(argv[2], &argv[2]);              /* execute specified program */
    } else
        execvp(argv[1], &argv[1]);                  /* execute specified program */

    perror("ile");                                  /* this executes if exec fails */
    exit(1);
}

/* Set up default key bindings and delimeters. */

void default_bindings() {
    int i;
    for (i = 0; i < CHAR_SET_SIZE; i++) {    /* clear delimiter vector and the action table */
        delimit[i] = FALSE;
        action_table[0][i].aors.action = insert;
        action_table[1][i].aors.action = bell;
        action_table[2][i].aors.action = bell;
        action_table[3][i].aors.action = bell;
        action_table[0][i].flag = is_action;
        action_table[1][i].flag = is_action;
        action_table[2][i].flag = is_action;
        action_table[3][i].flag = is_action;
    }

    delimit[' '] = TRUE;        /* blank */    /* default delimiters */
    delimit['/'] = TRUE;        /* slash */
    delimit['.'] = TRUE;        /* dot */
    delimit['-'] = TRUE;        /* dash */

    action_table[0][esc].aors.action= escape_1;        /* default action_table[0] */
    action_table[0][CA].aors.action = start_of_line;
    action_table[0][CB].aors.action = backward_char;
    action_table[0][CE].aors.action = end_of_line;
    action_table[0][CF].aors.action = forward_char;
    action_table[0][CK].aors.action = erase_to_end_of_line;
    action_table[0][CU].aors.action = erase_line;
    action_table[0][CL].aors.action = retype_line;
    action_table[0][CN].aors.action = forward_history;
    action_table[0][CP].aors.action = backward_history;
    action_table[0][CR].aors.action = search_backward_history;
    action_table[0][CT].aors.action = transpose_chars;
    action_table[0][CV].aors.action = quote;
    action_table[0][del].aors.action= delete_char;
    action_table[0][cr].aors.action = add_to_history;
    action_table[0][nl].aors.action = add_to_history;
    action_table[0][CX].aors.action = delete_char_under;

    action_table[0][CC].aors.action = pass;
    action_table[0][CD].aors.action = pass;
    action_table[0][CQ].aors.action = pass;
    action_table[0][CS].aors.action = pass;
    action_table[0][CZ].aors.action = pass;

    action_table[1]['['].aors.action = escape_2;         /* default action_table[1] ^[ */
    action_table[1]['b'].aors.action = backward_word;
    action_table[1]['f'].aors.action = forward_word;
    action_table[1][del].aors.action = delete_word;
    action_table[1]['u'].aors.action = upper_word;
    action_table[1]['l'].aors.action = lower_word;
    action_table[1]['c'].aors.action = capitalize_word;
    action_table[1][esc].aors.action = complete_file;
    action_table[1]['s'].aors.action = complete_file_full;
    action_table[1]['d'].aors.action = show_files;

    action_table[2]['A'].aors.action = search_backward_history; /* default action_table[2] ^[ [ */
    action_table[2]['B'].aors.action = search_forward_history;
    action_table[2]['C'].aors.action = forward_char;
    action_table[2]['D'].aors.action = backward_char;

/*  action_table[2]['1'].aors.action = start_of_line;
    action_table[2]['4'].aors.action = end_of_line;
    action_table[2]['2'].aors.action = toggle_insert_mode; NEED TO IMPLEMENT
    action_table[2]['3'].aors.action = delete_char_under;
    action_table[2]['5'].aors.action = backward_history; SHOULD BE INCREMENTAL
    action_table[2]['6'].aors.action = forward_history;  SEARCHING WITH PGUP/DN */
}
/*
Return a character or EOF. This routine reads characters from input
and converts them into a character using the following rules.

The character may be a single character, a control
character indicated by ^x, an octal number starting with \, or an
escaped character indictated by \x.
*/
int scan_char(FILE *input) {
    int ch;
    int value;

    ch = fgetc(input);
    switch (ch) {
    case '^':        /* it is a control character */
        for (ch = fgetc(input); '@' <= ch; ch = ch - '@');
        break;
    case '\\':        /* octal or an escaped character? */
        ch = fgetc(input);
        if (('0' <= ch) && (ch <= '7')) {              /* its an octal number */
            value = 0;
            while (('0' <= ch) && (ch <= '7')) {
                value = (value * 8) + (ch - '0');
                ch = fgetc(input);
            }
            (void) ungetc(ch, input);
            ch = value & 0177;                         /* make sure it is in range */
        } else
            ch = fgetc(input);                         /* its an escaped character */
        break;
    case '\n':
        ch = EOL;                                      /* the real end of the line */
        break;
    default:                                           /* it is just itself */
        break;
    }
    return (ch);
}
/* Set key bindings and delimiters from the users file. */

#define NAME_SIZE 40

void user_bindings(FILE *file) {
    static struct action_name_table {
        char *name;
        void (*action) ();
    } action_name_table[] = {
        { "complete_file_full", complete_file_full },
        { "complete_file", complete_file },
        { "show_files", show_files },
        { "bell", bell },
        { "pass", pass },
        { "insert", insert },
        { "transpose_chars", transpose_chars },
        { "delete_char", delete_char },
        { "delete_char_under", delete_char_under },
        { "quote", quote },
        { "escape_1", escape_1 },
        { "escape_2", escape_2 },
        { "escape_3", escape_3 },
        { "delete_word", delete_word },
        { "upper_word", upper_word },
        { "lower_word", lower_word },
        { "capitalize_word", capitalize_word },
        { "forward_word", forward_word },
        { "backward_word", backward_word },
        { "start_of_line", start_of_line },
        { "backward_char", backward_char },
        { "end_of_line", end_of_line },
        { "forward_char", forward_char },
        { "add_to_history", add_to_history },
        { "erase_line", erase_line },
        { "erase_to_end_of_line", erase_to_end_of_line },
        { "retype_line", retype_line },
        { "forward_history", forward_history },
        { "backward_history", backward_history },
        { "search_backward_history", search_backward_history },
        { "search_forward_history", search_forward_history },
        { "", NULL }
    };

    char name[NAME_SIZE];
    int ch, i, linecount, table, entry;

    for (i = 0; i < CHAR_SET_SIZE; i++)    /* First clear the default delimiters */
        delimit[i] = FALSE;
/* Now read the delimiter characters */
    while (((int) (ch = fgetc(file)) != EOF) && (ch != '\n'))
        delimit[ch] = TRUE;
    linecount = 2;
/* Now read the character binding pairs */
    while ((int) (ch = fgetc(file)) != EOF)
        switch (ch) {
        case '\n': linecount++; break;            /* skipping a blank line */
        case '0':
        case '1':
        case '2':
        case '3':
            table = ch - '0';            /* which table is this entry directed to? */
            entry = scan_char(file);            /* get the character code */
            ch = fgetc(file);            /* make sure the '=' is there */
            if (ch != '=') {
                (void) fprintf(stderr, "ile: '=' missing on line %d\n", linecount);
                exit(1);
            }
            /* collect the action name or string */

            for(ch=scan_char(file),i=0; ((int)ch!=EOL)&&(i<(NAME_SIZE-1)); ch=scan_char(file),i++) {
                name[i] = ch;
                name[i + 1] = '\0';
            }
            for(i=0;            /* look it up in the action_name_table */
                (action_name_table[i].action!=NULL)&&(strcmp(name,action_name_table[i].name)!=0);
                i++);
            /* if it was found, put it in the action array */

            if (action_name_table[i].action == NULL) {                /* must be a string */
                action_table[table][entry].flag = is_string;
                action_table[table][entry].aors.string =
                    (char *) malloc((unsigned) strlen(name) + 1);
                (void) strcpy(action_table[table][entry].aors.string, name);
            } else {                /* its an action */
                action_table[table][entry].flag = is_action;
                action_table[table][entry].aors.action =
                    action_name_table[i].action;
            }
            linecount++;        /* count the line */
            break;

        default:
            (void) fprintf(stderr, "\nile: error in initialization file on line %d\n", linecount);
            exit(1);
        } 

    (void) fclose(file);
}

/* Initialize key bindings and delimiters. */

void initialize(char *argv[]) {
    FILE *file;
    char name[BUFFER_SIZE];
    char *pwd;

    default_bindings();    /* set up the default bindings */

    /* Look for an initialization file. If it's there, load it. */

    name[0] = '\0';
    homedir = getenv("HOME");
    if (homedir == NULL)
        name[0] = '\0';                        /* no home dir, use / instead */
    else
        (void) strcpy(name, homedir);
    (void) strcat(name, "/.ilerc");

/*  pwd = getenv("PWD");
    if (pwd == NULL)
        (void) strcpy(currentdir, homedir);
    else
        (void) strcpy(currentdir, pwd);
*/
    if ((argv[1] != NULL) && (*argv[1] == '-') && ((file = fopen(argv[1] + 1, "r")) != NULL))
        user_bindings(file);
    else if (((file = fopen("./.ilerc", "r")) != NULL) || ((file = fopen(name, "r")) != NULL))
        user_bindings(file);
}

int main(int argc, char *argv[]) {
    int childpid;    /* Child process id */

    getpty(&master_pty, &slave_tty);            /* create the tty/pty pair */
    get_termcap();                              /* get control sequences from termcap */
    initialize(argv);                           /* initialize the dispatch vectors */
    childpid = fork();                          /* create the child process */

    switch (childpid) {
        case 0:  child(argv);   break;          /* child process */
        case -1: perror("ile"); exit(1);        /* fork failed */
        default: sprintf(currentdir,"/proc/%d/cwd",childpid);
                 ile();
                 break;          /* parent process */
    
    
    }
}
