/* safedelete.c
  This program will make a compressed copy of a file and save it
  in a user-specified location and then remove it from the original
  location.  This will allow a file to be "undeleted" if the user so
  wishes.
*/

#include <stdlib.h>
#include <sys/types.h>
#include <unistd.h>
#include <fcntl.h>
#include <sys/stat.h>
#include <time.h>
#include <stdio.h>
#include <pwd.h>
#include <string.h>
#include <ctype.h>
#include <getopt.h>
#include "safedelete.h"

/* main routine

   Parses the command line parameters and safedeletes
   all the files listed.

   Passed: list of filenames

   Returns: 0 - all files safedelete successfully
            1 - one or more files not safedeleted
*/

int  main(int argc, char **argv)
{
int  i, j, k, randnum, ccode;
int  SafeDays, RetCode;

char *EnvValue, *LoginName, *HomeDir, c;
char *wrkptr, *palloc;

char WorkBuf[WORK_BUF_LEN];
char WorkBuf2[WORK_BUF_LEN];
char LogRec[WORK_BUF_LEN];
char FileName[FILE_NAME_LEN];
char GrungedFileName[GRUNGED_FILE_LEN];
char SafeDelLog[DEL_LOG_LEN];
char SafeDir[SAFE_DIR_LEN];
char SafeDelRC[SAFE_RC_LEN];

char CmdName[] = "safedelete";

char CompressCmd[41], UncompressCmd[41];
char FileSuff[9];

char *InfoLoc;

FILE *gfp;

time_t currtime;
struct tm *tm;
struct passwd *UserInfo;
struct stat DirInfo, FileInfo;

rcstruct *CompRClist, *SuffRClist, *DaysRClist;
safedays_struct *days;
compression_struct *compress;
suffix_struct *suffix;
file_struct Orig;

FILE *flog;

static int InteractiveFlag, VerboseFlag;

static struct option LongOpts[] =
{
    {"interactive", no_argument, &InteractiveFlag,   1},
    {"verbose"    , no_argument, &VerboseFlag    ,   1},
    {"force"      , no_argument, NULL            , 'f'},
    {"directory"  , no_argument, NULL            , 'd'},
    {"recursive"  , no_argument, NULL            , 'r'},
    {"help"       , no_argument, NULL            ,  98},
    {"version"    , no_argument, NULL            ,  99},
    {0            , 0          , 0               ,   0}
};

/* Initialize some variables and see
   if user specified any options.
*/

  InteractiveFlag=VerboseFlag=0;
  RetCode=ccode=0;
  DaysRClist=NULL;
  CompRClist=NULL;
  SuffRClist=NULL;
  days=NULL;
  compress=NULL;
  suffix=NULL;
  WorkBuf2[0]='\0';

  while((i=getopt_long(argc, argv, "dfivrR", LongOpts, &j)) != -1)
  {
    switch(i)
    {
	case 99  : printf("SafeDelete version 1.1a\n");
	case 'd' :
	case 'r' :
	case 'R' : strcpy(WorkBuf, "/bin/rm ");
                   for(i=1; i < argc; i++)
	  	   {
                     strcat(WorkBuf, argv[i]);
                     strcat(WorkBuf, " ");
		   }
                   ccode=system(WorkBuf);
                   return ccode;

	case 98  : printf("\nSafeDelete accepts the same options as the rm\n");
                   printf("command except for -d, -r and -R.\n");
                   printf("See the following for further details\n\n");
	           strcpy(WorkBuf, "/bin/rm ");
                   for(i=1; i < argc; i++)
	  	   {
                     strcat(WorkBuf, argv[i]);
                     strcat(WorkBuf, " ");
		   }
                   ccode=system(WorkBuf);
                   return ccode;

	case 'i' : InteractiveFlag=1;
                   break;

	case 'v' : VerboseFlag=1;
                   break;

	case '?' : if(isprint(optopt))
                     fprintf(stderr, "safedelete: unknown option '-%c'\n",
                             optopt);
                   else
                     fprintf(stderr, "safedelete: unknown option '\\x%x'\n",
                             optopt);
                   return 1;
	 default : break;
    }
  }

/* See if user specified a filename */

  if(optind == argc)
  {
    printf("safedelete: no filename specified\n");
    return 1;
  }

/* See if the variable $SAFEDAYS exists.  If not, default it to MAX_SAFE_DAYS.
   If it does exists, and it is zero, just call /bin/rm to delete the file.
*/

  EnvValue=getenv("SAFEDAYS");
  if(EnvValue == NULL)
    SafeDays=MAX_SAFE_DAYS;
  else
    SafeDays=atoi(EnvValue);

  if(SafeDays == 0)
  {
    strcpy(WorkBuf, "/bin/rm ");
    for(i=1; i < argc; i++)
    {
      strcat(WorkBuf, argv[i]);
      strcat(WorkBuf, " ");
    }
    ccode=system(WorkBuf);
    return ccode;
  }

/* Do some more initialization work
   before going any further
*/

  HomeDir=getenv("HOME");
  if(HomeDir == NULL)
  {
    LoginName=getlogin();
    UserInfo=getpwnam(LoginName);
    HomeDir=UserInfo->pw_dir;
  }
  strcpy(SafeDelLog, HomeDir);
  strcat(SafeDelLog, "/.safedelete.log");
  strcpy(SafeDelRC, HomeDir);
  strcat(SafeDelRC, "/.Safedelrc");

/* If the .safedelete.log already exists, make 
   sure we can write to it.
*/

  if(stat(SafeDelLog, &DirInfo) == 0)
    if(CheckPerms(NULL, SafeDelLog, CmdName) != 0)
      return 1;

/* Initialize the random number generator */

  srand(time(0));

/* set up directory for deleted files */

  strcpy(SafeDir, HomeDir);
  strcat(SafeDir, SAFE_DIR_NAME);
  
/* Now see if the actual directory exists and create
   it if needed.
*/

  ccode=stat(SafeDir, &DirInfo);
  if(ccode == 0)
  {
    if(! S_ISDIR(DirInfo.st_mode))
    {
      fprintf(stderr,
              "safedelete: %s is not a directory\n", SafeDir);
      return 1;
    }
    else
    {

/* Safedelete directory exists, make sure we can write to it */

      if(CheckPerms(SafeDir, NULL, CmdName) != 0)
        return 1;
    }
  }
  else
  {     
    if(ccode == -1 && errno == ENOENT)
    {
      if(mkdir(SafeDir, S_IRWXU) != 0)
      {
	fprintf(stderr, "safedelete: error creating %s\n", SafeDir);
	return 1;
      }
    }
    else
    {
      fprintf(stderr,
              "safedelete: error accessing %s\n", SafeDir);
      return 1;
    }
  }

/* If the users .safedelete.log has data in it, see
   if it's already in the format we need.  If it's not,
   invoke savecnvt to convert it.
*/

  ccode=stat(SafeDelLog, &FileInfo);
  if(ccode == 0 && FileInfo.st_size > 0)
  {
    palloc=malloc(FileInfo.st_size);
    flog=fopen(SafeDelLog, "r");
    fread(palloc, FileInfo.st_size, 1, flog);
    fclose(flog);
    if(*palloc == '/')
    {
      printf("safedelete: your .safedelete.log is still in the old format\n");
      printf("safedelete: it will be converted using the safecnvt command\n");
      system("safecnvt");

/* Now that it's converted, let's just double check */

      flog=fopen(SafeDelLog, "r");
      fread(palloc, FileInfo.st_size, 1, flog);
      fclose(flog);
      if(*palloc == '/')
      {
        printf("safedelete: conversion failed, cannot safely delete any files\n");
        free(palloc);
        return 1;
      }
      else
        printf("safedelete: your .safedelete.log was successfully converted\n");
    }
    free(palloc);
  }

/* Read the users .Safedelrc file */

  DaysRClist=ReadRC(SafeDelRC, "safedays");
  CompRClist=ReadRC(SafeDelRC, "compression");
  SuffRClist=ReadRC(SafeDelRC, "suffix");

/* At this point we have the number of days to
   keep each file and the directory where they are
   to be kept.  Now go through the filenames on
   the command line and process them.
*/

  for(i=optind; i < argc; i++)
  {

/* Set up .Safedelrc structure pointers */

     if(DaysRClist)
       days=DaysRClist->DaysStruct;
     if(CompRClist)
       compress=CompRClist->CompStruct;
     if(SuffRClist)
       suffix=SuffRClist->SuffStruct;

/* See if user specified absolute path name or relative path name.
   If relative, prefix the user-specified path name with $PWD.
*/

     if(argv[i][0] == '/')
      strcpy(FileName, argv[i]);
     else 
     { 

/* Filename does not start with /, build fully qualified filename */

       getcwd(FileName, 256);
       if(argv[i][0] == '.')
       {
         wrkptr=argv[i];
         while(*wrkptr == '.')
         {

/* See if user specified ./ */
	   
           if(strncmp(wrkptr, "./", 2) == 0)
	   {
             wrkptr+=2;
             continue;
	   }
	   
/* See if user specified ../ */

	   if(strncmp(wrkptr, "../", 3) == 0)
	   {
	     wrkptr+=3;
             j=ParseFileName(FileName);
             if(j == 0)
	       strcpy(FileName, "/");
	     else
	       FileName[j]='\0';
             continue;
           }

/* None of the above, must be a hidden file */

           break;
         }
         strcat(FileName, "/");
	 strcat(FileName, wrkptr);
       }
       else
       {
         strcat(FileName, "/");
         strcat(FileName, argv[i]);
       }
     }

/* At this point FileName contains an fully qualified file name,
   parse it into the file name and the directory path.
*/
     
     j=ParseFileName(FileName);
     *(FileName+j)='\0';

/* Check permissions of directory */

     if(CheckPerms(FileName, NULL, CmdName) != 0)
     {
       RetCode=1;
       break;
     }

/* Recreate the full path name of the file */

     *(FileName+j)='/';

/* Check permissions of file */

     if(CheckPerms(NULL, FileName, CmdName) != 0)
     {
       RetCode=1;
       break;
     }

/* If the user specified interactive mode, ask for permission
   to safedelete the file.
*/

     if(InteractiveFlag)
     {
       printf("safedelete: remove '%s'? ", FileName);
       c=getchar();                 /* get first character of response */
       while(getchar() != '\n');  /* ignore everything else except the enter key */
       if(c != 'y' && c != 'Y')
         continue;
     }

/* See if this file should be safedeleted by examining
   the SafeDays section of  the users .Safedelrc file.
   If the return code is 0, indicating the number of safe
   days for this file is 0, we just delete the file and go
   on to the next one.
*/

     k=SafeDays;
     if(DaysRClist)
     { 
       for(j=0; j < DaysRClist->NumDaysEnts; j++)
       {
         if((k=CheckFileDays(FileName, days)) == -1)
           days++;
         else
           break;
       }
     }
     if(k == 0)
     {
       if(VerboseFlag)
         printf("safedelete: %s\n", FileName);
       unlink(FileName);
       continue;
     }

/* Create a "grunged" file name with the following format:
   d<gregorian date>.t<time of day>.r<random number>

   Example:  d040595.t112715.r882719
*/

     currtime=time(0);
     tm=localtime(&currtime);
     strftime(WorkBuf, 7, "%m%d%y", tm);
     strcpy(GrungedFileName, "d");
     strcat(GrungedFileName, WorkBuf);
     strftime(WorkBuf, 7, "%H%M%S", tm);
     strcat(GrungedFileName, ".t");
     strcat(GrungedFileName, WorkBuf);
     randnum=rand();
     sprintf(WorkBuf, "%d", randnum);
     WorkBuf[6]='\0';
     strcat(GrungedFileName, ".r");
     strcat(GrungedFileName, WorkBuf);

/* Update the users $HOME/.safedelete.log with the following info:

       Original file name (fully qualified)
       Grunged file name (contains date/time file was deleted)
       File permissions
       UID of owner
       GID of owner
       size of file (in bytes)
       date/time file was last accessed
       date/time file was last modified
       contents of symbolic link, if file is a soft link
       uncompress command, if file was compressed
       compress command file suffix, if file was compressed

   We do this so the undelete command will work.  Without the
   log file, nothing can be undeleted.

   Since the record will contain binary data, we need to
   use the fwrite subroutine to write it to the log.  This
   also means that we can't use a terminating newline
   character to indicate where the end of the record is.
   We therefore reserve the first 2 bytes of each record for
   the length.  The length kept in these 2 bytes will be
   the length of the record + 2 bytes (making it the offset
   to the next record when reading the file).
*/

     wrkptr=LogRec+2;
     strcpy(wrkptr, FileName);
     wrkptr+=strlen(wrkptr)+1;
     strcpy(wrkptr, GrungedFileName);
     wrkptr+=strlen(wrkptr)+1;

/* Check the length of the record so far.  If not on
   an even byte boundary add '>' as a pad character.
   We do this to keep all the binary integer data on
   even boundaries since the 386/486 chips access data
   more efficiently if the data is aligned this way.
*/

     if((int)(wrkptr-LogRec)%2)
     {
       *wrkptr=127;
       wrkptr++;
     }
     InfoLoc=wrkptr;

/* Before going any further, we need to get some information
   from the file's inode.
*/

     if(lstat(FileName, &FileInfo) != 0)
       stat(FileName, &FileInfo);

/* We only work with regular files and links.  Anything
   else we leave to the usual /bin/rm command.
*/

     if(! S_ISREG(FileInfo.st_mode) && ! S_ISLNK(FileInfo.st_mode))
     {
       fprintf(stderr, "safedelete: only regular files and links can be safedeleted\n");
       fprintf(stderr, "safedelete: file %s skipped\n", FileName);
       continue;
     }

/* Save original file inode information */

     Orig.Version=VERSION;
     Orig.FileFlag=0;
     Orig.Mode=FileInfo.st_mode;
     Orig.Uid=FileInfo.st_uid;
     Orig.Gid=FileInfo.st_gid;
     Orig.Size=FileInfo.st_size;
     Orig.Atime=FileInfo.st_atime;
     Orig.Mtime=FileInfo.st_mtime;

/* If file is really a symbolic link, read it and save for use by undelete */

     if(S_ISLNK(FileInfo.st_mode))
     {
       if((j=readlink(FileName, WorkBuf2, FileInfo.st_size)) > 0)
       {
         *(WorkBuf2+j)='\0';

/* Write out the contents of the symbolic link as the file */

         k=ParseFileName(FileName);
         sprintf(WorkBuf, "%s/%s", SafeDir, FileName+k+1);
         gfp=fopen(WorkBuf, "w");
         fwrite(WorkBuf2, j, 1, gfp);
         fclose(gfp);
       }
     }

/* Only regular files are compressed (symbolic links are not).
   Use gzip for the default compress command
   and .gz as the default suffix.
*/

     if(S_ISREG(FileInfo.st_mode))
     {
       strcpy(CompressCmd, "gzip");
       strcpy(UncompressCmd, "gunzip");
       strcpy(FileSuff, ".gz");

/* Check the filename's pattern against what the
   user specified in the .Safedelrc file, if it exists.
*/

       if(CompRClist != NULL)
       {
         for(j=0; j < CompRClist->NumCompressEnts; j++)
         {

/* See if this file's suffix uses a particular compression routine */

           if(CheckFilePattern(FileName, compress) == 0)
           {
             strcpy(CompressCmd, compress->CompCmd);
             strcpy(UncompressCmd, compress->UncompCmd);
             if(strcmp(CompressCmd, "none") != 0)
             {
               if(SuffRClist != NULL)
               {
                 for(k=0; k < SuffRClist->NumSuffixes; k++)
                 {

/* See what suffix the compression routine appends to the filename */

                   if(strncmp(CompressCmd, suffix->CmdName, strlen(suffix->CmdName)) == 0)
                   {
                     strcpy(FileSuff, suffix->Suffix);
                     break;
                   }
                   suffix++;
                 }
               }
	     }
             break;
           }
           compress++;
	 }
       }
     }
     else
       strcpy(CompressCmd, "none");

/* If user specified verbose mode, let'em know we're
   gettin' rid of this file.
*/

     if(VerboseFlag)
       printf("safedelete: %s\n", FileName);

/* Copy the file to the safedelete directory */

     if(! S_ISLNK(FileInfo.st_mode))
     {
       sprintf(WorkBuf, "cp %s %s", FileName, SafeDir);
       system(WorkBuf);
     }

/* Compress the file, if we're supposed to */

     j=ParseFileName(FileName);
     if(strcmp(CompressCmd, "none") != 0)
     {
       sprintf(WorkBuf, "%s %s/%s 2>/dev/null", CompressCmd, SafeDir, FileName+j+1);
       system(WorkBuf);

/* Append the suffix, added by the compression command,
   then rename the file to the grunged file name.
*/

       sprintf(WorkBuf, "%s/%s%s", SafeDir, FileName+j+1, FileSuff);

/* If the compress failed, indicate this then use
   the original filename for the rename step.
*/

       if(stat(WorkBuf, &FileInfo) == 0)
       {
         memcpy(InfoLoc, &Orig, sizeof(Orig));
         wrkptr=InfoLoc+sizeof(Orig);
         strcpy(wrkptr, UncompressCmd);
         wrkptr+=strlen(wrkptr)+1;
         strcpy(wrkptr, FileSuff);
         wrkptr+=strlen(wrkptr)+1;
       }
       else
       {
         Orig.FileFlag|=NOCOMPRESS;
         memcpy(InfoLoc, &Orig, sizeof(Orig));
         wrkptr=InfoLoc+sizeof(Orig);
         sprintf(WorkBuf, "%s/%s", SafeDir, FileName+j+1);
       }
     }
     else
     {
       Orig.FileFlag|=NOCOMPRESS;
       memcpy(InfoLoc, &Orig, sizeof(Orig));
       wrkptr=InfoLoc+sizeof(Orig);
       if(strlen(WorkBuf2) > 0)
       {
         strcpy(wrkptr, WorkBuf2);
         wrkptr+=strlen(WorkBuf2)+1;
         *wrkptr='\0';
       }
       sprintf(WorkBuf, "%s/%s", SafeDir, FileName+j+1);
     }

     if((int)(wrkptr-LogRec)%2)
     {
       wrkptr++;
       *wrkptr='\0';
     }
     j=(int)(wrkptr-LogRec);
     memcpy(LogRec, &j, 2);

     sprintf(WorkBuf2, "%s/%s", SafeDir, GrungedFileName);

     rename(WorkBuf, WorkBuf2);

/* Now set up the permissions so that only the owner can read
   the files.
*/

     chmod(WorkBuf2, S_IRUSR);

/* Update the users .safedelete.log with the new entry */

     if(stat(SafeDelLog, &FileInfo) == 0)
       flog=fopen(SafeDelLog, "a");
     else
       flog=fopen(SafeDelLog, "w");

     memcpy(&j, LogRec, 2);

     fwrite(LogRec, j, 1, flog);

     fclose(flog);

     chmod(SafeDelLog, S_IRUSR | S_IWUSR);

/* Now that we're all done, really delete the original file */

     unlink(FileName);
  }

/* Free the RC list arrays */

  if(days != NULL)
    free(days);
  if(suffix != NULL)
    free(suffix);
  if(compress != NULL)
    free(compress);

/* All done deleting files */

  return RetCode;
}
