/* safedelchk.c
   This scans through the systems safedelete
   directory looking for files older than
   the number of days specified and deleting
   them.

   Usage:  safedelchk [-d nn] [-u]

           -d : number of days after which
                the saved files will be
                deleted. If not specified
		the maximum set at install
		time will be used.

           -u : force user mode when running
                as root.

   This command is meant to be run in 2
   distinct environments:
   - from each users .bash_profile
   - from a nightly scheduled crontab as root
   It can also be run on an ad hoc
   basis, if needed.
   
   Only root can run safedelchk with the
   -d and -u options.  All other users are
   forced to use either the $SAFEDAYS option
   or the installation-defined default.

   When examining the date of each file
   in the safedelete directory, the date
   embedded in the file name is used instead
   of the last-updated date in the directory.
   This allows the files to be deleted even
   though someone may "touch" every file in
   their subdirectory.
*/

#include <unistd.h>
#include <stdio.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <string.h>
#include <stdlib.h>
#include <time.h>
#include <dirent.h>
#include <pwd.h>
#include "safedelete.h"

/* ================================================= */

/* ComputeDate
   Compute the oldest date a safedeleted file can remain on
   the system.

   Passed: number of days
           pointer to stucture of type tm

   Returns: the correct date in the tm structure

   Example: passed NumDays=4  tm -> 05/12/95
            returns tm -> 05/08/95
*/

void ComputeDate(int NumDays, struct tm *tm)
{
int  DaysPerMonth[12] = {31, 28, 31, 30, 31, 30, 31, 31,
                       30, 31, 30, 31};

/* Allow for leap years */

  if(tm->tm_year%4 == 0 && tm->tm_year%8 != 0)
    DaysPerMonth[1]=29;

/* See if we go past a month/year boundary */

  tm->tm_mday-=NumDays;
  if(tm->tm_mday <= 0)
  {
    tm->tm_mon=(tm->tm_mon == 1 ? 12 : --tm->tm_mon);
    tm->tm_mday+=DaysPerMonth[tm->tm_mon];
    if(tm->tm_mon == 12)
      tm->tm_year=(tm->tm_year == 0 ? 99 : --tm->tm_year);
  }

}

/* ============================================================= */

/* PurgeFiles
   Purge all the files in the specified directory which are
   older than the provided date.

   Passed: pointer to safedelete directory name
           pointer to the high level qualifier containing the
           date of the oldest files  (e.g. "d041395")
           pointer to the user's safedelete log file name to update

   Returns: 0 if purge went OK
            1 if purge failed
*/

int PurgeFiles(char *DirName, char *OldestDate, char *SafeDelLog, rcstruct *rc)
{
DIR *dir;
struct dirent *dirent;
struct stat FileInfo;
struct tm *tm;
time_t currtime;
int  i, j, k, retcode;
int  GotMatch, NumDels;
FILE *flog;
char *LogList, *palloc, *wrkptr;
char WorkBuf[WORK_BUF_LEN];
char CmdName[] = "safedelchk";

safedays_struct *days;

/* Check permissions to the safedelete directory and
   to the user's log file.
*/

  if(CheckPerms(DirName, SafeDelLog, CmdName) != 0)
    return 1;

  retcode=0;

/* Read in the entire safedelete log so it's easier
   to scan and update.  The format will be a null
   terminated entry for each line with a null
   delimiter to indicate the end of list.
*/

  stat(SafeDelLog, &FileInfo);
  LogList=palloc=malloc(FileInfo.st_size);

  flog=fopen(SafeDelLog, "r");
  fread(LogList, FileInfo.st_size, 1, flog);
  fclose(flog);

/* First make sure the .safedelete.log is in the
   format we require.
*/

  if(*LogList == '/')
  {
    free(palloc);

    printf("safedelchk: %s is still in the old format\n", SafeDelLog);
    printf("safedelchk: it will be converted using the safecnvt command\n");
    system("safecnvt");

/* Now that it's converted, let's just double check */

    stat(SafeDelLog, &FileInfo);
    LogList=palloc=malloc(FileInfo.st_size);

    flog=fopen(SafeDelLog, "r");
    fread(LogList, FileInfo.st_size, 1, flog);
    fclose(flog);

    if(*LogList == '/')
    {
      printf("safedelchk: conversion failed, cannot continue\n");
      free(palloc);
      return 1;
    }
    else
      printf("safedelchk: %s was successfully converted\n", SafeDelLog);
  }

/* Go through the .safedelete.log and flag all entries
   which are to be removed.
*/

  NumDels=0;
  j=k=0;
  while(j < FileInfo.st_size)
  {
    memcpy(&k, LogList, 2);
    wrkptr=LogList+2+strlen(LogList+2)+1;   /* point to grunged filename */
    GotMatch=0;

/* If user has a .Safedelrc file, scan the Safedays section */

    if(rc)
    {
      days=rc->DaysStruct;
      for(i=0; i < rc->NumDaysEnts; i++)
      {
        if(CheckFileDays(LogList+2, days) >= 0)
        {
          GotMatch=1;
          currtime=time(0);
          tm=localtime(&currtime);
          ComputeDate(days->Days, tm);
          strcpy(WorkBuf, "d");
          strftime(WorkBuf+1, 7, "%m%d%y", tm);
          if(strncmp(wrkptr, WorkBuf, 8) <= 0)
          {
            *(LogList+2)='$';
            sprintf(WorkBuf, "%s/%s", DirName, wrkptr);
            unlink(WorkBuf);
            NumDels++;
          }
          break;   
        }
        days++;
      }
    }

    if(! GotMatch)
    {
      if(strncmp(wrkptr, OldestDate, 8) < 0)
      {
        *(LogList+2)='$';
        sprintf(WorkBuf, "%s/%s", DirName, wrkptr);
        unlink(WorkBuf);
        NumDels++;
      }
    }
    j+=k;
    LogList+=k;
  }
 
  printf("safedelchk: %d file(s) removed\n", NumDels);

/* Read the safedelete directory and delete the old files.
   Also, make sure any remaining files are listed in the
   log.
*/

  dir=opendir(DirName);
  while((dirent=readdir(dir)))
  {
    if(dirent->d_name[0] == '.')
      continue;

/* Here we compare the filename in the directory against
   the in-storage list read from the .safedelete.log.  If an
   entry in the log was deleted (first byte = "$") then
   it is ignored.  If the filename from the directory matches
   an entry in the log, we flag it by setting the first byte
   to "f".  We can get away with this because each of the
   entries is a fully qualified path beginning with "/".  We'll
   restore the "/" just before we rewrite the log.
*/

    else
    {
      i=j=k=0;
      LogList=palloc;
      while(j < FileInfo.st_size)
      {
        memcpy(&k, LogList, 2);
        if(*(LogList+2) != '$')
        {
          wrkptr=LogList+2+strlen(LogList+2)+1;
          if(strncmp(wrkptr, dirent->d_name, strlen(wrkptr)) == 0)
          {
            i=1;
            *(LogList+2)='f';
            break;
          }
        }
        j+=k;
        LogList+=k;
      }
      if(! i)
      {
        fprintf(stderr,
                "safedelchk: .safedelete.log has no entry for saved file %s\n",
                dirent->d_name);
        retcode=1;
      } 
    }
  }
  closedir(dir);

/* Now we go through the in-storage log list
   looking for any entries that were not found
   in the safedelete directory.  We also reset the
   found flag (first char = "f") back to "/".  Again,
   we ignore any files that were deleted (first
   byte = "$");
*/

  j=k=0;
  LogList=palloc;
  while(j < FileInfo.st_size)
  {
    memcpy(&k, LogList, 2);
    if(*(LogList+2) != '$')
    {
      if(*(LogList+2) == 'f')
        *(LogList+2)='/';
      else
      {
        fprintf(stderr,
                "safedelchk: safedelete file for %s not found in %s\n",
                LogList+2, DirName);
        retcode=1;
      }
    }
    j+=k;
    LogList+=k;
  }

/* Finally, we reset the file pointer to the beginning
   of the file and write out the remaining log entries.
*/

  flog=fopen(SafeDelLog, "w");
  j=k=0;
  LogList=palloc;
  while(j < FileInfo.st_size)
  {
    memcpy(&k, LogList, 2);
    if(*(LogList+2) != '$')
      fwrite(LogList, k, 1, flog);
    j+=k;
    LogList+=k;
  }
  fclose(flog);
  free(palloc);
  return retcode;
}

   
/* ============================================================= */  

int main(int argc, char *argv[])
{
int  c, ccode, NumDays, retcode;
char *EnvValue, *HomeDir, *LoginName = NULL;
char SafeDir[SAFE_DIR_LEN];
char SafeDelLog[DEL_LOG_LEN];
char SafeDelRC[SAFE_RC_LEN];
char OldestDate[9];

int  UserMode;

time_t currtime;
struct tm *tm;
struct passwd *UserInfo;
struct stat DirInfo;
uid_t uid;

rcstruct *DaysRClist;

/* Do some initializtion work */

  currtime=time(0);
  tm=localtime(&currtime);
  retcode=0;
  NumDays=MAX_SAFE_DAYS;

  printf("safedelchk: scan beginning\n");

/* First we get the uid of the person running
   the command.  If it's superuser, then we
   look for any options.  If it's Joe User,
   we look for the $SAFEDAYS variable.
*/

  uid=getuid();
  if(uid == 0)
  {
    
/* See if superuser specified any options */

    UserMode=0;

    while((c=getopt(argc, argv, "d:u")) != -1)
    {
      switch(c)
      {
        case 'd' : NumDays=atoi(optarg);
                   break;

        case 'u' : UserMode=1;
                   break;

        case '?' :
        default  : fprintf(stderr, "safedelchk: invalid option %c\n", c);
                   return 1;
      }
    }
  }
  else
  {

/* Normal user -- just look for $SAFEDAYS */
    
    UserMode=1;
    if((EnvValue=getenv("SAFEDAYS")))
      NumDays=atoi(EnvValue);
  }

  NumDays=(NumDays <= MAX_SAFE_DAYS ? NumDays : MAX_SAFE_DAYS);

/* Now get the date of NumDays ago */

  ComputeDate(NumDays, tm);

/* And build a file name out of it */

  strcpy(OldestDate, "d");
  strftime(OldestDate+1, 7, "%m%d%y", tm);


/* Now we're ready to start deleting files and
   log entries.  If running in user mode, everything
   is all set up.  If running administrator mode,
   we go through the /etc/passwd file and examine 
   the directories and logs for each user with
   safedeleted files.
*/
  if(UserMode)
  {

/* Finish setting up things for
   user mode processing
*/

    HomeDir=getenv("HOME");
    if(HomeDir == NULL)
    {
      LoginName=getlogin();
      UserInfo=getpwnam(LoginName);
      HomeDir=UserInfo->pw_dir;
    }
    strcpy(SafeDelLog, HomeDir);
    strcat(SafeDelLog, "/.safedelete.log");
    strcpy(SafeDir, HomeDir);
    strcat(SafeDir, SAFE_DIR_NAME);
    strcpy(SafeDelRC, HomeDir);
    strcat(SafeDelRC, "/.Safedelrc");

    DaysRClist=ReadRC(SafeDelRC, "safedays");

    if(DaysRClist && DaysRClist->NumDaysEnts > 0)
      printf("safedelete: using safedays values in your .Safedelrc\n");
    else
      printf("safedelchk: removing files safedeleted before %c%c/%c%c/%c%c\n",
           OldestDate[1], OldestDate[2], OldestDate[3], OldestDate[4],
           OldestDate[5], OldestDate[6]);


/* Purge the files and prune the log for the user */

    ccode=PurgeFiles(SafeDir, OldestDate, SafeDelLog, DaysRClist);

/* Release the memory obtained by ReadRC */

    if(DaysRClist)
      free(DaysRClist->DaysStruct);

    if(ccode != 0)
      retcode=1;
  }
  else
  {
    
/* Administrator mode -- check all users */

    setpwent();
    while((UserInfo=getpwent()) != NULL)
    {
      strcpy(SafeDir, UserInfo->pw_dir);
      strcat(SafeDir, SAFE_DIR_NAME);
      if(stat(SafeDir, &DirInfo) == 0)
      {
        printf("safedelchk: examining files for user %s\n",
               UserInfo->pw_name);

        strcpy(SafeDelLog, UserInfo->pw_dir);
        strcat(SafeDelLog, "/.safedelete.log");
        strcpy(SafeDelRC, UserInfo->pw_dir);
        strcat(SafeDelRC, "/.Safedelrc");

        DaysRClist=ReadRC(SafeDelRC, "safedays");

        ccode=PurgeFiles(SafeDir, OldestDate, SafeDelLog, DaysRClist);

        if(DaysRClist)
          free(DaysRClist->DaysStruct);

        if(ccode != 0)
          retcode=1;
      }
    }
    endpwent();
  }   

  printf("safedelchk: scan completed\n");
  return retcode;
}
    

