/* utmp.c: handles the utmp file and changing tty modes.
 *
 * todd j. derr <tjd@haven.zets.org>
 *
 * 	This program is free software; you can redistribute it and/or
 *	modify it under the terms of the GNU General Public License
 *	as published by the Free Software Foundation; either version
 *	2 of the License, or (at your option) any later version.
 */

#include <unistd.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <fcntl.h>
#include <utmp.h>
#include <syslog.h>
#include <sys/stat.h>
#include <sys/types.h>

#include "config.h"

#ifdef TTY
/* if you have to change this, see below */
#define VTNAME "/dev/tty%d"
#define NR_VCS 63
static pid_t ttypid[NR_VCS+1]={0,};
#endif

/* getutent() and friends are not nice things at all. */

/* we call this before spawning a console to create a utmp entry.
 * doesn't touch wtmp, we assume login will do that. (?)
 */

#ifdef TTY
void create_utmp(pid_t pid, char *device,int ttyno,gid_t tty_gid)
#else
void create_utmp(pid_t pid, char *device)
#endif
{
	struct utmp tmp;
	int fd,found=0;

#ifdef TTY
	if(tty_gid != (gid_t) -1) {		/* try to be nice */
		chown(device,-1,tty_gid);
		chmod(device,TTYMODE);
	}
	ttypid[ttyno]=pid;
#endif

	if((fd=open(UTMP_FILE,O_RDWR)) < 0)
	{
		syslog(LOG_WARNING,"open(%s): %m\r\n",UTMP_FILE);
		return;
	}

	while(read(fd,&tmp,sizeof(struct utmp))==sizeof(struct utmp))
	{
		if(tmp.ut_pid==pid || tmp.ut_type < 5 || tmp.ut_type == 8 ||
		    (tmp.ut_id[0]==device[8] && tmp.ut_id[1]==device[9]
				&& tmp.ut_type != 5))
		{
			found=1;
			break;
		}
	}

	bzero(&tmp,sizeof(struct utmp));
	tmp.ut_time=time(NULL);
	tmp.ut_pid=pid;
	tmp.ut_type=LOGIN_PROCESS;
	strcpy(tmp.ut_line,device+5);  /* skip /dev/ */
	strncpy(tmp.ut_id,device+8,2); /* skip 'tty' */

	if(found)
		lseek(fd,-sizeof(struct utmp), SEEK_CUR);
	else
		lseek(fd,0,SEEK_END);

	write(fd,&tmp,sizeof(struct utmp));

	close(fd);
}


/* and this gets rid of a utmp entry.  Sets the type to 0, which seems
 * to be what SysVinit likes to do...
 * also makes a wtmp entry.
 */
#ifdef TTY
void remove_utmp(pid_t pid,uid_t tty_uid,gid_t tty_gid)
#else
void remove_utmp(pid_t pid)
#endif
{
	int fd;
	struct utmp tmp,blank;

	if((fd=open(UTMP_FILE,O_RDWR)) < 0)
	{
		syslog(LOG_WARNING,"open(%s): %m\r\n",UTMP_FILE);
		return;
	}

	while(read(fd,&tmp,sizeof(struct utmp))==sizeof(struct utmp))
	{
		if(tmp.ut_pid == pid)
			break;
	}

	if(tmp.ut_pid != pid)
	{
		syslog(LOG_WARNING,"no utmp entry for pid %d\r\n",pid);
		return;
	}
#ifdef TTY
	else	/* try to restore the tty mode... */
	{
		
		char device[sizeof VTNAME +2];
		unsigned int i;
		/* don't trust utmp... */	
		for(i=0;i<=NR_VCS;++i)
		{
			if(ttypid[i]==pid) break;
		}
		/* not worth complaining if we didn't find it, but... */
	
		if(i<=NR_VCS && ttypid[i]>1 )
		{
			sprintf(device,VTNAME,i);
			ttypid[i]=1;
			chown(device,tty_uid,tty_gid);
			chmod(device,TTYMODE);
		}
	}
#endif

	tmp.ut_type=0;
	bzero(tmp.ut_name,UT_NAMESIZE);
	tmp.ut_time=time(NULL);

	/* backtrack */
	lseek(fd, -sizeof(struct utmp), SEEK_CUR);
	bzero(&blank,sizeof(struct utmp));
	write(fd,&blank,sizeof(struct utmp));
	close(fd);
		
	/* now do wtmp */
	if((fd=open(WTMP_FILE,O_WRONLY|O_APPEND)) >= 0)
	{
		write(fd,&tmp,sizeof(struct utmp));
		close(fd);
	}
	else
	{
		syslog(LOG_WARNING,"open %s: %m\r\n",WTMP_FILE);
	}
}
