/*
 * $Id: utils.c,v 1.40 1995/05/30 06:21:41 coleman Exp coleman $
 *
 * utils.c - miscellaneous utilities
 *
 * This file is part of zsh, the Z shell.
 *
 * Copyright (c) 1992-1995 Paul Falstad
 * All rights reserved.
 *
 * Permission is hereby granted, without written agreement and without
 * license or royalty fees, to use, copy, modify, and distribute this
 * software and its documentation for any purpose, provided that the
 * above copyright notice and the following two paragraphs appear in
 * all copies of this software.
 *
 * In no event shall Paul Falstad or the Zsh Development Group be liable
 * to any party for direct, indirect, special, incidental, or consequential
 * damages arising out of the use of this software and its documentation,
 * even if Paul Falstad and the Zsh Development Group have been advised of
 * the possibility of such damage.
 *
 * Paul Falstad and the Zsh Development Group specifically disclaim any
 * warranties, including, but not limited to, the implied warranties of
 * merchantability and fitness for a particular purpose.  The software
 * provided hereunder is on an "as is" basis, and Paul Falstad and the
 * Zsh Development Group have no obligation to provide maintenance,
 * support, updates, enhancements, or modifications.
 *
 */

#include "zsh.h"

/* source a file */

/**/
int
source(char *s)
{
    int fd        = SHIN;            /* store the shell input fd                   */
    int cj        = thisjob;         /* store our current job number               */
    int oldlineno = lineno;          /* store our current lineno                   */
    int oldshst   = opts[SHINSTDIN]; /* store current value of this option         */
    FILE *obshin  = bshin;           /* store file handle for buffered shell input */
    int osubsh    = subsh;           /* store whether we are in a subshell         */
    /* Current while, for, ... loops are invisible to the source'd script          */
    int oloops    = loops;           /* stored the # of nested loops we are in     */

    loops  = 0;
    lineno = 0;
    opts[SHINSTDIN] = OPT_UNSET;
    if ((SHIN = movefd(open(s, O_RDONLY))) == -1) {
	SHIN = fd;
	thisjob = cj;
	opts[SHINSTDIN] = oldshst;
	return 1;
    }
    bshin = fdopen(SHIN, "r");
    subsh = 0;
    sourcelevel++;
    loop(0);                    /* now we got everything set up, so loop through *
                                 * the file to be sourced                        */
    sourcelevel--;
    fclose(bshin);
    bshin = obshin;             /* restore file handle for buffered shell input */
    subsh = osubsh;             /* restore whether we are in a subshell         */
    opts[SHINSTDIN] = oldshst;  /* restore old value of this option             */
    SHIN = fd;                  /* restore the shell input fd                   */
    thisjob = cj;               /* restore current job number                   */
    loops = oloops;             /* restore the # of nested loops we are in      */
    lineno = oldlineno;         /* restore our current lineno                   */
    errflag = 0;
    retflag = 0;
    return 0;
}

/* try to source a file in the home directory */

/**/
void
sourcehome(char *s)
{
    char buf[PATH_MAX];
    char *h;

    if (!(h = getsparam("ZDOTDIR")))
	h = home;
    sprintf(buf, "%s/%s", h, s);
    (void)source(buf);
}

/* print an error */

/**/
void
zwarnnam(char *cmd, char *fmt, char *str, int num)
{
    int waserr = errflag;

    zerrnam(cmd, fmt, str, num);
    errflag = waserr;
}

/**/
void
zerrnam(char *cmd, char *fmt, char *str, int num)
{
    if (cmd) {
	if (errflag || noerrs)
	    return;
	errflag = 1;
	trashzle();
	if (isset(SHINSTDIN))
	    fprintf(stderr, "%s: ", cmd);
	else
	    fprintf(stderr, "%s: %s: ", argzero, cmd);
    }
    while (*fmt)
	if (*fmt == '%') {
	    fmt++;
	    switch (*fmt++) {
	    case 's':
		while (*str)
		    niceputc(*str++, stderr);
		break;
	    case 'l':
		while (num--)
		    niceputc(*str++, stderr);
		break;
	    case 'd':
		fprintf(stderr, "%d", num);
		break;
	    case '%':
		putc('%', stderr);
		break;
	    case 'c':
		niceputc(num, stderr);
		break;
	    case 'e':
		if (num == EINTR) {
		    fputs("interrupt\n", stderr);
		    errflag = 1;
		    return;
		}
		if (num == EIO)
		    fputs(sys_errlist[num], stderr);
		else {
		    fputc(tulower(sys_errlist[num][0]), stderr);
		    fputs(sys_errlist[num] + 1, stderr);
		}
		break;
	    }
	} else
	    putc(*fmt++, stderr);
    if (unset(SHINSTDIN) && lineno)
	fprintf(stderr, " [%ld]\n", lineno);
    else
	putc('\n', stderr);
    fflush(stderr);
}

/**/
void
zerr(char *fmt, char *str, int num)
{
    if (errflag || noerrs)
	return;
    errflag = 1;
    trashzle();
    fprintf(stderr, "%s: ", (isset(SHINSTDIN)) ? "zsh" : argzero);
    zerrnam(NULL, fmt, str, num);
}

/**/
void
niceputc(int c, FILE *f)
{
    if (itok(c)) {
	if (c >= Pound && c <= Comma)
	    putc(ztokens[c - Pound], f);
	return;
    }
    c &= 0xff;
    if (isprint(c))
	putc(c, f);
    else if (c == '\n') {
	putc('\\', f);
	putc('n', f);
    } else {
	putc('^', f);
	putc(c | '@', f);
    }
}

/* get a symlink-free pathname for s relative to PWD */

/**/
char *
findpwd(char *s)
{
    char *t;

    if (*s == '/')
	return xsymlink(s);
    s = tricat((pwd[1]) ? pwd : "", "/", s);
    t = xsymlink(s);
    zsfree(s);
    return t;
}

/* Check whether a string contains the *
 * name of the present directory.      */

/**/
int
ispwd(char *s)
{
    struct stat sbuf, tbuf;

    if (stat(s, &sbuf) == 0 && stat(".", &tbuf) == 0)
	if (sbuf.st_dev == tbuf.st_dev && sbuf.st_ino == tbuf.st_ino)
	    return 1;
    return 0;
}

static char xbuf[PATH_MAX];

/* expand symlinks in s, and remove other weird things */

/**/
char *
xsymlink(char *s)
{
    if (unset(CHASELINKS))
	return ztrdup(s);
    if (*s != '/')
	return NULL;
    strcpy(xbuf, "");
    if (xsymlinks(s + 1, 1))
	return ztrdup(s);
    if (!*xbuf)
	return ztrdup("/");
    return ztrdup(xbuf);
}

/**/
char **
slashsplit(char *s)
{
    char *t, **r, **q;
    int t0;

    if (!*s)
	return (char **)zcalloc(sizeof(char **));

    for (t = s, t0 = 0; *t; t++)
	if (*t == '/')
	    t0++;
    q = r = (char **)zalloc(sizeof(char **) * (t0 + 2));

    while ((t = strchr(s, '/'))) {
	*t = '\0';
	*q++ = ztrdup(s);
	*t = '/';
	while (*t == '/')
	    t++;
	if (!*t) {
	    *q = NULL;
	    return r;
	}
	s = t;
    }
    *q++ = ztrdup(s);
    *q = NULL;
    return r;
}

/* expands symlinks and .. or . expressions */
/* if flag = 0, only expand .. and . expressions */

/**/
int
xsymlinks(char *s, int flag)
{
    char **pp, **opp;
    char xbuf2[PATH_MAX], xbuf3[PATH_MAX];
    int t0;

    opp = pp = slashsplit(s);
    for (; *pp; pp++) {
	if (!strcmp(*pp, ".")) {
	    zsfree(*pp);
	    continue;
	}
	if (!strcmp(*pp, "..")) {
	    char *p;

	    zsfree(*pp);
	    if (!strcmp(xbuf, "/"))
		continue;
	    p = xbuf + strlen(xbuf);
	    while (*--p != '/');
	    *p = '\0';
	    continue;
	}
	if (unset(CHASELINKS)) {
	    strcat(xbuf, "/");
	    strcat(xbuf, *pp);
	    zsfree(*pp);
	    continue;
	}
	sprintf(xbuf2, "%s/%s", xbuf, *pp);
	t0 = readlink(xbuf2, xbuf3, PATH_MAX);
	if (t0 == -1 || !flag) {
	    strcat(xbuf, "/");
	    strcat(xbuf, *pp);
	    zsfree(*pp);
	} else {
	    xbuf3[t0] = '\0';	/* STUPID */
	    if (*xbuf3 == '/') {
		strcpy(xbuf, "");
		if (xsymlinks(xbuf3 + 1, flag))
		    return 1;
	    } else if (xsymlinks(xbuf3, flag))
		return 1;
	    zsfree(*pp);
	}
    }
    free(opp);
    return 0;
}

/* print a directory */

/**/
void
fprintdir(char *s, FILE *f)
{
    int t0;

    t0 = finddir(s);
    if (t0 == -1) {
	fputs(s, f);
    } else {
	putc('~', f);
	fputs(namdirs[t0].name, f);
	fputs(s + namdirs[t0].len, f);
    }
}

/**/
void
printdir(char *s)
{
    fprintdir(s, stdout);
}

/**/
int
findname(char *s)
{
    int t0;

    for (t0 = 0; t0 < userdirct; t0++)
	if (!strcmp(namdirs[t0].name, s))
	    return t0;
    return -1;
}

/* see if a path has a named directory as its prefix */

/**/
int
finddir(char *s)
{
    int t0, min, max;
    struct nameddirs ss;
    static int last = -1;
    static char previous[PATH_MAX] = "\0";

    if (!s) {			/* Invalidate directory cache */
	*previous = '\0';
	return last = -1;
    }
    if (!strcmp(s, previous))
	return last;
    ss.name = "";
    ss.namelen = ss.homedir = 0;
    ss.dir = s;
    ss.len = strlen(s);
    if (lencmp(&ss, &namdirs[0]) < 0)
	return -1;

/* This binary search doesn't seem to make much difference but... */

    min = 0;
    max = userdirct;
    while ((t0 = (min + max) >> 1) != min)
	if (lencmp(&ss, &namdirs[t0]) < 0)
	    max = t0;
	else
	    min = t0;

/* Binary search alone doesn't work because we want the best match, i.e. the
match latest in the namdirs array. */

    for (t0 = min; t0 >= 0; t0--)
	if (!dircmp(namdirs[t0].dir, s)) {
	    strcpy(previous, s);
	    return last = namdirs[t0].namelen<namdirs[t0].len ? t0 : -1;
	}
    return -1;
}

/* this static variable is used when addusernames() calls adduserdir() */
static nosortdirs = 0;

/* add the usernames to the named directory array */

/**/
void
addusernames(void)
{
#ifdef CACHE_USERNAMES
    struct passwd *pw;

    if (!usernamescached) {
	setpwent();
	nosortdirs = 1;
	while ((pw = getpwent()) != NULL && !errflag)
	    adduserdir(dupstring(pw->pw_name), pw->pw_dir, 1, 1);
	endpwent();
	nosortdirs = 0;
	qsort((void *) namdirs, userdirct, sizeof *namdirs,
	      (int (*) _((const void *, const void *)))lencmp);
	usernamescached = 1;
    }
#endif /* CACHE_USERNAMES */

    return;
}

/* add a named directory */

/**/
void
adduserdir(char *s, char *t, int ishomedir, int always)
{
    int t0 = -1, t1, t2;

    if (!interact)
	return;

    if (ishomedir) {
	if ((t0 = findname(s)) != -1)
	    return;
    } else if (!t || *t != '/') {
	if ((t0 = findname(s)) != -1) {	/* remove the name */
	    zsfree(namdirs[t0].name);
	    zsfree(namdirs[t0].dir);
	    for (; t0 < userdirct - 1; t0++)
		memcpy((void *) & namdirs[t0], (void *) & namdirs[t0 + 1],
		       sizeof *namdirs);
	    userdirct--;
	    finddir(0);
	}
	return;
    }
    if (unset(AUTONAMEDIRS) && findname(s) == -1 && always == 2)
	return;

    t2 = strlen(t);
    if (!ishomedir && t2 < PATH_MAX && (t0 = findname(s)) != -1) {

    /* update value */

	zsfree(namdirs[t0].dir);
	namdirs[t0].dir = ztrdup(t);
	t1 = namdirs[t0].len;
	namdirs[t0].len = t2;
	if (!nosortdirs)
	    if (t2 < t1)
		qsort((void *) namdirs, t0 + 1, sizeof *namdirs,
		      (int (*) _((const void *, const void *)))lencmp);
	    else
		qsort((void *) & namdirs[t0], userdirct - t0, sizeof *namdirs,
		      (int (*) _((const void *, const void *)))lencmp);

	finddir(0);		/* Invalidate directory cache */
	return;
    }
 /* add the name */

    if (userdirsz == userdirct) {
	userdirsz *= 2;
	namdirs = (Nameddirs) realloc((void *) namdirs,
				      sizeof *namdirs * userdirsz);
	if (!namdirs)
	    return;
    }
    for (t0 = 0; t0 < userdirct; t0++)
	if (namdirs[t0].len > t2)
	    break;
    for (t1 = userdirct; t1 > t0; t1--)
	memcpy((void *) & namdirs[t1], (void *) & namdirs[t1 - 1],
	       sizeof *namdirs);
    namdirs[t0].len = t2;
    namdirs[t0].namelen = strlen(s);
    namdirs[t0].name = ztrdup(s);
    namdirs[t0].dir = ztrdup(t);
    namdirs[t0].homedir = ishomedir;
    userdirct++;
    if (t0 && namdirs[t0 - 1].len == t2 && !nosortdirs)
	qsort((void *) namdirs, t0 + 1, sizeof *namdirs,
	      (int (*) _((const void *, const void *)))lencmp);
    finddir(0);
}

/**/
int
dircmp(char *s, char *t)
{
    if (s) {
	for (; *s == *t; s++, t++)
	    if (!*s)
		return 0;
	if (!*s && *t == '/')
	    return 0;
    }
    return 1;
}

/**/
int
lencmp(void *first, void *sec)
{
#ifdef HIDE_NAMES
    /* Directories are sorted only by path length, so the name actually used
    for a directory is the last one added to the table that matches the longest
    possible part of the path. */
    return ((Nameddirs) first)->len - ((Nameddirs) sec)->len;
#else
# ifdef OLD_NAME_SORT
    int i;

    /* Directories are sorted first by path length, and then in reverse order
    by name length, so that the shortest available name for the longest
    possible matching path is used. */
    if ((i = ((Nameddirs) first)->len - ((Nameddirs) sec)->len))
	return i;
    else
	return ((Nameddirs) sec)->namelen - ((Nameddirs) first)->namelen;
# else
    /* Directories are sorted by the difference between the path length and the
    name length, so that directories are referred to in the shortest possible
    way. */
    return ((Nameddirs) first)->len + ((Nameddirs) sec)->namelen -
	((Nameddirs) first)->namelen - ((Nameddirs) sec)->len;
# endif
#endif
}

/* do pre-prompt stuff */

/**/
void
preprompt(void)
{
    int diff;
    List list;
    struct schedcmd *sch, *schl;

    if (unset(NOTIFY))
	scanjobs();
    if (errflag)
	return;
    if ((list = getshfunc("precmd")))
	doshfunc(list, NULL, 0, 1);
    if (errflag)
	return;
    if (period && (time(NULL) > lastperiod + period) &&
	(list = getshfunc("periodic"))) {
	doshfunc(list, NULL, 0, 1);
	lastperiod = time(NULL);
    }
    if (errflag)
	return;
    if (watch) {
	diff = (int)difftime(time(NULL), lastwatch);
	if (diff > logcheck) {
	    dowatch();
	    lastwatch = time(NULL);
	}
    }
    if (errflag)
	return;
    diff = (int)difftime(time(NULL), lastmailcheck);
    if (diff > mailcheck) {
	if (mailpath && *mailpath && **mailpath)
	    checkmailpath(mailpath);
	else if (mailfile && *mailfile) {
	    char *x[2];

	    x[0] = mailfile;
	    x[1] = NULL;
	    checkmailpath(x);
	}
	lastmailcheck = time(NULL);
    }
    for (schl = (struct schedcmd *)&schedcmds, sch = schedcmds; sch;
	 sch = (schl = sch)->next) {
	if (sch->time < time(NULL)) {
	    execstring(sch->cmd, 0);
	    schl->next = sch->next;
	    zsfree(sch->cmd);
	    zfree(sch, sizeof(struct schedcmd));

	    sch = schl;
	}
	if (errflag)
	    return;
    }
}

/* Return the number of elements in an *
 * array of pointers.                  */

/**/
int
arrlen(char **s)
{
    int count;

    for (count = 0; *s; s++, count++);
    return count;
}

/**/
void
checkmailpath(char **s)
{
    struct stat st;
    char *v, *u, c;

    while (*s) {
	for (v = *s; *v && *v != '?'; v++);
	c = *v;
	*v = '\0';
	if (c != '?')
	    u = NULL;
	else
	    u = v + 1;
	if (**s == 0) {
	    *v = c;
	    zerr("empty MAILPATH component: %s", *s, 0);
	} else if (stat(*s, &st) == -1) {
	    if (errno != ENOENT)
		zerr("%e: %s", *s, errno);
	} else if (S_ISDIR(st.st_mode)) {
	    Lklist l;
	    DIR *lock = opendir(*s);
	    char buf[PATH_MAX * 2], **arr, **ap;
	    struct dirent *de;
	    int ct = 1;

	    if (lock) {
		pushheap();
		heapalloc();
		l = newlist();
		readdir(lock);
		readdir(lock);
		while ((de = readdir(lock))) {
		    if (errflag)
			break;
		    if (u)
			sprintf(buf, "%s/%s?%s", *s, de->d_name, u);
		    else
			sprintf(buf, "%s/%s", *s, de->d_name);
		    addnode(l, dupstring(buf));
		    ct++;
		}
		closedir(lock);
		ap = arr = (char **)alloc(ct * sizeof(char *));

		while ((*ap++ = (char *)ugetnode(l)));
		checkmailpath(arr);
		popheap();
	    }
	} else {
	    if (st.st_size && st.st_atime <= st.st_mtime &&
		st.st_mtime > lastmailcheck)
		if (!u) {
		    fprintf(stderr, "You have new mail.\n");
		    fflush(stderr);
		} else {
		    char *z = u;

		    while (*z)
			if (*z == '$' && z[1] == '_') {
			    fprintf(stderr, "%s", *s);
			    z += 2;
			} else
			    fputc(*z++, stderr);
		    fputc('\n', stderr);
		    fflush(stderr);
		}
	    if (isset(MAILWARNING) && st.st_atime > st.st_mtime &&
		st.st_atime > lastmailcheck && st.st_size) {
		fprintf(stderr, "The mail in %s has been read.\n", *s);
		fflush(stderr);
	    }
	}
	*v = c;
	s++;
    }
}

/**/
void
saveoldfuncs(char *x, Cmdnam y)
{
    Cmdnam cc;

    if (y->flags & (SHFUNC | DISABLED)) {
	cc = (Cmdnam) zcalloc(sizeof *cc);
	*cc = *y;
	y->u.list = NULL;
	addhnode(ztrdup(x), cc, cmdnamtab, freecmdnam);
    }
}

/* create command hashtable */

/**/
void
newcmdnamtab(void)
{
    Hashtab oldcnt;

    oldcnt = cmdnamtab;
    permalloc();
    cmdnamtab = newhtable(101);
    addbuiltins();
    if (oldcnt) {
	listhtable(oldcnt, (HFunc) saveoldfuncs);
	freehtab(oldcnt, freecmdnam);
    }
    lastalloc();
    pathchecked = path;
}

/**/
void
freecmdnam(void *a)
{
    struct cmdnam *c = (struct cmdnam *)a;

    if (c->flags & SHFUNC) {
	if (c->u.list)
	    freestruct(c->u.list);
    } else if ((c->flags & HASHCMD) == HASHCMD)
	zsfree(c->u.cmd);

    zfree(c, sizeof(struct cmdnam));
}

/**/
void
freecompcond(void *a)
{
    Compcond cc = (Compcond) a;
    Compcond and, or, c;
    int n;

    for (c = cc; c; c = or) {
	or = c->or;
	for (; c; c = and) {
	    and = c->and;
	    if (c->type == CCT_POS ||
		c->type == CCT_NUMWORDS) {
		free(c->u.r.a);
		free(c->u.r.b);
	    } else if (c->type == CCT_CURSUF ||
		       c->type == CCT_CURPRE) {
		for (n = 0; n < c->n; n++)
		    if (c->u.s.s[n])
			zsfree(c->u.s.s[n]);
		free(c->u.s.s);
	    } else if (c->type == CCT_RANGESTR ||
		       c->type == CCT_RANGEPAT) {
		for (n = 0; n < c->n; n++)
		    if (c->u.l.a[n])
			zsfree(c->u.l.a[n]);
		free(c->u.l.a);
		for (n = 0; n < c->n; n++)
		    if (c->u.l.b[n])
			zsfree(c->u.l.b[n]);
		free(c->u.l.b);
	    } else {
		for (n = 0; n < c->n; n++)
		    if (c->u.s.s[n])
			zsfree(c->u.s.s[n]);
		free(c->u.s.p);
		free(c->u.s.s);
	    }
	    zfree(c, sizeof(struct compcond));
	}
    }
}

/**/
void
freecompctl(void *a)
{
    Compctl cc = (Compctl) a;

    if (cc == &cc_default ||
 	cc == &cc_first ||
	cc == &cc_compos ||
	--cc->refc > 0)
	return;

    zsfree(cc->keyvar);
    zsfree(cc->glob);
    zsfree(cc->str);
    zsfree(cc->func);
    zsfree(cc->explain);
    zsfree(cc->prefix);
    zsfree(cc->suffix);
    zsfree(cc->hpat);
    zsfree(cc->subcmd);
    if (cc->cond)
	freecompcond(cc->cond);
    if (cc->ext) {
	Compctl n, m;

	n = cc->ext;
	do {
	    m = (Compctl) (n->next);
	    freecompctl(n);
	    n = m;
	}
	while (n);
    }
    if (cc->xor && cc->xor != &cc_default)
	freecompctl(cc->xor);
    zfree(cc, sizeof(struct compctl));
}

/**/
void
freecompctlp(void *a)
{
    Compctlp ccp = (Compctlp) a;

    freecompctl(ccp->cc);
    zfree(ccp, sizeof(struct compctlp));
}

/**/
void
freestr(void *a)
{
    zsfree(a);
}

/**/
void
freeanode(void *a)
{
    struct alias *c = (struct alias *)a;

    zsfree(c->text);
    zfree(c, sizeof(struct alias));
}

/**/
void
freepm(void *a)
{
    struct param *pm = (Param) a;

    zfree(pm, sizeof(struct param));
}

/**/
void
gettyinfo(struct ttyinfo *ti)
{
    if (SHTTY != -1) {
#ifdef HAVE_TERMIOS_H
# ifdef HAVE_TCGETATTR
	if (tcgetattr(SHTTY, &ti->tio) == -1)
# else
	if (ioctl(SHTTY, TCGETS, &ti->tio) == -1)
# endif
	    zerr("bad tcgets: %e", NULL, errno);
#else
# ifdef HAVE_TERMIO_H
	ioctl(SHTTY, TCGETA, &ti->tio);
# else
	ioctl(SHTTY, TIOCGETP, &ti->sgttyb);
	ioctl(SHTTY, TIOCLGET, &ti->lmodes);
	ioctl(SHTTY, TIOCGETC, &ti->tchars);
	ioctl(SHTTY, TIOCGLTC, &ti->ltchars);
# endif
#endif
#ifdef TIOCGWINSZ
/*	if (ioctl(SHTTY, TIOCGWINSZ, &ti->winsize) == -1)
	    	zerr("bad tiocgwinsz: %e",NULL,errno);*/
	ioctl(SHTTY, TIOCGWINSZ, (char *)&ti->winsize);
#endif
    }
}

/**/
void
settyinfo(struct ttyinfo *ti)
{
    if (SHTTY != -1) {
#ifdef HAVE_TERMIOS_H
# ifdef HAVE_TCGETATTR
#  ifndef TCSADRAIN
#   define TCSADRAIN 1	/* XXX Princeton's include files are screwed up */
#  endif
	tcsetattr(SHTTY, TCSADRAIN, &ti->tio);
    /* if (tcsetattr(SHTTY, TCSADRAIN, &ti->tio) == -1) */
# else
	ioctl(SHTTY, TCSETS, &ti->tio);
    /* if (ioctl(SHTTY, TCSETS, &ti->tio) == -1) */
# endif
	/*	zerr("settyinfo: %e",NULL,errno)*/ ;
#else
# ifdef HAVE_TERMIO_H
	ioctl(SHTTY, TCSETA, &ti->tio);
# else
	ioctl(SHTTY, TIOCSETN, &ti->sgttyb);
	ioctl(SHTTY, TIOCLSET, &ti->lmodes);
	ioctl(SHTTY, TIOCSETC, &ti->tchars);
	ioctl(SHTTY, TIOCSLTC, &ti->ltchars);
# endif
#endif
    }
}

#ifdef TIOCGWINSZ
extern winchanged;
#endif

/* check the size of the window and adjust if necessary */

/**/
void
adjustwinsize(void)
{
#ifdef TIOCGWINSZ
    int oldcols = columns, oldrows = lines;

    if (SHTTY == -1)
	return;

    ioctl(SHTTY, TIOCGWINSZ, (char *)&shttyinfo.winsize);
    if (shttyinfo.winsize.ws_col)
	columns = shttyinfo.winsize.ws_col;
    if (shttyinfo.winsize.ws_row)
	lines = shttyinfo.winsize.ws_row;
    if (oldcols != columns) {
	if (zleactive) {
	    resetneeded = winchanged = 1;
	    refresh();
	}
	setintenv("COLUMNS", columns);
    }
    if (oldrows != lines)
	setintenv("LINES", lines);
#endif   /*  TIOCGWINSZ */
}

/**/
int
zyztem(char *s, char *t)
{
    int cj = thisjob;

    s = tricat(s, " ", t);
    execstring(s, 0);		/* Depends on recursion condom in execute() */
    zsfree(s);
    thisjob = cj;
    return lastval;
}

/* move a fd to a place >= 10 */

/**/
int
movefd(int fd)
{
    int fe;

    if (fd == -1)
	return fd;
#ifdef F_DUPFD
    fe = fcntl(fd, F_DUPFD, 10);
#else
    if ((fe = dup(fd)) < 10)
	fe = movefd(fe);
#endif
    close(fd);
    return fe;
}

/* move fd x to y */

/**/
void
redup(int x, int y)
{
    if (x != y) {
	dup2(x, y);
	close(x);
    }
}


/* copy len chars from t into s, and null terminate */

/**/
void
ztrncpy(char *s, char *t, int len)
{
    while (len--)
	*s++ = *t++;
    *s = '\0';
}

/* copy t into *s and update s */

/**/
void
strucpy(char **s, char *t)
{
    char *u = *s;

    while ((*u++ = *t++));
    *s = u - 1;
}

/**/
void
struncpy(char **s, char *t, int n)
{
    char *u = *s;

    while (n--)
	*u++ = *t++;
    *s = u;
    *u = '\0';
}

/**/
int
checkrmall(char *s)
{
    fflush(stdin);
    if (*s == '/')
	fprintf(shout, "zsh: sure you want to delete all the files in %s? ",
		s);
    else
	fprintf(shout, "zsh: sure you want to delete all the files in %s/%s? ",
		(pwd[1]) ? pwd : "", s);
    fflush(shout);
    feep();
    return (getquery() == 'y');
}

/**/
int
getquery(void)
{
    char c, d;
    int isem = !strcmp(term, "emacs");

#ifdef FIONREAD
    int val;
#endif

    attachtty(mypgrp);
    if (!isem)
	setcbreak();

#ifdef FIONREAD
    ioctl(SHTTY, FIONREAD, (char *)&val);
    if (val) {
	if (!isem)
	    settyinfo(&shttyinfo);
	write(SHTTY, "n\n", 2);
	return 'n';
    }
#endif
    if (read(SHTTY, &c, 1) == 1)
	if (c == 'y' || c == 'Y' || c == '\t')
	    c = 'y';
    if (isem) {
	if (c != '\n')
	    while (read(SHTTY, &d, 1) == 1 && d != '\n');
    } else {
	settyinfo(&shttyinfo);
	if (c != '\n')
	    write(2, "\n", 1);
    }
    return (int)c;
}

static int d;
static char *guess, *best;

/**/
void
spscan(char *s, char *junk)
{
    int nd;

    nd = spdist(s, guess, (int)strlen(guess) / 4 + 1);
    if (nd <= d) {
	best = s;
	d = nd;
    }
}

/* spellcheck a word */
/* fix s and s2 ; if s2 is non-null, fix the history list too */

/**/
void
spckword(char **s, char **s2, char **tptr, int cmd, int ask)
{
    char *t, *u;
    char firstchar;
    int x;
    int pram = 0;

    if (**s == '-' || **s == '%')
	return;
    if (!strcmp(*s, "in"))
	return;
    if (!(*s)[0] || !(*s)[1])
	return;
    if (gethnode(*s, cmdnamtab) || gethnode(*s, aliastab)
	|| gethnode(*s, reswdtab))
	return;
    else if (isset(HASHLISTALL)) {
	fullhash();
	if (gethnode(*s, cmdnamtab))
	    return;
    }
    t = *s;
    if (*t == Tilde || *t == Equals || *t == String)
	t++;
    for (; *t; t++)
	if (itok(*t))
	    return;
    best = NULL;
    for (t = *s; *t; t++)
	if (*t == '/')
	    break;
    if (**s == String) {
	if (*t)
	    return;
	pram = 1;
	guess = *s + 1;
	while (*guess == '+' || *guess == '^' ||
	       *guess == '#' || *guess == '~' ||
	       *guess == '=')
	    guess++;
	d = 100;
	listhtable(paramtab, spscan);
    } else {
	if ((u = spname(guess = *s)) != *s)
	    best = u;
	if (!*t && !cmd) {
	    if (access(*s, F_OK) == 0)
		return;
	    if (hashcmd(*s, pathchecked))
		return;
	    guess = *s;
	    d = 100;
	    listhtable(reswdtab, spscan);
	    listhtable(aliastab, spscan);
	    listhtable(cmdnamtab, spscan);
	}
    }
    if (errflag)
	return;
    if (best && (int)strlen(best) > 1 && strcmp(best, guess)) {
	if (ask) {
	    char *pptbuf;
	    int pptlen;
	    rstring = best;
	    Rstring = guess;
	    firstchar = *guess;
	    if (*guess == Tilde)
		*guess = '~';
	    else if (*guess == String)
		*guess = '$';
	    else if (*guess == Equals)
		*guess = '=';
	    pptbuf = putprompt(sprompt, &pptlen, NULL, 1);
	    fwrite(pptbuf, pptlen, 1, stderr);
	    fflush(stderr);
	    *guess = firstchar;
	    feep();
	    x = getquery();
	} else
	    x = 'y';
	if (x == 'y' || x == ' ') {
	    if (!pram) {
		*s = dupstring(best);
	    } else {
		*s = (char *)alloc(strlen(best) + 2);
		strcpy(*s + 1, best);
		**s = String;
	    }
	    if (s2) {
		if (*tptr && !strcmp(hlastw, *s2) && hlastw < hptr) {
		    char *z;

		    hptr = hlastw;
		    if (pram)
			hwaddc('$');
		    for (z = best; *z; z++)
			hwaddc(*z);
		    hwaddc(HISTSPACE);
		    *tptr = hptr - 1;
		    **tptr = '\0';
		}
		*s2 = dupstring(best);
	    }
	} else if (x == 'a') {
	    histdone |= HISTFLAG_NOEXEC;
	} else if (x == 'e') {
	    histdone |= HISTFLAG_NOEXEC | HISTFLAG_RECALL;
	}
    }
}

/**/
int
ztrftime(char *buf, int bufsize, char *fmt, struct tm *tm)
{
    static char *astr[] =
    {"Sun", "Mon", "Tue", "Wed", "Thu", "Fri", "Sat"};
    static char *estr[] =
    {"Jan", "Feb", "Mar", "Apr", "May", "Jun", "Jul",
     "Aug", "Sep", "Oct", "Nov", "Dec"};
    static char *lstr[] =
    {"12", " 1", " 2", " 3", " 4", " 5", " 6", " 7", " 8", " 9",
     "10", "11"};
    char tmp[3];

#ifdef HAVE_STRFTIME
    char *origbuf = buf;
#endif

    tmp[0] = '%';
    tmp[2] = '\0';
    while (*fmt)
	if (*fmt == '%') {
	    fmt++;
	    switch (*fmt++) {
	    case 'a':
		strucpy(&buf, astr[tm->tm_wday]);
		break;
	    case 'b':
		strucpy(&buf, estr[tm->tm_mon]);
		break;
	    case 'd':
		*buf++ = '0' + tm->tm_mday / 10;
		*buf++ = '0' + tm->tm_mday % 10;
		break;
	    case 'e':
		if (tm->tm_mday > 9)
		    *buf++ = '0' + tm->tm_mday / 10;
		*buf++ = '0' + tm->tm_mday % 10;
		break;
	    case 'k':
		if (tm->tm_hour > 9)
		    *buf++ = '0' + tm->tm_hour / 10;
		*buf++ = '0' + tm->tm_hour % 10;
		break;
	    case 'l':
		strucpy(&buf, lstr[tm->tm_hour % 12]);
		break;
	    case 'm':
		*buf++ = '0' + (tm->tm_mon + 1) / 10;
		*buf++ = '0' + (tm->tm_mon + 1) % 10;
		break;
	    case 'M':
		*buf++ = '0' + tm->tm_min / 10;
		*buf++ = '0' + tm->tm_min % 10;
		break;
	    case 'p':
		*buf++ = (tm->tm_hour > 11) ? 'p' : 'a';
		*buf++ = 'm';
		break;
	    case 'S':
		*buf++ = '0' + tm->tm_sec / 10;
		*buf++ = '0' + tm->tm_sec % 10;
		break;
	    case 'y':
		*buf++ = '0' + tm->tm_year / 10;
		*buf++ = '0' + tm->tm_year % 10;
		break;
	    default:
#ifdef HAVE_STRFTIME
		*buf = '\0';
		tmp[1] = fmt[-1];
		strftime(buf, bufsize - strlen(origbuf), tmp, tm);
		buf += strlen(buf);
#else
		*buf++ = '%';
		*buf++ = fmt[-1];
#endif
		break;
	    }
	} else
	    *buf++ = *fmt++;
    *buf = '\0';
    return 0;
}

/**/
char *
join(char **arr, int delim)
{
    int len = 0;
    char **s, *ret, *ptr;
    static char *lastmem = NULL;

    for (s = arr; *s; s++)
	len += strlen(*s) + 1;
    if (!len)
	return "";
    zsfree(lastmem);
    lastmem = ptr = ret = (char *)zalloc(len);
    for (s = arr; *s; s++) {
	strucpy(&ptr, *s);
	*ptr++ = delim;
    }
    ptr[-1] = '\0';
    return ret;
}

/**/
char *
spacejoin(char **s)
{
    return join(s, *ifs);
}

/**/
char **
colonsplit(char *s)
{
    int ct;
    char *t, **ret, **ptr;

    for (t = s, ct = 0; *t; t++)
	if (*t == ':')
	    ct++;
    ptr = ret = (char **)zalloc(sizeof(char **) * (ct + 2));

    t = s;
    do {
	for (s = t; *t && *t != ':'; t++);
	*ptr = (char *)zalloc((t - s) + 1);
	ztrncpy(*ptr++, s, t - s);
    }
    while (*t++);
    *ptr = NULL;
    return ret;
}

/**/
char **
spacesplit(char *s)
{
    int ct;
    char *t, **ret, **ptr;

    for (t = s, ct = 0; *t; t++)
	if (isep(*t))
	    ct++;
    ptr = ret = (char **)ncalloc(sizeof(char **) * (ct + 2));

    t = s;
    do {
	for (s = t; *t && !isep(*t); t++);
	*ptr = (char *)ncalloc((t - s) + 1);
	ztrncpy(*ptr++, s, t - s);
    }
    while (*t++);
    *ptr = NULL;
    return ret;
}

/**/
int
findsep(char **s, char *sep)
{
    int i;
    char *t, *tt;

    if (!sep) {
	for (t = *s, i = 1; i && *t;) {
	    for (tt = ifs, i = 1; i && *tt; tt++)
		if (*tt == *t)
		    i = 0;
	    if (i)
		t++;
	}
	i = t - *s;
	*s = t;
	return i;
    }
    if (!sep[0]) {
	return **s ? ++*s, 1 : -1;
    }
    for (i = 0; **s; (*s)++, i++) {
	for (t = sep, tt = *s; *t && *tt && *t == *tt; t++, tt++);
	if (!*t)
	    return i;
    }
    return -1;
}

/**/
char *
findword(char **s, char *sep)
{
    char *r, *t, *tt;
    int f, sl;

    if (!**s)
	return NULL;

    if (sep) {
	sl = strlen(sep);
	r = *s;
	while (!(f = findsep(s, sep))) {
	    r = *s += sl;
	}
	return r;
    }
    for (t = *s, f = 1; f && *t;) {
	for (tt = ifs, f = 0; !f && *tt; tt++)
	    if (*tt == *t)
		f = 1;
	if (f)
	    t++;
    }
    *s = t;
    findsep(s, sep);
    return t;
}

/**/
int
wordcount(char *s, char *sep, int mul)
{
    int r = 1, sl, c, cc;
    char *t = s, *ti;

    if (sep) {
	sl = strlen(sep);
	for (; (c = findsep(&t, sep)) >= 0; t += sl)
	    if ((c && *(t + sl)) || mul)
		r++;
    } else {
	if (!mul)
	    for (c = 1; c && *t;) {
		for (c = 0, ti = ifs; !c && *ti; ti++)
		    if (*ti == *t)
			c = 1;
		if (c)
		    t++;
	    }
	if (!*t)
	    return 0;
	for (; *t; t++) {
	    for (c = 0, ti = ifs; !c && *ti; ti++)
		if (*ti == *t)
		    c = 1;
	    if (c && !mul) {
		for (cc = 1, t++; cc && *t;) {
		    for (cc = 0, ti = ifs; !cc && *ti; ti++)
			if (*ti == *t)
			    cc = 1;
		    if (cc)
			t++;
		}
		if (*t)
		    r++;
	    } else if (c)
		r++;
	}
    }
    return r;
}

/**/
char *
sepjoin(char **s, char *sep)
{
    char *r, *p, **t;
    int l, sl, elide = 0;
    static char *lastmem = NULL;
    char sepbuf[2];

    if (!*s)
	return "";
    if (!sep) {
	elide = 1;
	sep = sepbuf;
	sepbuf[0] = *ifs;
	sepbuf[1] = '\0';
    }
    sl = strlen(sep);
    for (t = s, l = 1 - sl; *t; l += strlen(*t) + sl, t++);
    if (l == 1)
	return "";
    zsfree(lastmem);
    lastmem = r = p = (char *)zalloc(l);
    t = s;
    if (elide)
	while (*t && !**t)
	    t++;
    for (; *t; t++) {
	strucpy(&p, *t);
	if (t[1] && (!elide || t[1][0]))
	    strucpy(&p, sep);
    }
    return r;
}

/**/
char **
sepsplit(char *s, char *sep)
{
    int n, sl, f;
    char *t, *tt, **r, **p;

    if (!sep)
	return spacesplit(s);

    sl = strlen(sep);
    n = wordcount(s, sep, 1);
    r = p = (char **)ncalloc((n + 1) * sizeof(char *));

    for (t = s; n--;) {
	tt = t;
	f = findsep(&t, sep);	/* f set not but not used? ++jhi; */
	*p = (char *)ncalloc(t - tt + 1);
	strncpy(*p, tt, t - tt);
	(*p)[t - tt] = '\0';
	p++;
	t += sl;
    }
    *p = NULL;

    return r;
}

/**/
List
getshfunc(char *nam)
{
    Cmdnam x = (Cmdnam) gethnode(nam, cmdnamtab);

    if (x && (x->flags & SHFUNC)) {
	if (x->flags & PMFLAG_u) {
	    List l;

	    if (!(l = getfpfunc(nam))) {
		zerr("function not found: %s", nam, 0);
		return NULL;
	    }
	    x->flags &= ~PMFLAG_u;
	    permalloc();
	    x->u.list = (List) dupstruct(l);
	    lastalloc();
	}
	return x->u.list;
    }
    return NULL;
}

/* allocate a tree element */

static int sizetab[N_COUNT] =
{
    sizeof(struct list),
    sizeof(struct sublist),
    sizeof(struct pline),
    sizeof(struct cmd),
    sizeof(struct redir),
    sizeof(struct cond),
    sizeof(struct forcmd),
    sizeof(struct casecmd),
    sizeof(struct ifcmd),
    sizeof(struct whilecmd),
    sizeof(struct varasg)};

static int flagtab[N_COUNT] =
{
    NT_SET(N_LIST, 1, NT_NODE, NT_NODE, 0, 0),
    NT_SET(N_SUBLIST, 2, NT_NODE, NT_NODE, 0, 0),
    NT_SET(N_PLINE, 1, NT_NODE, NT_NODE, 0, 0),
    NT_SET(N_CMD, 2, NT_STR | NT_LIST, NT_NODE, NT_NODE | NT_LIST, NT_NODE | NT_LIST),
    NT_SET(N_REDIR, 3, NT_STR, 0, 0, 0),
    NT_SET(N_COND, 1, NT_NODE, NT_NODE, 0, 0),
    NT_SET(N_FOR, 1, NT_STR, NT_NODE, 0, 0),
    NT_SET(N_CASE, 0, NT_STR | NT_ARR, NT_NODE | NT_ARR, 0, 0),
    NT_SET(N_IF, 0, NT_NODE | NT_ARR, NT_NODE | NT_ARR, 0, 0),
    NT_SET(N_WHILE, 1, NT_NODE, NT_NODE, 0, 0),
    NT_SET(N_VARASG, 1, NT_STR, NT_STR, NT_STR | NT_LIST, 0)};

/**/
void *
allocnode(int type)
{
    struct node *n = (struct node *)alloc(sizetab[type]);

    memset((char *)n, 0, sizetab[type]);
    n->type = flagtab[type];
    if (useheap)
	n->type |= NT_HEAP;

    return (void *) n;
}

/**/
void *
dupstruct(void *a)
{
    struct node *n = (struct node *)a, *r;

    if (!a || ((List) a) == &dummy_list)
	return (void *) a;

    if ((n->type & NT_HEAP) && !useheap) {
	heapalloc();
	n = (struct node *)dupstruct2((void *) n);
	permalloc();
	n = simplifystruct(n);
    }
    r = (struct node *)dupstruct2((void *) n);

    if (!(n->type & NT_HEAP) && useheap)
	r = expandstruct(r, N_LIST);

    return (void *) r;
}

/**/
struct node *
simplifystruct(struct node *n)
{
    if (!n || ((List) n) == &dummy_list)
	return n;

    switch (NT_TYPE(n->type)) {
    case N_LIST:
	{
	    List l = (List) n;

	    l->left = (Sublist) simplifystruct((struct node *)l->left);
	    if (l->type == Z_SYNC && !l->right)
		return (struct node *)l->left;
	}
	break;
    case N_SUBLIST:
	{
	    Sublist sl = (Sublist) n;

	    sl->left = (Pline) simplifystruct((struct node *)sl->left);
	    if (sl->type == END && !sl->flags && !sl->right)
		return (struct node *)sl->left;
	}
	break;
    case N_PLINE:
	{
	    Pline pl = (Pline) n;

	    pl->left = (Cmd) simplifystruct((struct node *)pl->left);
	    if (pl->type == END && !pl->right)
		return (struct node *)pl->left;
	}
	break;
    case N_CMD:
	{
	    Cmd c = (Cmd) n;
	    int i = 0;

	    if (empty(c->args))
		c->args = NULL, i++;
	    if (empty(c->redir))
		c->redir = NULL, i++;
	    if (empty(c->vars))
		c->vars = NULL, i++;

	    c->u.list = (List) simplifystruct((struct node *)c->u.list);
	    if (i == 3 && !c->flags &&
		(c->type == CWHILE || c->type == CIF ||
		 c->type == COND))
		return (struct node *)c->u.list;
	}
	break;
    case N_FOR:
	{
	    Forcmd f = (Forcmd) n;

	    f->list = (List) simplifystruct((struct node *)f->list);
	}
	break;
    case N_CASE:
	{
	    struct casecmd *c = (struct casecmd *)n;
	    List *l;

	    for (l = c->lists; *l; l++)
		*l = (List) simplifystruct((struct node *)*l);
	}
	break;
    case N_IF:
	{
	    struct ifcmd *i = (struct ifcmd *)n;
	    List *l;

	    for (l = i->ifls; *l; l++)
		*l = (List) simplifystruct((struct node *)*l);
	    for (l = i->thenls; *l; l++)
		*l = (List) simplifystruct((struct node *)*l);
	}
	break;
    case N_WHILE:
	{
	    struct whilecmd *w = (struct whilecmd *)n;

	    w->cont = (List) simplifystruct((struct node *)w->cont);
	    w->loop = (List) simplifystruct((struct node *)w->loop);
	}
    }

    return n;
}

/**/
struct node *
expandstruct(struct node *n, int exp)
{
    struct node *m;

    if (!n || ((List) n) == &dummy_list)
	return n;

    if (exp != N_COUNT && exp != NT_TYPE(n->type)) {
	switch (exp) {
	case N_LIST:
	    {
		List l;

		m = (struct node *)allocnode(N_LIST);
		l = (List) m;
		l->type = Z_SYNC;
		l->left = (Sublist) expandstruct(n, N_SUBLIST);

		return (struct node *)l;
	    }
	case N_SUBLIST:
	    {
		Sublist sl;

		m = (struct node *)allocnode(N_SUBLIST);
		sl = (Sublist) m;
		sl->type = END;
		sl->left = (Pline) expandstruct(n, N_PLINE);

		return (struct node *)sl;
	    }
	case N_PLINE:
	    {
		Pline pl;

		m = (struct node *)allocnode(N_PLINE);
		pl = (Pline) m;
		pl->type = END;
		pl->left = (Cmd) expandstruct(n, N_CMD);

		return (struct node *)pl;
	    }
	case N_CMD:
	    {
		Cmd c;

		m = (struct node *)allocnode(N_CMD);
		c = (Cmd) m;
		switch (NT_TYPE(n->type)) {
		case N_WHILE:
		    c->type = CWHILE;
		    break;
		case N_IF:
		    c->type = CIF;
		    break;
		case N_COND:
		    c->type = COND;
		}
		c->u.list = (List) expandstruct(n, NT_TYPE(n->type));
		c->args = newlist();
		c->vars = newlist();
		c->redir = newlist();

		return (struct node *)c;
	    }
	}
    } else
	switch (NT_TYPE(n->type)) {
	case N_LIST:
	    {
		List l = (List) n;

		l->left = (Sublist) expandstruct((struct node *)l->left,
						 N_SUBLIST);
		l->right = (List) expandstruct((struct node *)l->right,
					       N_LIST);
	    }
	    break;
	case N_SUBLIST:
	    {
		Sublist sl = (Sublist) n;

		sl->left = (Pline) expandstruct((struct node *)sl->left,
						N_PLINE);
		sl->right = (Sublist) expandstruct((struct node *)sl->right,
						   N_SUBLIST);
	    }
	    break;
	case N_PLINE:
	    {
		Pline pl = (Pline) n;

		pl->left = (Cmd) expandstruct((struct node *)pl->left,
					      N_CMD);
		pl->right = (Pline) expandstruct((struct node *)pl->right,
						 N_PLINE);
	    }
	    break;
	case N_CMD:
	    {
		Cmd c = (Cmd) n;

		if (!c->args)
		    c->args = newlist();
		if (!c->vars)
		    c->vars = newlist();
		if (!c->redir)
		    c->redir = newlist();

		switch (c->type) {
		case CFOR:
		case CSELECT:
		    c->u.list = (List) expandstruct((struct node *)c->u.list,
						    N_FOR);
		    break;
		case CWHILE:
		    c->u.list = (List) expandstruct((struct node *)c->u.list,
						    N_WHILE);
		    break;
		case CIF:
		    c->u.list = (List) expandstruct((struct node *)c->u.list,
						    N_IF);
		    break;
		case CCASE:
		    c->u.list = (List) expandstruct((struct node *)c->u.list,
						    N_CASE);
		    break;
		case COND:
		    c->u.list = (List) expandstruct((struct node *)c->u.list,
						    N_COND);
		    break;
		case ZCTIME:
		    c->u.list = (List) expandstruct((struct node *)c->u.list,
						    N_SUBLIST);
		    break;
		default:
		    c->u.list = (List) expandstruct((struct node *)c->u.list,
						    N_LIST);
		}
	    }
	    break;
	case N_FOR:
	    {
		Forcmd f = (Forcmd) n;

		f->list = (List) expandstruct((struct node *)f->list,
					      N_LIST);
	    }
	    break;
	case N_CASE:
	    {
		struct casecmd *c = (struct casecmd *)n;
		List *l;

		for (l = c->lists; *l; l++)
		    *l = (List) expandstruct((struct node *)*l, N_LIST);
	    }
	    break;
	case N_IF:
	    {
		struct ifcmd *i = (struct ifcmd *)n;
		List *l;

		for (l = i->ifls; *l; l++)
		    *l = (List) expandstruct((struct node *)*l, N_LIST);
		for (l = i->thenls; *l; l++)
		    *l = (List) expandstruct((struct node *)*l, N_LIST);
	    }
	    break;
	case N_WHILE:
	    {
		struct whilecmd *w = (struct whilecmd *)n;

		w->cont = (List) expandstruct((struct node *)w->cont,
					      N_LIST);
		w->loop = (List) expandstruct((struct node *)w->loop,
					      N_LIST);
	    }
	}

    return n;
}

/* duplicate a syntax tree node of given type, argument number */

/**/
void *
dupnode(int type, void *a, int argnum)
{
    if (!a)
	return NULL;
    switch (NT_N(type, argnum)) {
    case NT_NODE:
	return (void *) dupstruct2(a);
    case NT_STR:
	return (useheap) ? ((void *) dupstring(a)) :
	    ((void *) ztrdup(a));
    case NT_LIST | NT_NODE:
	if (type & NT_HEAP)
	    if (useheap)
		return (void *) duplist(a, (VFunc) dupstruct2);
	    else
		return (void *) list2arr(a, (VFunc) dupstruct2);
	else if (useheap)
	    return (void *) arr2list(a, (VFunc) dupstruct2);
	else
	    return (void *) duparray(a, (VFunc) dupstruct2);
    case NT_LIST | NT_STR:
	if (type & NT_HEAP)
	    if (useheap)
		return (void *) duplist(a, (VFunc) dupstring);
	    else
		return (void *) list2arr(a, (VFunc) ztrdup);
	else if (useheap)
	    return (void *) arr2list(a, (VFunc) dupstring);
	else
	    return (void *) duparray(a, (VFunc) ztrdup);
    case NT_NODE | NT_ARR:
	return (void *) duparray(a, (VFunc) dupstruct2);
    case NT_STR | NT_ARR:
	return (void *) duparray(a, (VFunc) (useheap ? dupstring : ztrdup));
    default:
	abort();
    }
}

/* Free a syntax tree node of given type, argument number */

/**/
void
freenode(int type, void *a, int argnum)
{
    if (!a)
	return;
    switch (NT_N(type, argnum)) {
    case NT_NODE:
	freestruct(a);
	break;
    case NT_STR:
	zsfree(a);
	break;
    case NT_LIST | NT_NODE:
    case NT_NODE | NT_ARR:
	{
	    char **p = (char **)a;

	    while (*p)
		freestruct(*p++);
	    free(a);
	}
	break;
    case NT_LIST | NT_STR:
    case NT_STR | NT_ARR:
	freearray(a);
	break;
    default:
	abort();
    }
}

/* duplicate a syntax tree */

/**/
void **
dupstruct2(void *a)
{
    struct node *n = (struct node *)a, *m;
    int type;

    if (!n || ((List) n) == &dummy_list)
	return a;
    type = n->type;
    m = (struct node *)alloc(sizetab[NT_TYPE(type)]);
    m->type = (type & ~NT_HEAP);
    if (useheap)
	m->type |= NT_HEAP;
    switch (NT_TYPE(type)) {
    case N_LIST:
	{
	    List nl = (List) n;
	    List ml = (List) m;

	    ml->type = nl->type;
	    ml->left = (Sublist) dupnode(type, nl->left, 0);
	    ml->right = (List) dupnode(type, nl->right, 1);
	}
	break;
    case N_SUBLIST:
	{
	    Sublist nsl = (Sublist) n;
	    Sublist msl = (Sublist) m;

	    msl->type = nsl->type;
	    msl->flags = nsl->flags;
	    msl->left = (Pline) dupnode(type, nsl->left, 0);
	    msl->right = (Sublist) dupnode(type, nsl->right, 1);
	}
	break;
    case N_PLINE:
	{
	    Pline npl = (Pline) n;
	    Pline mpl = (Pline) m;

	    mpl->type = npl->type;
	    mpl->left = (Cmd) dupnode(type, npl->left, 0);
	    mpl->right = (Pline) dupnode(type, npl->right, 1);
	}
	break;
    case N_CMD:
	{
	    Cmd nc = (Cmd) n;
	    Cmd mc = (Cmd) m;

	    mc->type = nc->type;
	    mc->flags = nc->flags;
	    mc->lineno = nc->lineno;
	    mc->args = (Lklist) dupnode(type, nc->args, 0);
	    mc->u.generic = (void *) dupnode(type, nc->u.generic, 1);
	    mc->redir = (Lklist) dupnode(type, nc->redir, 2);
	    mc->vars = (Lklist) dupnode(type, nc->vars, 3);
	}
	break;
    case N_REDIR:
	{
	    Redir nr = (Redir) n;
	    Redir mr = (Redir) m;

	    mr->type = nr->type;
	    mr->fd1 = nr->fd1;
	    mr->fd2 = nr->fd2;
	    mr->name = (char *)dupnode(type, nr->name, 0);
	}
	break;
    case N_COND:
	{
	    Cond nco = (Cond) n;
	    Cond mco = (Cond) m;

	    mco->type = nco->type;
	    mco->left = (void *) dupnode(type, nco->left, 0);
	    mco->right = (void *) dupnode(type, nco->right, 1);
	}
	break;
    case N_FOR:
	{
	    Forcmd nf = (Forcmd) n;
	    Forcmd mf = (Forcmd) m;

	    mf->inflag = nf->inflag;
	    mf->name = (char *)dupnode(type, nf->name, 0);
	    mf->list = (List) dupnode(type, nf->list, 1);
	}
	break;
    case N_CASE:
	{
	    struct casecmd *ncc = (struct casecmd *)n;
	    struct casecmd *mcc = (struct casecmd *)m;

	    mcc->pats = (char **)dupnode(type, ncc->pats, 0);
	    mcc->lists = (List *) dupnode(type, ncc->lists, 1);
	}
	break;
    case N_IF:
	{
	    struct ifcmd *nic = (struct ifcmd *)n;
	    struct ifcmd *mic = (struct ifcmd *)m;

	    mic->ifls = (List *) dupnode(type, nic->ifls, 0);
	    mic->thenls = (List *) dupnode(type, nic->thenls, 1);

	}
	break;
    case N_WHILE:
	{
	    struct whilecmd *nwc = (struct whilecmd *)n;
	    struct whilecmd *mwc = (struct whilecmd *)m;

	    mwc->cond = nwc->cond;
	    mwc->cont = (List) dupnode(type, nwc->cont, 0);
	    mwc->loop = (List) dupnode(type, nwc->loop, 1);
	}
	break;
    case N_VARASG:
	{
	    Varasg nva = (Varasg) n;
	    Varasg mva = (Varasg) m;

	    mva->type = nva->type;
	    mva->name = (char *)dupnode(type, nva->name, 0);
	    mva->str = (char *)dupnode(type, nva->str, 1);
	    mva->arr = (Lklist) dupnode(type, nva->arr, 2);
	}
	break;
    }
    return (void **) m;
}

/* free a syntax tree */

/**/
void
freestruct(void *a)
{
    struct node *n = (struct node *)a;
    int type;

    if (!n || ((List) n) == &dummy_list)
	return;

    type = n->type;
    switch (NT_TYPE(type)) {
    case N_LIST:
	{
	    List nl = (List) n;

	    freenode(type, nl->left, 0);
	    freenode(type, nl->right, 1);
	}
	break;
    case N_SUBLIST:
	{
	    Sublist nsl = (Sublist) n;

	    freenode(type, nsl->left, 0);
	    freenode(type, nsl->right, 1);
	}
	break;
    case N_PLINE:
	{
	    Pline npl = (Pline) n;

	    freenode(type, npl->left, 0);
	    freenode(type, npl->right, 1);
	}
	break;
    case N_CMD:
	{
	    Cmd nc = (Cmd) n;

	    freenode(type, nc->args, 0);
	    freenode(type, nc->u.generic, 1);
	    freenode(type, nc->redir, 2);
	    freenode(type, nc->vars, 3);
	}
	break;
    case N_REDIR:
	{
	    Redir nr = (Redir) n;

	    freenode(type, nr->name, 0);
	}
	break;
    case N_COND:
	{
	    Cond nco = (Cond) n;

	    freenode(type, nco->left, 0);
	    freenode(type, nco->right, 1);
	}
	break;
    case N_FOR:
	{
	    Forcmd nf = (Forcmd) n;

	    freenode(type, nf->name, 0);
	    freenode(type, nf->list, 1);
	}
	break;
    case N_CASE:
	{
	    struct casecmd *ncc = (struct casecmd *)n;

	    freenode(type, ncc->pats, 0);
	    freenode(type, ncc->lists, 1);
	}
	break;
    case N_IF:
	{
	    struct ifcmd *nic = (struct ifcmd *)n;

	    freenode(type, nic->ifls, 0);
	    freenode(type, nic->thenls, 1);

	}
	break;
    case N_WHILE:
	{
	    struct whilecmd *nwc = (struct whilecmd *)n;

	    freenode(type, nwc->cont, 0);
	    freenode(type, nwc->loop, 1);
	}
	break;
    case N_VARASG:
	{
	    Varasg nva = (Varasg) n;

	    freenode(type, nva->name, 0);
	    freenode(type, nva->str, 1);
	    freenode(type, nva->arr, 2);
	}
	break;
    }
    zfree(n, sizetab[NT_TYPE(type)]);
}

/**/
Lklist
duplist(Lklist l, VFunc func)
{
    Lklist ret;
    Lknode node;

    ret = newlist();
    for (node = firstnode(l); node; incnode(node))
	addnode(ret, func(getdata(node)));
    return ret;
}

/**/
char **
duparray(char **arr, VFunc func)
{
    char **ret = (char **)alloc((arrlen(arr) + 1) * sizeof(char *)), **rr;

    for (rr = ret; *arr;)
	*rr++ = (char *)func(*arr++);
    *rr = NULL;

    return ret;
}

/**/
char **
list2arr(Lklist l, VFunc func)
{
    char **arr, **r;
    Lknode n;

    arr = r = (char **)alloc((countnodes(l) + 1) * sizeof(char *));

    for (n = firstnode(l); n; incnode(n))
	*r++ = (char *)func(getdata(n));
    *r = NULL;

    return arr;
}

/**/
Lklist
arr2list(char **arr, VFunc func)
{
    Lklist l = newlist();

    while (*arr)
	addnode(l, func(*arr++));

    return l;
}

/**/
char **
mkarray(char *s)
{
    char **t = (char **)zalloc((s) ? (2 * sizeof s) : (sizeof s));

    if ((*t = s))
	t[1] = NULL;
    return t;
}

/**/
void
feep(void)
{
    if (unset(NOBEEP))
	write(2, "\07", 1);
}

/**/
void
freearray(char **s)
{
    char **t = s;

    while (*s)
	zsfree(*s++);
    free(t);
}

/**/
int
equalsplit(char *s, char **t)
{
    for (; *s && *s != '='; s++);
    if (*s == '=') {
	*s++ = '\0';
	*t = s;
	return 1;
    }
    return 0;
}

/* see if the right side of a list is trivial */

/**/
void
simplifyright(List l)
{
    Cmd c;

    if (l == &dummy_list || !l->right)
	return;
    if (l->right->right || l->right->left->right ||
	l->right->left->flags || l->right->left->left->right ||
	l->left->flags)
	return;
    c = l->left->left->left;
    if (c->type != SIMPLE || full(c->args) || full(c->redir)
	|| full(c->vars))
	return;
    l->right = NULL;
    return;
}

/* initialize the ztypes table */

/**/
void
inittyptab(void)
{
    int t0;
    char *s;

    for (t0 = 0; t0 != 256; t0++)
	typtab[t0] = 0;
    for (t0 = 0; t0 != 32; t0++)
	typtab[t0] = typtab[t0 + 128] = ICNTRL;
    typtab[127] = ICNTRL;
    for (t0 = '0'; t0 <= '9'; t0++)
	typtab[t0] = IDIGIT | IALNUM | IWORD | IIDENT | IUSER;
    for (t0 = 'a'; t0 <= 'z'; t0++)
	typtab[t0] = typtab[t0 - 'a' + 'A'] = IALPHA | IALNUM | IIDENT | IUSER | IWORD;
    for (t0 = 0240; t0 != 0400; t0++)
	typtab[t0] = IALPHA | IALNUM | IIDENT | IUSER | IWORD;
    typtab['_'] = IIDENT | IUSER;
    typtab['-'] = IUSER;
    typtab[' '] |= IBLANK | INBLANK;
    typtab['\t'] |= IBLANK | INBLANK;
    typtab['\n'] |= INBLANK;
    for (t0 = (int)STOUC(ALPOP); t0 <= (int)STOUC(Nularg);
	 t0++)
	typtab[t0] |= ITOK;
    for (s = ifs; *s; s++)
	typtab[(int)(unsigned char)*s] |= ISEP;
    for (s = wordchars; *s; s++)
	typtab[(int)(unsigned char)*s] |= IWORD;
    for (s = SPECCHARS; *s; s++)
	typtab[(int)(unsigned char)*s] |= ISPECIAL;
}

/**/
char **
arrdup(char **s)
{
    char **x, **y;

    y = x = (char **)ncalloc(sizeof(char *) * (arrlen(s) + 1));

    while ((*x++ = dupstring(*s++)));
    return y;
}

/* next few functions stolen (with changes) from Kernighan & Pike */
/* "The UNIX Programming Environment" (w/o permission) */

/**/
char *
spname(char *oldname)
{
    char *p, spnameguess[PATH_MAX + 1], spnamebest[PATH_MAX + 1];
    static char newname[PATH_MAX + 1];
    char *new = newname, *old;

    if (itok(*oldname)) {
	int ne = noerrs;

	noerrs = 1;
	singsub(&oldname);
	noerrs = ne;
	if (!oldname)
	    return NULL;
    }
    if (access(oldname, F_OK) == 0)
	return NULL;
    old = oldname;
    for (;;) {
	while (*old == '/')
	    *new++ = *old++;
	*new = '\0';
	if (*old == '\0')
	    return newname;
	p = spnameguess;
	for (; *old != '/' && *old != '\0'; old++)
	    if (p < spnameguess + PATH_MAX)
		*p++ = *old;
	*p = '\0';
	if (mindist(newname, spnameguess, spnamebest) >= 3)
	    return NULL;
	for (p = spnamebest; (*new = *p++);)
	    new++;
    }
}

/**/
int
mindist(char *dir, char *mindistguess, char *mindistbest)
{
    int mindistd, nd;
    DIR *dd;
    struct dirent *de;
    char buf[PATH_MAX];

    if (dir[0] == '\0')
	dir = ".";
    mindistd = 100;
    sprintf(buf, "%s/%s", dir, mindistguess);
    if (access(buf, F_OK) == 0) {
	strcpy(mindistbest, mindistguess);
	return 0;
    }
    if (!(dd = opendir(dir)))
	return mindistd;
    while ((de = readdir(dd))) {
	nd = spdist(de->d_name, mindistguess,
		    (int)strlen(mindistguess) / 4 + 1);
	if (nd <= mindistd) {
	    strcpy(mindistbest, de->d_name);
	    mindistd = nd;
	    if (mindistd == 0)
		break;
	}
    }
    closedir(dd);
    return mindistd;
}

/**/
int
spdist(char *s, char *t, int thresh)
{
    char *p, *q;
    char *keymap =
    "\n\n\n\n\n\n\n\n\n\n\n\n\n\n\
\t1234567890-=\t\
\tqwertyuiop[]\t\
\tasdfghjkl;'\n\t\
\tzxcvbnm,./\t\t\t\
\n\n\n\n\n\n\n\n\n\n\n\n\n\n\
\t!@#$%^&*()_+\t\
\tQWERTYUIOP{}\t\
\tASDFGHJKL:\"\n\t\
\tZXCVBNM<>?\n\n\t\
\n\n\n\n\n\n\n\n\n\n\n\n\n\n";

    if (!strcmp(s, t))
	return 0;
/* any number of upper/lower mistakes allowed (dist = 1) */
    for (p = s, q = t; *p && tulower(*p) == tulower(*q); p++, q++);
    if (!*p && !*q)
	return 1;
    if (!thresh)
	return 200;
    for (p = s, q = t; *p && *q; p++, q++)
	if (*p == *q)
	    continue;		/* don't consider "aa" transposed, ash */
	else if (p[1] == q[0] && q[1] == p[0])	/* transpositions */
	    return spdist(p + 2, q + 2, thresh - 1) + 1;
	else if (p[1] == q[0])	/* missing letter */
	    return spdist(p + 1, q + 0, thresh - 1) + 2;
	else if (p[0] == q[1])	/* missing letter */
	    return spdist(p + 0, q + 1, thresh - 1) + 2;
	else if (*p != *q)
	    break;
    if ((!*p && strlen(q) == 1) || (!*q && strlen(p) == 1))
	return 2;
    for (p = s, q = t; *p && *q; p++, q++)
	if (p[0] != q[0] && p[1] == q[1]) {
	    int t0;
	    char *z;

	/* mistyped letter */

	    if (!(z = strchr(keymap, p[0])) || *z == '\n' || *z == '\t')
		return spdist(p + 1, q + 1, thresh - 1) + 1;
	    t0 = z - keymap;
	    if (*q == keymap[t0 - 15] || *q == keymap[t0 - 14] ||
		*q == keymap[t0 - 13] ||
		*q == keymap[t0 - 1] || *q == keymap[t0 + 1] ||
		*q == keymap[t0 + 13] || *q == keymap[t0 + 14] ||
		*q == keymap[t0 + 15])
		return spdist(p + 1, q + 1, thresh - 1) + 2;
	    return 200;
	} else if (*p != *q)
	    break;
    return 200;
}

/**/
char *
zgetenv(char *s)
{
    char **av, *p, *q;

    for (av = environ; *av; av++) {
	for (p = *av, q = s; *p && *p != '=' && *q && *p == *q; p++, q++);
	if (*p == '=' && !*q)
	    return p + 1;
    }
    return NULL;
}

/**/
int
tulower(int c)
{
    c &= 0xff;
    return (isupper(c) ? tolower(c) : c);
}

/**/
int
tuupper(int c)
{
    c &= 0xff;
    return (islower(c) ? toupper(c) : c);
}

/* set cbreak mode, or the equivalent */

/**/
void
setcbreak(void)
{
    struct ttyinfo ti;

    ti = shttyinfo;
#ifdef HAS_TIO
    ti.tio.c_lflag &= ~ICANON;
    ti.tio.c_cc[VMIN] = 1;
    ti.tio.c_cc[VTIME] = 0;
#else
    ti.sgttyb.sg_flags |= CBREAK;
#endif
    settyinfo(&ti);
}

/* give the tty to some process */

/**/
void
attachtty(pid_t pgrp)
{
    static int ep = 0;

    if (jobbing) {
#ifdef HAVE_TCSETPGRP
	if (SHTTY != -1 && tcsetpgrp(SHTTY, pgrp) == -1 && !ep)
#else
# if ardent
	if (SHTTY != -1 && setpgrp() == -1 && !ep)
# else
	int arg = pgrp;

	if (SHTTY != -1 && ioctl(SHTTY, TIOCSPGRP, &arg) == -1 && !ep)
# endif
#endif
	{
	    if (kill(pgrp, 0) == -1)
		attachtty(mypgrp);
	    else {
		zerr("can't set tty pgrp: %e", NULL, errno);
		fflush(stderr);
		opts[MONITOR] = OPT_UNSET;
		ep = 1;
		errflag = 0;
	    }
	}
    }
}

/* get the tty pgrp */

/**/
pid_t
gettygrp(void)
{
    pid_t arg;

    if (SHTTY == -1)
	return -1;

#ifdef HAVE_TCSETPGRP
    arg = tcgetpgrp(SHTTY);
#else
    ioctl(SHTTY, TIOCGPGRP, &arg);
#endif

    return arg;
}

