/*
 * nqs_comms/all-systems/a6btoul.c
 * Convert a 6-bit name into an unsigned long.
 */

#include <nqs_generic/license.h>

/*
 * DESCRIPTION:
 *
 *	This module contains the functions:
 *
 *		a6btoul(), and
 *		is6bitstr()
 *
 *	which respectively evaluate a 6-bit name returning the long
 *	integer value, and return the boolean truth or falsehood of
 *	the specified name as a 6-bit name.
 *
 *	For more information, see pack6name.c
 */

/*
 *	7-bit Character to 6-bit value translation table.
 */

static short ctoi64 [128] = 
{
  127, 127, 127, 127, 127, 127, 127, 127,
  127, 127, 127, 127, 127, 127, 127, 127,
  127, 127, 127, 127, 127, 127, 127, 127,
  127, 127, 127, 127, 127, 127, 127, 127,
  127, 127, 127, 127, 127, 127, 127, 127,
  127, 127, 127,   0, 127, 127, 127, 127,
    1,   2,   3,   4,   5,   6,   7,   8,
    9,  10, 127, 127, 127, 127, 127, 127,
  127,  11,  12,  13,  14,  15,  16,  17,
   18,  19,  20,  21,  22,  23,  24,  25,
   26,  27,  28,  29,  30,  31,  32,  33,
   34,  35,  36, 127, 127, 127, 127,  37,
  127,  38,  39,  40,  41,  42,  43,  44,
   45,  46,  47,  48,  49,  50,  51,  52,
   53,  54,  55,  56,  57,  58,  59,  60,
   61,  62,  63, 127, 127, 127, 127, 127
};

/*** a6btoul
 *
 *
 *	unsigned long a6btoul():
 *
 *	Evaluate the specified 6-bit alphabet name returning the
 *	unsigned long value.
 *
 *	WARNING:
 *		No check is made to enforce the requirement that
 *		the name be constructed from the appropriate 6-bit
 *		alphabet!  Use is6bitstr() if this is required.
 *
 */
unsigned long a6btoul (char *name, int len)
{
  register short ch;
  register unsigned long res;

  ENTER_FUNCTION("a6btoul");
  TEST_ARG(name != NULL, 1);
    /* ensure that the name is not garbage */
  TEST_ARG(len > 0, 2);
    /* ensure that the length makes sense */
  
  res = 0;
  while (len--)
  {
    ch = *name++;
    ch &= 0x7F;
      /* Mask to 7-bits */
    res *= 64;
    res += ctoi64 [ch];
  }
  EXIT_FUNCTION;
  return (res);
}


/*** is6bitstr
 *
 *
 *	int is6bitstr():
 *
 *	Return non-zero (TRUE), if the specified string is a 6-bit
 *	alphabet name string.  Otherwise return zero (FALSE).
 */
int is6bitstr (char *name, int len)
{
  register short ch;

  ENTER_FUNCTION("is6bitstr");
  TEST_ARG(name != NULL, 1);
    /* ensure we are not passed garbage */
  TEST_ARG(len > 0);
    /* ensure that the string length makes sense */
  
  while (len--) 
  {
    ch = *name++;
    ch &= 0x7F;
      /* Mask to 7-bits */
    if (ctoi64 [ch] == 127)
    {
      /*
       *  The specified name is not a valid
       *  6-bit alphabet name.
       */
      EXIT_FUNCTION;
      return (0);
    }
  }
  /*
   *  The specified name is a valid 6-bit alphabet name.
   */
  EXIT_FUNCTION;
  return (1);
}
