/*
 * message.c
 * Message catalogue support
 */

/*
 * Author(s)
 * 
 * Stuart Herbert
 * Academic Computing Services
 * University of Sheffield
 * Email : S.Herbert@sheffield.ac.uk
 */

/*
 * The message catalogue interface is inspired by, but NOT compatible
 * with, the X/Open interface described in the X/Open Portability Guide.
 *
 * Future versions of UFC will provide support for the X/Open interface
 * once it becomes widely available and established.  For the moment,
 * the X/Open interface is not supported on some key older platforms.
 */

/* ***** ***** ***** ***** ***** ***** ***** ***** ***** ***** ***** *****
 * Modifications
 * 
 * 	1.00	SLH	06 Jan 95	Initial version
 * 	1.01	SLH	12 Jul 95	Moved from UFC into CATALOGUE
 */

#include <Catalogue/message.h>
#include <assert.h>
#include <stdio.h>

/* ***** ***** ***** ***** ***** ***** ***** ***** ***** ***** ***** *****
 * Local variables
 */

/* ***** ***** ***** *****
 * UFCV_ulCurrentLanguage - specifiy language set with RESF_CatOpen()
 * 
 * defaults to the first language in the message catalogue.
 */

static UFCT_Language UFCS_ulCurrentLanguage = 0;

/* ***** *****
 * UFCS_Languages - table about the various languages supported
 */

static UFCT_LanguagesTable UFCS_Languages[] =
{
  { "English/UK"	},
  { "English/USA"	}
};

unsigned int UFCF_GetLanguageNo(char *szLanguage)
{
  UFCT_LanguagesTable *pstWalker;
  unsigned int         ulIndex = 0;
  
  assert(szLanguage != NULL);
  
  for (pstWalker = UFCS_Languages; pstWalker->szMessage != NULL; pstWalker++)
  {
    if (strncmp(pstWalker->szMessage, szLanguage, strlen(pstWalker->szMessage)) == 0)
      return ulIndex;
    ulIndex++;
  }
}

/* ***** *****
 * Set a language
 */

void UFCF_SetLanguage(UFCT_Language uiLang)
{
  assert(uiLang < UFCD_LANG_MAX);
    /* ensure the user is requesting a legitimate language */
  
  UFCS_ulCurrentLanguage = uiLang;
}

/* ***** ***** ***** *****
 * Get a message
 *
 * stLangTable		- pointer to the message catalogue
 * ulMessage		- message required
 */

char * UFCF_GetMessage(UFCT_LanguageTable *stLangTable, 
		       UFCT_Message	   ulMessage)
{
  static UFCT_MessageTable *stMessTable = NULL;
    /* pointer to the last used message table - used to improve performance */
  static UFCT_LanguageTable *stLastLangTable = NULL;
    /* pointer to the last used language table - used to improve performance */
  static UFCT_Language ulLastLanguage = UFCD_LANG_MAX;
    /* value of the language used last time - used to improve performance */
  UFCT_LanguageTable *stIndex;
    /* search pointer */
  
  assert(stLangTable != NULL);
  
  /* 
   * this has the effect of a most-recently-used cache, which improves/
   * performance when messages repeatedly come from the same message 
   * catalogue
   */
  
  if ((ulLastLanguage != UFCS_ulCurrentLanguage) ||
      (stLangTable    != stLastLangTable))
  {

    /*
     * if we get to this point, then there has either been a change of
     * message catalogue (most likely), or a change of language (less
     * likely).
     *
     * we must search the language table for the message catalogue for
     * the current language.
     */
  
    ulLastLanguage = UFCS_ulCurrentLanguage;
    stIndex = stLastLangTable = stLangTable;
  
    while (stIndex->ulLanguage != UFCD_LANG_MAX)
      if (stIndex->ulLanguage == UFCS_ulCurrentLanguage)
      {
        stMessTable = stIndex->stMessageTable;
	break;
      }
  }
  
  return (stMessTable + ulMessage)->szMessage;
}
