/*
 *	Network Queueing System (NQS)
 *  This version of NQS is Copyright (C) 1992  John Roman
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 1, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
/*++ nqs_disreq.c - Network Queueing System
 *
 * $Source: /usr2/jrroma/nqs/nqs-3.30/src/RCS/nqs_disreq.c,v $
 *
 * DESCRIPTION:
 *
 *	Dispose of an NQS request or subrequest.  This can
 *	entail deleting the control and data files for the
 *	request (not a subrequest), as well as the memory-
 *	resident version of the request or subrequest structure,
 *	and associated transaction descriptor (not a subrequest).
 *	The Extreqcount global var is also updated as appropriate.
 *
 *
 *	Author:
 *	-------
 *	Brent A. Kingsbury, Sterling Software Incorporated.
 *	March 30, 1986.
 *
 *
 * STANDARDS VIOLATIONS:
 *   None.
 *
 * REVISION HISTORY: ($Revision: 1.3 $ $Date: 1992/12/22 15:39:52 $ $State: Exp $)
 * $Log: nqs_disreq.c,v $
 * Revision 1.3  1992/12/22  15:39:52  jrroma
 * Version 3.30
 *
 * Revision 1.2  92/06/18  17:31:00  jrroma
 * Added gnu header
 * 
 * Revision 1.1  92/02/12  16:14:32  jrroma
 * Initial revision
 * 
 *
 */

#include "nqs.h"			/* NQS constants and data types */
#include "nqsxvars.h"			/* NQS external vars and dirs  */
#include <malloc.h>

/*** nqs_disreq
 *
 *
 *	void nqs_disreq():
 *
 *	Dispose of an NQS request or subrequest.  This can
 *	entail deleting the control and data files for the
 *	request (not a subrequest), as well as the memory-
 *	resident version of the request or subrequest structure,
 *	and associated transaction descriptor (not a subrequest).
 *	The Extreqcount global var is also updated as appropriate.
 */
void nqs_disreq (struct request *request, short deleteflag)
{
	register int tid;		/* Transaction-id */

	/*
	 *  MOREHERE someday to delete child subrequest(s) as well.
	 */
	if ((request->status & RQF_SUBREQUEST) == 0) {
		/*
		 *  This request is not a subrequest.
		 */
		if (request->status & RQF_EXTERNAL) {
			/*
			 *  The request was queued from an external source.
			 *  Update the number of externally queued requests
			 *  as necessary.
			 */
			Extreqcount--;	/* One less externally queued */
		}			/* request */
		tid = request->v1.req.trans_id;
		/*
		 *  This request still has a transaction-id/descriptor
		 *  assigned to it.  Free up the transaction-id so that
		 *  it can be used by another request.
		 */
		tid_deallocate (tid);
		/*
		 *  Update the permanent memory NQS database to record
		 *  that the corresponding transaction descriptor is now
		 *  unallocated.
		 *
		 *  WARNING:  It is CRITICAL that the transaction-id/descr
		 *	      for a request be released PRIOR to actually
		 *	      deleting any files associated with the
		 *	      request.  This temporal sequence is
		 *	      CRITICAL and ESSENTIAL for the correct
		 *	      operation of nqs_rbuild.c.
		 */
		if (tra_release (tid) == -1) {
			/*
			 *  Something went wrong updating the database.
			 */
			printf ("E$Tra_release() error in nqs_disreq.\n");
			printf ("I$%s.\n", asciierrno());
			printf ("I$Transaction-id was: %1d.\n", tid);
			fflush (stdout);
		}
		if (deleteflag) {
			/*
			 *  The request is NOT being placed in the
			 *  failed directory for analysis.  The request
			 *  control and data files are to be deleted.
			 */
			nqs_delrfs (request->v1.req.orig_seqno,
				    request->v1.req.orig_mid,
				    request->v1.req.ndatafiles);
		}
	}
	free ((char *) request);	/* Delete the request structure */
}
