/*
 *	Network Queueing System (NQS)
 *  This version of NQS is Copyright (C) 1992  John Roman
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 1, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
/*
*  PROJECT:     Network Queueing System
*  AUTHOR:      John Roman
*
*  Modification history:
*
*       Version Who     When            Description
*       -------+-------+---------------+-------------------------
*       V01.10  JRR                     Initial version.
*       V01.2   JRR     12-Feb-1992	Fixed static routine references.
*       V01.3   JRR     14-Feb-1992	Make boundpreamble into rcm_bound...
*	V01.4	JRR	17-Jun-1992	Added header.
*	V01.5	JRR	01-Dec-1992	Modified for C prototypes.
*/
/*++ analyzercm.c - Network Queueing System
 *
 * $Source: /usr2/jrroma/nqs/nqs-3.30/lib/RCS/analyzercm.c,v $
 *
 * DESCRIPTION:
 *
 *
 *	Completely analyze and diagnose a request completion
 *	code.
 *
 *
 *	Author:
 *	-------
 *	Brent A. Kingsbury, Sterling Software Incorporated.
 *	April 28, 1986.
 *
 *
 * STANDARDS VIOLATIONS:
 *   None.
 *
 * REVISION HISTORY: ($Revision: 1.5 $ $Date: 1992/12/22 15:45:56 $ $State: Exp $)
 * $Log: analyzercm.c,v $
 * Revision 1.5  1992/12/22  15:45:56  jrroma
 * Version 3.30
 *
 * Revision 1.4  92/06/18  13:23:31  jrroma
 * Added gnu header
 * 
 * Revision 1.3  92/02/14  10:22:05  jrroma
 * Finally fix up static routines.
 * 
 * Revision 1.2  92/02/12  12:27:59  jrroma
 * Fixed references to static routines.
 * 
 * Revision 1.1  92/02/12  12:26:06  jrroma
 * Initial revision
 * 
 *
 */

#include "nqs.h"
#include "informcc.h"		/* NQS information completion codes */
#include "requestcc.h"		/* NQS request completion codes */

static void applypreamble ( FILE *stream, char *prefix );
static void boundpreamble ( FILE *stream, char *prefix );
static void routefaipreamble ( FILE *stream, char *prefix );

/*** analyzercm
 *
 *
 *	void analyzercm():
 *
 *	Completely analyze and diagnose a request completion
 *	code.
 */
void analyzercm (
	long code,		/* Completion code */
	FILE *stream,		/* Output stream */
	char *prefix)		/* Output prefix */
{

	register short reason;		/* Reason bits of completion code */

	reason = (code & XCI_REASON_MASK);
	if (reason < 0 || reason > (RCM_MAXRCM & XCI_REASON_MASK) ||
	   (code & XCI_TRANSA_MASK)) {
		/*
		 *  The request completion code is invalid.
		 */
		reason = (RCM_UNDEFINED & XCI_REASON_MASK);
		code = RCM_UNDEFINED;
	}
	else if (reason == (RCM_UNDEFINED & XCI_REASON_MASK)) {
		/*
		 *  Clear information bits if undefined request completion
		 *  code.
		 */
		code &= ~XCI_INFORM_MASK;
	}
	/*
	 *  Analyze and describe a specific request completion code.
	 */
	rcmmsgs (code, stream, prefix);
	/*
	 *  Analyze and describe any additional information bits
	 *  set in the request completion code.
	 */
	if (reason == RCM_DELIVERFAI || reason == RCM_DELIVERRETX) {
		fputc ('\n', stream);
		fputs (prefix, stream);
		fputs ("Transaction failure reason:\n", stream);
		fflush (stream);	/* For output to logfile */
		tcmmsgs (code >> 10, stream, prefix);
	}
	else if (reason == RCM_STAGEOUTBAK || reason == RCM_STAGEOUTFAI) {
		fputc ('\n', stream);
		fputs (prefix, stream);
		fputs ("Transaction failure reason at primary destination:\n",
			stream);
		fflush (stream);	/* For output to logfile */
		tcmmsgs (code >> 10, stream, prefix);
	}
	else if (reason == RCM_ROUTED && reason == RCM_ROUTEDLOC &&
		 reason == RCM_DELIVERED) {
		/*
		 *  For the moment, we allow RCM_DELIVERED
		 *  to have quota information bits in this
		 *  ****TRIAGE**** implementation.
		 */
		tcimsgs (code, stream, prefix, boundpreamble);
	}
	else if (reason == RCM_ROUTEFAI || reason == RCM_ROUTERETX) {
		rcimsgs (code, stream, prefix, routefaipreamble);
	}
	else if (reason == RCM_UNABLETOEXE) {
		tcimsgs (code, stream, prefix, applypreamble);
	}
}


/*** applypreamble
 *
 *
 *	void applypreamble():
 *	Display limit application preamble.
 */
static void applypreamble (FILE *stream, char *prefix)
{
	fputc ('\n', stream);
	fputs (prefix, stream);
	fputs ("Problem enforcing the following limit(s):\n\n", stream);
	fflush (stream);
}


/*** boundpreamble
 *
 *
 *	void boundpreamble():
 *	Display bounded limit preamble.
 */
static void boundpreamble (FILE *stream, char *prefix)
{
	fputc ('\n', stream);
	fputs (prefix, stream);
	fputs ("In order to successfully queue or deliver the request,\n",
		stream);
	fputs (prefix, stream);
	fputs ("it was necessary to alter the following explicit request\n",
		stream);
	fputs (prefix, stream);
	fputs ("quota limit(s), binding them to values within the ranges\n",
		stream);
	fputs (prefix, stream);
	fputs ("enforceable at the execution machine:\n\n", stream);
	fflush (stream);		/* For output to logfile */
}


/*** routefaipreamble
 *
 *
 *	void routefaipreamble():
 *	Display pipe queue request routing failure preamble.
 */
static void routefaipreamble (FILE *stream, char *prefix)
{
	fputc ('\n', stream);
	fputs (prefix, stream);
	fputs ("The request could not be routed to any of the possible\n",
		stream);
	fputs (prefix, stream);
	fputs (
	  "pipe queue destinations because of the following reason(s):\n\n",
	   stream);
	fflush (stream);		/* For output to logfile */
}
