/*
 *	Network Queueing System (NQS)
 *  This version of NQS is Copyright (C) 1992  John Roman
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 1, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
/*++ dir.4.2.h - 
 *
 * $Source: /usr2/jrroma/nqs/nqs-3.35.6/libndir/RCS/dir.4.2.h,v $
 *
 * DESCRIPTION:
 *
 *	Emulate the Berkeley UNIX directory structure on top
 *	of the System V fixed name length directory structure.
 *
 *	The following comments describe the structure of a Berkeley
 *	UNIX directory:
 *
 *	A directory consists of some number of blocks of DIRBLKSIZ
 *	bytes, where DIRBLKSIZ is chosen such that it can be transferred
 *	to disk in a single atomic operation (e.g. 512 bytes on most machines).
 *
 *	Each DIRBLKSIZ byte block contains some number of directory entry
 *	structures, which are of variable length.  Each directory entry has
 *	a struct direct at the front of it, containing its inode number,
 *	the length of the entry, and the length of the name contained in
 *	the entry.  These are followed by the name padded to a 4 byte boundary
 *	with null bytes.  All names are guaranteed null terminated.
 *	The maximum length of a name in a directory is MAXNAMLEN.
 *
 *	The macro DIRSIZ(dp) gives the amount of space required to represent
 *	a directory entry.  Free space in a directory is represented by
 *	entries which have dp->d_reclen >= DIRSIZ(dp).  All DIRBLKSIZ bytes
 *	in a directory block are claimed by the directory entries.  This
 *	usually results in the last entry in a directory having a large
 *	dp->d_reclen.  When entries are deleted from a directory, the
 *	space is returned to the previous entry in the same directory
 *	block by increasing its dp->d_reclen.  If the first entry of
 *	a directory block is free, then its dp->d_ino is set to 0.
 *	Entries other than the first in a directory do not normally have
 *	dp->d_ino set to 0.
 *
 *
 *	Author:
 *	-------
 *	Kirk McKusick, University of California, Berkeley.
 *	March 10, 1982.
 *
 *	Modifications:
 *	--------------
 *	Brent A. Kingsbury, Sterling Software Incorporated.
 *	September 10, 1985.
 *
 *	  Changed "#include <dir.h>" to read '#include "dir.4.2.h"' since
 *	  we are after the Berkeley definition, and not System V.
 *
 *
 * STANDARDS VIOLATIONS:
 *
 * REVISION HISTORY: ($Revision: 1.4 $ $Date: 1994/03/30 20:34:14 $ $State: Exp $)
 * $Log: dir.4.2.h,v $
 * Revision 1.4  1994/03/30  20:34:14  jrroma
 * Version 3.35.6
 *
 * Revision 1.3  92/06/18  13:33:30  jrroma
 * Added gnu header
 * 
 * Revision 1.2  92/03/20  10:56:25  jrroma
 * Done.
 * 
 * Revision 3.2  91/02/11  16:56:27  root
 * Version 2.0 Source
 * 
 * Revision 2.2  87/04/22  14:57:46  hender
 * Sterling version 4/22/87
 * 
 *
 */

/*
 * 
 *	Define the directory block allocation size in bytes.
 *
 *	WARNING:
 *	  Seven system types are defined:  BSD42, BSD43, SGI, SYS52,
 *	  ULTRIX, UNICOS, and UTS.  Accordingly, the flag: SYSTEM_TYPE
 *	  in the Makefile must be appropriately defined.
 *
 *
 */
#ifdef	BSD43		/* Berkeley UNIX */
#define	DIRBLKSIZ	512
#else
#ifdef	SYS52				/* AT&T UNIX */
#define	DIRBLKSIZ	512
#else
#ifdef	UTS				/* Amdahl Unix */
#define	DIRBLKSIZ	4096
#else
BAD SYSTEM TYPE
#endif
#endif
#endif
#define	MAXNAMLEN	255

#if	BSD43 | SYS52 | UTS
struct	direct {
	unsigned long d_ino;		/* inode number of entry */
	unsigned short d_reclen;	/* length of this record */
	unsigned short d_namlen;	/* length of string in d_name */
	char	d_name[MAXNAMLEN + 1];	/* name must be no longer than this */
};
#else
BAD SYSTEM TYPE
#endif
#endif

/*
 * The DIRSIZ macro gives the minimum record length which will hold
 * the directory entry.  This requires the amount of space in struct direct
 * without the d_name field, plus enough space for the name with a terminating
 * null byte (dp->d_namlen+1), rounded up to a 4 byte boundary.
 */
#undef DIRSIZ
#define DIRSIZ(dp) \
    ((sizeof (struct direct) - (MAXNAMLEN+1)) + (((dp)->d_namlen+1 + 3) &~ 3))

#ifndef /* KERNEL */
/*
 * Definitions for library routines operating on directories.
 */
typedef struct _dirdesc {
	int	dd_fd;
	long	dd_loc;
	long	dd_size;
	char	dd_buf[DIRBLKSIZ];
} DIR;
#ifndef NULL
#define NULL 0
#endif
extern	DIR *opendir();
extern	struct direct *readdir();
extern	long telldir();
extern	void seekdir();
#define rewinddir(dirp)	seekdir((dirp), (long)0)
extern	void closedir();
#endif KERNEL
#endif	/* BSD43 | SYS52 | UTS */

