/*
 *	Network Queueing System (NQS)
 *  This version of NQS is Copyright (C) 1992  John Roman
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 1, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
/*++ writehdr.c - Network Queueing System
 *
 * $Source: /usr2/jrroma/nqs/nqs-3.30/lib/RCS/writehdr.c,v $
 *
 * DESCRIPTION:
 *
 *	[Re]write the common header portion of a request control
 *	file.
 *
 *
 *	Author:
 *	-------
 *	Brent A. Kingsbury, Sterling Software Incorporated.
 *	August 12, 1985.
 *
 *
 * STANDARDS VIOLATIONS:
 *   None.
 *
 * REVISION HISTORY: ($Revision: 1.3 $ $Date: 1992/12/22 15:46:38 $ $State: Exp $)
 * $Log: writehdr.c,v $
 * Revision 1.3  1992/12/22  15:46:38  jrroma
 * Version 3.30
 *
 * Revision 1.2  92/06/18  13:25:02  jrroma
 * Added gnu header
 * 
 * Revision 1.1  92/06/18  12:51:44  jrroma
 * Initial revision
 * 
 *
 */

#include "nqs.h"
#include <errno.h>
#include <unistd.h>


/*** writehdr
 *
 *
 *	int writehdr():
 *
 *	[Re]write the common header portion of a request control
 *	file.
 *
 *	Unless errno is returned non-zero, the lseek() position
 *	of the control file descriptor upon return, is ALWAYS at
 *	the first byte of the varying portion of the control file.
 *
 *
 *	Returns:
 *		0: if successful;
 *	       -1: otherwise.
 *
 *			Note:  errno will be returned 0 by this function
 *			       unless a system call error has occurred.
 */
int writehdr (int fd, struct rawreq *rawreq)
{
	register int size;		/* Size of common request header */

	errno = 0;
	if (lseek (fd, 0L, 0) == -1) {	/* Seek to start of control file */
		return (-1);		/* Return EBADF */
	}
	/*
	 *  [Re]write the common header portion of an NQS control file,
	 *  in one single write() system call.
	 *
	 *  We do this for reasons of efficiency, and also so that the first
	 *  block of the control file is [re]written in a single atomic
	 *  I/O operation so that Qmod updates, and multiple readers of
	 *  the control file (who are also carefully reading the first
	 *  block of the control in an indivisible fashion), will not
	 *  step on each other.
	 */
	size = ((char *) &rawreq->v) - ((char *) rawreq);
	if (write (fd, (char *) rawreq, size) != size) {
		if (errno == 0) {
			/*
			 *  We presume that the receiving device has been
			 *  completely filled-up, and artifically set errno
			 *  to ENOSPC, for the benefit of the caller.
			 */
			errno = ENOSPC;	/* Insufficient space */
		}
		return (-1);		/* Write() failed */
	}
	/*
	 *  Leave the file pointer pointing at the varying part
	 *  of the request control file.
	 */
	if (rawreq->type == RTYPE_DEVICE) {
		/*
		 *  We are [re]writing the control file common
		 *  header of a device-oriented request.
		 */
		size = sizeof (struct rawdevreq);
	}
	else {
		/*
		 *  We are [re]writing the control file common
		 *  header of a non device-oriented request.
		 */
		size = sizeof (struct rawbatreq);
	}
	lseek (fd, (long) size, 1);	/* Relative seek */
	errno = 0;			/* Extreme paranoia */
	return (0);
}
