/*
 *	Network Queueing System (NQS)
 *  This version of NQS is Copyright (C) 1992  John Roman
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 1, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
/*
*  PROJECT:     Network Queueing System
*  AUTHOR:      John Roman
*
*  Modification history:
*
*       Version Who     When            Description
*       -------+-------+---------------+-------------------------
*       V01.10  JRR                     Initial version.
*       V01.20  JRR     16-Jan-1992	Added support for RS6000.
*	V01.3	JRR	17-Jun-1992	Added header.
*	V01.4	JRR			Modified for C prototypes.	
*	V01.5	JRR
*	V01.6	JRR	23-Dec-1992	Change index=>strchr
*	V01.7	JRR	02-Mar-1994	Fixed up NETDB.
*/
/*++ machpath.c - Network Queueing System
 *
 * $Source: /usr2/jrroma/nqs/nqs-3.35.6/lib/RCS/machpath.c,v $
 *
 * DESCRIPTION:
 *
 *	Determine the machine-id of the explicit or implied machine
 *	specification in the given path specification:
 *
 *		path_name			OR
 *		machine_name:path_name
 *
 *
 *	Author:
 *	-------
 *	Brent A. Kingsbury, Sterling Software Incorporated.
 *	November 5, 1985.
 *
 *
 * STANDARDS VIOLATIONS:
 *   None.
 *
 * REVISION HISTORY: ($Revision: 1.7 $ $Date: 1994/03/30 20:32:36 $ $State: Exp $)
 * $Log: machpath.c,v $
 * Revision 1.7  1994/03/30  20:32:36  jrroma
 * Version 3.35.6
 *
 * Revision 1.6  93/02/05  23:13:15  jrroma
 * Version 3.31
 * 
 * Revision 1.5  92/12/22  15:46:11  jrroma
 * Version 3.30
 * 
 * Revision 1.4  92/11/06  13:31:13  jrroma
 * Added support for C prototypes.
 * 
 * Revision 1.3  92/06/18  13:24:06  jrroma
 * Added gnu header
 * 
 * Revision 1.2  92/01/16  15:59:29  jrroma
 * Added support for RS6000.
 * 
 * Revision 1.1  92/01/16  15:58:30  jrroma
 * Initial revision
 * 
 *
 */

#include "nqs.h"
#include <string.h>
#include <netdb.h>			/* Network database */
#include "nmap.h"			/* Network mapping codes */

/*** machpath
 *
 *
 *	int machpath():
 *
 *	Determine the machine-id of the explicit or implied machine
 *	specification in the given path specification.
 *
 *		path_name			OR
 *		machine_name:path_name
 *
 *	Returns:
 *		1: if successful, and an explicit machine-name
 *		   appeared in the pathname specification.
 *		   The machine-id parameter is appropriately
 *		   updated.
 *
 *		0: if successful, but no explicit machine-name
 *		   appeared in the pathname specification (the
 *		   local machine is implicitly assumed).  The
 *		   machine-id parameter is appropriately
 *		   updated.
 *
 *	       -1: if a null machine-name was specified in the
 *		   path-name:
 *
 *			:path_name
 *
 *	       -2: if the explicit or implied machine-specification
 *		   is not recognized by the local system (NMAP_ENOMAP).
 *
 *	       -3: if the Network Mapping Procedures (NMAP_)
 *		   deny access to the caller (NMAP_ENOPRIV).
 *
 *	       -4: if some other general NMAP_ error occurs.
 */
int machpath (char *path_name, Mid_t *machine_id)
{
	struct hostent *ent;	/* Host table entry structure */
	char *end_machine;	/* End of machine-name */

	if ((end_machine = strchr (path_name, ':')) == (char *) 0) {
		/*
		 *  No machine-specification was given.  The
		 *  local machine is assumed.
		 */
		return (localmid (machine_id));
	}
	/*
	 *  A machine-name specification appears.
	 */
	if (path_name == end_machine) {
		/*
		 *  Missing machine-specification.
		 */
		return (-1);		/* Bad syntax */
	}
	/*
	 *  Determine the machine-id of the destination machine.
	 */
	*end_machine = '\0';		/* Temporarily zap ':' */
	ent = gethostbyname (path_name);/* Lookup machine-name */
	*end_machine = ':';		/* Restore ':' */
	if (ent == (struct hostent *) 0) {
		return (-2);		/* This machine is not */
					/* known to us */
	}
	switch (nmap_get_mid (ent, machine_id)) {
	case NMAP_SUCCESS:		/* Successfully got local machine-id */
		return (1);		/* for explicitly named machine */
	case NMAP_ENOMAP:		/* What?  No local mid defined! */
		return (-2);
	case NMAP_ENOPRIV:		/* No privilege */
		return (-3);
	}
	return (-4);			/* General NMAP_ error */
}
