/*
 *	Network Queueing System (NQS)
 *  This version of NQS is Copyright (C) 1992  John Roman
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 1, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
/*
*  PROJECT:     Network Queueing System
*  AUTHOR:      John Roman
*
*  Modification history:
*
*       Version Who     When            Description
*       -------+-------+---------------+-------------------------
*       V01.10  JRR                     Initial version.
*       V01.20  JRR     16-Jan-1992	Added support for RS6000.
*	V01.3	JRR	17-Jun-1992	Added header.
*	V01.4	JRR			Placeholder.
*	V01.5	JRR	01-Mar-1994	Added support for SOLARIS.
*/
/*++ nqs_fndreq.c - Network Queueing System
 *
 * $Source: /usr2/jrroma/nqs/nqs-3.35.6/src/RCS/nqs_fndreq.c,v $
 *
 * DESCRIPTION:
 *
 *	Find an NQS request in one of the local queues by sequence
 *	number, and machine-id only.
 *
 *
 *	Author:
 *	-------
 *	Brent A. Kingsbury, Sterling Software Incorporated.
 *	August 12, 1985.
 *
 *
 * STANDARDS VIOLATIONS:
 *   None.
 *
 * REVISION HISTORY: ($Revision: 1.5 $ $Date: 1994/03/30 20:36:38 $ $State: Exp $)
 * $Log: nqs_fndreq.c,v $
 * Revision 1.5  1994/03/30  20:36:38  jrroma
 * Version 3.35.6
 *
 * Revision 1.4  92/06/18  17:31:06  jrroma
 * Added gnu header
 * 
 * Revision 1.3  92/05/06  10:39:57  jrroma
 *  Version 3.20
 * 
 * Revision 1.2  92/01/17  10:57:27  jrroma
 * Added support for RS6000.
 * 
 * Revision 1.1  92/01/17  10:56:41  jrroma
 * Initial revision
 * 
 *
 */

#include "nqs.h"			/* NQS constants and data types */
#include "nqsxvars.h"			/* Global vars */

/*** nqs_fndreq
 *
 *
 *	struct request *nqs_fndreq():
 *
 *	Find an NQS request in one of the local queues by original
 *	sequence number, and original machine-id.
 *
 *	Returns:
 *		A pointer to the request structure for the request,
 *		if the request was located in one of the local NQS
 *		queues.  Otherwise, a NIL request pointer is returned.
 */
struct request *nqs_fndreq (orig_seqno, orig_mid, predecessor, state)
register long orig_seqno;		/* Original req sequence number. */
register Mid_t orig_mid;		/* Original machine-id */
struct request **predecessor;		/* If the request is located, then */
					/* we return a pointer to the */
					/* predecessor request structure */
					/* for the request.  If the request */
					/* is the first member in the set, */
					/* then *predecessor upon return */
					/* will be (struct request *) 0. */
int *state;				/* This parameter is a "value/return"*/
					/* parameter.  *State initially */
					/* identifies the set of request */
					/* queue states in which the request */
					/* may currently reside (RQS_). */
					/* Upon completion, *state has a */
					/* single RQS_ state value, the */
					/* state of the located request */
					/* (if the request is indeed found) */
{
	register struct nqsqueue *queue;	/* Queue set to traverse */
	register struct request *req;
	register struct request *pred;
	register short set;		/* Request set [0..6] */
	int setstate;			/* State type for search set */

	queue = Nonnet_queueset;	/* The request will always be */
					/* located in a non-network queue */
					/* if it exists (note that subre- */
					/* quests are never ever placed in */
					/* non-network queues). Subrequests*/
					/* are only found in network queues*/
					/* and they ALWAYS have a parent in*/
					/* a non-network queue */
	while (queue != (struct nqsqueue *) 0) {
		set = 0;
		while (set <= 6) {
			/*
			 *  Search the next set for the request.
			 */
			req = (struct request *) 0;
			switch (set) {
			case 0:	if (*state & RQS_DEPARTING) {
					setstate = RQS_DEPARTING;
					req = queue->departset;
				}
				break;
			case 1:	if (*state & RQS_RUNNING) {
					setstate = RQS_RUNNING;
					req = queue->runset;
				}
				break;
			case 2: if (*state & RQS_STAGING) {
					setstate = RQS_STAGING;
					req = queue->stageset;
				}
				break;
			case 3: if (*state & RQS_QUEUED) {
					setstate = RQS_QUEUED;
					req = queue->queuedset;
				}
				break;
			case 4: if (*state & RQS_WAITING) {
					setstate = RQS_WAITING;
					req = queue->waitset;
				}
				break;
			case 5: if (*state & RQS_HOLDING) {
					setstate = RQS_HOLDING;
					req = queue->holdset;
				}
				break;
			case 6: if (*state & RQS_ARRIVING) {
					setstate = RQS_ARRIVING;
					req = queue->arriveset;
				}
				break;
			}
			pred = (struct request *) 0;
			while (req != (struct request *) 0 &&
			      (req->v1.req.orig_seqno != orig_seqno ||
			       req->v1.req.orig_mid != orig_mid)) {
				pred = req;
				req = req->next;
			}
			if (req != (struct request *) 0) {
				/*
				 *  We found the request.
				 */
				*predecessor = pred;
				*state = setstate;
				return (req);
			}
			set++;		/* Examine the next set (state) */
		}
		/*
		 *  Examine the next queue in the queue set.
		 */
		queue = queue->next;
	}
	return ((struct request *) 0);	/* Not found */
}
