/*
 *	Network Queueing System (NQS)
 *  This version of NQS is Copyright (C) 1992  John Roman
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 1, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
/*
*  PROJECT:     Network Queueing System
*  AUTHOR:      John Roman
*
*  Modification history:
*
*       Version Who     When            Description
*       -------+-------+---------------+-------------------------
*       V01.10  JRR                     Initial version.
*       V01.20  JRR     16-Jan-1992	Added support for RS6000.
*       V01.3   JRR     02-Mar-1992	Added Cosmic V2 changes.
*       V01.4   JRR     03-Mar-1992	Changed mid_t to Mid_t.
*       V01.5   JRR     18-Mar-1992	Fixed modify request:  when change
*					per request cpu limit need to fix
*					infinite per process cpu limit flag.
*       V01.6   JRR     10-Apr-1992     Added CERN enhancments.
*	V01.7	JRR	26-May-1992	Fixed support for RS6000.
*					Added header.
*	V01.8	JRR	10-Nov-1992	Added support for HPUX.
*	V01.9	JRR	06-Apr-1993	Added support for DECOSF.
*	V01.10	JRR	05-Aug-1994	Fixes per Bruno Wolff.
*/
/*++ nqs_modreq.c - Network Queueing System
 *
 * $Source: /usr2/jrroma/nqs/nqs-3.36/src/RCS/nqs_modreq.c,v $
 *
 * DESCRIPTION:
 *
 *	Modify a queued NQS request.
 *
 *
 *	Author:
 *	-------
 *	Brent A. Kingsbury, Sterling Software Incorporated.
 *	October 8, 1985.
 *
 *
 * STANDARDS VIOLATIONS:
 *   None.
 *
 * REVISION HISTORY: ($Revision: 1.10 $ $Date: 1994/09/02 17:39:42 $ $State: Exp $)
 * $Log: nqs_modreq.c,v $
 * Revision 1.10  1994/09/02  17:39:42  jrroma
 * Version 3.36
 *
 * Revision 1.9  1993/07/13  21:33:54  jrroma
 * Version 3.34
 *
 * Revision 1.8  92/12/22  15:41:23  jrroma
 * Version 3.30
 * 
 * Revision 1.7  92/06/18  17:31:10  jrroma
 * Added gnu header
 * 
 * Revision 1.6  92/05/06  10:41:18  jrroma
 *  Version 3.20
 * 
 * Revision 1.5  92/03/20  10:59:14  jrroma
 * *** empty log message ***
 * 
 * Revision 1.4  92/03/03  17:04:20  jrroma
 * Changed mid_t to Mid_t.
 * 
 * Revision 1.3  92/03/02  15:13:15  jrroma
 * Added Cosmic V2 changes.
 * 
 * Revision 1.2  92/01/17  11:02:16  jrroma
 * Added support for RS6000.
 * 
 * Revision 1.1  92/01/17  11:01:30  jrroma
 * Initial revision
 * 
 *
 */

#include "nqs.h"			/* NQS constants and data types */
#include <errno.h>
#include "transactcc.h"			/* Transaction completion codes */
#include "nqsxvars.h"			/* Global vars */


/*** nqs_modreq
 *
 *
 *	long nqs_modreq():
 *	Modify a queued NQS request.
 *
 *	Returns:
 *		TCML_COMPLETE:	 if the request was successfully
 *				 modified.
 *		TCML_NOSUCHREQ:	 if the specified request does not
 *				 exist on this machine.
 *		TCML_NOTREQOWN:	 if the mapped user-id does not
 *				 match the current mapped user-id
 *				 of the request owner.
 *		TCML_REQRUNNING: if the specified request is running.
 */
long nqs_modreq (mapped_uid, orig_seqno, orig_mid, modtype, param1, param2, inf)
uid_t mapped_uid;                       /* Mapped owner user-id */
long orig_seqno;                        /* Original request sequence# */
Mid_t orig_mid;                         /* Original machine-id of req */
long modtype;                           /* Type of modification */
long param1;                            /* First modification parameter */
long param2;                            /* Second modification parameter */
int inf;                                /* Set to infinite flag */
{
	struct request *predecessor;	/* Predecessor in req set in queue */
	int state;			/* Request queue state: RQS_ */
	struct request *req;		/* Ptr to request structure for */
					/* located req, if found */
        char path[MAX_PATHNAME+1];      /* pathname of control file */
        int cfd;                        /* file descriptor for control file */
        struct rawreq rawreq;           /* raw request structure */
        long    lim;                    /* New limit */


	/*
	 *  Locate the request.
	 */
	state = RQS_STAGING | RQS_QUEUED | RQS_WAITING | RQS_HOLDING
	      | RQS_ARRIVING | RQS_RUNNING;
        predecessor = (struct request *) 0;
	if ((req = nqs_fndreq (orig_seqno, orig_mid, &predecessor,
			       &state)) == (struct request *) 0) {
		/*
		 *  The request was not found in any of the local queues.
		 */
		return (TCML_NOSUCHREQ);
	}
        if (mapped_uid && req->v1.req.uid != mapped_uid) {
		/*
		 *  This request cannot be affected by the client.
		 */
		return (TCML_NOTREQOWN);
	}
        if (state == RQS_RUNNING) {
                return (TCML_REQRUNNING);
        }

        /*
         *  Make the requested modification.
         */
        pack6name (path, Nqs_control,
                  (int) (orig_seqno % MAX_CTRLSUBDIRS), (char *) 0,
                  (long) orig_seqno, 5, (long) orig_mid, 6, 0, 0);
        if ((cfd = open( path, O_RDWR )) < 0) {
                printf("E$nqs_modreq(): Unable to open control file (%s).", path);
                printf("  Errno = %d\n", errno);
                fflush(stdout);
                return (TCML_UNAFAILURE);
        }
        if (readreq(cfd, &rawreq) < 0) {
                close(cfd);
                printf("E$nqs_modreq(): Read of request header failed.");
                printf("  Errno = %d\n", errno);
                return (TCML_UNAFAILURE);
        }

        /*
         *  Set the new limits.
         */
        rawreq.v.bat.infinite &= ~modtype;
        rawreq.v.bat.explicit |= modtype;
        if (inf) {                              /* if infinite flag set */
                rawreq.v.bat.infinite |= modtype;
        } else {
                switch (modtype) {
/* added cases for PPDATA and PPCPUT. Bruno Wolff 12/21/93 */
                case LIM_PPDATA:
                        rawreq.v.bat.ppdatasize.max_quota = param1;
                        rawreq.v.bat.ppdatasize.warn_quota = param1;
                        rawreq.v.bat.ppdatasize.max_units = param2;
                        rawreq.v.bat.ppdatasize.warn_units = param2;
                      break;
                case LIM_PPCPUT:
                        rawreq.v.bat.ppcputime.max_seconds = param1;
                        rawreq.v.bat.ppcputime.warn_seconds = param1;
                        rawreq.v.bat.ppcputime.max_ms = param2;
                        rawreq.v.bat.ppcputime.warn_ms = param2;
                      break;
                case LIM_PRMEM:
                        rawreq.v.bat.prmemsize.max_quota = param1;
                        rawreq.v.bat.prmemsize.warn_quota = param1;
                        rawreq.v.bat.prmemsize.max_units = param2;
                        rawreq.v.bat.prmemsize.warn_units = param2;
                        break;
                case LIM_PPNICE:
                        lim = rawreq.v.bat.ppnice;      /* get previous value */
                        rawreq.v.bat.ppnice = param1;
                        break;
                case LIM_PRCPUT:
                        rawreq.v.bat.prcputime.max_seconds = param1;
                        rawreq.v.bat.prcputime.warn_seconds = param1;
                        rawreq.v.bat.prcputime.max_ms = param2;
                        rawreq.v.bat.prcputime.warn_ms = param2;
                        break;
                default:
                        close(cfd);
                        return (TCML_NOSUCHQUO);
                }
        }
        if (writereq(cfd, &rawreq) < 0) {
                printf("E$nqs_modreq(): Write of request header failed.");
                printf("  Errno = %d\n", errno);
                close(cfd);
                return (TCML_UNAFAILURE);
        }
        close(cfd);

        return (TCML_COMPLETE);
}
