/*
 *	Network Queueing System (NQS)
 *  This version of NQS is Copyright (C) 1992  John Roman
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 1, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
/*++ fetchgrdb.c - Network Queueing System
 *
 * $Source: /usr2/jrroma/nqs/nqs-3.30/lib/RCS/fetchgrdb.c,v $
 *
 * DESCRIPTION:
 *
 *	This module contains three functions:
 *
 *		fetchgrgid(),
 *		fetchgrnam(), and
 *		closegrdb()
 *
 *	that respectively return the group structure for the specified
 *	local group-id, return the group structure for the specified
 *	local group-name, and close the group-database opened by fetchgrgid()
 *	or fetchgrnam().
 *
 *	This module is provided as an "onion-skin layer" residing above
 *	the locally implemented group-database access functions.
 *
 *
 *	Author:
 *	-------
 *	Brent A. Kingsbury, Sterling Software Incorporated.
 *	January 20, 1986.
 *
 *
 * STANDARDS VIOLATIONS:
 *   None.
 *
 * REVISION HISTORY: ($Revision: 1.3 $ $Date: 1992/12/22 15:45:58 $ $State: Exp $)
 * $Log: fetchgrdb.c,v $
 * Revision 1.3  1992/12/22  15:45:58  jrroma
 * Version 3.30
 *
 * Revision 1.2  92/06/18  13:23:43  jrroma
 * Added gnu header
 * 
 * Revision 1.1  92/06/18  10:17:34  jrroma
 * Initial revision
 * 
 *
 */

#include "nqs.h"
#include <grp.h>			/* Password file structures */
#include <string.h>

/*** fetchgrgid
 *
 *
 *	struct group *fetchgrgid():
 *	Return struct group structure for the specified local group-id.
 *
 *	NOTE:	This function does not open and close the
 *		group-database on successive invocations.
 *		It is up to the caller to invoke closegrdb()
 *		when access is no longer required to the
 *		group-database.
 */
struct group *fetchgrgid (int gid)
{
	register struct group *p;

	setgrent();			/* Seek to start of group file */
	while ((p = getgrent()) != (struct group *) 0 && p->gr_gid != gid)
		;
	return (p);
}


/*** fetchgrnam
 *
 *
 *	struct group *fetchgrnam():
 *	Return struct group structure for the specified group-name.
 *
 *	NOTE:	This function does not open and close the
 *		group-database on successive invocations.
 *		It is up to the caller to invoke closegrdb()
 *		when access is no longer required to the
 *		group-database.
 */
struct group *fetchgrnam (char *name)
{
	register struct group *p;

	setgrent();			/* Seek to start of group file */
	while ((p = getgrent()) && strcmp (p->gr_name, name))
		;
	return(p);
}


/*** closegrdb
 *
 *
 *	void closegrdb():
 *
 *	Close the local group-database opened by fetchgrgid(), or
 *	by fetchgrnam().
 */
void closegrdb()
{
	endgrent();			/* Close the local group file */
}
