/*
	JoSH's implementation of identd (RFC 1413)

	Inspired by qpliu's identd.c for Linux, v0.1, which didn't work
	on my system, so I decided to write my own.

	JoSH Lehan
	jlehan@galaxy.csc.calpoly.edu
	
	v1.0  8/11/95
	Distribution/copyright: I place this program under the
	                        GNU Public License (v2 or later)
*/

#define PROGNAME "jidentd"
#define VERSION "1.0"
#define EMAIL "jlehan@galaxy.csc.calpoly.edu"

#include <sys/types.h>
#include <sys/socket.h>
#include <sys/time.h>
#include <sys/wait.h>
#include <netinet/in.h>
#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>
#include <string.h>
#include <unistd.h>
#include <syslog.h>
#include <pwd.h>
#include <signal.h>

#define LIARFILE "/etc/jidentd.liar"
#define HIDEFILE "/etc/jidentd.hidden"
#define TCPFILE "/proc/net/tcp"
#define IDENTPORT 113
#define BUFSIZE 1024
#define IPSIZE 17			/* for IP addresses, e.g. "127.0.0.1" */
#define NAMESIZE 256		/* for usernames, should be 9? */

#define NO 0
#define YES 1

typedef unsigned char Flag;		/* variable types */
typedef unsigned char Byte;
typedef unsigned short Word;
typedef unsigned long Quad;

int SocketFD, ConnectFD;
struct sockaddr_in Sock, Conn;
Quad LocalAddr, RemoteAddr;
Word LocalPort, RemotePort;
char Buf[BUFSIZE];
char Incoming[BUFSIZE];
char LocalName[IPSIZE], RemoteName[IPSIZE];
Word PortHere, PortThere;
uid_t UserID;	/* assume short.  If ever enlarged, rewrite scanf! */
char UserName[NAMESIZE], LiedName[NAMESIZE];

char InputBuf[BUFSIZE];		/* this should be static, or better */
int InputRemain;				/* yet, a CLASS in C++! */
char *InputPtr;

int Argc;
char **Argv;

/* default options */
Word Timeout = 0;			/* seconds before dropping connection */
Word IdentPort = IDENTPORT;	/* the port this server uses */
Word DebugLvl = 0;		/* debugging level, 0-10 */
Flag SingleQuery = NO;	/* disable multiple queries per connection? */
Flag QuitAfter = NO;		/* quit program after connection ends? */
Flag AlwaysLie = NO;		/* always avoid using real name? */
Flag Verbose = NO;		/* write detailed info to syslog? */
Flag Truthful = NO;		/* disable usage of liarfile/hidefile? */
char Hidefile[BUFSIZE];	/* file names */
char Liarfile[BUFSIZE];
char TCPfile[BUFSIZE];

void End(int RetCode)
/* To be consistent, have the program land here every time it quits */
{
	closelog();
	exit(RetCode);
}

Word StrToNum(char *Str, Word *Num)
/* Converts a string to a Word, returns 0=good 1=error */
{
	long Val;
	char *ErrPtr;
		
	Val = strtol(Str, &ErrPtr, 10);
	if (*Str == '\0' || *ErrPtr != '\0')
		return 1;
	if (Val < 0 || Val > 65535)
		return 1;
	*Num = (Word)Val;
	return 0;
}

void Dprintf(int DebugMin, char *format, ...)
/* wrapper for printf() -- prints only if DebugLvl is high enough */
{
	va_list args;
	
	va_start(args, format);
	if (DebugLvl >= DebugMin)
		vfprintf(stderr, format, args);
	va_end(args);
}

void Dsyslog(int Priority, char *format, ...)
/* wrapper for syslog() -- always log errors, log other stuff and
	echo to stderr only if Verbose is set */
{
	int Err = errno;
	char Str[BUFSIZE];
	va_list args;

	va_start(args, format);

	/* take care of va_ stuff, syslog can't handle it */
	/* Does NOT check for overflow! */
	vsprintf(Str, format, args);
	if (Priority == LOG_ERR)
	{
		/* add error message, doesn't check for overflow */
		strcat(Str, " ");
		strcat(Str, strerror(Err));
	}
		
	if (Priority == LOG_ERR || (Verbose == YES && Priority == LOG_INFO)
			|| DebugLvl > 0)
		syslog(Priority, "%s", Str);
	if (DebugLvl > 0)
		fprintf(stderr, "=LOG %d= %s\n", Priority, Str);
		
	va_end(args);
}

void Die(char *Why)
/* Aborts program, with error message */
{
	Dsyslog(LOG_ERR, Why);
	Dprintf(1, "Program failed!  Aborting!\n");
	End(1);
}

void AddrToStr(Quad Addr, char *Str)
/* Converts a 32-bit IP address to a string (e.g. "129.65.100.169") */
{
	Word B1, B2, B3, B4;		/* to make it easy for sprintf */

	/* assuming we're in MSB-first (network) byte order!! */
	B1 = Addr >> 24;
	B2 = (Addr >> 16) & 0xFF;
	B3 = (Addr >> 8) & 0xFF;
	B4 = Addr & 0xFF;
	
	/* does not check for overflow of Str */
	sprintf(Str, "%hu.%hu.%hu.%hu", B1, B2, B3, B4); 
}

void GetLocAddr(int FD)
/* Fills in LocalAddr/Port */
{
	struct sockaddr_in SockDummy;
	int Size = sizeof(struct sockaddr_in);
	
	if (getsockname(FD, (struct sockaddr *)&SockDummy, &Size) != 0)
		Die("Couldn't get local hostname:");
	LocalAddr = ntohl(SockDummy.sin_addr.s_addr);
	LocalPort = ntohs(SockDummy.sin_port);
	AddrToStr(LocalAddr, LocalName);
	Dprintf(5, "Local addr is %s, port=%hu\n", LocalName, LocalPort);
}

void GetRemAddr(int FD)
/* same as GetLocAddr except for other side, fills in RemoteAddr/Port */
{
	struct sockaddr_in SockDummy;
	int Size = sizeof(struct sockaddr_in);

	if (getpeername(FD, (struct sockaddr *)&SockDummy, &Size) != 0)
		Die("Couldn't get remote hostname:");
	RemoteAddr = ntohl(SockDummy.sin_addr.s_addr);
	RemotePort = ntohs(SockDummy.sin_port);
	AddrToStr(RemoteAddr, RemoteName);
	Dprintf(5, "Remote addr is %s, port=%hu\n", RemoteName, RemotePort);	
}

int WaitFor(int FD, Word Secs)
/* using select(), wait for the file descriptor FD to have something
	readable.  If Secs=0, wait forever (unlike select())
	Returns 0=FD now readable  1=timeout occured */
{
	fd_set FDMask;
	struct timeval Time;
	int Result;

	Dprintf(2, "Waiting on file #%d for %hd seconds...\n", FD, Secs);
	
	do
	{
		FD_ZERO(&FDMask);
		FD_SET(FD, &FDMask);
		Time.tv_usec = 0;
		if (Secs == 0)
			Time.tv_sec = 2000000000;	/* forever */
		else
			Time.tv_sec = (long)Secs;

		Result = select(FD + 1, &FDMask, NULL, NULL, &Time);
		Dprintf(3, "Done with select, got %d %d\n", Result, errno);

		/* restart call if interrupted */
	}	while (Result < 0 && errno == EINTR);
		
	if (Result < 0)
		Die("Problem while waiting for input:");	
	if (Result == 0)
		return 1;
	return 0;
}

void CreateSocket(void)
/* Sets up SocketFD for incoming connections */
{
	int NumListens;

	/* create an Internet socket */
	SocketFD = socket(AF_INET, SOCK_STREAM, 0);
	if (SocketFD < 0)
		Die("Couldn't create socket:");
	Dprintf(3, "Socket created, fd=%d\n", SocketFD);
	Dprintf(8, "We're gonna try for port %hu\n", IdentPort);

	/* bind it to the IDENT port */
	Sock.sin_family = AF_INET;
	Sock.sin_addr.s_addr = INADDR_ANY;
	Sock.sin_port = htons(IdentPort);	
	if (bind(SocketFD, (struct sockaddr *)&Sock, sizeof(Sock)) < 0)
		Die("Couldn't bind socket to port:");

	/* this is just for debugging */
	GetLocAddr(SocketFD);

	/* set it up for connections */
	if (QuitAfter == YES)
		NumListens = 1;
	else
		NumListens = 5;	/* let multiple connections stack up */
	if (listen(SocketFD, NumListens) < 0)
		Die("Socket is deaf:");
	Dprintf(1, "Now listening for connections...\n");
}

void ClearLine(void)
/* Clears input buffers */
{
	bzero(InputBuf, BUFSIZE);
	InputPtr = InputBuf;
	InputRemain = BUFSIZE - 1;		/* allow for terminating NULL char */
}

void GrabSocket(void)
/* Sets up a connection on a socket */
{
	int Size = sizeof(struct sockaddr_in);

	ConnectFD = accept(SocketFD, (struct sockaddr *)&Conn, &Size);
	if (ConnectFD < 0)
		Die("Problem accepting connection:");

	GetLocAddr(ConnectFD);
	GetRemAddr(ConnectFD);

	Dsyslog(LOG_INFO, "Incoming connection from %s", RemoteName);
	Dsyslog(LOG_DEBUG, "Us: %s %hu <--> Them: %s %hu", LocalName,
			LocalPort, RemoteName, RemotePort);

	/* clear input buffers */
	ClearLine();
}

int FetchLine(int FD, char *Line)
/* Reads and assembles a Line of input from a file descriptor FD */
/* Removes the trailing CR/LF from it */
/* Assumes ClearLine()'s already been called */
/* Returns: 0=got line  1=got line, but this is the FINAL line
            2=got line, but buffer overflowed  -1=error
            -2=timeout occured */
{
	int Count;
	char *OrigPtr = InputPtr;
	char *Ptr;

	Dprintf(9, "Starting this mess, %d remaining\n", InputRemain);
	while(InputRemain > 0)
	{
		if (WaitFor(ConnectFD, Timeout) != 0)
			return -2;	/* timeout */			
		
		Count = read(ConnectFD, InputPtr, InputRemain);
		if (Count <= 0)
		{	
			Dprintf(9, "Ending input, count=%d\n", Count);
			memcpy(Line, InputBuf, InputPtr - InputBuf);
			Line[InputPtr - InputBuf] = '\0';
			memmove(InputBuf, InputPtr, InputPtr - InputBuf);
			InputRemain += (InputPtr - InputBuf);
			InputPtr = InputBuf;
			if (Count < 0)
				return -1;
			else
				return 1;
		}
		
		InputPtr += Count;
		InputRemain -= Count;
		Dprintf(9, "Read in %d bytes, %d remaining\n", Count,
				InputRemain);

		for (Ptr = OrigPtr; Ptr < InputPtr; Ptr ++)
		{
			if (*Ptr == '\n')			/* LF, weed out and return */
			{
				Dprintf(10, "Hit LF, now returning\n");
				memcpy(Line, InputBuf, Ptr - InputBuf);
				Line[Ptr - InputBuf] = '\0';
				memmove(InputBuf, Ptr + 1, (Ptr + 1) - InputBuf);
				InputPtr -= ((Ptr + 1) - InputBuf);
				InputRemain += ((Ptr + 1) - InputBuf); 
				return 0;
			}
			else if (*Ptr == '\r')	/* CR, weed this out */
			{
				Dprintf(10, "Hit CR, weeding it out\n");
				memmove(Ptr, Ptr + 1, InputPtr - Ptr);
				Ptr --;
				InputPtr --;
				InputRemain ++;
			}
			else
			{
				Dprintf(10, "Processed '%c' (%d)\n", *Ptr, (int)(*Ptr));
			}
		}	
	}

	Dprintf(9, "Input buffer overflowed, returning what we got\n");
	memcpy(Line, InputBuf, BUFSIZE);
	Line[BUFSIZE - 1] = '\0';			/* insurance */
	InputPtr = InputBuf;
	InputRemain = BUFSIZE - 1;	
	return 2;	/* buffer overflow */
}

int ParsePorts(char *In)
/* Break down input line, fill in PortHere and PortThere */
/* Returns 0=good -1=INVALID-PORT */
{
	int BigHere, BigThere;
	int Count, NumChars;
	char *Ptr = In;
	
	PortHere = PortThere = 0;	/* something to use in error msg's */
	
	Count = sscanf(Ptr, "%d%n", &BigHere, &NumChars);
	if (Count != 1)
	{
		Dprintf(5, "Missed first number (%d)\n", Count);
		return 1;
	}
	
	/* skip over whitespace, commas, etc. */
	Ptr += NumChars;
	while(*Ptr != '\0')
	{
		if (*Ptr >= '0' && *Ptr <= '9')
			break;
		Ptr ++;
	}
		
	Count = sscanf(Ptr, "%d", &BigThere);
	if (Count != 1)
	{
		Dprintf(5, "Missed second number (%d)\n", Count);
		return 1;
	}

	if (BigHere < 1 || BigHere > 65535)
	{
		Dprintf(5, "First number out of range: %d\n", BigHere);
		return 1;
	}
	else
		PortHere = (Word)BigHere;
		
	if (BigThere < 1 || BigThere > 65535)
	{
		Dprintf(5, "Second number out of range: %d\n", BigHere);
		return 1;
	}
	else
		PortThere = (Word)BigThere;

	return 0;
}

void SendError(char *Text)
/* Sends the error message in proper format */
{
	Dsyslog(LOG_DEBUG, "Sending error message: %s", Text);

	/* Neither of these sprintfs check for overflow of Buf! */
	sprintf(Buf, "%hu , %hu : ERROR : %s\r\n", PortHere, PortThere,
			Text);
	if (write(ConnectFD, Buf, strlen(Buf)) != strlen(Buf))
		Die("Couldn't write error message through the socket:");
}

void SendGood(char *Name)
/* Sends a username in the proper format */
{
	Dsyslog(LOG_DEBUG, "Sending name of == %s ==", Name);

	sprintf(Buf, "%hu , %hu : USERID : UNIX : %s\r\n", PortHere,
			PortThere, Name);
	if (write(ConnectFD, Buf, strlen(Buf)) != strlen(Buf))
		Die("Couldn't write username message through the socket:");
}

int LookupPort(void)
/* Looks up PortHere and PortThere in the system TCP file */
/* Assigns UserID */
/* Returns 0=success, -1=no user found, -2=error in file */
{
	FILE *File;
	Quad LocIP, RemIP;
	Word LocPort, RemPort;
	int Count;
		
	File = fopen(TCPfile, "r");
	if (File == NULL)
		return -2;	/* couldn't read file */

	/* skip past first line of file */
	if (fgets(Buf, BUFSIZE, File) == NULL)
		return -2;

	Dprintf(6, "We're looking for these IP's:  %8lx  %8lx\n",
			LocalAddr, RemoteAddr);
			
	while(fgets(Buf, BUFSIZE, File) != NULL)
	{
		Dprintf(7, "Checking: \"%s\"\n", Buf);

		Count = sscanf(Buf, "%*d: %lx:%hx %lx:%hx %*x %*x:%*x %*x:%*x %*x %hd",
				&LocIP, &LocPort, &RemIP, &RemPort, &UserID);		
		if (Count != 5)
		{
			Dprintf(6, "Only %d fields parsed!\n", Count);
			continue;
		}
		
		/* IP addresses are stored backwards, for some reason */
		LocIP = htonl(LocIP);
		RemIP = htonl(RemIP);

		Dprintf(7, "%8lx %5hd   %8lx %5hd   %5hd\n", LocIP, LocPort,
				RemIP, RemPort, UserID);

		if ((LocIP == LocalAddr) && (RemIP == RemoteAddr)
				&& (LocPort == PortHere) && (RemPort == PortThere))
		{
			Dprintf(4, "Dirty pool, old man, I like it!  %d\n", UserID);
			return 0;
		}
	}

	Dprintf(4, "You MISSED!\n");	
	return -1;	/* no user found */
}

int LookupName(void)
/* Looks up UserID in the system password file */
/* Fills in UserName */
/* Returns 0=success, -1=no username found or error reading */
{
	struct passwd *Pass;

	Pass = getpwuid(UserID);
	if (Pass == NULL)
		return -1;

	strncpy(UserName, Pass -> pw_name, NAMESIZE - 1);
	return 0;
}

int Lie(void)
/* If liarfile is valid, change LiedName to it, otherwise
	leave unchanged */
/* Returns 0=lied -1=couldn't lie */
{
	/* "I don't know why I feel the need to lie,
		 And cause you so much pain...
		 Maybe it's something inside
		 Maybe it's something I can't explain
		 'Cause all I do
		 Is mess you up and lie to you */
		 
	FILE *File;
	
	File = fopen(Liarfile, "r");
	if ((File == NULL) || (fgets(Buf, BUFSIZE, File) == NULL))
	{
		return -1;
	}

	/* kludge around fgets adding an extra newline */
	if (Buf[strlen(Buf) - 1] == '\n')
		Buf[strlen(Buf) - 1] = '\0';

	strncpy(LiedName, Buf, NAMESIZE - 1);
	return 0;
}

int HideName(void)
/* Look up UserName in hidefile, act like Lie() and change LiedName
	or signal HIDDEN-USER if there's only one field
	Returns 0=changed name -1=name not found -2=err 1=hide this name! */
{
	FILE *File;
	int Count;
	char SrcName[NAMESIZE], DstName[NAMESIZE];
				
	File = fopen(Hidefile, "r");
	if (File == NULL)
		return -2;			/* error reading file */

	while(fgets(Buf, BUFSIZE, File) != NULL)
	{
		/* Does not check for overflow!! */
		Count = sscanf(Buf, "%s %s", SrcName, DstName);
		if ((Count > 0) && (strncmp(SrcName, UserName, NAMESIZE) == 0))
		{
			if (Count == 1)
				return 1;	/* use HIDDEN-USER */
			strncpy(LiedName, DstName, NAMESIZE - 1);
			return 0;
		}
	}
	
	return -1;	/* name not found */
}
	
void ProcessIt(char *In)
/* Do what the program was intended to do */
{
	int Status;
	
	if (ParsePorts(In) != 0)
	{
		Dsyslog(LOG_DEBUG, "Couldn't understand what they sent: \"%s\"", In);
		SendError("INVALID-PORT");
		return;
	}
	Dsyslog(LOG_INFO, "They want to know: %hu (our port) <--> %hu (theirs)",
			PortHere, PortThere);		

	Status = LookupPort();
	if (Status < 0)
	{
		if (Status == -2)
		{
			Dsyslog(LOG_ERR, "Couldn't read %s:", TCPfile);
			SendError("UNKNOWN-ERROR");
		}
		else
		{
			Dsyslog(LOG_INFO, "Those ports are unused, though!");
			SendError("NO-USER");
		}
		return;
	}

	if (Truthful == NO || AlwaysLie == YES)
	{
		if (Lie() != 0)
			Dsyslog(LOG_DEBUG, "Unable to read liarfile, ignoring it...");
		else
		{		
			Dsyslog(LOG_INFO, "Lying!  Using liarfile, user %d is now: %s\n",
					UserID, LiedName);
			SendGood(LiedName);
			return;
		}
	}
			
	if (LookupName() != 0)
	{
		Dsyslog(LOG_ERR, "Couldn't find name for userid %d:", UserID);
		SendError("UNKNOWN-ERROR");
		return;
	}

	if (Truthful == NO || AlwaysLie == YES)
		Status = HideName();
	else
	{
		/* If we get here, Truthful == YES && AlwaysLie == NO */
		Dsyslog(LOG_DEBUG, "Ignoring liarfile and hidefile.");
		SendGood(UserName);
		return;
	}

	/* the following is very hairy; it's even worse as a switch/case
		statement, though! */
	if (Status < 0)
	{
		if (Status == -2)		/* -2 = error */
			Dsyslog(LOG_DEBUG, "Unable to read hidefile, ignoring it...");
		else						/* -1 = name not found */
			Dsyslog(LOG_DEBUG, "Name not found in hidefile: %s", UserName);

		if (AlwaysLie == YES)
		{
			Dsyslog(LOG_INFO, "Sending error message instead of real name (%s).",
					UserName);
			SendError("UNKNOWN-ERROR");
		}
		else
		{
			Dsyslog(LOG_INFO, "Sending real name for: %s", UserName);
			SendGood(UserName);
		}
	}			
	else
	{
		if (Status == 0)		/* 0 = name found and altered */
		{
			Dsyslog(LOG_INFO, "Lying!  Altering name %s to: %s",
					UserName, LiedName);
			SendGood(LiedName);
		}
		else						/* 1 = name found, hide it */
		{
			Dsyslog(LOG_INFO, "Hiding the name %s as HIDDEN-USER!",
					UserName);
			SendError("HIDDEN-USER");  
		}
	}
}

void DoSocket(void)
/* Now act on that connection! */
{
	int Status;
	
	Dprintf(1, "Now reading from socket!!\n");	

	Status = FetchLine(ConnectFD, Incoming);
	while (Status >= 0)
	{
		ProcessIt(Incoming);
		
		/* hang up if we're done, or we're in single-query mode */
		if (Status == 1 || SingleQuery == YES)
			break;
		Status = FetchLine(ConnectFD, Incoming);
	}
	
	if (Status < 0)
	{
		if (Status == -2)
			Dsyslog(LOG_INFO, "Timed out while waiting; closing connection.");
		else
			Die("Problem reading from socket:");
	}
	
	Dprintf(1, "Done reading from socket...\n");
	close(ConnectFD);
}

void Usage(void)
{
/* strange syntax here just to make it easier to read */
#define FS fprintf(stderr,
FS "Usage: %s [options]\n", Argv[0]);
FS "-f <file> = use <file> for TCP lookups instead of %s\n", TCPFILE);
FS "-i <file> = use <file> as hidefile instead of %s\n", HIDEFILE);
FS "-l <file> = use <file> as liarfile instead of %s\n", LIARFILE);
FS "-t <secs> = set idle timeout for connections (and entire program if -q)\n");
FS "-d <level> = enable debugging output to stderr, verbosity 0-10\n");
FS "-p <port> = connect server to <port> instead of %d\n", IDENTPORT);
FS "-s = close each connection after only a single query (disable multiple)\n");
FS "-q = accept only a single connection, then quit (disable forking)\n");
FS "-v = increase verbosity when logging to syslog\n");
FS "-n = disable liarfile and hidefile; behave like conventional identd\n");
FS "-a = always lie; never send real userid (overrides -n)\n");
FS "-h = show this help\n");
FS "%s %s Copyright (C) 1995 Joshua Lehan    (%s)\n",
		PROGNAME, VERSION, EMAIL);
FS "This program is free software, and may be modified and/or redistributed\n");
FS "under the terms of the GNU General Public License, version 2 or later.\n");
#undef FS
}

void DoArgs(void)
/* Parses options and sets global variables */
{
	extern char *optarg;
	extern int optind;
	extern int opterr;
	int Option;
		
	opterr = 0;
	optind = 1;

	/* set up default filenames */
	strcpy(Hidefile, HIDEFILE);
	strcpy(Liarfile, LIARFILE);
	strcpy(TCPfile, TCPFILE);

	while ((Option = getopt(Argc, Argv, "sqavnht:p:i:l:f:d:")) != EOF)
	{
		Dprintf(2, "Now processing '%c' (%s)\n", (char)Option,
				optarg);
		switch((char)Option)
		{
			case 's': case 'S':
				SingleQuery = YES;
				break;
			case 'q': case 'Q':
				QuitAfter = YES;
				break;
			case 'a': case 'A':
				AlwaysLie = YES;
				break;
			case 'v': case 'V':
				Verbose = YES;
				break;
			case 'n': case 'N':
				Truthful = YES;
				break;
			case 'h': case 'H':
				Usage();
				End(0);
				break;	/* NOTREACHED */
			case 'i': case 'I':
				strncpy(Hidefile, optarg, BUFSIZE - 1);
				break;
			case 'l': case 'L':
				strncpy(Liarfile, optarg, BUFSIZE - 1);
				break;
			case 'f': case 'F':
				strncpy(TCPfile, optarg, BUFSIZE - 1);
				break;
			case 't': case 'T':
				if (StrToNum(optarg, &Timeout) != 0)
					Die("Timeout value must be in range 0-65535\n");
				break;
			case 'd': case 'D':
				if (StrToNum(optarg, &DebugLvl) != 0)
					Die("Debug level must be in range 0-65535\n");
				break;
			case 'p': case 'P':
				if ((StrToNum(optarg, &IdentPort) != 0) ||
						IdentPort < 1)
					Die("Port must be in range 1-65535\n");
				break;
			default:
				Dsyslog(LOG_INFO, "Unrecognized option '%c'",
						(char)Option);
				Usage();
				End(1);
		}
	}

	Dprintf(2, "optind now at %d\n", optind);
}

void MainLoop(void)
{
	pid_t Forked;
		
	if (QuitAfter == YES)
	{
		if (Timeout != 0)
		{
			/* enforce timeout */
			if (WaitFor(SocketFD, Timeout) != 0)
			{
				Dsyslog(LOG_INFO, "Timed out while waiting for connection...");
				return;
			}			
		}
		
		/* block if necessary, we aren't forking */	
		GrabSocket();	/* does accept() */
		DoSocket();		/* processes and closes socket */
	}
	else	
	{
		while(1)		/* infinite loop */
		{
			/* don't use timeout here, but we want to select anyway */
			if (WaitFor(SocketFD, 0) != 0)	/* wait forever */
				Die("Something happened while waiting!");
			
			/* fork off a child to handle the connection */		
			Dprintf(1, "Tapping 2 red mana...\n");		
			Forked = fork();
			if (Forked == -1)
				Die("Counterspell!  Fork Failed!");
			else if (Forked == 0)
			{
				/* child */
				Dprintf(2, "Child active!\n");
				GrabSocket();
				DoSocket();
				End(0);
			}
			else
			{
				/* parent */
				Dprintf(2, "Parent active, child's PID is %d\n", Forked);
				sleep(5);
				/* RACE CONDITION here??  We want the child to have
					picked up the connection by now, so we don't try
					to fork the same connection again... */
			}	
		}
	}
}

/* forward decl. for ChildHandler */
void DoSignals(void);

void ChildHandler(int Sig)
{
	pid_t Pid;
	int Status;
	
	Dprintf(2, "In child handler... now what?\n");
	Pid = waitpid(-1, &Status, WNOHANG);
	if (Pid == -1)
		Die("Problem waiting for children:");
	else if (Pid != 0)
		Dprintf(1, "Child process %d did something...\n", Pid);

	/* reset handlers */
	DoSignals();
}

void DoSignals(void)
{
	/* the return value of signal() is funky, so ignore it for now,
		it shouldn't error anyway */	
	signal(SIGCHLD, ChildHandler);
	/* don't do any other signals, if they go off, not much we can
		do anyway */
}

int main(int argc, char **argv)
{
	Argc = argc;
	Argv = argv;

	DoSignals();	
	openlog(PROGNAME, LOG_PID | LOG_CONS, LOG_AUTHPRIV);

	DoArgs();
	Dprintf(10, "Starting program\n");
		
	CreateSocket();
	MainLoop();
			
	Dprintf(1, "Program ending!\n");
	End(0);

	/* NOTREACHED */
	return 1;
}
