/*
 *
 *	new.c
 *
 *	Create new views and edit defaults.
 *
 *	HNMS User Interface
 *	HNMS 2.0
 *
 *	February 1994
 *
 *	Leslie Schlecht
 *	Computer Sciences Corporation
 *	Numerical Aerodynamic Simulation Systems Division
 *	NASA Ames Research Center
 *
 *	Copyright (c) 1994 Leslie Schlecht
 *
 *	This program is free software; you can redistribute it and/or modify
 *	it under the terms of the GNU General Public License as published by
 *	the Free Software Foundation; either version 1, or (at your option)
 *	any later version.
 *
 *	This program is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *	GNU General Public License for more details.
 *
 *	You should have received a copy of the GNU General Public License
 *	along with this program; if not, write to the Free Software
 *	Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

#include	<stdio.h>
#include	<string.h>

#include	<X11/StringDefs.h>
#include	<X11/Intrinsic.h>
#include	<X11/Shell.h>
#include	<X11/Core.h>
#include	<Xm/Xm.h>
#include	<Xm/Label.h>
#include	<Xm/Text.h>
#include	<Xm/Form.h>
#include	<Xm/Frame.h>
#include	<Xm/List.h>
#include	<Xm/PushB.h>
#include	<Xm/RowColumn.h>
#include	<Xm/ScrolledW.h>


#include	"defines.h"
#include	"externs.h"
#include	"object.h"
#include	"viewpanel.h"
#include	"xsupport.h"
#include	"view.h"
#include	"new.h"

static Widget		left_form;
static Widget		right_form;

static Widget		class_frame, class_menu, class_button[NUM_OBJCLASSES];
static Widget		interval_frame, interval_text;
static Widget		object_frame;
static Widget		objlist_frame, addobject, removeobject, object_list;
static Widget		objlist_label;
static Widget		objtext_frame, obj_text_label, object_text;
static Widget		background_frame, background_menu;
static Widget		variable_frame;
static Widget		varlist_frame, addvariable, removevariable,
			variable_list;
static Widget		vartext_frame, var_text_label, variable_text;
static Widget		miblist_frame, mib_list;

static VIEWPANEL	*newpanel;
static VIEW		*newview, *oldview;
static int		*view_class_help[NUM_VIEWCLASSES];

static int		view_class_count[NUM_VIEWCLASSES];
static int		current_objclass = 0;
static char	*view_class_names[NUM_VIEWCLASSES] = {
		"",
		"wanstatus",
		"customstatus",
		"sitestatus",
		"dump",
		"data",
		"default",
		"statuslog",
		};
static char	*view_class_titles[NUM_VIEWCLASSES] = {
		"",
		"WAN Status",
		"Custom Status",
		"Site Status",
		"Object Dump",
		"Object Data",
		"Object Default",
		"Object Reach Status Log",
		};


/*
 *	Add to the edit panel object list.
 */
static
void
AddToObjList(widg, a1, a2)
Widget	widg;
caddr_t	a1, a2;
	{
	char		*t;
	XmString	xs;

	t = XmTextGetString(object_text);
	if (!t) return;
	if (strlen(t)) {
		xs = X_STR(t);
		if (!XmListItemExists(object_list, xs))
			XmListAddItem(object_list, xs, 0);
		XmStringFree(xs);
		}
	XtFree(t);
	}


/*
 *	Add to the edit panel variable list.
 */
static
void
AddToVarList(widg, a1, a2)
Widget	widg;
caddr_t	a1, a2;
	{
	char		*t;
	XmString	xs;

	t = XmTextGetString(variable_text);
	if (!t) return;
	if (strlen(t)) {
		xs = X_STR(t);
		if (!XmListItemExists(variable_list, xs))
			XmListAddItem(variable_list, xs, 0);
		XmStringFree(xs);
		}
	XtFree(t);
	}


/*
 *	Apply the information from the edit panel to the view.
 */
void
ApplyNewPanel(widg, a1, a2)
Widget	widg;
caddr_t	a1, a2;
	{
	char	*s, buf[BUFSIZ];
	register	i;
	int	objclass;

	objclass = current_objclass;
	SelectObjectClass(NULL, 0, NULL);
	buf[0] = '\n';
	buf[1] = '\0';
	switch (newview->class) {
	case CUSTOMSTATUS:
		AddViewButton(newview->panel, newview->panel->name);
		if (newview->objstr[objclass])
			strcpy(buf, newview->objstr[objclass]);
		s = gettoken(buf);
		SetViewPanelTitle(newview->panel, s);
		XtVaSetValues(newview->panel->titletext,
			XmNsensitive, True,
			XmNeditable, True,
			XmNcursorPositionVisible, True,
			NULL);
		oldview = AllocateView();
		break;
	case SITESTATUS:
		AddViewButton(newview->panel, newview->panel->name);
		if (newview->objstr[OBJ_site])
			strcpy(buf, newview->objstr[OBJ_site]);
		s = gettoken(buf);
		SetViewPanelTitle(newview->panel, s);
		oldview = AllocateView();
		break;
	case WANSTATUS:
		AddViewButton(newview->panel, newview->panel->name);
		if (newview->objstr[OBJ_processor])
			strcpy(buf, newview->objstr[OBJ_processor]);
		s = gettoken(buf);
		SetViewPanelTitle(newview->panel, s);
		XtVaSetValues(newview->panel->titletext,
			XmNsensitive, True,
			XmNeditable, True,
			XmNcursorPositionVisible, True,
			NULL);
		newview->background = GetCurrentBackground(&s);
		oldview = AllocateView();
		break;
	case OBJECTDATA:
		AddViewButton(newview->panel, newview->panel->name);
		if (newview->objstr[objclass])
			strcpy(buf, newview->objstr[objclass]);
		s = gettoken(buf);
		SetViewPanelTitle(newview->panel, s);
		for (i=1; i<NUM_OBJCLASSES; i++)
			if (i != objclass) {
				if (newview->objstr[i]) {
					free(newview->objstr[i]);
					newview->objstr[i] = NULL;
					}
				if (newview->varstr[i]) {
					free(newview->varstr[i]);
					newview->varstr[i] = NULL;
					}
				}
		oldview = AllocateView();
		break;
	case DEFAULT:
		for (i=0; i<NUM_OBJCLASSES; i++) {
			free(oldview->objstr[i]);
			oldview->objstr[i] = NULL;
			free(oldview->varstr[i]);
			oldview->varstr[i] = NULL;
			}
		break;
	default:;
		}
	s = XmTextGetString(interval_text);
	if (strlen(s)) {
		sscanf(s, "%d", &(newview->interval));
		XtFree(s);
		}
	bcopy(newview, oldview, sizeof(VIEW));
	for (i=0; i<NUM_OBJCLASSES; i++) {
		newview->objstr[i] = NULL;
		newview->varstr[i] = NULL;
		}
	CloseViewPanel(NULL, newpanel, NULL);
	LoadView(oldview);
	AnnounceObjectsToView(oldview);
	OpenViewPanel(NULL, oldview->panel, NULL);
	}


/*
 *	Close the edit panel.
 */
void
CloseNewPanel(panel)
VIEWPANEL	*panel;
	{
	register	i;
	int		viewclass;
	XmString	xs;

	viewclass = newview->class;
	SelectObjectClass(NULL, 0, NULL);
	for (i=0; i<NUM_OBJCLASSES; i++) {
		free(newview->objstr[i]);
		free(newview->varstr[i]);
		}
	free(newview);
	switch (viewclass) {
	case WANSTATUS:
		XtSetSensitive(class_button[OBJ_processor], 0);
		XtSetSensitive(class_button[OBJ_ipaddr], 0);
		XtManageChild(right_form);
		XtVaSetValues(left_form,
			XmNrightPosition, 50,
			NULL);
		break;
	case CUSTOMSTATUS:
		XtSetSensitive(class_button[OBJ_processor], 0);
		XtSetSensitive(class_button[OBJ_network], 0);
		XtSetSensitive(class_button[OBJ_subnet], 0);
		XtSetSensitive(class_button[OBJ_site], 0);
		XtSetSensitive(class_button[OBJ_ipaddr], 0);
		XtManageChild(right_form);
		XtVaSetValues(left_form,
			XmNrightPosition, 50,
			NULL);
		XtManageChild(background_frame);
		XtVaSetValues(object_frame,
			XmNbottomAttachment, XmATTACH_WIDGET,
			XmNbottomWidget, background_frame,
			NULL);
		break;
	case SITESTATUS:
		XtSetSensitive(class_button[OBJ_site], 0);
		XtSetSensitive(class_button[OBJ_ipaddr], 0);
		XtManageChild(right_form);
		XtVaSetValues(left_form,
			XmNrightPosition, 50,
			NULL);
		XtManageChild(background_frame);
		XtVaSetValues(object_frame,
			XmNbottomAttachment, XmATTACH_WIDGET,
			XmNbottomWidget, background_frame,
			NULL);
		break;
	case OBJECTDATA:
		for (i=1; i<NUM_OBJCLASSES; i++)
			XtSetSensitive(class_button[i], 0);
		XtManageChild(background_frame);
		XtVaSetValues(object_frame,
			XmNbottomAttachment, XmATTACH_WIDGET,
			XmNbottomWidget, background_frame,
			NULL);
		XtVaSetValues(objlist_label,
			XmNlabelString, (xs=X_STR("Object Mask")),
			NULL);
		XmStringFree(xs);
		XtManageChild(addobject);
		XtManageChild(removeobject);
		break;
	case DEFAULT:
		for (i=1; i<NUM_OBJCLASSES; i++)
			XtSetSensitive(class_button[i], 0);
		break;
	default:;
		}
	EnableEdit();
	}


/*
 *	Create a new view.
 */
void
CreateNewView(viewclass)
int	viewclass;
	{
	OBJECT		*obj;
	register	i;

	oldview = NULL;
	newview = (VIEW*)myalloc(NULL, 1, sizeof(VIEW));
	InitNewView(newview, NULL, viewclass, 0);
	switch (viewclass) {
	case WANSTATUS:
		newview->objstr[OBJ_processor] = NULL;
		newview->objstr[OBJ_ipaddr] = newstr("*\n");
		OpenNewViewPanel(OBJ_processor);
		break;
	case CUSTOMSTATUS:
		newview->objstr[OBJ_ipaddr] = newstr("*\n");
		OpenNewViewPanel(OBJ_processor);
		break;
	case SITESTATUS:
		newview->objstr[OBJ_ipaddr] = newstr("*\n");
		GetCurrentObject(&obj);
		if (obj) {
			if (obj->class == OBJ_site)
				XmTextSetString(object_text, obj->name);
			}
		OpenNewViewPanel(OBJ_site);
		break;
	case OBJECTDATA:
		GetCurrentObject(&obj);
		if (obj) {
			XmTextSetString(object_text, obj->name);
			OpenNewViewPanel(obj->class);
			}
		else
			OpenNewViewPanel(OBJ_agent);
		break;
	case DEFAULT:
		oldview = AllocateView();
		bcopy(newview, oldview, sizeof(VIEW));
		for (i=1; i<NUM_OBJCLASSES; i++) {
			oldview->objstr[i] = newstr("*\n");
			oldview->announce[i] = 1;
			}
		LoadView(oldview);
		break;
	case STATUSLOG:
		oldview = AllocateView();
		bcopy(newview, oldview, sizeof(VIEW));
		LoadView(oldview);
		break;
	default:;
		}
	}


/*
 *	Create edit panel.
 */
void
CreateNewViewPanel()
	{
	XmString	xs;
	Widget		w, b, f, m, l;
	register	i;

	newpanel = CreateViewPanel("EDIT:", "edit", 0, 0, "edit", 0, OK_PIXMAP,
		0);
	XtAddCallback(newpanel->optionbutton, XmNactivateCallback,
		ApplyNewPanel, NULL);
	newpanel->close = CloseNewPanel;
	newpanel->title = newstr("New View Panel");
	f = XtVaCreateManagedWidget("form",
		xmFormWidgetClass,
		newpanel->viewform,
		XmNtopAttachment, XmATTACH_FORM,
		XmNbottomAttachment, XmATTACH_FORM,
		XmNleftAttachment, XmATTACH_FORM,
		XmNrightAttachment, XmATTACH_FORM,
		XmNrubberPositioning, True,
		NULL);
	left_form = XtVaCreateManagedWidget("form",
		xmFormWidgetClass,
		f,
		XmNtopAttachment, XmATTACH_FORM,
		XmNbottomAttachment, XmATTACH_FORM,
		XmNleftAttachment, XmATTACH_FORM,
		XmNrightAttachment, XmATTACH_POSITION,
		XmNrightPosition, 50,
		NULL);
	right_form = XtVaCreateManagedWidget("form",
		xmFormWidgetClass,
		f,
		XmNtopAttachment, XmATTACH_FORM,
		XmNbottomAttachment, XmATTACH_FORM,
		XmNrightAttachment, XmATTACH_FORM,
		XmNleftAttachment, XmATTACH_POSITION,
		XmNleftPosition, 50,
		NULL);
	class_frame = XtVaCreateManagedWidget("frame",
		xmFrameWidgetClass,
		left_form,
		XmNtopAttachment, XmATTACH_FORM,
		XmNbottomAttachment, XmATTACH_NONE,
		XmNleftAttachment, XmATTACH_FORM,
		XmNrightAttachment, XmATTACH_FORM,
		XmNshadowType, XmSHADOW_OUT,
		XmNmarginWidth, 5,
		XmNmarginHeight, 5,
		NULL);
	interval_frame = XtVaCreateManagedWidget("frame",
		xmFrameWidgetClass,
		left_form,
		XmNtopAttachment, XmATTACH_WIDGET,
		XmNtopWidget, class_frame,
		XmNbottomAttachment, XmATTACH_NONE,
		XmNleftAttachment, XmATTACH_FORM,
		XmNrightAttachment, XmATTACH_FORM,
		XmNshadowType, XmSHADOW_OUT,
		XmNmarginWidth, 5,
		XmNmarginHeight, 5,
		NULL);
	background_frame = XtVaCreateManagedWidget("frame",
		xmFrameWidgetClass,
		left_form,
		XmNtopAttachment, XmATTACH_NONE,
		XmNbottomAttachment, XmATTACH_FORM,
		XmNleftAttachment, XmATTACH_FORM,
		XmNrightAttachment, XmATTACH_FORM,
		XmNshadowType, XmSHADOW_OUT,
		XmNmarginWidth, 5,
		XmNmarginHeight, 5,
		NULL);
	object_frame = XtVaCreateManagedWidget("frame",
		xmFrameWidgetClass,
		left_form,
		XmNtopAttachment, XmATTACH_WIDGET,
		XmNtopWidget, interval_frame,
		XmNbottomAttachment, XmATTACH_WIDGET,
		XmNbottomWidget, background_frame,
		XmNleftAttachment, XmATTACH_FORM,
		XmNrightAttachment, XmATTACH_FORM,
		XmNshadowType, XmSHADOW_OUT,
		XmNmarginWidth, 5,
		XmNmarginHeight, 5,
		NULL);
	f = XtVaCreateManagedWidget("form",
		xmFormWidgetClass,
		object_frame,
		NULL);
	objtext_frame = XtVaCreateManagedWidget("frame",
		xmFrameWidgetClass,
		f,
		XmNtopAttachment, XmATTACH_NONE,
		XmNbottomAttachment, XmATTACH_FORM,
		XmNleftAttachment, XmATTACH_FORM,
		XmNrightAttachment, XmATTACH_FORM,
		XmNmarginWidth, 5,
		XmNmarginHeight, 5,
		NULL);
	objlist_frame = XtVaCreateManagedWidget("frame",
		xmFrameWidgetClass,
		f,
		XmNtopAttachment, XmATTACH_FORM,
		XmNbottomAttachment, XmATTACH_WIDGET,
		XmNbottomWidget, objtext_frame,
		XmNleftAttachment, XmATTACH_FORM,
		XmNrightAttachment, XmATTACH_FORM,
		XmNmarginWidth, 5,
		XmNmarginHeight, 5,
		NULL);
	variable_frame = XtVaCreateManagedWidget("frame",
		xmFrameWidgetClass,
		right_form,
		XmNtopAttachment, XmATTACH_FORM,
		XmNbottomAttachment, XmATTACH_POSITION,
		XmNbottomPosition, 50,
		XmNleftAttachment, XmATTACH_FORM,
		XmNrightAttachment, XmATTACH_FORM,
		XmNshadowType, XmSHADOW_OUT,
		XmNmarginWidth, 5,
		XmNmarginHeight, 5,
		NULL);
	f = XtVaCreateManagedWidget("form",
		xmFormWidgetClass,
		variable_frame,
		NULL);
	vartext_frame = XtVaCreateManagedWidget("frame",
		xmFrameWidgetClass,
		f,
		XmNtopAttachment, XmATTACH_NONE,
		XmNbottomAttachment, XmATTACH_FORM,
		XmNleftAttachment, XmATTACH_FORM,
		XmNrightAttachment, XmATTACH_FORM,
		XmNmarginWidth, 5,
		XmNmarginHeight, 5,
		NULL);
	varlist_frame = XtVaCreateManagedWidget("frame",
		xmFrameWidgetClass,
		f,
		XmNtopAttachment, XmATTACH_FORM,
		XmNbottomAttachment, XmATTACH_WIDGET,
		XmNbottomWidget, vartext_frame,
		XmNleftAttachment, XmATTACH_FORM,
		XmNrightAttachment, XmATTACH_FORM,
		XmNmarginWidth, 5,
		XmNmarginHeight, 5,
		NULL);
	miblist_frame = XtVaCreateManagedWidget("frame",
		xmFrameWidgetClass,
		right_form,
		XmNtopAttachment, XmATTACH_POSITION,
		XmNtopPosition, 50,
		XmNbottomAttachment, XmATTACH_FORM,
		XmNleftAttachment, XmATTACH_FORM,
		XmNrightAttachment, XmATTACH_FORM,
		XmNshadowType, XmSHADOW_OUT,
		XmNmarginWidth, 5,
		XmNmarginHeight, 5,
		NULL);

	/* object class option menu */
	f = XtVaCreateManagedWidget("form",
		xmFormWidgetClass,
		class_frame,
		NULL);
	m = XmCreatePulldownMenu(f, "menu", NULL, 0);
	for (i=1; i<NUM_OBJCLASSES; i++) {
		class_button[i] = pbutton("pb", m, (char*)obj_class_names[i]);
		XtAddCallback(class_button[i], XmNactivateCallback,
			SelectObjectClass, i);
		XtSetSensitive(class_button[i], 0);
		}	
	class_menu = XmCreateOptionMenu(f, "menu", NULL, 0);
	XtVaSetValues(class_menu,
		XmNtopAttachment, XmATTACH_FORM,
		XmNleftAttachment, XmATTACH_FORM,
		XmNrightAttachment, XmATTACH_FORM,
		XmNbottomAttachment, XmATTACH_FORM,
		XmNlabelString, (xs=X_STR("Object Class")),
		XmNsubMenuId, m,
		NULL);
	XtManageChild(class_menu);

	/* view update interval */
	f = XtVaCreateManagedWidget("form",
		xmFormWidgetClass,
		interval_frame,
		NULL);
	interval_text = labeled_text("interval", f, 8, "Update Interval");
	XtVaSetValues(interval_text,
		XmNeditable, True,
		XmNcursorPositionVisible, True,
		XmNshadowThickness, 2,
		XmNhighlightThickness, 2,
		XmNleftAttachment, XmATTACH_NONE,
		NULL);

	/* view background menu */
	f = XtVaCreateManagedWidget("form",
		xmFormWidgetClass,
		background_frame,
		NULL);
	LoadBackgrounds(f, &m);
	background_menu = XmCreateOptionMenu(f, "menu", NULL, 0);
	XtVaSetValues(background_menu,
		XmNtopAttachment, XmATTACH_FORM,
		XmNleftAttachment, XmATTACH_FORM,
		XmNrightAttachment, XmATTACH_FORM,
		XmNbottomAttachment, XmATTACH_FORM,
		XmNlabelString, (xs=X_STR("Background")),
		XmNsubMenuId, m,
		NULL);
	XtManageChild(background_menu);
	
	/* object list */
	f = XtVaCreateManagedWidget("form",
		xmFormWidgetClass,
		objlist_frame,
		NULL);
	objlist_label = XtVaCreateManagedWidget("label",
		xmLabelWidgetClass,
		f,
		XmNtopAttachment, XmATTACH_FORM,
		XmNleftAttachment, XmATTACH_FORM,
		XmNrightAttachment, XmATTACH_FORM,
		XmNbottomAttachment, XmATTACH_NONE,
		XmNlabelString, (xs=X_STR("Object Mask")),
		XmNalignment, XmALIGNMENT_BEGINNING,
		NULL);
	XmStringFree(xs);
	w = XtVaCreateManagedWidget("form",
		xmFormWidgetClass,
		f,
		XmNtopAttachment, XmATTACH_NONE,
		XmNleftAttachment, XmATTACH_FORM,
		XmNrightAttachment, XmATTACH_FORM,
		XmNbottomAttachment, XmATTACH_FORM,
		NULL);
	addobject = pbutton("pb", w, " Add ");
	XtAddCallback(addobject, XmNactivateCallback, AddToObjList, NULL);
	XtVaSetValues(addobject,
		XmNtopAttachment, XmATTACH_FORM,
		XmNleftAttachment, XmATTACH_FORM,
		XmNrightAttachment, XmATTACH_NONE,
		XmNbottomAttachment, XmATTACH_FORM,
		NULL);
	removeobject = pbutton("pb", w, " Remove ");
	XtAddCallback(removeobject, XmNactivateCallback, RemoveFromObjList,
		NULL);
	XtVaSetValues(removeobject,
		XmNtopAttachment, XmATTACH_FORM,
		XmNleftAttachment, XmATTACH_NONE,
		XmNrightAttachment, XmATTACH_FORM,
		XmNbottomAttachment, XmATTACH_FORM,
		NULL);
	object_list = XmCreateScrolledList(f, "list", NULL, 0);
	XtVaSetValues(XtParent(object_list),
		XmNtopAttachment, XmATTACH_WIDGET,
		XmNtopWidget, objlist_label,
		XmNtopOffset, 5,
		XmNleftAttachment, XmATTACH_FORM,
		XmNleftOffset, 5,
		XmNrightAttachment, XmATTACH_FORM,
		XmNrightOffset, 5,
		XmNbottomAttachment, XmATTACH_WIDGET,
		XmNbottomWidget, w,
		XmNbottomOffset, 5,
		XmNscrollBarDisplayPolicy, XmSTATIC,
		NULL);
	XtManageChild(object_list);

	/* object text */
	f = XtVaCreateManagedWidget("form",
		xmFormWidgetClass,
		objtext_frame,
		NULL);
	obj_text_label = XtVaCreateManagedWidget("label",
		xmLabelWidgetClass,
		f,
		XmNtopAttachment, XmATTACH_FORM,
		XmNleftAttachment, XmATTACH_FORM,
		XmNrightAttachment, XmATTACH_FORM,
		XmNbottomAttachment, XmATTACH_NONE,
		XmNlabelString, (xs=X_STR("Selection")),
		XmNalignment, XmALIGNMENT_BEGINNING,
		NULL);
	XmStringFree(xs);
	object_text = XtVaCreateManagedWidget("text",
		xmTextWidgetClass,
		f,
		XmNtopAttachment, XmATTACH_WIDGET,
		XmNtopWidget, obj_text_label,
		XmNleftAttachment, XmATTACH_FORM,
		XmNrightAttachment, XmATTACH_FORM,
		XmNbottomAttachment, XmATTACH_FORM,
		NULL);
	
	/* variable list */
	f = XtVaCreateManagedWidget("form",
		xmFormWidgetClass,
		varlist_frame,
		NULL);
	l = XtVaCreateManagedWidget("label",
		xmLabelWidgetClass,
		f,
		XmNtopAttachment, XmATTACH_FORM,
		XmNleftAttachment, XmATTACH_FORM,
		XmNrightAttachment, XmATTACH_FORM,
		XmNbottomAttachment, XmATTACH_NONE,
		XmNlabelString, (xs=X_STR("Variable List")),
		XmNalignment, XmALIGNMENT_BEGINNING,
		NULL);
	XmStringFree(xs);
	w = XtVaCreateManagedWidget("form",
		xmFormWidgetClass,
		f,
		XmNtopAttachment, XmATTACH_NONE,
		XmNleftAttachment, XmATTACH_FORM,
		XmNrightAttachment, XmATTACH_FORM,
		XmNbottomAttachment, XmATTACH_FORM,
		NULL);
	addvariable = pbutton("pb", w, " Add ");
	XtAddCallback(addvariable, XmNactivateCallback, AddToVarList, NULL);
	XtVaSetValues(addvariable,
		XmNtopAttachment, XmATTACH_FORM,
		XmNleftAttachment, XmATTACH_FORM,
		XmNrightAttachment, XmATTACH_NONE,
		XmNbottomAttachment, XmATTACH_FORM,
		NULL);
	removevariable = pbutton("pb", w, " Remove ");
	XtAddCallback(removevariable, XmNactivateCallback, RemoveFromVarList,
		NULL);
	XtVaSetValues(removevariable,
		XmNtopAttachment, XmATTACH_FORM,
		XmNleftAttachment, XmATTACH_NONE,
		XmNrightAttachment, XmATTACH_FORM,
		XmNbottomAttachment, XmATTACH_FORM,
		NULL);
	variable_list = XmCreateScrolledList(f, "list", NULL, 0);
	XtVaSetValues(XtParent(variable_list),
		XmNtopAttachment, XmATTACH_WIDGET,
		XmNtopWidget, l,
		XmNtopOffset, 5,
		XmNleftAttachment, XmATTACH_FORM,
		XmNleftOffset, 5,
		XmNrightAttachment, XmATTACH_FORM,
		XmNrightOffset, 5,
		XmNbottomAttachment, XmATTACH_WIDGET,
		XmNbottomWidget, w,
		XmNbottomOffset, 5,
		NULL);
	XtManageChild(variable_list);

	/* variable text */
	f = XtVaCreateManagedWidget("form",
		xmFormWidgetClass,
		vartext_frame,
		NULL);
	var_text_label = XtVaCreateManagedWidget("label",
		xmLabelWidgetClass,
		f,
		XmNtopAttachment, XmATTACH_FORM,
		XmNleftAttachment, XmATTACH_FORM,
		XmNrightAttachment, XmATTACH_FORM,
		XmNbottomAttachment, XmATTACH_NONE,
		XmNlabelString, (xs=X_STR("Selection")),
		XmNalignment, XmALIGNMENT_BEGINNING,
		NULL);
	XmStringFree(xs);
	variable_text = XtVaCreateManagedWidget("text",
		xmTextWidgetClass,
		f,
		XmNtopAttachment, XmATTACH_WIDGET,
		XmNtopWidget, var_text_label,
		XmNleftAttachment, XmATTACH_FORM,
		XmNrightAttachment, XmATTACH_FORM,
		XmNbottomAttachment, XmATTACH_FORM,
		NULL);
	
	/* mib list */
	f = XtVaCreateManagedWidget("form",
		xmFormWidgetClass,
		miblist_frame,
		NULL);
	l = XtVaCreateManagedWidget("label",
		xmLabelWidgetClass,
		f,
		XmNtopAttachment, XmATTACH_FORM,
		XmNleftAttachment, XmATTACH_FORM,
		XmNrightAttachment, XmATTACH_FORM,
		XmNbottomAttachment, XmATTACH_NONE,
		XmNlabelString, (xs=X_STR("MIB List")),
		XmNalignment, XmALIGNMENT_BEGINNING,
		NULL);
	XmStringFree(xs);
	mib_list = XmCreateScrolledList(f, "list", NULL, 0);
	XtVaSetValues(XtParent(mib_list),
		XmNtopAttachment, XmATTACH_WIDGET,
		XmNtopWidget, l,
		XmNtopOffset, 5,
		XmNleftAttachment, XmATTACH_FORM,
		XmNleftOffset, 5,
		XmNrightAttachment, XmATTACH_FORM,
		XmNrightOffset, 5,
		XmNbottomAttachment, XmATTACH_FORM,
		XmNbottomOffset, 5,
		NULL);
	XtManageChild(mib_list);

	XtAddCallback(mib_list, XmNbrowseSelectionCallback, SelectMibItem,NULL);
	XtAddCallback(object_list, XmNbrowseSelectionCallback, SelectObjectItem,
		NULL);
	XtAddCallback(variable_list, XmNbrowseSelectionCallback,
		SelectVariableItem, NULL);
	
	for (i=0; i<NUM_VIEWCLASSES; i++)
		view_class_count[i] = 0;
	GetHelp(newpanel, 1, 1);
	view_class_help[DEFAULT] = newpanel->help;
	GetHelp(newpanel, 2, 0);
	view_class_help[WANSTATUS] = newpanel->help;
	GetHelp(newpanel, 3, 0);
	view_class_help[CUSTOMSTATUS] = newpanel->help;
	GetHelp(newpanel, 4, 0);
	view_class_help[SITESTATUS] = newpanel->help;
	GetHelp(newpanel, 5, 0);
	view_class_help[OBJECTDATA] = newpanel->help;
	}


/*
 *	Edit the ui defaults.
 */
void
EditDefaults(view)
VIEW	*view;
	{
	register	i;

	oldview = view;
	newview = (VIEW*)myalloc(NULL, 1, sizeof(VIEW));
	bcopy(view, newview, sizeof(VIEW));
	for (i=0; i<NUM_OBJCLASSES; i++) {
		if (view->objstr[i])
			newview->objstr[i] = newstr(view->objstr[i]);
		if (view->varstr[i])
			newview->varstr[i] = newstr(view->varstr[i]);
		}
	XmTextSetString(newpanel->titletext, "Object Defaults");
	OpenNewViewPanel(OBJ_agent);
	}


/*
 *	Initialize a new view.
 */
void
InitNewView(v, viewname, viewclass, viewid)
VIEW	*v;
char	*viewname;
int	viewclass;
int	viewid;
	{
	char		name[BUFSIZ];
	OBJECT		*obj;
	register	i;

	v->class = viewclass;
	if (view_class_count[viewclass] < viewid)
		view_class_count[viewclass] = viewid;
	v->interval = GetDefaultInterval();
	if (viewname)
		strcpy(name, viewname);
	else
		sprintf(name, "%s%d", view_class_names[viewclass],
			view_class_count[viewclass]);
	view_class_count[viewclass] ++;
	v->name = newstr(name);
	switch (viewclass) {
	case WANSTATUS:
		v->panel = CreateViewPanel("WAN:", name, 640, 580,
			view_class_names[viewclass], 1, MORE_PIXMAP, 1);
		v->background = GetDefaultBackground();
		break;
	case CUSTOMSTATUS:
		v->panel = CreateViewPanel("CUSTOM:", name, 640, 580,
			view_class_names[viewclass], 1, MORE_PIXMAP, 1);
		break;
	case SITESTATUS:
		v->panel = CreateViewPanel("SITE:", name, 300, 300,
			view_class_names[viewclass], 1, MORE_PIXMAP, 0);
		break;
	case OBJECTDATA:
		v->panel = CreateViewPanel("DATA:", name, 300, 300,
			view_class_names[viewclass], 1, MORE_PIXMAP, 0);
		break;
	case DEFAULT:
		v->background = GetDefaultBackground();
		break;
	case STATUSLOG:
		v->panel = CreateViewPanel("LOG:", name, 650, 0,
			view_class_names[viewclass], 1, MORE_PIXMAP, 0);
		v->recvonly = 1;
		break;
	default:;
		}
	}



/*
 *	Open the edit panel for a view initializing it to a specific object
 *	class.
 */
static
void
OpenNewViewPanel(objclass)
int	objclass;
	{
	register	i;
	char		buf[32];
	XmString	xs;

	DisableEdit();
	newpanel->help = view_class_help[newview->class];
	switch (newview->class) {
	case WANSTATUS:
		XmTextSetString(newpanel->titletext, "New WAN Status View");
		XtSetSensitive(class_button[OBJ_processor], 1);
		XtSetSensitive(class_button[OBJ_ipaddr], 1);
		XtUnmanageChild(right_form);
		XtVaSetValues(left_form,
			XmNrightPosition, 100,
			NULL);
		XtVaSetValues(class_menu,
			XmNmenuHistory, class_button[objclass],
			NULL);
		SetCurrentBackground(newview->background);
		XtVaSetValues(newpanel->shell,
			XmNwidth, 375,
			XmNheight, 375,
			NULL);
		break;
	case CUSTOMSTATUS:
		XmTextSetString(newpanel->titletext, "New Custom Status View");
		XtSetSensitive(class_button[OBJ_processor], 1);
		XtSetSensitive(class_button[OBJ_network], 1);
		XtSetSensitive(class_button[OBJ_subnet], 1);
		XtSetSensitive(class_button[OBJ_site], 1);
		XtSetSensitive(class_button[OBJ_ipaddr], 1);
		XtUnmanageChild(right_form);
		XtVaSetValues(left_form,
			XmNrightPosition, 100,
			NULL);
		XtUnmanageChild(background_frame);
		XtVaSetValues(object_frame,
			XmNbottomAttachment, XmATTACH_FORM,
			NULL);
		XtVaSetValues(class_menu,
			XmNmenuHistory, class_button[objclass],
			NULL);
		XtVaSetValues(newpanel->shell,
			XmNwidth, 375,
			XmNheight, 375,
			NULL);
		break;
	case SITESTATUS:
		XmTextSetString(newpanel->titletext, "New Site Status View");
		XtSetSensitive(class_button[OBJ_site], 1);
		XtSetSensitive(class_button[OBJ_ipaddr], 1);
		XtUnmanageChild(right_form);
		XtVaSetValues(left_form,
			XmNrightPosition, 100,
			NULL);
		XtUnmanageChild(background_frame);
		XtVaSetValues(object_frame,
			XmNbottomAttachment, XmATTACH_FORM,
			NULL);
		XtVaSetValues(class_menu,
			XmNmenuHistory, class_button[objclass],
			NULL);
		XtVaSetValues(newpanel->shell,
			XmNwidth, 375,
			XmNheight, 375,
			NULL);
		break;
	case DEFAULT:
		for (i=1; i<NUM_OBJCLASSES; i++)
			XtSetSensitive(class_button[i], 1);
		XtVaSetValues(class_menu,
			XmNmenuHistory, class_button[objclass],
			NULL);
		SetCurrentBackground(newview->background);
		XtVaSetValues(newpanel->shell,
			XmNwidth, 600,
			XmNheight, 415,
			NULL);
		break;
	case OBJECTDATA:
		XmTextSetString(newpanel->titletext, "New Object Data View");
		for (i=1; i<NUM_OBJCLASSES; i++)
			XtSetSensitive(class_button[i], 1);
		XtUnmanageChild(background_frame);
		XtVaSetValues(object_frame,
			XmNbottomAttachment, XmATTACH_FORM,
			NULL);
		XtVaSetValues(class_menu,
			XmNmenuHistory, class_button[objclass],
			NULL);
		XtVaSetValues(newpanel->shell,
			XmNwidth, 600,
			XmNheight, 415,
			NULL);
		XtVaSetValues(objlist_label,
			XmNlabelString, (xs=X_STR("Object List")),
			NULL);
		XmStringFree(xs);
		XtUnmanageChild(addobject);
		XtUnmanageChild(removeobject);
		break;
	default:;
		}
	sprintf(buf, "%d", newview->interval);
	XmTextSetString(interval_text, buf);
	SelectObjectClass(NULL, objclass, NULL);
	OpenViewPanel(NULL, newpanel, NULL);
	}


/*
 *	Separate the view name into class and id.
 */
int
ParseViewName(str, class, id)
char	*str;
int	*class, *id;
	{
	register	i;

	for (i=1; i<NUM_VIEWCLASSES; i++)
		if (strncmp(view_class_names[i], str, 4) == 0) {
			*class = i;
			*id = atoi(str+strlen(view_class_names[i]));
			return(1);
			}
	return(0);
	}


/*
 *	Remove from the object list.
 */
static
void
RemoveFromObjList(widg, a1, a2)
Widget	widg;
caddr_t	a1, a2;
	{
	char		*t;
	XmString	xs;

	t = XmTextGetString(object_text);
	if (!t) return;
	if (strlen(t)) {
		xs = X_STR(t);
		if (XmListItemExists(object_list, xs))
			XmListDeleteItem(object_list, xs);
		XmStringFree(xs);
		}
	XtFree(t);
	}


/*
 *	Remove from the variable list.
 */
static
void
RemoveFromVarList(widg, a1, a2)
Widget	widg;
caddr_t	a1, a2;
	{
	char		*t;
	XmString	xs;

	t = XmTextGetString(variable_text);
	if (!t) return;
	if (strlen(t)) {
		xs = X_STR(t);
		if (XmListItemExists(variable_list, xs))
			XmListDeleteItem(variable_list, xs);
		XmStringFree(xs);
		}
	XtFree(t);
	}


/*
 *	Select a object from the object list.
 */
static
void
SelectMibItem(widg, a, cb)
Widget	widg;
caddr_t	a;
XmListCallbackStruct	*cb;
	{
	char	buf[BUFSIZ];

	strcpy(buf, exstr(cb->item));
	XmTextSetString(variable_text, buf);
	if (XtIsManaged(variable_list)) XmListDeselectAllItems(variable_list);
	}


/*
 *	Select the object class to be edited.
 */
static
void
SelectObjectClass(widg, objclass, a)
Widget	widg;
int	objclass;
caddr_t	a;
	{
	register	i;
	XmStringTable	item;
	XmString	xs;
	int		c;
	char		*t;

	if (objclass == current_objclass) return;
	if (current_objclass) {
		switch (newview->class) {
		case SITESTATUS:
			if (current_objclass == OBJ_site) {
				XmListDeleteAllItems(object_list);
				AddToObjList(NULL, NULL, NULL);
				XtManageChild(addobject);
				XtManageChild(removeobject);
				XtVaSetValues(objlist_label,
					XmNlabelString,
					(xs=X_STR("Object Mask")),
					NULL);
				XmStringFree(xs);
				}
		case WANSTATUS:
		case CUSTOMSTATUS:
			/* objects */
			if (newview->objstr[current_objclass]) {
				free(newview->objstr[current_objclass]);
				newview->objstr[current_objclass] = NULL;
				}
			XtVaGetValues(object_list,
				XmNitems, &item,
				XmNitemCount, &c,
				NULL);
			if (c) {
				ConvertTableToStr(item, c,
					&(newview->objstr[current_objclass]));
				XmListDeleteAllItems(object_list);
				}
			XmTextSetString(object_text, "");
			break;
		case OBJECTDATA:
			XmListDeleteAllItems(object_list);
			AddToObjList(NULL, NULL, NULL);
			XtManageChild(addobject);
			XtManageChild(removeobject);
		case DEFAULT:
			/* objects */
			if (newview->objstr[current_objclass]) {
				free(newview->objstr[current_objclass]);
				newview->objstr[current_objclass] = NULL;
				}
			XtVaGetValues(object_list,
				XmNitems, &item,
				XmNitemCount, &c,
				NULL);
			if (c) {
				ConvertTableToStr(item, c,
					&(newview->objstr[current_objclass]));
				XmListDeleteAllItems(object_list);
				}
			XmTextSetString(object_text, "");

			/* variables */
			if (newview->varstr[current_objclass]) {
				free(newview->varstr[current_objclass]);
				newview->varstr[current_objclass] = NULL;
				}
			XtVaGetValues(variable_list,
				XmNitems, &item,
				XmNitemCount, &c,
				NULL);
			XmTextSetString(variable_text, "");

			if (c) {
				ConvertTableToStr(item, c,
					&(newview->varstr[current_objclass]));
				newview->varcnt[current_objclass] = c;
				XmListDeleteAllItems(variable_list);
				}
			else
				newview->varcnt[current_objclass] = 0;
			XmListDeleteAllItems(mib_list);
			break;
		default:;
			}
		}

	current_objclass = objclass;
	if (!objclass) return;

	switch (newview->class) {
	case SITESTATUS:
		if (objclass == OBJ_site) {
			if (newview->objstr[OBJ_site]) {
				t = gettoken(newview->objstr[OBJ_site]);
				if (t) XmTextSetString(object_text, t);
				}
			XtVaSetValues(object_list,
				XmNitems, GetObjectListItems(objclass),
				XmNitemCount, GetObjectListItemCount(objclass),
				NULL);
			XmListDeselectAllItems(object_list);
			XtVaSetValues(objlist_label,
				XmNlabelString, (xs=X_STR("Sites")),
				NULL);
			XmStringFree(xs);
			XtUnmanageChild(addobject);
			XtUnmanageChild(removeobject);
			}
		break;
	case WANSTATUS:
	case CUSTOMSTATUS:
		StrToList(newview->objstr[objclass], object_list);
		break;
	case OBJECTDATA:
		if (newview->objstr[objclass]) {
			t = gettoken(newview->objstr[objclass]);
			if (t) XmTextSetString(object_text, t);
			}
		XtVaSetValues(object_list,
			XmNitems, GetObjectListItems(objclass),
			XmNitemCount, GetObjectListItemCount(objclass),
			NULL);
		XmListDeselectAllItems(object_list);
		StrToList(newview->varstr[objclass], variable_list);
		XtVaSetValues(mib_list,
			XmNitems, GetMibItems(objclass),
			XmNitemCount, GetMibItemCount(objclass),
			NULL);
		XmListDeselectAllItems(variable_list);
		XtUnmanageChild(addobject);
		XtUnmanageChild(removeobject);
		break;
	case DEFAULT:
		StrToList(newview->objstr[objclass], object_list);
		StrToList(newview->varstr[objclass], variable_list);
		XtVaSetValues(mib_list,
			XmNitems, GetMibItems(objclass),
			XmNitemCount, GetMibItemCount(objclass),
			NULL);
		XmListDeselectAllItems(variable_list);
		break;
	default:;
		}
	}


/*
 *	Select a object from the object list.
 */
static
void
SelectObjectItem(widg, a, cb)
Widget	widg;
caddr_t	a;
XmListCallbackStruct	*cb;
	{
	char	buf[BUFSIZ];

	strcpy(buf, exstr(cb->item));
	XmTextSetString(object_text, buf);
	}


/*
 *	Select a object from the object list.
 */
static
void
SelectVariableItem(widg, a, cb)
Widget	widg;
caddr_t	a;
XmListCallbackStruct	*cb;
	{
	char	buf[BUFSIZ];

	strcpy(buf, exstr(cb->item));
	XmTextSetString(variable_text, buf);
	if (XtIsManaged(mib_list)) XmListDeselectAllItems(mib_list);
	}


/*
 *	Set the object text to the current object.
 */
void
SetNewCurrentObject(name, class)
char	*name;
int	class;
	{
	if (class == current_objclass)
		XmTextSetString(object_text, name);
	}


/*
 *	Get a view class name.
 */
char*
ViewClassName(class)
int	class;
	{
	return(view_class_names[class]);
	}


/*
 *	Get a view class title.
 */
char*
ViewClassTitle(class)
int	class;
	{
	return(view_class_titles[class]);
	}
