/*  blinkenlights - blink keyboard lights 

    Copyright (C) 1994 Greg Hankins <gregh@cc.gatech.edu> 

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#include <stdio.h>
#include <linux/tty.h>
#include <linux/fcntl.h>
#include <linux/kd.h>
#include <unistd.h>
#include <signal.h> 

#define RIGHT 		0				
#define LEFT 		1
#define BOUNCE 		2
#define CONVERGE 	3
#define BLINK 		4
#define DELAY 		125000	

/* KDSETLED values:
	0 - off
	1 - LED_SCR
	2 - LED_NUM
	3 - LED_NUM + LED_SCR
	4 - LED_CAP
	5 - LED_CAP + LED_SCR
	6 - LED_CAP + LED_NUM
	7 - all on
*/ 

int ioctl();
char *strcat();
int strcmp();
int open();
int atoi();
void usage();
void right();
void left();
void converge();
void blink();
time_t time();
int rand();
void srand();

unsigned char savedleds = 0; 	/* saved led states */
int ttyfd;			/* fd for the tty */

/* reset leds on signals */
void reset_leds(int signal)
{
	ioctl(ttyfd,KDSETLED,savedleds);
	close(ttyfd);
	exit(0);
}
 
int main(int argc, char *argv[])		
{
	int dir;	/* direction we are going */
	int delay;	/* delay between lights */
	char c;		/* char for getopt() */
	char tty[10] = "/dev/";	/* device name */
	extern char *optarg;	/* for getopt() */

	/* check args and print out message if necessary */
	if(strcmp("-h",argv[argc - 1]) == 0 || strcmp("--help",argv[argc - 1]) == 0
           || argc == 1) usage(argv[0]);

	/* try to open tty */
	strcat(tty,argv[argc - 1]);
	if((ttyfd = open(tty,O_RDWR)) < 0) {	
	       	fprintf(stderr,"opening keyboard %s: ",tty); 
		perror("");
	        usage(argv[0]);
	}

	/* set up defaults, signal handlers, and get the current led states */
	dir = BOUNCE;
	delay = DELAY;
	signal(SIGINT,reset_leds);
	signal(SIGQUIT,reset_leds);
	signal(SIGTERM,reset_leds);
	signal(SIGHUP,reset_leds);
	ioctl(ttyfd,KDGETLED,&savedleds);

	/* get options */
	while((c = getopt(argc,argv,"rlbkcd:")) != -1)
		switch(c) {
		case 'd':
			/* set new delay */
			if(optarg != NULL) delay = atoi(optarg);
			else delay = DELAY;
			break;
		case 'r':
			dir = RIGHT;
			break;
		case 'l':
			dir = LEFT;
			break;
		case 'b':
			dir = BOUNCE;
			break;
		case 'c':
			dir = CONVERGE;
			break;
		case 'k':
			dir = BLINK;
			break;
		default:
			break;
		}

	/* call blinking functions */
	switch(dir) {
		case RIGHT:
			while(1>0) right(delay);
			break;
		case LEFT:
			while(1>0) left(delay);
			break;
		case BOUNCE:
			while(1>0) {		
				right(delay);
				left(delay);
			}
			break;
		case CONVERGE:
			while(1>0) converge(delay);
			break;
		case BLINK:
			while(1>0) blink(delay);
			break;
		default:
			exit(-1);
	}

	exit(0);
}

/* blink to the right */
void right(int delay)
{
	ioctl(ttyfd,KDSETLED,LED_NUM);
	usleep(delay);

	ioctl(ttyfd,KDSETLED,LED_CAP);
	usleep(delay);
	
	ioctl(ttyfd,KDSETLED,LED_SCR);
	usleep(delay);
}

/* blink to the left */
void left(int delay)	
{
	ioctl(ttyfd,KDSETLED,LED_SCR);
	usleep(delay);

	ioctl(ttyfd,KDSETLED,LED_CAP);
	usleep(delay);
	
	ioctl(ttyfd,KDSETLED,LED_NUM);
	usleep(delay);
}

/* blink randomly */
void blink(int delay)
{
    	int random;	/* random number */
        time_t now;	/* current time */
	unsigned char leds;	/* led mask */

	/* get time and seed random number generator */ 
        now = time(&now) / rand();
        srand(getpid() + (int)((now >> 16) + now + time(&now)));

	/* make a random number between 1 and 7 */
	while((random=rand()) == 0);
	if (random % 7 == 0) random = 7;
	else random = random % 7;
	leds = (unsigned char)random;	

	/* set leds */
	ioctl(ttyfd,KDSETLED,leds);
	usleep(delay);	
}

/* converge to center */
void converge(int delay)
{
	ioctl(ttyfd,KDSETLED,3);
	usleep(delay);

	ioctl(ttyfd,KDSETLED,LED_CAP);
	usleep(delay);
}

/* print helpful information */
void usage(char *name)
{
	printf("blinkenlights version 1.0, Copyright (C) 1994 Greg Hankins\n");
	printf("Usage: %s [-rlbck] [-d time] <device>\n",name);
	printf("-r	blink right\n");
	printf("-l	blink left\n");
	printf("-b	bounce (right, then left)\n");
	printf("-c	converge to center\n");
	printf("-k	blink randomly\n");
	printf("-d	delay value in milliseconds (default is 125000)\n");
	printf("-h	prints out this message\n");
	printf("<device> is the tty you are logged in on (without \"/dev/\").\n");
	exit(0);
}

