/* tua.h -- Constants and structs definition, functions prototype.
 *
 * This file is part of TUA.
 *
 *   Copyright (C) 1991,92,93  Lele Gaifax (lele@nautilus.sublink.org)
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the license, or (at
 *   your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

#ifndef _TUA_H
#define _TUA_H

/*
 * Check if the user defined the right things
 */
#if defined(USE_READLINE) && !defined(USE_TCL)
#error It is silly to link in the Readline stuff without TCL
#endif

#include "config.h"

#if STDC_HEADERS || HAVE_STRING_H
#include <string.h>
/* An ANSI string.h and pre-ANSI memory.h might conflict.  */
#if !STDC_HEADERS && HAVE_MEMORY_H
#include <memory.h>
#endif /* not STDC_HEADERS and HAVE_MEMORY_H */
#ifndef NeXT /* NeXT already does the same */
#define index strchr
#define rindex strrchr
#define bcopy(s, d, n) memcpy ((d), (s), (n))
#define bcmp(s1, s2, n) memcmp ((s1), (s2), (n))
#define bzero(s, n) memset ((s), 0, (n))
#endif /* NeXT */
#else /* not STDC_HEADERS and not HAVE_STRING_H */
#include <strings.h>
/* memory.h and strings.h conflict on some systems.  */
#endif /* not STDC_HEADERS and not HAVE_STRING_H */

#if HAVE_SIZE_T_IN_STDDEF_H
#include <stddef.h>
#else
#if HAVE_SIZE_T_IN_TYPES_H
#include <sys/types.h>
#if HAVE_TIME_T_IN_TYPES_H
#define _SYS_TYPES_H
#endif
#else
typedef unsigned int size_t;
#endif
#endif

#if HAVE_TIME_T_IN_TIME_H
#include <time.h>
#else
#if HAVE_TIME_T_IN_TYPES_H && !defined(_SYS_TYPES_H)
#include <sys/types.h>
#else
typedef long time_t;
#endif
#endif

#include <stdio.h>
#include "ansidecl.h"
#include "customize.h"
#include "trees.h"
#include "debug.h"

typedef unsigned char BOOLEAN;

#ifdef USE_TCL
#include "tcl.h"

typedef struct tcl_command_def
{
  char *name;
  Tcl_CmdProc *function;
  char *help;
} tcl_command_def_t;

#endif

#define OK	    0
#define ERROR	    1

#if !defined	(TRUE)
#define FALSE 0
#define TRUE 1
#endif

#define TIMESLICES  ((60 / 3) * 24)	/* A sample every 20 mins */

typedef double HourlyAct_t[TIMESLICES];

typedef struct port_rec		/* Port's activity */
{
  char *Port;			/* Port's name */
  BOOLEAN Killed;
  HourlyAct_t Activity;
} port_rec_t;

typedef enum
{
  LOCAL_SYSTEM,
  REMOTE_SYSTEM
} payer_t;

typedef enum
{
  SENT,
  RECEIVED
} direction_t;

typedef struct Date
{
  unsigned int Year:11;
  unsigned int Month:4;
  unsigned int Day:5;
  unsigned int DayOfWeek:3;
} Date_t;

typedef struct Time
{
  unsigned int Hour:5;
  unsigned int Min:6;
  unsigned int Sec:6;
#ifdef TAYLOR_UUCP
  unsigned int Cent:7;
#endif
} Time_t;

typedef double Julian_t;

typedef struct moved_stuff
{
  unsigned int Files;
  double Bytes;
  double Time;
} moved_stuff_t;

typedef struct system_history		/* System's history info */
{
  moved_stuff_t In;
  moved_stuff_t Out;
  double MonthlyActivity[12];	/* Last 12 months activity */
  int LastMonthProcessed;
} system_history_t;

typedef struct command_rec		/* Command's tree node */
{
  char *Command;		/* Command's name */
  unsigned int Number;		/* # of invocation */
} command_rec_t;

typedef struct day_rec		/* Daily table */
{
  Date_t Date;
  moved_stuff_t In;
  moved_stuff_t Out;
  BTREE Commands;		/* This day commands */
} day_rec_t;

typedef struct system_rec
{
  char *System;			/* System's name */
  BOOLEAN Killed;		/* TRUE: do not print this system */
  moved_stuff_t In;
  moved_stuff_t Out;
  double TimeConnect;		/* Total connection time in seconds */
  double * PhoneCost;
  double TimePayedBy[2];
  unsigned int Calls;		/* # of calls */
  unsigned int CallsIn;		/* # of inbound calls */
  unsigned int CallsOut;	/* # of outboud calls */
  unsigned int CallsOK;		/* Correctly finished ones */
  unsigned int CallsFAIL;	/*
                                 * # of calls that did not reached the
                                 * login phase.
                                 */
  unsigned int CallsSTOPPED;	/* Uncorrectly finished ones */
  Julian_t LastConnection;
  system_history_t History;		/* System's historical datas */
  BTREE Commands;
} system_rec_t;

typedef struct user_rec
{
  char *User;			/* User's name */
  BOOLEAN Killed;		/* TRUE: do not print this 
                                 * user */
  moved_stuff_t Out;
#ifdef TAYLOR_UUCP  
  moved_stuff_t In;
#endif  
  BTREE Commands;
} user_rec_t;

enum uucico_status
{
  TUUS_NONE,
  TUUS_OK,
  TUUS_INCOMING,
  TUUS_CALLING,
  TUUS_HANDSHAKE,
  TUUS_LOGIN,
  TUUS_SENDING,
  TUUS_ERROR
};

typedef struct process_status
{
  int ProcessId;
  
  BOOLEAN Killed;
  
  system_rec_t * System;
  
  unsigned int WhoIsPaying:1;
  unsigned int LoginPassed:1;
  unsigned int Status:3;
#ifdef HDB_UUCP
  Julian_t StartTime, EndTime;
#endif
} process_status_t;

#if defined(HDB_ERR_FACT)
extern double ERRFACT;		/* Correction factor for
                                 * buggy HDB */
#endif

/* #define CMO Command Line Option ;-) */

#if defined (BOTH_OF_THEM)

/* CMO: TRUE iff we are parsing a Taylor Uucp Log */
extern int is_taylor_uucp;

#endif

/* CMO: Show also a report for every command called */
extern int do_command_report_opt;

/* CMO: Build a separated command report */
extern int separated_command_report_opt;

/* CMO: Show just some system in the reports */
extern int just_some_system_opt;

/* CMO: Use the Uucp Logs under this directory */
extern char *logs_prefix_path_opt;

/* Name of the host where TUA is running */
extern char SystemName[];

extern char TheEpoc[];

/* A string representing the oldest time TUA's found in the logs */
extern char *Since;

/* The current month, extracted from CurrentTime */
extern int CurrentMonth;

/* CMO: Vertical Size of the Charts */
extern int chart_size_opt;

/* CMO: TRUE when the user wants to see the progress of the parsing */
#ifdef DEBUG
extern int be_verbose;
#endif

/* When really the log began... */
extern Julian_t StatStartingTime;

/* ...and ended */
extern Julian_t StatEndingTime;

/* Difference in days between StatStartingTime and StatEndingTime */
extern int DaysSinceLastClean;

/* Compilation time */
extern char compilation_date[];

/* Julian Date&Time */
extern Julian_t EXFUN(d_to_julian, (Date_t date, Julian_t * julian));
extern Julian_t EXFUN(dt_to_julian, (Date_t date, Time_t time, Julian_t * julian));
extern Date_t   EXFUN(julian_to_d, (Julian_t julian, Date_t * date));
extern void     EXFUN(julian_to_dt, (Julian_t julian, Date_t * date, Time_t * time));
extern CONST char * EXFUN(julian_to_asc, (Julian_t julian));
extern int      EXFUN(julian_dow, (Julian_t julian));
extern Julian_t EXFUN(julian_current_date, (NOARGS));
     
#ifdef USE_TCL

/* The TCL Interpreter for TUA */
extern Tcl_Interp *TUA_interp;

#endif

/* Killing */
extern void EXFUN (kill_system, (CONST char *));
extern void EXFUN (kill_user, (CONST char *));
extern void EXFUN (kill_port, (CONST char *));
extern void EXFUN (kill_process, (int procId));

/* Inserting... */     
extern command_rec_t *EXFUN (insert_command, (BTREE *, CONST char *));
extern user_rec_t *EXFUN (insert_user, (CONST char *));
extern system_rec_t *EXFUN (insert_system, (CONST char *));
extern day_rec_t *EXFUN (insert_day, (Date_t));
extern port_rec_t *EXFUN (insert_port, (CONST char *));
extern process_status_t * EXFUN (insert_process, (int procId));

/* ... Aliases... */     
extern CONST char *EXFUN (insert_user_alias, (CONST char *, CONST char *));
extern CONST char *EXFUN (insert_system_alias, (CONST char *, CONST char *));
extern CONST char *EXFUN (insert_port_alias, (CONST char *, CONST char *));
     
/* ... Glob aliases */
extern void EXFUN (insert_user_glob_alias, (CONST char *, CONST char *));
extern void EXFUN (insert_system_glob_alias, (CONST char *, CONST char *));
extern void EXFUN (insert_port_glob_alias, (CONST char *, CONST char *));

/* Enquiring... */     
extern void EXFUN (enquire_systems, (traverse_func_t funct));
extern void EXFUN (enquire_users, (traverse_func_t funct));
extern void EXFUN (enquire_commands, (BTREE, traverse_func_t funct));
extern void EXFUN (enquire_days, (traverse_func_t funct));
extern void EXFUN (enquire_ports, (traverse_func_t funct));
extern void EXFUN (enquire_processes, (traverse_func_t funct));
     
/* ... Aliases... */     
extern CONST char *EXFUN (get_possible_user_alias, (CONST char *));
extern CONST char *EXFUN (get_possible_system_alias, (CONST char *));
extern CONST char *EXFUN (get_possible_port_alias, (CONST char *));

/* ... Glob aliases */     
extern CONST char *EXFUN (get_possible_user_glob_alias, (CONST char *));
extern CONST char *EXFUN (get_possible_system_glob_alias, (CONST char *));
extern CONST char *EXFUN (get_possible_port_glob_alias, (CONST char *));

/* Searching */     
extern system_rec_t *EXFUN (search_system, (CONST char *));
extern user_rec_t *EXFUN (search_user, (CONST char *));
extern port_rec_t *EXFUN (search_port, (CONST char *));

extern size_t EXFUN (number_of_ports, (NOARGS));

/* Processing HDB Uucp */     
extern int EXFUN (read_xferstats_log, (void));
extern int EXFUN (read_uucico_log, (void));
extern int EXFUN (read_uuxqt_log, (void));
extern int EXFUN (read_uucp_log, (void));
extern int EXFUN (read_uux_log, (void));

/* Processing Taylor Uucp */     
extern int EXFUN (read_stats_log, (void));
extern int EXFUN (read_log_log, (void));

/* Reporting */     
extern void EXFUN (systems_report, (void));
extern void EXFUN (users_report, (void));
extern void EXFUN (daily_report, (void));
extern void EXFUN (systems_command_report, (void));
extern void EXFUN (users_command_report, (void));
extern void EXFUN (daily_command_report, (void));
extern void EXFUN (systems_summary, (void));
extern void EXFUN (systems_summary_table, (void));
extern void EXFUN (system_history_table, (void));
extern void EXFUN (hourly_activity_charts, (void));
extern void EXFUN (global_hourly_activity_chart, (void));
extern void EXFUN (monthly_history_table, (void));
extern void EXFUN (print_the_report, (void));

/* Remembering */     
extern int EXFUN (read_system_history, (void));
extern int EXFUN (write_system_history, (int));

/* Other(ing) */     
extern Julian_t EXFUN (parse_date, (CONST char *));
extern int EXFUN (get_phone_price, (int, int, int));
extern int EXFUN (gethostname, (char *, int));

extern void EXFUN (system_alloc_phone_costs, (system_rec_t * sys));
     
#ifdef GLOB_ALIASES
extern int EXFUN (wildmat, (CONST char *, CONST char *));
#endif

#ifdef USE_READLINE
extern void EXFUN (initialize_readline, (CONST char *));
extern char *EXFUN (readline_gets, (char *));
#endif

#ifdef USE_TCL
extern void DEFUN (initialize_TCL, (NOARGS));     
extern Tcl_CmdProc tcl_insert_port_alias;
extern Tcl_CmdProc tcl_insert_port_glob_alias;
extern Tcl_CmdProc tcl_kill_port;
extern Tcl_CmdProc tcl_systems_report;
extern Tcl_CmdProc tcl_users_report;
extern Tcl_CmdProc tcl_daily_report;
extern Tcl_CmdProc tcl_system_summary;
extern Tcl_CmdProc tcl_system_summary_table;
extern Tcl_CmdProc tcl_report;
extern Tcl_CmdProc tcl_insert_system_alias;
extern Tcl_CmdProc tcl_insert_system_glob_alias;
extern Tcl_CmdProc tcl_kill_system;
extern Tcl_CmdProc tcl_do_option;
extern Tcl_CmdProc interactive;
extern Tcl_CmdProc tcl_insert_user_alias;
extern Tcl_CmdProc tcl_insert_user_glob_alias;
extern Tcl_CmdProc tcl_kill_user;
extern Tcl_CmdProc tcl_show_help;
extern Tcl_CmdProc interactive_mode;
#endif

/* Useful macros & funcs */

extern int EXFUN (strbegcmp, (CONST char *, CONST char *));
extern int EXFUN (strcasecmp, (CONST char *, CONST char *));
extern int EXFUN (getdomainname, (char * buf, int size));
extern int EXFUN (gethostname, (char * buf, int size));
extern char * EXFUN (savestring, (CONST char *));     
extern PTR EXFUN (xmalloc, (size_t size));
extern void EXFUN (xfree, (PTR ptr));
     
/* Phone Category Definition */
extern int EXFUN (parse_pcd_file, (CONST char * pathname));
     
#ifdef __GNUC__

static __inline__ int
fstrcmp (register CONST char * s1, register CONST char * s2)
{
  if (*s1 && *s1 == *s2)
    return strcmp (s1+1, s2+1);
  else
    return (*s1 - *s2);
}

#else

#define fstrcmp(s1, s2) (*(s1) && *(s1) == *(s2) ? strcmp ((s1)+1, (s2)+1) : (*(s1) - *(s2)))

#endif /* __GNUC__ */

#if HAVE_FUNCTION_MACRO
#define LOG(fmt, arg)	fprintf(stderr, "TUA (" __FUNCTION__ "): " fmt "\n", arg)
#else

#ifdef __STDC__
#define LOG(fmt, arg)	fprintf(stderr, "TUA: " fmt "\n", arg)
#else
#define LOG(fmt, arg)   fprintf(stderr, fmt, arg); fprintf(stderr, "\n");
#endif

#endif /* HAVE_FUNCTION_MACRO */

#endif /* !_TUA_H */
