/* systree.c -- Systems tree management.
 *
 * This file is part of TUA.
 * 
 *   Copyright (C) 1991,92,93  Lele Gaifax (lele@nautilus.sublink.org)
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the license, or (at
 *   your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

#include    "tua.h"
#include    "pcdl.h"

static BTREE Root = NULL;

static int
DEFUN (system_compare, (sys1, sys2),
       CONST system_rec_t * sys1 AND CONST system_rec_t * sys2)
{
  int ret = fstrcmp (sys1->System, sys2->System);

  return ret;
}

void
DEFUN (system_alloc_phone_costs, (sys),
       system_rec_t * sys)
{
  int idx;
  
  sys->PhoneCost = (double *) xmalloc (sizeof(double) * pcd_names_count);
  for (idx = 0; idx < pcd_names_count; idx++)
    sys->PhoneCost[idx] = 0.0;
}
  				 
static PTR
DEFUN (system_new, (data),
       CONST PTR data)
{
  system_rec_t *sr;
  int idx;
  
  sr = (system_rec_t *) xmalloc (sizeof (system_rec_t));

  sr->System = savestring (((system_rec_t *) data)->System);
  sr->Killed = FALSE;
  sr->Out.Files = sr->In.Files = 0;
  sr->Out.Bytes = sr->In.Bytes = 0.0;
  sr->Out.Time = sr->In.Time = sr->TimeConnect = 0.0;
  sr->Calls = sr->CallsOK = sr->CallsFAIL = sr->CallsSTOPPED = 0;
  sr->CallsIn = sr->CallsOut = 0;
  sr->Commands = NULL;
  sr->History.Out.Files = sr->History.In.Files = 0;
  sr->History.Out.Bytes = sr->History.In.Bytes = 0.0;
  sr->History.Out.Time = sr->History.In.Time = 0.0;
  for (idx = 0; idx <= 11; idx++)
    sr->History.MonthlyActivity[idx] = -1.0;

  if (pcd_names_count)
    system_alloc_phone_costs (sr);
  else
    sr->PhoneCost = 0;
  
  sr->History.LastMonthProcessed = -1;
  sr->TimePayedBy[LOCAL_SYSTEM] = sr->TimePayedBy[REMOTE_SYSTEM] = 0.0;

  return (PTR) sr;
}  
       
system_rec_t *
DEFUN (insert_system, (sys),
       CONST char *sys)
{
  system_rec_t sr;
  
  if (Root == NULL)
    Root = btree_new ((compare_func_t) system_compare, (makenew_func_t) system_new);

  sr.System = (char *) get_possible_system_alias (sys);
  return (system_rec_t *) btree_insert (Root, &sr);
}

void
DEFUN (kill_system, (sys),
       CONST char *sys)
{
  system_rec_t sr;
  
  if (Root == NULL)
    Root = btree_new ((compare_func_t) system_compare, (makenew_func_t) system_new);

  sr.System = (char *) get_possible_system_alias (sys);
  ((system_rec_t *) (btree_insert (Root, &sr)))->Killed = TRUE;
}

static traverse_func_t originalTraverseFunction;

static void
DEFUN (FilteredEnquiry, (sys),
       CONST system_rec_t * sys)
{
  if (! sys->Killed)
    (*originalTraverseFunction) (sys);
}

void
DEFUN (enquire_systems, (funct),
       traverse_func_t funct)
{
  originalTraverseFunction = funct;
  btree_traverse (Root, (traverse_func_t) FilteredEnquiry);
}

system_rec_t *
DEFUN (search_system, (sys),
       CONST char *sys)
{
  system_rec_t sr;

  sr.System = (char *) get_possible_system_alias (sys);
  return (system_rec_t *) btree_search (Root, &sr);
}

#ifdef USE_TCL

int
DEFUN (tcl_kill_system, (clientData, interp, argc, argv),
       ClientData clientData AND
       Tcl_Interp * interp AND
       int argc AND
       char *argv[])

{
  if (argc != 2)
    {
      sprintf (interp->result, "Wrong number of Parameters: %s needs just one SystemName", argv[0]);
      return TCL_ERROR;
    }
  kill_system (argv[1]);
  return TCL_OK;
}

#endif /* ifdef USE_TCL */
