/*
 * mknbi.c  -  MaKe NetBoot Image
 *
 * Copyright (C) 1995 Gero Kuhlmann <gero@gkminix.han.de>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <stdio.h>
#include <unistd.h>
#include <string.h>
#include <getopt.h>
#include <fcntl.h>
#include <errno.h>
#include <malloc.h>
#include <netdb.h>
#include <sys/param.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include "mknbi.h"


char *progname;
char nfsdir[MAXPATHLEN];	/* Directory to mount as root via NFS */
char kname[MAXPATHLEN];		/* Name of kernel image file */
char outname[MAXPATHLEN];	/* Name of output file */
int kimage;			/* File handle for kernel image */
int outfile;			/* File handle for output file */

struct load_header header;			/* load header */
struct load_record *rec_list[NUM_RECORDS];	/* list of records */



/*
 * Write a buffer into the output file and update the load record
 */
static void putrec(int recnum, char *src, int size)
{
  long isize;
  char *buf;

  isize = ((size / (SECTSIZE + 1)) + 1) * SECTSIZE;
  if ((buf = malloc(isize)) == NULL) {
	perror(progname);
	exit(1);
  }
  memset(buf, isize, 0);
  memcpy(buf, src, size);
  if (write(outfile, buf, isize) != isize) {
	perror(outname);
	exit(1);
  }
  rec_list[recnum]->ilength += isize;
  rec_list[recnum]->mlength += isize;
}


/*
 * Copy a certain number of bytes from the kernel image file into the
 * boot file
 */
static int copyrec(int recnum, int remaining)
{
  int bytes_read, size, i = 1;
  char buf[SECTSIZE];

  for (bytes_read = 0; remaining > 0 && i > 0;) {
	size = (remaining > SECTSIZE) ? SECTSIZE : remaining;
	if ((i = read(kimage, buf, size)) < 0) {
		perror(kname);
		exit(1);
	}
	putrec(recnum, buf, i);
	bytes_read += i;
	remaining -= i;
  }
  return (bytes_read);
}


/*
 * Initialize a load record
 */
static void initrec(int recnum, int segment, int flags)
{
  rec_list[recnum]->rlength = sizeof(struct load_record) / sizeof(unsigned long);
  rec_list[recnum]->rtag1   = recnum + VENDOR_OFFSET;
  rec_list[recnum]->rflags  = flags;
  rec_list[recnum]->address = (unsigned long) segment << 4;
}


/*
 * Process kernel image file
 */
static void do_kernel(void)
{
  /* Process the floppy boot loader code */
  initrec(INITNUM, DEF_INITSEG, 0);
  if (copyrec(INITNUM, INITLSIZE) != INITLSIZE) {
	fprintf(stderr, "%s: Unexpected end of kernel image file\n", progname);
	exit(1);
  }
  rec_list[INITNUM]->mlength = INITMSIZE;

  /* Process the setup code */
  initrec(SETUPNUM, DEF_SETUPSEG, 0);
  if (copyrec(SETUPNUM, SETUPLSIZE) != SETUPLSIZE) {
	fprintf(stderr, "%s: Unexpected end of kernel image file\n", progname);
	exit(1);
  }
  rec_list[SETUPNUM]->mlength = SETUPMSIZE;

  /* Process the kernel. This is the last record */
  initrec(KERNELNUM, DEF_SYSSEG, FLAG_EOF);
  while (copyrec(KERNELNUM, SECTSIZE) == SECTSIZE) ;
  if (rec_list[KERNELNUM]->ilength > SYSLSIZE) {
	fprintf(stderr, "%s: Kernel image too large\n", progname);
	exit(1);
  }
  rec_list[KERNELNUM]->mlength = SYSMSIZE;
}


/*
 * Print the usage
 */
static void usage(void)
{
  fprintf(stderr, "Usage: %s [-d rom|kernel|<dir>]\n", progname);
  fprintf(stderr, "		[-i rom|kernel|<client:server:gateway:netmask>]\n");
  fprintf(stderr, "		[-a <append>] [[-k] <kernel>] [[-o] <outfile>]\n");
  exit(1);
}


/*
 * Main program
 */
void main(int argc, char **argv)
{
  struct load_record *lrp;
  char *cmdline = NULL;		/* Command line to pass to the kernel */
  char *append = NULL;		/* String to append to end of command line */
  char *addrs = NULL;		/* String containing various addresses */
  char *cp, *ip;
  int vendor_size;
  int i, len;

  /* Determine my own name for error output */
  if ((cp = strrchr(argv[0], '/')) == NULL)
	progname = argv[0];
  else
	progname = ++cp;

  /* Initialize option argments */
  strcpy(nfsdir, DFLT_DIR);
  strcpy(kname, DFLT_IMAGE);
  strcpy(outname, "");

  /* Initialize array of pointers into load header */
  vendor_size = (sizeof(VENDOR_ID) / sizeof(unsigned long) + 1) * sizeof(unsigned long);
  lrp = (struct load_record *) (&header.dummy[vendor_size]);
  for (i = 0; i < NUM_RECORDS; i++)
	rec_list[i] = lrp++;

  /* Parse options */
  opterr = 0;
  while ((i = getopt(argc, argv, "a:d:hi:k:o:")) != EOF)
	switch (i) {
		case 'd': strncpy(nfsdir, optarg, MAXPATHLEN-1);
		          nfsdir[MAXPATHLEN-1] = '\0';
		          break;
		case 'k': strncpy(kname, optarg, MAXPATHLEN-1);
		          kname[MAXPATHLEN-1] = '\0';
		          break;
		case 'o': strncpy(outname, optarg, MAXPATHLEN-1);
		          outname[MAXPATHLEN-1] = '\0';
		          break;
		case 'a': if ((append = strdup(optarg)) == NULL) {
		          	perror(progname);
		          	exit(1);
		          }
		          break;
		case 'i': if ((addrs = strdup(optarg)) == NULL) {
				perror(progname);
				exit(1);
		          }
		          break;
		case 'h':
		default:  usage();
	}

  /* Parse additional arguments */
  if (optind < argc) {
	strncpy(kname, argv[optind++], MAXPATHLEN-1);
	kname[MAXPATHLEN-1] = '\0';
  }
  if (optind < argc) {
	strncpy(outname, argv[optind++], MAXPATHLEN-1);
	outname[MAXPATHLEN-1] = '\0';
  }
  if (optind != argc)
	usage();

  /* Parse the IP address option */
  if (addrs != NULL && strcmp(addrs, "rom") && strcmp(addrs, "kernel")) {
	struct hostent *hp;
	char *buf, *bp;

	if ((buf = malloc(16 * 5)) == NULL) {
		perror(progname);
		exit(1);
	}
	i = 0;
	ip = addrs;
	bp = buf;
	while (ip != NULL && *ip) {
		if ((cp = strchr(ip, ':')) != NULL) *cp++ = '\0';
		if (strlen(ip) > 0) {
			if ((hp = gethostbyname(ip)) == NULL) {
				fprintf(stderr, "%s: invalid hostname %s\n", progname, ip);
				exit(1);
			}
			if (hp->h_length != sizeof(struct in_addr)) {
				fprintf(stderr, "%s: invalid host address type\n", progname);
				exit(1);
			}
			strcpy(bp, inet_ntoa(*(struct in_addr *) hp->h_addr));
			bp += strlen(bp);
		}
		ip = cp;
		if (i < 3) *bp++ = ':';
		if (i >= 3) break;
		i++;
	}
	for (; i < 3; i++)
		*bp++ = ':';
	*bp = '\0';
	free(addrs);
	addrs = buf;
  }
  if (addrs == NULL)
	addrs = DFLT_ADDRS;

  /* Open the input and output files */
  if ((kimage = open(kname, O_RDONLY)) < 0) {
	perror(kname);
	exit(1);
  }
  if (strlen(outname) == 0 || !strcmp(outname, "-")) {
	outfile = STDOUT_FILENO;
	strcpy(outname, "stdout");
  } else if ((outfile = creat(outname, 0664)) < 0) {
	perror(outname);
	exit(1);
  }

  /* Construct command line to pass to the kernel */
  len = strlen(DFLT_CMDL) + 1;
  if (append != NULL)
	len += strlen(append) + 1;
  if (strlen(nfsdir) > 0)
	len += strlen(NFS_ROOT) + strlen(nfsdir) + 1;
  if (strlen(addrs) > 0)
	len += strlen(NFS_ADDRS) + strlen(addrs) + 1;
  if (len > CMDLLSIZE) {
	fprintf(stderr, "%s: Command line too long\n", progname);
	exit(1);
  }
  if ((cmdline = malloc(len + 2)) == NULL) {
	perror(progname);
	exit(1);
  }

  strcpy(cmdline, DFLT_CMDL);
  if (strlen(nfsdir) > 0) {
	cp = cmdline + strlen(cmdline);
	sprintf(cp, "%s%s", NFS_ROOT, nfsdir);
  }
  if (strlen(addrs) > 0) {
	cp = cmdline + strlen(cmdline);
	sprintf(cp, "%s%s", NFS_ADDRS, addrs);
  }
  if (append != NULL && strlen(append) > 0) {
	cp = cmdline + strlen(cmdline);
	sprintf(cp, " %s", append);
  }

  /* Initialize the boot header */
  memset(&header, sizeof(header), 0);
  header.magic           = HEADER_MAGIC;
  header.hlength         = (unsigned char)(((int) &header.dummy - (int) &header) / sizeof(unsigned long)) & 0x0f;
  header.hlength        |= (unsigned char)((vendor_size / sizeof(unsigned long)) << 4) & 0xf0;
  header.locn.segment    = DEF_HEADERSEG;
  header.locn.offset     = 0;
  header.execute.segment = DEF_BOOTLSEG;
  header.execute.offset  = 0;
  header.bootsig         = BOOT_SIGNATURE;
  memcpy(&header.dummy, VENDOR_ID, sizeof(VENDOR_ID));
  if (write(outfile, &header, sizeof(header)) < 0) {
	perror(outname);
	exit (1);
  }

  /* Process the boot loader record */
  if (first_data_size > BOOTLLSIZE) {
	fprintf(stderr, "%s: Boot loader too large\n", progname);
	exit(1);
  }
  initrec(BOOTLNUM, DEF_BOOTLSEG, 0);
  putrec(BOOTLNUM, (char *) &first_data, first_data_size);
  rec_list[BOOTLNUM]->mlength = BOOTLMSIZE;

  /* Process the command line */
  initrec(CMDLNUM, DEF_CMDLSEG, 0);
  putrec(CMDLNUM, cmdline, strlen(cmdline) + 1);
  rec_list[CMDLNUM]->mlength = CMDLMSIZE;

  /* Process the kernel image */
  do_kernel();

  /* After writing out all these stuff, finally update the boot header */
  if (lseek(outfile, 0, 0) != 0) {
	perror(outname);
	exit(1);
  }
  if (write(outfile, &header, sizeof(header)) < 0) {
	perror(outname);
	exit (1);
  }
  exit(0);
}
