/*********************************************************************
 * Reboot/Program Killer Joystick Daemon (jsr.c)
 * 
 * (c) 1994  Scott Heavner (sdh@po.cwru.edu)
 *           Version 1.0 - December 5, 1994
 *********************************************************************
 * 
 *    jsr is a daemon that will perform certain commands when it 
 * detects joystick button action.  The actions I have chosen to
 * watch for are both buttons down and the release of one or
 * both buttons after both have been detected.
 *
 *    As it is set up now, if you press and release both buttons,
 * the computer will reboot.  If you press both buttons, release
 * one, but hold the other for longer than a second, then release
 * it, it will grep the process list for "X :0" and try to kill
 * that process, which might be useful if the X server hangs.
 * You are welcome to plug in your own commands (You might have to
 * if your X process doesn't show up as "X :0").  Feel free to 
 * adjust the defaults #defined below.
 * 
 *    All the defaults can also be set with command line arguments.
 * jsr -r "my new reboot command"            (or --reboot-cmd)
 * jsr -a "my new alt command"               (or --alt-cmd)
 * jsr -t my_new_button_timeout_in_seconds   (or --button-timeout)
 * jsr -d path_to_joystick_device            (or --device)
 *
 *    The commands are executed via a system() call and can be
 * any command you would issue to sh on the command line (script files,
 * executable files, sh builtins) with any arguments you like.
 *********************************************************************
 *    The joystick should probably be looked upon as a temporary
 * solution.  I'm not using my joystick for anything and 
 * didn't feel like buiding a switch or circuit to simulate a UPS
 * shutdown request on the serial port.
 *
 *    Currently, the program keeps the joystick device open all the 
 * time and other programs cannot use it.  It might be nicer to check
 * less frequently and open/close the device, waiting for the
 * completion of any process which might have opened it.  For my use,
 * it is safer this way, I'm not using the joystick for anything else,
 * so I'd rather have my reboot available at all times.
 *
 *    Someone might say why stop at two button combinations -- because
 * two is enough and I don't want to have to remember even that many. 
 * My current choice is orientation independent and pretty easy to 
 * remember.  I also thought about a cancel option, but I don't really
 * need that either.  Accidents might happen, but it will also be 
 * pretty hard to enter a cancel code after it's accidentally triggered,
 * so too bad.  Maybe we should some form of Morse code and encode the
 * signals temporally.
 *********************************************************************
 * Compile with: gcc -O2 -s -Wall -DLINUX jsr.c -o jsr
 * Run:          jsr & (from rc after loading module)
 *********************************************************************
 * Requires:     joystick-0.7 module
 *********************************************************************/

#define DEFAULT_DEVICE  "/dev/js0" /* Which joystick to use (usually js0 or js1) */
#define DEFAULT_TIMEOUT 1          /* Timeout for altcmd selection 
                                    *  set 0 for always or huge for never (seconds) */
#define REBOOT_CMD     "/etc/reboot"

/* Choose your alt_cmd from below or the commmand line */
/* This kills an X session named 'X :0', can be used to kill other progs by changing name */
#define ALT_CMD        "kill -HUP `ps  -ax | grep 'X :0' | grep -v grep | awk '{print $1}'`" /* */
/* Change to a sane VT */
/* #define ALT_CMD        "chvt 1" /* */
/* Reset after a SVGALIB crash */
/* #define ALT_CMD        "textmode" /* */

#include <linux/joystick.h>
#include <fcntl.h>
#include <unistd.h>
#include <stdio.h>
#include <stdlib.h>
#include <time.h>
#include <getopt.h>

/* Open the joystick device, exit on error */
int open_js(char *devicename)
{
  int fd;
  
  /* open device file */
  fd = open (devicename, O_RDONLY);
  if (fd < 0) {
    perror ("js");
    exit (1);
  }

  return fd;
}

/* Read the joystick device, exit on error */
void read_js(int fd, struct JS_DATA_TYPE *js)
{
    if( read (fd, js, JS_RETURN) != JS_RETURN) {
      perror ("js");
      exit (1);
    }
}

/* Make some noise */
void js_beep()
{
  printf("\007");
  fflush(stdout);
}

int main(int argc, char **argv)
{
  int    fd, done=0;
  struct JS_DATA_TYPE js;
  time_t time1, button_timeout = DEFAULT_TIMEOUT;
  char   *reboot_cmd = REBOOT_CMD;
  char   *alt_cmd    = ALT_CMD;
  char   *devicename = DEFAULT_DEVICE;
  struct option long_options[] = {
      {"reboot-cmd", 1, 0, 'r'},
      {"alt-cmd", 1, 0, 'a'},
      {"device", 1, 0, 'd'},
      {"button-timeout", 1, 0, 't'},
      {0,0,0,0}
  };

  /* Parse the command line options */
  while(!done) {
    switch(getopt_long (argc, argv, "r:a:d:t:n", long_options, &fd)) {
      case -1:
        done = 1;
        break;
      case 'r':
        reboot_cmd = optarg;
        break;
      case 'a':
        alt_cmd = optarg;
        break;
      case 'd':
        devicename = optarg;
        break;
      case 't':
        sscanf(optarg,"%ld",&button_timeout);
        break;
    }
  }
    
  /* open device file */
  fd = open_js(devicename);

  while (1) {
    /* Check the joystick device */
    read_js(fd,&js); 

    /* Check for both buttons down, wait for at least one to release */
    if ((js.buttons&3)==3) {
      /* Output a beep */
      js_beep();

      /* Wait for at least one release */
      while( (js.buttons&3)==3)
        read_js(fd,&js); 
     
      /* Grab the current time */ 
      time1 = time(NULL);

      /* Wait for the other button to be released */
      while( (js.buttons&3))
        read_js(fd,&js);
     
      /* If the single button was held more than one second,
       *      exec the alternate command, else reboot */ 
      if ((time(NULL)-time1)>=button_timeout) {
        /* Output another beep */
        js_beep();
	system(alt_cmd);
      } else {
	system(reboot_cmd);
      }
    }
    
    /* give other processes a chance - we should poll @ 10Hz maybe??  Start with 100Hz
     * It's not like we are really doing anything to load the system */
    usleep(10000UL);
  }
 
  return 0; 
}
