/*************************************************************************/
/* ntape - a tape archiver                                               */
/* Module:  watch.c                                                      */
/*          Function for the so called watch mode.                       */
/*          That means logging the tape operation in an extra window.    */
/* Author:  Matthias Hanisch                                             */
/*************************************************************************/
/*                                                                       */
/* This program is free software; you can redistribute it and/or modify  */
/* it under the terms of the GNU General Public License as published by  */
/* the Free Software Foundation; either version 2 of the License, or     */
/* (at your option) any later version.                                   */
/*                                                                       */
/* This program is distributed in the hope that it will be useful,       */
/* but WITHOUT ANY WARRANTY; without even the implied warranty of        */
/* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         */
/* GNU General Public License for more details.                          */
/*                                                                       */
/* You should have received a copy of the GNU General Public License     */
/* along with this program; if not, write to the Free Software           */
/* Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.             */
/*                                                                       */
/*************************************************************************/

#include <unistd.h>
#include <ntape.h>

#define WATCH_WIN_WIDTH 60
#define WATCH_WIN_HEIGHT 10

static WINDOW *watch_win = NULL;
static char watch_win_title[WATCH_WIN_WIDTH];

/*************************************************************************/
/* Name:     create_watch_win                                            */
/* Purpose:  Creates the watch window in the upper right corner.         */
/*************************************************************************/
void create_watch_win(char *title)
{
    strcpy(watch_win_title,title);
    
    watch_win = newwin(WATCH_WIN_HEIGHT, WATCH_WIN_WIDTH, 3,
		       COLS - WATCH_WIN_WIDTH - 2);
    leaveok(watch_win,TRUE);
    scrollok(watch_win,TRUE);
    keypad(watch_win,TRUE);
    set_color_pair(watch_win,TAPELOGWINDOW);
    winclr(watch_win);
    box(watch_win,ACS_VLINE,ACS_HLINE);
    mvwaddstr(watch_win,0,(WATCH_WIN_WIDTH - strlen(watch_win_title))/2,
	      watch_win_title);
    wmove(watch_win,1,1);
    touchwin(watch_win);
    wrefresh(watch_win);
}


/*************************************************************************/
/* Name:     print_watch_win                                             */
/* Purpose:  Prints a message in the watch window.                       */
/*************************************************************************/
void print_watch_win(char *msg)
{
    char line[WATCH_WIN_WIDTH+1];
    int x,y,k,j,i = 0;
    
    if (watch_win)
    {
	while (msg[i] != '\0')
	{
	    getyx(watch_win,y,x);
#ifdef DEBUG
	    fprintf(stderr,"print_watch_win: message = '%s'. (x/y) = (%d/%d)\n"
		    ,&msg[i],x,y);
#endif
	    for (j = x, k = 0; msg[i] != '\0' && msg[i] != '\n' && 
			    j < WATCH_WIN_WIDTH - 1; i++, j++, k++)
		line[k] = msg[i];
	    
	    line[k] = '\0';
	    
	    waddstr(watch_win,line);
	    if (msg[i] == '\n' || j == WATCH_WIN_WIDTH - 1)
	    {
		if (j != WATCH_WIN_WIDTH - 1)
		    i++;
		getyx(watch_win,y,x);
		winclrtoeol(watch_win);
	    }
	    getyx(watch_win,y,x);
	    if (x == 0)
		wmove(watch_win,y,1);
	}

	getyx(watch_win,y,x);
	if (y == WATCH_WIN_HEIGHT - 1 && x > 1)
	{
	    winclrtoeol(watch_win);
	    y--;
	}
#ifdef DEBUG
	fprintf(stderr,"Leaving with position (%d/%d)\n",x,y);
#endif
	
	box(watch_win,ACS_VLINE,ACS_HLINE);
	mvwaddstr(watch_win,0,(WATCH_WIN_WIDTH - strlen(watch_win_title))/2,
		  watch_win_title);
	wmove(watch_win,y,x);
	
	touchwin(watch_win);
	wrefresh(watch_win);
    }
}


/*************************************************************************/
/* Name:     destroy_watch_win                                           */
/* Purpose:  Destroys the watch window.                                  */
/*************************************************************************/
void destroy_watch_win()
{
    if (watch_win)
    {
	sleep(1);  /* You should be able to read the last line */
	delwin(watch_win);
	watch_win = NULL;
    }
}


/*************************************************************************/
/* Copyright (C) 1995 Matthias Hanisch, Wuerzburg                   */
/*************************************************************************/
