/*************************************************************************/
/* ntape - a tape archiver                                               */
/* Module:  scroll.c                                                     */
/* Author:  Matthias Hanisch                                             */
/*************************************************************************/
/*                                                                       */
/* This program is free software; you can redistribute it and/or modify  */
/* it under the terms of the GNU General Public License as published by  */
/* the Free Software Foundation; either version 2 of the License, or     */
/* (at your option) any later version.                                   */
/*                                                                       */
/* This program is distributed in the hope that it will be useful,       */
/* but WITHOUT ANY WARRANTY; without even the implied warranty of        */
/* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         */
/* GNU General Public License for more details.                          */
/*                                                                       */
/* You should have received a copy of the GNU General Public License     */
/* along with this program; if not, write to the Free Software           */
/* Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.             */
/*                                                                       */
/*************************************************************************/

#include <stdlib.h>
#include <strings.h>
#include <ctype.h>
#include <ntape.h>

/* 
   Declaration of variables:
   first:     pointer to the first entry of the shown elements
   focus:     pointer to the focussed entry of the shown elements
   NoEntries: Number of shown entries
   ListSize:  Number of list elements
   hilit:     Function that highlightes the selected item and
              unhighlights the previous selected one.
   show:      Function that shows the list on screen
*/


/*************************************************************************/
/* Name:     top_pos                                                     */
/* Purpose:  First Element of list box.                                  */
/*************************************************************************/
void top_pos(focus,first,NoEntries,ListSize,hilit,show)
    int *focus;
    int *first;
    int NoEntries;
    int ListSize;
    void (*hilit)(int,int);
    void (*show)(void);
{
    if (!ListSize)
	return;
    if (*first != 0)
    {
	*first = 0;
	(*show)();
    }
    if (*focus > 0)
    {
	(*hilit)(*focus,0);
	*focus = 0;
    }
}


/*************************************************************************/
/* Name:     bottom_pos                                                  */
/* Purpose:  Last Element of list                                        */
/*************************************************************************/
void bottom_pos(focus,first,NoEntries,ListSize,hilit,show)
    int *focus;
    int *first;
    int NoEntries;
    int ListSize;
    void (*hilit)(int,int);
    void (*show)(void);
{
    if (!ListSize)
	return;
    if (*first != ListSize - NoEntries)
    {
	*first = ListSize - NoEntries;
	if (*first < 0)
	    *first = 0;
	(*show)();
    }
    if (*focus < ListSize - 1)
    {
	(*hilit)(*focus,ListSize - 1);
	*focus = ListSize - 1;
    }
}


/*************************************************************************/
/* Name:     scroll_down                                                 */
/* Purpose:  Srolls down one element                                     */
/*************************************************************************/
void scroll_down(focus,first,NoEntries,ListSize,hilit,show)
    int *focus;
    int *first;
    int NoEntries;
    int ListSize;
    void (*hilit)(int,int);
    void (*show)(void);
{
    if (!ListSize)
	return;
    if (*focus >= ListSize - 1)
	return;
    if (*focus - *first == NoEntries - 1)
    {
	*first += NoEntries / 2;
	if (*first + NoEntries > ListSize)
	    *first = ListSize - NoEntries;
	(*show)();
    }
    *focus += 1;
    (*hilit)(*focus - 1,*focus);
}

/*************************************************************************/
/* Name:     scroll_up                                                   */
/* Purpose:  Scrolls up one element                                      */
/*************************************************************************/
void scroll_up(focus,first,NoEntries,ListSize,hilit,show)
    int *focus;
    int *first;
    int NoEntries;
    int ListSize;
    void (*hilit)(int,int);
    void (*show)(void);
{
    if (!ListSize)
	return;
    if (*focus <= 0)
	return;
    if (*focus == *first)
    {
	*first -= NoEntries / 2;
	if (*first < 0)
	    *first = 0;
	(*show)();
    }
    *focus -= 1;
    (*hilit)(*focus + 1,*focus);
}

/*************************************************************************/
/* Name:     page_down                                                   */
/* Purpose:  Pages down list                                             */
/*************************************************************************/
void page_down(focus,first,NoEntries,ListSize,hilit,show)
    int *focus;
    int *first;
    int NoEntries;
    int ListSize;
    void (*hilit)(int,int);
    void (*show)(void);
{
    int tmp;

    if (!ListSize)
	return;
    if (*first + NoEntries >= ListSize)
    {
	tmp = *focus;
	*focus = ListSize - 1;
	if (tmp != *focus)
	    (*hilit)(tmp,*focus);
    }
    else
    {
	*first += NoEntries;
	if (*first + NoEntries > ListSize)
	    *first = ListSize - NoEntries;
	(*show)();
	tmp = *focus;
	*focus = *first;
	if (tmp >= *focus)
	{
	    *focus = *first + NoEntries - 1;
	    if (*focus >= ListSize)
		*focus = ListSize - 1;
	}
	(*hilit)(-1,*focus);
    }
}

/*************************************************************************/
/* Name:     page_up                                                     */
/* Purpose:  Pages up list                                               */
/*************************************************************************/
void page_up(focus,first,NoEntries,ListSize,hilit,show)
    int *focus;
    int *first;
    int NoEntries;
    int ListSize;
    void (*hilit)(int,int);
    void (*show)(void);
{
    int tmp;

    if (!ListSize)
	return;
    if (*first == 0)
    {
	tmp = *focus;
	*focus = 0;
	if (tmp != *focus)
	    (*hilit)(tmp,*focus);
      
    }
    else
    {
	*first -= NoEntries;
	if (*first < 0)
	    *first = 0;
	(*show)();
	tmp = *focus;
	*focus = *first + NoEntries - 1;
	if (tmp <= *focus)
	    *focus = *first;
	if (*focus > ListSize - 1)
	    *focus = ListSize - 1;
	(*hilit)(-1,*focus);
    }
}


/*************************************************************************/
/* Copyright (C) 1994,1995 Matthias Hanisch, Wuerzburg                   */
/*************************************************************************/
