/*****************************************************************************
	      idled - idle login killer daemon for the Linux OS
       (C)opyright 1994-1995, B. Scott Burkett, All Rights Reserved
 ***************************************************************************** 
 $Header: /usr3/src/idled/RCS/process.c,v 1.4 1995/05/08 17:43:07 root Exp $
 *****************************************************************************
 $Log: process.c,v $
 * Revision 1.4  1995/05/08  17:43:07  root
 * Revised code to use UT_NAMESIZE
 *
 * Revision 1.3  1995/04/20  22:17:11  root
 * Fixed problem with calling WarnNoMore() prematurely
 *
 * Revision 1.2  1995/03/25  06:01:28  root
 * Added RCS ident strings to comments at top
 *
 *****************************************************************************
  
   Copyright (c), 1994-1995, by B. Scott Burkett
   All rights reserved.
  
   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions
   are met:
  
   1. Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
   2. Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
   3. All modifications to the source code must be clearly marked as
      such.  Binary redistributions based on modified source code
      must be clearly marked as modified versions in the documentation
      and/or other materials provided with the distribution.
   4. All advertising materials mentioning features or use of this software
      must display the following acknowledgment:
        This product includes software developed by B. Scott Burkett and
        other unpaid contributors.
   5. The name of B. Scott Burkett may not be used to endorse or promote
      products derived from this software without specific prior
      written permission.
  
   THIS SOFTWARE IS PROVIDED BY B. SCOTT BURKETT AND CONTRIBUTORS "AS IS" AND
   ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
   IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
   ARE DISCLAIMED.  IN NO EVENT SHALL B. SCOTT BURKETT OR CONTRIBUTORS BE LIABLE
   FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
   DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
   OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
   HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
   LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
   OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
   SUCH DAMAGE.
 
 *****************************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <utmp.h>
#include <fcntl.h>
#include <unistd.h>
#include <signal.h>
#include <utime.h>
#include <errno.h>
#include <ctype.h>
#include <regex.h>
#include <sys/types.h>
#include <sys/wait.h>
#include <sys/stat.h>
#include <sys/time.h>
#include <sys/syslog.h>
#include <sys/ioctl.h>
#include <sys/param.h>

#include "idled.h"

/***************************************************************************
 *                                                                         *
 * void Process(void);                                                     *
 *                                                                         *
 * Main processing loop of the daemon       			           *
 *                                                                         *
 ***************************************************************************/

void Process(void)
{
	int ndx;

	while( 1 )
	{
		sleep(_sleepdelay);
		ReadCFGFile();

		if(_debug)
		{
			sprintf(logbuf, "DEBUG: Waking up...");
			err_sys(logbuf);
		}

		while( 1 )
		{
			userrec = getutent();
	
			if(userrec == NULL) 
				break;
	
			if(userrec->ut_type == USER_PROCESS)
			{
				/* In 1.00a, we just checked the time. */
				/* if( GetTimes() >= _maxidletime ) */

				if ( JudgementDay( &ndx ) )
					Guillotine( ndx );
			}
		
		}

		endutent();
	}

}

int JudgementDay( int *cfgndx )
{
	int ndx=0;
	int _match=FALSE;
	char nt_user[9];

	if(_debug)
	{
		sprintf(logbuf, "DEBUG: Judging process #%d\n", 
			userrec->ut_pid);	
		err_sys(logbuf);
	}
	
	/* Find the entry in the CFG table that matches this user */
	for(ndx=0; ndx<=highentry; ndx++)
	{
		/* Compile regular expression pattern */
		re_comp( cfg[ndx]._username );

		/* Build null terminated user name (Thanks to A.P. Harris
		   for pointing out my obvious blunder... :-) */
		sprintf(nt_user, "%.*s", UT_NAMESIZE, userrec->ut_user);
		
		/* Check for user name match */
		if(re_exec( nt_user ))
		{
			/* Got a match, let's check the terminal line */
			/* Compile regular expression pattern */
			re_comp( cfg[ndx]._ttydevice );
			
			/* Check for terminal device match */
			if(re_exec( userrec->ut_line ))
			{
				/* This is the one! */
				if(_debug)
				{
					sprintf(logbuf, "DEBUG: Matched user %.*s (%s)",
						UT_NAMESIZE, nt_user, userrec->ut_line);
					err_sys(logbuf);
				}

				_match = TRUE;
				break;
			}
		}
	}

	if(!_match)
		return(FALSE);
	else {
		*cfgndx = ndx;
		return(TRUE);
	}
}

/***************************************************************************
 *                                                                         *
 * time_t GetTimes(void);                                                  *
 *                                                                         *
 * Returns the time difference between the change time of the user's tty   *
 * device and the current time.  This is the idle time (essentially).      *
 *                                                                         *
 ***************************************************************************/

time_t GetTimes(void)
{
	char buf[80];
	struct stat statbuf;
	time_t currtime;

	/* Build device file name */
	sprintf(buf, "/dev/%s", userrec->ut_line);

	/* Stat the device file */ 
	stat(buf, &statbuf);

	/* Get current time */
	currtime = time(NULL);

	/* Calculate Idletime */
	if(_debug)
	{
		sprintf(logbuf, "DEBUG: Idle time is %ld\n", 
			(long)currtime-statbuf.st_atime);
		err_sys(logbuf);
	}

	return(currtime-statbuf.st_atime);
}

/***************************************************************************
 *                                                                         *
 * void Guillotine( int cfgndx );                                          *
 *                                                                         *
 * Function responsible for issuing warnings and killing processes.        *
 *                                                                         *
 ***************************************************************************/

void Guillotine( int cfgndx )
{
	int ndx, _killnow=FALSE, mins=0;
	time_t   currtime = time(NULL);

	/* Check the criteria */
	if(cfg[cfgndx]._idle == 0)
		return;

	/* Check session time limit, if necessary */
	if( cfg[cfgndx]._sessionlimit != 0)
	{
		/* Calculate in minutes */
		mins = (currtime-userrec->ut_time) / 60;
		if(cfg[cfgndx]._sessionlimit <= mins)
		{
			sprintf(logbuf, "%.*s reached session limit of %d minutes",
					UT_NAMESIZE, userrec->ut_user, mins);
			err_sys(logbuf);
			SendFile("/etc/idled.limit");
			_killnow = TRUE;			
		}

	}

	/* Has this process already been warned? */
	if((ndx=BeenWarned( userrec->ut_pid )) != -1 && !_killnow)
	{
		if(GetTimes() < cfg[cfgndx]._grace)
			return;

		_killnow=TRUE;
	}
	else
	if(!_killnow)
		if(GetTimes() < cfg[cfgndx]._idle )
			return;
	   
	if(_debug && !_killnow)
	{
		sprintf(logbuf, "DEBUG: Putting %.*s (PID %d) on the Guillotine",
			UT_NAMESIZE, userrec->ut_user, userrec->ut_pid);
		err_sys(logbuf);
	}

	/* Make sure the process is still there before sending any signals! */
	
	/* Issue a warning, or "behead" the process */
	if( !_warnfirst || _killnow )
	{
		WarnNoMore( ndx );
		if(!ProcActive(userrec->ut_pid))
			return;			
		
		currtime=time(NULL);	
		sprintf(logbuf, "Killing %.*s on %s", UT_NAMESIZE, userrec->ut_user,
					         userrec->ut_line);
		err_sys(logbuf);
		MessageKill();

		kill(userrec->ut_pid, SIGHUP);
		sleep(2);

		if(ProcActive(userrec->ut_pid)) {
			kill(userrec->ut_pid, SIGTERM);
			sleep(2);
		}

		if(ProcActive(userrec->ut_pid)) {
			kill(userrec->ut_pid, SIGKILL);
			sleep(2);
		}
		
		if(ProcActive(userrec->ut_pid)) {
			sprintf(logbuf, "Possible runaway process: USER=%.*s PID=%d", 
					UT_NAMESIZE, userrec->ut_user, userrec->ut_pid);
			err_sys(logbuf);
		}
		else
		{
			/* Ok, notify by mail if necessary */
			if (cfg[cfgndx]._notifybymail == 'Y')
			{
				sprintf(logbuf, "/bin/mail %.*s < /etc/idled.mail",
					UT_NAMESIZE, userrec->ut_user);
				err_sys(logbuf);
				system(buf);
			}
		}
	}
	else
	{
		if(!ProcActive(userrec->ut_pid))
			return;
			
		currtime=time(NULL);	
		sprintf(logbuf, "Warning %.*s on %s", UT_NAMESIZE, userrec->ut_user,
					         userrec->ut_line);
		err_sys(logbuf);
		MessageWarn();
		MarkAsWarned( userrec->ut_pid );
	}
}

/***************************************************************************
 *                                                                         *
 * void MessageWarn(void);                                                 *
 *                                                                         *
 * Sends the /etc/idled.warn file to the user's tty.                       *
 *                                                                         *
 ***************************************************************************/

void MessageWarn(void)
{
	SendFile("/etc/idled.warn");
}

/***************************************************************************
 *                                                                         *
 * void MessageKill(void);                                                 *
 *                                                                         *
 * Sends the /etc/idled.kill file to the user's tty.                       *
 *                                                                         *
 ***************************************************************************/

void MessageKill(void)
{
	SendFile("/etc/idled.kill");
}

/***************************************************************************
 *                                                                         *
 * void err_sys(char *msg);                                                *
 *                                                                         *
 * Logs the passed msg buffer into the syslog daemon                       *
 *                                                                         *
 ***************************************************************************/

void err_sys(char *msg)
{
	openlog("idled", LOG_PID|LOG_CONS, LOG_DAEMON);
	syslog(LOG_PID|LOG_ERR, msg);
	closelog();
}

/***************************************************************************
 *                                                                         *
 * int BeenWarned(int pid);                                                *
 *                                                                         *
 * Returns the ndx number (0-WARNED_TABLE_SIZE) of a process that has      *
 * already been issued a warning.  Returns -1 if not in the table.         *
 *                                                                         *
 ***************************************************************************/

int BeenWarned(int pid)
{
	int ndx;

	for(ndx=0; ndx<=highentry; ndx++)
	{
		if(warned[ndx].pid == pid)
			return(ndx);
	}

	return(-1);

}

/***************************************************************************
 *                                                                         *
 * void MarkAsWarned(int pid);                                             *
 *                                                                         *
 * Creates an entry in the warned table for the passed process ID #.       *
 * This denotes that the process has been issued a warning.                *
 *                                            				   * 
 ***************************************************************************/

void MarkAsWarned(int pid)
{
	int ndx;

	for(ndx=0; ndx<=highentry; ndx++)
	{
		/* Find the first empty slot and fill it */
		if(warned[ndx].pid == 0)
		{
			warned[ndx].pid = pid;
			warned[ndx].wflag = TRUE;
			return;
		}
	}

	/* If we get here, the table is full - ack! */
	sprintf(logbuf, "WARNING: Internal warning table is full!  Can't warn!");
	err_sys(logbuf);

	return;
}

/***************************************************************************
 *                                                                         *
 * void WarnNoMore( int ndx );                                             *
 *                                                                         *
 * Resets an entry in the warned table.                                    *
 *                                            				   * 
 ***************************************************************************/

void WarnNoMore( int ndx )
{
	/* Zero out the entry for the process in the warning table */
	warned[ndx].pid   = 0;
	warned[ndx].wflag = 0;
}

/***************************************************************************
 *                                                                         *
 * void SendFile(char *filename);                                          *
 *                                                                         *
 * Sends the contents of filename to the user's tty device.  This function *
 * is called by MessageWarn() and MessageKill().		           * 
 *                                            				   * 
 ***************************************************************************/

void SendFile(char *filename)
{
	FILE 	*tmp;
	int	pid, status, fd;
	char 	localhost[20], termdevice[30];
	time_t  currtime = time(NULL);
	struct	utimbuf timebuf;
	
	struct	stat statbuf;

	pid = fork();

	if(pid < 0) 
	{
		err_sys("Unable to fork child process");
		return;
	}

	if(pid)
	{
		wait(&status);
		return;
	}
	else
	{
		sprintf(termdevice, "/dev/%s", userrec->ut_line);

		if((tmp = fopen(filename, "rt")) == NULL)
		{
			err_sys("Unable to open warning message file!");
			exit(0);
		}

		gethostname(localhost, sizeof(localhost));

		if((fd = open(termdevice, O_WRONLY)) == -1)
		{
			err_sys("Unable to open user terminal!");	
			exit(1);
		}
		
		sprintf(buf, "\n\r\n\rMessage from idledaemon.%s at %s\n\r",
			localhost, ctime(&currtime));
		write(fd, buf, strlen(buf));

		while (1)
		{
			fgets(buf, 80, tmp);
			if(feof(tmp)) break;
	
			write(fd, buf, strlen(buf));
	
		}
	
		fclose(tmp);
		fflush(tty);
		close(fd);

		/* Update access time on device file.  We have to explicitly
		   do this, as it seems some (most?) of the Linux calls do not
		   modify the atime member of a file when accessed.  This seems
		   to be the case with the write() system call, at least.
	        */

		currtime=time(NULL);
		stat(termdevice, &statbuf);
		timebuf.actime = currtime;
		timebuf.modtime = statbuf.st_mtime;	/* Same as before */
		utime(termdevice, (struct utimbuf *)&timebuf);

		exit(0);
	}

}

/***************************************************************************
 *                                                                         *
 * int ProcActive(int pid);                                                *
 *                                                                         *
 * Returns TRUE if passed PID is active.  This requires the proc psuedo    *
 * file system to be in place and mounted.                   		   * 
 *                                            				   * 
 ***************************************************************************/

int ProcActive(int pid)
{
	char filename[80];
	struct stat statbuf;

	/* Build proc_fs style filename */
	sprintf(filename, "/proc/%d", pid);

	if(stat(filename, &statbuf))
		return(FALSE);
	else
		return(TRUE);
}

