/*
 * Copyright (c) 1995 David I. Bell
 * Permission is granted to use, distribute, or modify this source,
 * provided that this copyright notice remains intact.
 */

#ifdef	X11

#include <memory.h>

#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include <X11/keysym.h>
#include <X11/StringDefs.h>

#include "life.h"


#define	ALLMODMASKS	(ShiftMask | LockMask | ControlMask | Mod1Mask)

#define	WMRESX		20	/* X pixels to reserve for window manager */
#define	WMRESY		30	/* Y pixels to reserve for window manager */
#define	MINSCALE	-30	/* minimum scale */
#define	DEFAULTSCALE	-8	/* default scale */
#define	GRIDMAG		3	/* magnification when grid can be shown */
#define	TEXTOFF		2	/* pixels for offsetting text from edges */
#define	DRAWMAX		100	/* maximum things to draw in one X call */


/*
 * Actions to perform for a row of cells (a set of flags, even though any
 * particular cell only has one of them set).
 */
#define	CELLIGNORE	0x0	/* cell is not to be changed */
#define	CELLCLEAR	0x1	/* cell is to be cleared */
#define	CELLNORMAL	0x2	/* cell is to be set normal */
#define	CELLMARK	0x4	/* cell is to be set marked */


/*
 * Device interface routines for the rest of the life program to call.
 */
static	BOOL	X11Open PROTO((DEV *));
static	void	X11Close PROTO((DEV *));
static	void	X11Update PROTO((DEV *, BOOL));
static	void	X11Refresh PROTO((DEV *));
static	BOOL	X11InputIsReady PROTO((DEV *));
static	int	X11ReadChar PROTO((DEV *, BOOL));
static	void	X11MoveCursor PROTO((DEV *, SCALE, COORD, COORD));
static	void	X11ShowView PROTO((DEV *, OBJECT *));
static	void	X11ShowStatus PROTO((DEV *, char *));
static	void	X11AddStatus PROTO((DEV *, char *));
static	void	X11ShowHelp PROTO((DEV *));
static	void	X11AddHelp PROTO((DEV *, char *));
static	BOOL	X11Assign PROTO((DEV *, VALUE, UCHAR));
static	void	X11WriteAssign PROTO((DEV *, FILE *));


DEV	x11dev = {
	X11Open, X11Close, X11Update, X11Refresh, X11InputIsReady,
	X11ReadChar, X11MoveCursor, X11ShowView, X11ShowStatus,
	X11AddStatus, X11ShowHelp, X11AddHelp, X11Assign, X11WriteAssign,
	0, 0, 0, 0, MINSCALE, MAXSCALE, DEFAULTSCALE
};


/*
 * Local variables.
 */
static	Display *	display;	/* display for X */
static	int		screen;		/* screen number */
static	Window		rootwid;	/* top level window id */
static	Window		mainwid;	/* main window id */
static	Window		viewwid;	/* window id for view of cells */
static	Window		statwid;	/* window id for status line */
static	GC		statgc;		/* GC used for status line */
static	GC		viewgc;		/* GC used for view of cells */
static	GC		viewselectgc;	/* GC used for view of selected cells */
static	GC		viewgridgc;	/* GC used for view of grid */
static	GC		cursorgc;	/* GC used for cursor */
static	GC		pathgc;		/* GC used for paths */
static	XFontStruct *	font;		/* font for text */
static	Colormap	cmap;		/* default color map */
static	long		black;		/* black pixel value */
static	long		white;		/* white pixel value */
static	long		red;		/* red pixel value */
static	long		blue;		/* blue pixel value */
static	long		grey;		/* grey pixel value */
static	long		green;		/* green pixel value */
static	int		charheight;	/* height of characters */
static	int		charascent;	/* ascent of characters */
static	int		spacewidth;	/* width of a space */
static	int		statwidth;	/* width of status window */
static	int		statheight;	/* height of status window */
static	int		viewwidth;	/* width of view window */
static	int		viewheight;	/* height of view window */
static	int		maxviewwidth;	/* maximum view width so far */
static	int		usedstatus;	/* space used in status line so far */
static	int		cursorx;	/* current cursor x location */
static	int		cursory;	/* current cursor y location */
static	int		helpx;		/* x position for new help text */
static	int		helpy;		/* y position for new help text */
static	int		helpcol;	/* column number for new help text */
static	BOOL		cursorshown;	/* TRUE if cursor is being shown */
static	BOOL		mapped;		/* TRUE if main window is mapped */
static	BOOL		needclear;	/* TRUE if need to clear window */
static	UCHAR *		genstr;		/* current char of generated string */
static	UCHAR		genstrbuf[32];	/* generated string to return */
static	UCHAR		assigns[7];	/* macro assignments for buttons */
static	OBJECT *	seenobj;	/* copy of object currently seen */
static	char *		cellactions;	/* actions to do for a row of cells */


/*
 * Local procedures.
 */
static	int	DoEvent PROTO((DEV *, XEvent *));
static	int	DoKeyDown PROTO((XKeyEvent *));
static	void	DoExposure PROTO((XExposeEvent *));
static	int	DoButtonDown PROTO((XButtonEvent *));
static	void	DoConfigure PROTO((DEV *, XConfigureEvent *));
static	void	ViewNormal PROTO((OBJECT *, SCALE));
static	void	ViewScale PROTO((OBJECT *, SCALE));
static	void	DrawCursor PROTO((OBJECT *, SCALE, COORD, COORD));
static	void	DrawRow PROTO((OBJECT *, COORD, int, SCALE));
static	void	DrawPath PROTO((OBJECT *, SCALE));
static	void	ClearRow PROTO((OBJECT *, COORD, SCALE));
static	void	ClearArea PROTO((OBJECT *, COORD, COORD, COUNT, COUNT, SCALE));
static	COORD	ScreenCellToPixel PROTO((COORD, SCALE));
static	int	Mod10 PROTO((COORD));


/*
 * Open the X11 graphics screen.
 * Returns TRUE if successful.
 */
static BOOL
X11Open(dev)
	DEV *	dev;
{
	XColor		cellcolor;
	XColor		namecolor;
	XSizeHints	sizehints;
	XWMHints	wmhints;

	display = XOpenDisplay(NULL);

	if (display == NULL) {
		fprintf(stderr, "Cannot open graphics\n");

		return FALSE;
	}

	screen = DefaultScreen(display);

	font = XLoadQueryFont(display, "fixed");

	if (font == NULL) {
		fprintf(stderr, "Cannot load \"fixed\" font\n");

		return FALSE;
	}

	charascent = font->ascent;
	charheight = charascent + font->descent;
	spacewidth = font->max_bounds.width;

	/*
	 * Allocate colors, defaulting them if they are not available.
	 */
	black = BlackPixel(display, screen);
	white = WhitePixel(display, screen);
	cmap = DefaultColormap(display, screen);

	red = white;
	green = white;
	blue = black;
	grey = black;

	if (XAllocNamedColor(display, cmap, "red", &cellcolor, &namecolor)) {
		if (cellcolor.pixel != black)
			red = cellcolor.pixel;
	}

	if (XAllocNamedColor(display, cmap, "blue", &cellcolor, &namecolor)) {
		if (cellcolor.pixel != white)
			blue = cellcolor.pixel;
	}

	if (XAllocNamedColor(display, cmap, "grey", &cellcolor, &namecolor)) {
		if (cellcolor.pixel != white)
			grey = cellcolor.pixel;
	}

	if (XAllocNamedColor(display, cmap, "green", &cellcolor, &namecolor)) {
		if (cellcolor.pixel != black)
			green = cellcolor.pixel;
	}

	viewwidth = DisplayWidth(display, screen) - WMRESX;
	statwidth = viewwidth;
	statheight = charheight + TEXTOFF * 2;
	viewheight = DisplayHeight(display, screen) - statheight - WMRESY;

	rootwid = RootWindow(display, screen);

	mainwid = XCreateSimpleWindow(display, rootwid,
		0, 0, viewwidth, viewheight + statheight,
		0, 0, 0);

	statwid = XCreateSimpleWindow(display, mainwid,
		0, 0, viewwidth, statheight,
		0, 0, grey);

	viewwid = XCreateSimpleWindow(display, mainwid,
		0, statheight, viewwidth, viewheight,
		0, 0, blue);

	XSelectInput(display, mainwid, KeyPressMask | StructureNotifyMask);

	XSelectInput(display, statwid, ExposureMask | KeyPressMask);

	XSelectInput(display, viewwid, ExposureMask | KeyPressMask |
		ButtonPressMask);

	sizehints.flags = PSize | PMinSize;
	sizehints.width = viewwidth;
	sizehints.min_width = 25;
	sizehints.height = viewheight + statheight;
	sizehints.min_height = statheight + 25;

	XSetStandardProperties(display, mainwid, "life", "life", None,
		(char **) "", 0, &sizehints);

	wmhints.flags = InputHint;
	wmhints.input = True;

	XSetWMHints(display, mainwid, &wmhints);

	XMapWindow(display, mainwid);
	XMapWindow(display, statwid);
	XMapWindow(display, viewwid);

	statgc = XCreateGC(display, statwid, 0, NULL);
	viewgc = XCreateGC(display, viewwid, 0, NULL);
	viewgridgc = XCreateGC(display, viewwid, 0, NULL);
	viewselectgc = XCreateGC(display, viewwid, 0, NULL);
	cursorgc = XCreateGC(display, viewwid, 0, NULL);
	pathgc =  XCreateGC(display, viewwid, 0, NULL);

	XSetForeground(display, viewgc, white);
	XSetForeground(display, statgc, blue);
	XSetForeground(display, viewgridgc, (grey != black) ? grey : white);
	XSetForeground(display, viewselectgc, red);
	XSetForeground(display, cursorgc, white ^ blue);
	XSetForeground(display, pathgc, green);

	XSetFunction(display, cursorgc, GXxor);
	XSetFont(display, statgc, font->fid);

	dev->rows = viewheight;
	dev->cols = viewwidth;
	dev->textrows = (viewheight - TEXTOFF * 2) / charheight;
	dev->textcols = (viewwidth - TEXTOFF * 2) / spacewidth;

	genstr = genstrbuf;
	mapped = FALSE;
	needclear = FALSE;

	seenobj = AllocateObject();

	cellactions = malloc(viewwidth + 1);

	if (cellactions == NULL) {
		fprintf(stderr, "Cannot allocate memory\n");

		return FALSE;
	}

	maxviewwidth = viewwidth;

	return TRUE;
}


static void
X11Close(dev)
	DEV *	dev;
{
	if (display)
		XCloseDisplay(display);
}


static void
X11Update(dev, inputflag)
	DEV *	dev;
	BOOL	inputflag;
{
	XSync(display, False);
}


static void
X11Refresh(dev)
	DEV *	dev;
{
	XSync(display, False);
}


/*
 * Read the next character from the terminal, waiting if requested.
 * If not waiting and no character is ready, then EOF is returned.
 * Returns characters from a previously generated string first, if present.
 * Button press events are translated into generated string.
 * This also handles other types of X11 events which are seen.
 */
static int
X11ReadChar(dev, wait)
	DEV *	dev;
	BOOL	wait;
{
	XEvent	event;
	int	ch;

	if (*genstr)
		return *genstr++;

	do {
		if (!wait && (XEventsQueued(display, QueuedAfterFlush) <= 0))
			return EOF;

		XNextEvent(display, &event);

		ch = DoEvent(dev, &event);
	} while (ch == EOF);

	return ch;
}


/*
 * Check whether input is ready from the terminal.  Returns TRUE if so.
 * Input consists of either key presses or button presses, both of which
 * are left in the event queue for later reading.  All other types of
 * X11 events are processed here when seen.
 */
static BOOL
X11InputIsReady(dev)
	DEV *	dev;
{
	XEvent	event;

	if (*genstr)
		return TRUE;

	while (TRUE) {
		if (XEventsQueued(display, QueuedAfterFlush) <= 0)
			return FALSE;

		XPeekEvent(display, &event);

		if ((event.type == KeyPress) || (event.type == ButtonPress))
			return TRUE;

		XNextEvent(display, &event);

		DoEvent(dev, &event);
	}
}


/*
 * Assign a macro to be automatically executed by a mouse button.
 * Returns TRUE if the button number is legal, or FALSE if it is illegal.
 * The first three numbers represent the unmodified buttons, whereas
 * the second three numbers are for the buttons used with any modifier
 * such as control or shift.
 */
static BOOL
X11Assign(dev, button, macro)
	DEV *	dev;
	VALUE	button;
	UCHAR	macro;
{
	if ((button <= 0) || (button > 6))
		return FALSE;

	assigns[button] = macro;

	return TRUE;
}


/*
 * Write button macro assignments to the specified FILE handle.
 */
static void
X11WriteAssign(dev, fp)
	DEV *	dev;
	FILE *	fp;
{
	int	button;
	int	macro;

	fputs("! button assignments\n", fp);

	for (button = 1; button < 7; button++) {
		macro = assigns[button];

		if (macro)
			fprintf(fp, ";assign %d %c\n", button, macro);
	}
}


/*
 * Move the cursor to the specified location using the specified scale factor.
 * This is easy to do because we use XOR to draw it.
 */
static void
X11MoveCursor(dev, scale, row, col)
	DEV *	dev;
	SCALE	scale;
	COORD	row;
	COORD	col;
{
	if (!mapped)
		return;

	if (cursorshown)
		DrawCursor(curobj, scale, cursorx, cursory);

	cursory = ScreenCellToPixel(row, scale);
	cursorx = ScreenCellToPixel(col, scale);

	DrawCursor(curobj, scale, cursorx, cursory);
}


/*
 * Draw the cursor at the specified location.  If the scale factor is
 * large enough, then the cursor is made larger.  The cursor is drawn
 * using XOR, so that if the same call is made twice, it will be removed.
 */
static void
DrawCursor(obj, scale, x, y)
	OBJECT *obj;
	SCALE	scale;
	COORD	x;
	COORD	y;
{
	cursorshown = !cursorshown;

	DrawPath(obj, scale);

	if (scale > -GRIDMAG) {
		XDrawPoint(display, viewwid, cursorgc, x, y);
		return;
	}

	XDrawPoint(display, viewwid, cursorgc, x, y);
	XDrawPoint(display, viewwid, cursorgc, x - 1, y);
	XDrawPoint(display, viewwid, cursorgc, x + 1, y);
	XDrawPoint(display, viewwid, cursorgc, x, y - 1);
	XDrawPoint(display, viewwid, cursorgc, x, y + 1);
}


/*
 * Show the currently defined path on the window, if any.
 * The endpoints of each segment of the path are in the centers of the cells.
 * The path can actually consist of more than one loop of segments.
 */
static void
DrawPath(obj, scale)
	OBJECT *obj;
	SCALE	scale;
{
	PATH *	path;
	COORD	mincol;
	COORD	minrow;
	int	x1;
	int	y1;
	int	x2;
	int	y2;
	int	i;

	minrow = obj->minrow;
	mincol = obj->mincol;

	path = &obj->path;

	i = 0;

	while (i + 1 < path->count) {
		if (path->locs[i].flag) {
			i++;
			continue;
		}

		x1 = ScreenCellToPixel(path->locs[i].col - mincol, scale);
		y1 = ScreenCellToPixel(path->locs[i].row - minrow, scale);

		i++;

		x2 = ScreenCellToPixel(path->locs[i].col - mincol, scale);
		y2 = ScreenCellToPixel(path->locs[i].row - minrow, scale);

		XDrawLine(display, viewwid, pathgc, x1, y1, x2, y2);
	}
}


/*
 * Show the view around the current window location.
 */
static void
X11ShowView(dev, obj)
	DEV *	dev;
	OBJECT *obj;
{
	if (!mapped)
		return;

	if (obj->scale > 1)
		ViewScale(obj, obj->scale);
	else if (obj->scale >= -1)
		ViewNormal(obj, (SCALE) 1);
	else
		ViewNormal(obj, -obj->scale);
}


/*
 * Update the status line in the status window.
 */
static void
X11ShowStatus(dev, str)
	DEV *	dev;
	char *	str;
{
	int	len;
	int	width;

	if (!mapped)
		return;

	len = strlen(str);

	XClearWindow(display, statwid);
	width = XTextWidth(font, str, len);
	XDrawString(display, statwid, statgc, TEXTOFF, charascent + TEXTOFF, str, len);
	usedstatus = width + TEXTOFF;
}


/*
 * Add to the status line.
 * The assumption is made that this routine is used only for input,
 * and thus the text is ended with an underscore as a prompt.
 */
static void
X11AddStatus(dev, str)
	DEV *	dev;
	char *	str;
{
	int		len;
	static	int	uswidth;

	len = strlen(str);

	if (uswidth == 0)
		uswidth = XTextWidth(font, "_", 1);

	XClearArea(display, statwid, usedstatus, 0, uswidth, 0, False);

	XDrawString(display, statwid, statgc, usedstatus, charascent + TEXTOFF,
		str, len);

	usedstatus += XTextWidth(font, str, len);

	XDrawString(display, statwid, statgc, usedstatus, charascent + TEXTOFF,
		"_", 1);
}


/*
 * Begin to show help information.
 * We use the view window for this.
 */
static void
X11ShowHelp(dev)
	DEV *	dev;
{
	XClearWindow(display, viewwid);

	cursorshown = FALSE;
	needclear = TRUE;
	helpcol = 0;
	helpx = TEXTOFF;
	helpy = charascent + TEXTOFF;
}


/*
 * Add information to the help display.
 * We must interpret certain control codes.
 */
static void
X11AddHelp(dev, str)
	DEV *	dev;
	char *	str;
{
	char *	cp;
	int	len;

	cp = str;

	for (;;) {
		while (*cp >= ' ')
			cp++;

		len = cp - str;

		if (len > 0) {
			XDrawString(display, viewwid, viewgc, helpx, helpy,
				str, len);

			helpcol += len;
			helpx += XTextWidth(font, str, len);
		}

		switch (*cp++) {
			case '\0':
				return;

			case '\n':
				helpcol = 0;
				helpx = TEXTOFF;
				helpy += charheight;
				break;

			case '\t':
				len = 8 - (helpcol + 1) % 8;

				XDrawString(display, viewwid, viewgc,
					helpx, helpy, "        ", len);

				helpcol += len;
				helpx += spacewidth * len;
				break;
		}
		str = cp;
	}
}


/*
 * Handle the specified event.
 * Returns a character generated by the event, or else EOF if
 * the event does not produce a character.
 */
static int
DoEvent(dev, event)
	DEV *		dev;
	XEvent *	event;
{
	switch (event->type) {
		case Expose:
			mapped = TRUE;
			DoExposure(&event->xexpose);
			return NULL_CMD;

		case ButtonPress:
			return DoButtonDown(&event->xbutton);
			break;

		case KeyPress:
			return DoKeyDown(&event->xkey);

		case MapNotify:
			mapped = TRUE;
			break;

		case UnmapNotify:
			mapped = FALSE;
			break;

		case ConfigureNotify:
			DoConfigure(dev, &event->xconfigure);
			return NULL_CMD;
	}

	return EOF;
}


static void
DoExposure(ep)
	XExposeEvent *	ep;
{
	if (ep->count > 0)
		return;

	if (ep->window == statwid)
		curobj->update |= U_STAT;

	if (ep->window == viewwid)
		curobj->update |= U_REDRAW;
}


/*
 * Handle a button down event.
 * This fakes up a string which moves the pointer to the mouse position,
 * and possibly executes a macro.
 */
static int
DoButtonDown(bp)
	XButtonEvent *	bp;
{
	OBJECT *obj;
	char *	cp;
	SCALE	scale;
	COORD	buttonrow;
	COORD	buttoncol;
	COUNT	deltarow;
	COUNT	deltacol;
	UCHAR *	theassign;

	if (bp->window != viewwid)
		return -1;

	obj = curobj;

	scale = obj->scale;

	if (scale >= 0) {
		buttonrow = bp->y * scale;
		buttoncol = bp->x * scale;
	} else {
		buttonrow = bp->y / (-scale);
		buttoncol = bp->x / (-scale);
	}

	deltarow = buttonrow - (obj->currow - obj->minrow);
	deltacol = buttoncol - (obj->curcol - obj->mincol);

	if (scale > 1) {
		deltarow /= scale;
		deltacol /= scale;
	}

	sprintf(genstrbuf, "%ldj%ldl", deltarow, deltacol);

	/*
	 * If a macro has been assigned to the button, then add the
	 * sequence to execute the macro.  Use any modifier key used
	 * with the buttons as three further possible macros.
	 */
	switch (bp->button)
	{
		case Button1:
			theassign = &assigns[1];
			break;

		case Button2:
			theassign = &assigns[2];
			break;

		case Button3:
			theassign = &assigns[3];
			break;

		default:
			theassign = NULL;
			break;
	}

	if (theassign) {
		if (bp->state & ALLMODMASKS)
			theassign += 3;

		if (*theassign) {
			cp = genstrbuf + strlen(genstrbuf);
			*cp++ = ESC;
			*cp++ = *theassign;
			*cp = '\0';
		}
	}

	genstr = genstrbuf;

	return NULL_CMD;
}


/*
 * Return a key typed in our window, or EOF if no such key was typed.
 */
static int
DoKeyDown(kp)
	XKeyEvent *	kp;
{
	KeySym	keysym;
	int	len;
	int	ch;
	int	macro;
	UCHAR	buf[12];

	len = XLookupString(kp, buf, sizeof(buf), &keysym, NULL);

	/*
	 * Look for special keys which are either cursor movement keys,
	 * or else function keys to execute macros.
	 */
	macro = '\0';

	switch (keysym) {
		case XK_Up:
			return (kp->state & ALLMODMASKS) ? 'K' : 'k';

		case XK_Down:
			return (kp->state & ALLMODMASKS) ? 'J' : 'j';

		case XK_Left:
			return (kp->state & ALLMODMASKS) ? 'H' : 'h';

		case XK_Right:
			return (kp->state & ALLMODMASKS) ? 'L' : 'l';

		case XK_F1:	macro = '1'; break;
		case XK_F2:	macro = '2'; break;
		case XK_F3:	macro = '3'; break;
		case XK_F4:	macro = '4'; break;
		case XK_F5:	macro = '5'; break;
		case XK_F6:	macro = '6'; break;
		case XK_F7:	macro = '7'; break;
		case XK_F8:	macro = '8'; break;
		case XK_F9:	macro = '9'; break;
	}

	/*
	 * If it was a function key, then set the associated macro name as
	 * the next input character, and return the macro invocation
	 * character now.
	 */
	if (macro) {
		genstrbuf[0] = macro;
		genstrbuf[1] = '\0';
		genstr = genstrbuf;

		return ESC;
	}

	/*
	 * It was not a special key, so check for a normal key.
	 */
	if ((len != 1) || IsModifierKey(keysym))
		return EOF;

	ch = buf[0];

	if (ch == '\r')
		ch = '\n';

	return ch;
}


/*
 * Here for a configuration change to our window.
 * This is typically a resize.
 */
static void
DoConfigure(dev, cp)
	DEV *			dev;
	XConfigureEvent *	cp;
{
	char *	newactions;

	if ((cp->height == (viewheight + statheight)) &&
		(cp->width == viewwidth))
			return;

	/*
	 * If the view is growing, then reallocate the actions table.
	 * If that fails, just ignore the reconfiguration.
	 */
	if (cp->width > maxviewwidth) {
		newactions = realloc(cellactions, cp->width + 1);

		if (newactions == NULL)
			return;

		cellactions = newactions;

		maxviewwidth = cp->width;
	}

	/*
	 * OK, remember the new sizes and resize the viewing and status
	 * windows to fit the new size.
	 */
	viewwidth = cp->width;
	statwidth = viewwidth;
	viewheight = cp->height - statheight;

	if (viewheight <= 0)
		viewheight = 1;

	XResizeWindow(display, statwid, statwidth, statheight);
	XResizeWindow(display, viewwid, viewwidth, viewheight);

	/*
	 * Remember the new sizes for the rest of the program to know about.
	 * Then redraw the current object which will take into account the
	 * new size.
	 */
	dev->rows = viewheight;
	dev->cols = viewwidth;
	dev->textrows = (viewheight - TEXTOFF * 2) / charheight;
	dev->textcols = (viewwidth - TEXTOFF * 2) / spacewidth;

	SetScale(curobj, curobj->scale);
}


/*
 * Show the cells around the cursor normally (showing each cell).
 * The specified magnification factor is applied.
 */
static void
ViewNormal(obj, mag)
	OBJECT *obj;
	SCALE	mag;
{
	ROW  *	rp;		/* current row */
	CELL *	cp;		/* current cell */
	COORD	row;		/* current row number */
	COORD	col;		/* current column number */
	COORD	screenrow;	/* row on screen */
	ROW *	srp;		/* current row seen */
	CELL *	scp;		/* current cell seen */
	OBJECT *sobj;		/* currently seen object */
	COUNT	totalcols;	/* total number of columns in display */
	int	rowactions;	/* actions to perform on a row */
	int	action;
	int	saction;
	COORD	tmp;

	if ((obj->scale != seenobj->scale) ||
		(obj->minrow != seenobj->minrow) ||
		(obj->mincol != seenobj->mincol))
			needclear = TRUE;

	if (needclear | (obj->update & U_CLEAR)) {
		ZeroObject(seenobj);

		ClearArea(obj, 0, 0, (obj->maxrow - obj->minrow + 2),
			(obj->maxcol - obj->mincol + 2), mag);

		cursorshown = FALSE;
		needclear = FALSE;
	}

	rp = obj->firstrow;
	row = obj->minrow;

	while (row > rp->row)
		rp = rp->next;

	sobj = seenobj;
	srp = sobj->firstrow;

	while (row > srp->row)
		srp = srp->next;

	totalcols = obj->maxcol - obj->mincol + 1;

	seecount = 0;

	while (TRUE) {
		/*
		 * Find the next row which exists in either of the new or
		 * old objects.
		 */
		if ((row != rp->row) && (row != srp->row)) {
			tmp = rp->row;

			if (tmp > srp->row)
				tmp = srp->row;

			row = tmp;
		}

		/*
		 * If the row is beyond the display area, then we are done.
		 */
		if (row > obj->maxrow)
			break;

		/*
		 * Now simultaneously walk the old and new cells for this row.
		 * For every cell which differs, we either draw the new cell
		 * or clear the new cell.  If the row doesn't exist in one of
		 * the new or old objects, then use an empty row to avoid
		 * having special cases.
		 */
		screenrow = row - obj->minrow;
		col = obj->mincol;

		cp = termcell;
		if (row == rp->row) {
			cp = rp->firstcell;

			while (col > cp->col)
				cp = cp->next;

			rp = rp->next;
		}

		scp = termcell;

		if (row == srp->row) {
			scp = srp->firstcell;

			while (col > scp->col)
				scp = scp->next;

			srp = srp->next;
		}

		memset(cellactions, CELLIGNORE, totalcols + 1);
		rowactions = CELLIGNORE;

		while (TRUE) {
			/*
			 * Find the next column which exists in either the
			 * new or old objects.
			 */
			if ((col != cp->col) && (col != scp->col)) {
				tmp = cp->col;

				if (tmp > scp->col)
					tmp = scp->col;

				col = tmp;
			}

			/*
			 * If the column is off of the window, then the
			 * scan of the row is done.
			 */
			if (col > obj->maxcol)
				break;

			/*
			 * Determine what has happened to the cell at this
			 * column.  Set the action appropriately.
			 */
			action = CELLCLEAR;
			if (col == cp->col) {
				seecount++;

				if (cp->marks & MARK_SEE)
					action = CELLMARK;
				else
					action = CELLNORMAL;

				cp = cp->next;
			}

			saction = CELLCLEAR;

			if (col == scp->col) {
				if (scp->marks & MARK_SEE)
					saction = CELLMARK;
				else
					saction = CELLNORMAL;

				scp = scp->next;
			}

			/*
			 * If there has been a change, then remember it.
			 */
			if (action != saction) {
				cellactions[col - obj->mincol] = action;
				rowactions |= action;
			}
		}

		/*
		 * Now that the row is scanned, make the indicated changes.
		 * Separate passes are used for supposed X11 efficiency.
		 * Clear cells, then set normal ones, then set marked ones.
		 * First clear the cursor if anything changed.
		 */
		if ((rowactions != CELLIGNORE) && cursorshown)
			DrawCursor(obj, -mag, cursorx, cursory);

		if (rowactions & CELLCLEAR)
			ClearRow(obj, screenrow, mag);

		if (rowactions & CELLNORMAL)
			DrawRow(obj, screenrow, CELLNORMAL, mag);

		if (rowactions & CELLMARK)
			DrawRow(obj, screenrow, CELLMARK, mag);
	}

	/*
	 * Now save the current object as the seen object for the next update.
	 */
	CopyObject(obj, seenobj);
}


/*
 * Show the view around the cursor with an arbitrary scale factor.
 * Each pixel then represents one or more live cells in an N by N area.
 * If any of those cells are marked, then the pixel will show it.
 */
static void
ViewScale(obj, scale)
	OBJECT *obj;
	SCALE	scale;
{
	CELL *	cp;		/* current cell structure */
	COORD	row;		/* current row number */
	COORD	col;		/* current column number */
	int	sum;		/* number of cells in square */
	CELL **	cpp;		/* pointer into cell table */
	CELL **	endcpp;		/* end of cell table */
	ROW *	rp;		/* row pointer */
	COUNT	blankrows;	/* number of blank rows */
	COORD	screenrow;	/* row on screen */
	COORD	screencol;	/* column on screen */
	BOOL	cleared;	/* TRUE if line has been cleared */
	GC	gc;		/* GC for points */
	CELL *	cptab[MAXSCALE];	/* table of rows */

	needclear = TRUE;
	row = obj->minrow;
	col = obj->mincol;
	endcpp = &cptab[scale];
	seecount = 0;
	screenrow = 0;
	blankrows = 0;

	for (rp = obj->firstrow; (rp->row < row); rp = rp->next)
		;

	while (row <= obj->maxrow) {
		/*
		 * If there is a large gap to the next row number then
		 * the graphics line is empty.
		 */
		if (rp->row >= (row + scale)) {	/* no rows here */
			row += scale;
			blankrows++;
			screenrow++;

			continue;
		}

		/*
		 * Collect the rows to be searched for one graphics line.
		 * Dummy up empty rows if necessary.
		 */
		for (cpp = cptab; cpp < endcpp; cpp++) {
			*cpp = termcell;

			if (rp->row > row++)
				continue;

			*cpp = rp->firstcell;
			rp = rp->next;
		}

		/*
		 * Advance along each row to the next range of columns,
		 * adding cells found to get the result for each square.
		 */
		screencol = 0;
		cleared = FALSE;

		for (col = obj->mincol; col <= obj->maxcol; col += scale) {
			gc = viewgc;
			sum = 0;

			for (cpp = cptab; cpp < endcpp; cpp++) {
				cp = *cpp;

				while (col > cp->col)
					cp = cp->next;

				while ((col + scale) >= cp->col) {
					sum++;

					if (cp->marks & MARK_SEE)
						gc = viewselectgc;

					cp = cp->next;
				}

				*cpp = cp;
			}

			if (sum == 0) {		/* no cells in square */
				screencol++;
				continue;
			}

			if (!cleared) {
				XClearArea(display, viewwid, 0,
					screenrow - blankrows, viewwidth,
					blankrows + 1, False);

				blankrows = 0;
				cleared = TRUE;
			}

			XDrawPoint(display, viewwid, gc, screencol, screenrow);

			seecount += sum;
			screencol++;
		}

		if (!cleared)
			blankrows++;

		screenrow++;
	}

	XClearArea(display, viewwid, 0, screenrow - blankrows, 
		viewwidth, viewheight - (screenrow - blankrows), False);

	cursorshown = FALSE;
}


/*
 * Clear the cells in a row that require that done.
 */
static void
ClearRow(obj, screenrow, mag)
	OBJECT *obj;
	COORD	screenrow;
	SCALE	mag;
{
	char *	cp;
	COORD	screencol;
	int	count;
	int	maxcols;

	if (mag < 1)
		mag = 1;

	cp = cellactions;
	screencol = 0;

	maxcols = obj->maxcol - obj->mincol + 1;

	while (screencol < maxcols) {
		cp = memchr(cp, CELLCLEAR, maxcols - screencol);

		if (cp == NULL)
			return;

		screencol = (cp - cellactions);
		count = 0;

		while ((screencol < maxcols) && (*cp == CELLCLEAR)) {
			screencol++;
			cp++;
			count++;
		}

		ClearArea(obj, screenrow, screencol - count, 1, count, mag);
	}
}


/*
 * Clear the cells in the specified rectangular region of the screen.
 * This first blanks out the area, and then puts back the grid points
 * if the grid is being shown.
 */
static void
ClearArea(obj, screenrow, screencol, rowcount, colcount, mag)
	OBJECT *obj;
	COORD	screenrow;
	COORD	screencol;
	COUNT	rowcount;
	COUNT	colcount;
	SCALE	mag;
{
	COORD	xp;
	COORD	yp;
	COORD	col;
	COUNT	count;
	int	point1count;
	int	point2count;
	XPoint	points1[DRAWMAX];
	XPoint	points2[DRAWMAX];

	XClearArea(display, viewwid, screencol * mag, screenrow * mag,
		mag * colcount, mag * rowcount, False);

	/*
	 * If we don't show a grid then we are done.
	 */	
	if ((mag < GRIDMAG) || (gridchar == ' '))
		return;

	/*
	 * Show the grid by putting dots at the bottom right corner of
	 * each cell.  Use another color for every tenth row and column.
	 */
	point1count = 0;
	point2count = 0;

	yp = screenrow * mag + mag - 1;

	while (rowcount-- > 0) {
		xp = screencol * mag + mag - 1;
		count = colcount;
		col = screencol;

		while (count-- > 0) {
			/*
			 * If either of the point tables are full,
			 * then draw both sets of points to empty them.
			 */
			if ((point1count >= DRAWMAX) ||
				(point2count >= DRAWMAX))
			{
				if (point1count > 0)
					XDrawPoints(display, viewwid,
						viewgridgc, points1,
						point1count, CoordModeOrigin);

				if (point2count > 0)
					XDrawPoints(display, viewwid,
						viewselectgc, points2,
						point2count, CoordModeOrigin);

				point1count = 0;
				point2count = 0;
			}

			/*
			 * Add the point to the appropriate table of points,
			 * depending on the current screen row and column.
			 * Make the differing grid colors depend on the
			 * absolute coordinates so that they don't change
			 * when the view is shifted.
			 */
			if ((Mod10(obj->minrow + screenrow) == 0) ||
				(Mod10(obj->mincol + col) == 9))
			{
				points1[point1count].x = xp;
				points1[point1count].y = yp;
				point1count++;
			} else {
				points2[point2count].x = xp;
				points2[point2count].y = yp;
				point2count++;
			}

			xp += mag;
			col++;
		}

		yp += mag;
		screenrow++;
	}

	/*
	 * Finish drawing any leftover points.
	 */
	if (point1count > 0)
		XDrawPoints(display, viewwid, viewgridgc, points1,
			point1count, CoordModeOrigin);

	if (point2count > 0)
		XDrawPoints(display, viewwid, viewselectgc, points2,
			point2count, CoordModeOrigin);
}


/*
 * Calculate the residue of the specifed coordinate value mod 10.
 * This is slightly complicated because it has to work for negatives too.
 */
static int
Mod10(coord)
	COORD	coord;
{
	if (coord >= 0)
		return coord % 10;

	coord = (-coord) % 10;

	if (coord)
		coord = 10 - coord;

	return coord;
}


/*
 * Draw the cells in a row that have the specified action for them.
 * This is used either for normal cells, or for selected cells.
 */
static void
DrawRow(obj, screenrow, action, mag)
	OBJECT *obj;
	COORD	screenrow;
	int	action;
	SCALE	mag;
{
	char *		cp;
	COORD		screencol;
	int		count;
	int		size;
	int		maxcols;
	GC		gc;
	XPoint *	pt;
	XRectangle *	rect;
	XRectangle	rectangles[DRAWMAX];
	XPoint		points[DRAWMAX];

	if (action == CELLNORMAL)
		gc = viewgc;
	else
		gc = viewselectgc;

	if (mag < 1)
		mag = 1;

	size = ((mag < 3) ? mag : (mag - 1));

	cp = cellactions;
	screencol = 0;

	maxcols = obj->maxcol - obj->mincol + 2;

	while (screencol < maxcols) {
		cp = memchr(cp, action, maxcols - screencol);

		if (cp == NULL)
			return;

		screencol = (cp - cellactions);
		rect = rectangles;
		pt = points;
		count = 0;

		while ((screencol < maxcols) && (*cp == action) &&
			(count < DRAWMAX))
		{
			if (mag == 1) {
				pt->x = screencol;
				pt->y = screenrow;
				pt++;
			} else {
				rect->x = screencol * mag;
				rect->y = screenrow * mag;
				rect->width = size;
				rect->height = size;
				rect++;
			}

			screencol++;
			cp++;
			count++;
		}

		if (mag == 1)
			XDrawPoints(display, viewwid, gc, points, count,
				CoordModeOrigin);
		else
			XFillRectangles(display, viewwid, gc, rectangles,
				count);
	}
}


/*
 * Convert a screen cell offset into a pixel offset.
 * The returned pixel is in the center of the cell.
 */
static COORD
ScreenCellToPixel(coord, scale)
	COORD	coord;
	SCALE	scale;
{
	if ((scale == 0) || (scale == -1))
		scale = 1;

	if (scale > 0)
		return (coord / scale);

	scale = -scale;

	return (coord * scale + ((scale - 1) / 2));
}

#endif

/* END CODE */
