/*
 * Copyright (c) 1995 David I. Bell
 * Permission is granted to use, distribute, or modify this source,
 * provided that this copyright notice remains intact.
 */

#include "life.h"


#ifdef	TERMIOS
#include <termios.h>
#endif

#ifdef	TERMIO
#include <sys/termio.h>
#endif

#ifdef	SGTTY
#include <sys/ioctl.h>
#include <sgtty.h>
#endif


static	int	(*scanroutine)();	/* routine to read characters */
static	jmp_buf *scanjumpbuf;		/* jump buffer to use in ScanChar */
static	UCHAR *	scanreadptr;		/* current read pointer */
static	UCHAR *	scanwriteptr;		/* current write pointer */
static	UCHAR	scanbuffer[SCAN_SIZE+1];/* storage for characters */


/*
 * Initialize for later calls to ScanChar.
 */
void
ScanInit(routine, jumpbuf)
	int	(*routine)();		/* routine to get characters */
	jmp_buf	jumpbuf;		/* jump buffer to use later */
{
	/*
	 * Init some static variables.
	 */
	scanroutine = routine;
	scanjumpbuf = jumpbuf;
	scanwriteptr = scanbuffer;
	scanreadptr = scanbuffer;

	/*
	 * Ask the system for the settings of the editing characters.
	 * All of this is within a pair of brackets so that local data
	 * structures can be used.
	 */
	{
#ifdef	TERMIOS
		struct	termios	term;

		if (tcgetattr(STDIN, &term) == 0)
#endif

#ifdef	TERMIO
		struct	termio	term;

		if (ioctl(STDIN, TCGETA, &term) == 0)
#endif

#if	defined(TERMIO) || defined(TERMIOS)
		{
			vkill = term.c_cc[VKILL];
			verase1 = term.c_cc[VERASE];
			veof = term.c_cc[VEOF];
#ifdef	VWERASE
			vwerase = term.c_cc[VWERASE];
#endif

#ifdef	VLNEXT
			vlnext = term.c_cc[VLNEXT];
#endif
		}
#endif


#ifdef	SGTTY
		struct	sgttyb	sgbuf;
		struct	ltchars	ltbuf;

		if (ioctl(STDIN, TIOCGETP, &sgbuf) == 0) {
			verase1 = sgbuf.sg_erase;
			vkill = sgbuf.sg_kill;
		}

		if (ioctl(STDIN, TIOCGLTC, &ltbuf) == 0) {
			vwerase = ltbuf.t_werasc;
			vlnext = ltbuf.t_lnextc;
		}
#endif
	}

	/*
	 * Undefine any values are aren't set to control characters.
	 * Things like the ancient # and @ kill characters will interfere
	 * with our own commands.
	 */
	if (vkill >= ' ')
		vkill = '\0';

	if (verase1 >= ' ')
		verase1 = '\0';

	if (veof >= ' ')
		veof = '\0';

	if (vwerase >= ' ')
		vwerase = '\0';

	if (vlnext >= ' ')
		vlnext = '\0';

	/*
	 * Now set some defaults if they weren't defined by the system.
	 * Set the second erase character to the other DEL or BS key if the
	 * first erase char is one of them, otherwise set it identically.
	 */
	if (!verase1)
		verase1 = '\b';

	verase2 = verase1;

	if (verase1 == '\b')
		verase2 = DEL;

	if (verase1 == DEL)
		verase2 = '\b';

	if (!vkill)
		vkill = 'U'-'@';

	if (!vwerase)
		vwerase = 'W'-'@';

	if (!vlnext)
		vlnext = 'V'-'@';

	if (!veof)
		veof = 'D'-'@';
}


/*
 * Read the next input character.  If it is an editing character,
 * abort the current context and longjmp back to the last setjmp.
 * NOTE: for proper results, the caller should not alter the global
 * state until the full command has been read in.  This includes such
 * things as prompting for input or saving values.  Otherwise, improper
 * results will occur if the user edits the command.
 */
int
ScanChar()
{
	int	ch;		/* current character */

loop:	
	if (scanreadptr < scanwriteptr)	/* get saved char if have any */
		return *scanreadptr++;

	ch = (*scanroutine)();		/* get new character */

	if (ch == EOF)			/* no character means eof */
		ScanEof();

	if (ch == vlnext) {		/* literal input */
		ch = (*scanroutine)();

		goto store;
	}

	if ((ch == verase1) || (ch == verase2)) {	/* character erase */
		if (scanwriteptr <= scanbuffer) {
			Beep();

			goto loop;
		}

		scanwriteptr--;
		scanreadptr = scanbuffer;

		longjmp(scanjumpbuf[0], SCAN_EDIT);
	}

	if (ch == vwerase) {		/* word erase */
		if (scanwriteptr <= scanbuffer)
			goto loop;

		while ((--scanwriteptr >= scanbuffer) &&
			isblank(*scanwriteptr))
		    		;

		scanwriteptr++;
		while ((--scanwriteptr >= scanbuffer) &&
			!isblank(*scanwriteptr))
			    	;

		scanwriteptr++;
		scanreadptr = scanbuffer;

		longjmp(scanjumpbuf[0], SCAN_EDIT);
	}

	if (ch == vkill) {		/* line erase */
		if (scanwriteptr <= scanbuffer)
			goto loop;

		scanwriteptr = scanbuffer;
		scanreadptr = scanbuffer;

		longjmp(scanjumpbuf[0], SCAN_EDIT);
	}

store:	
	if (scanwriteptr >= scanbuffer + SCAN_SIZE) {
		Beep();

		goto loop;
	}

	*scanwriteptr++ = ch;

	return *scanreadptr++;
}


/*
 * Abort reading of the current command
 */
void
ScanAbort()
{
	scanreadptr = scanbuffer;
	scanwriteptr = scanbuffer;

	longjmp(scanjumpbuf[0], SCAN_ABORT);
}


/*
 * Indicate no more characters ready yet
 */
void
ScanEof()
{
	scanreadptr = scanbuffer;

	longjmp(scanjumpbuf[0], SCAN_EOF);
}


/*
 * Simply reset input and output pointers without longjmping
 */
void
ScanReset()
{
	scanreadptr = scanbuffer;
	scanwriteptr = scanbuffer;
}


void
Beep()
{
	fputc('\007', stderr);
	fflush(stderr);
}

/* END CODE */
