/*
 * Copyright (c) 1995 David I. Bell
 * Permission is granted to use, distribute, or modify this source,
 * provided that this copyright notice remains intact.
 */

#include "life.h"

#define	PATHALLOC	20


static	BOOL	PointIsInsidePath PROTO((OBJECT *, COORD, COORD));
static	BOOL	PointIsOnPath PROTO((OBJECT *, COORD, COORD));
static	BOOL	Intersects PROTO((COORD, COORD, COORD, COORD, COORD));
static	BOOL	AllocatePath PROTO((PATH *, int));
static	void	FindBeginLoop PROTO((PATH *));


/*
 * Initialize the path for an object.
 * This creates an empty path.
 */
void
InitializePath(obj)
	OBJECT *obj;
{
	obj->path.count = 0;
	obj->path.maxcount = 0;
	obj->path.oldcount = 0;
	obj->path.begin = 0;
	obj->path.locs = NULL;
}


/*
 * Free the path for an object.
 * This is used when an object is being deleted.
 */
void
FreePath(obj)
	OBJECT *obj;
{
	if (obj->path.maxcount > 0)
		free((char *) obj->path.locs);

	obj->path.count = 0;
	obj->path.maxcount = 0;
	obj->path.oldcount = 0;
	obj->path.begin = 0;
	obj->path.locs = NULL;
}


/*
 * Copy a path from one object to another.
 */
void
CopyPath(srcobj, destobj)
	OBJECT *srcobj;
	OBJECT *destobj;
{
	PATH *	srcpath;
	PATH *	destpath;

	if (srcobj == destobj)
		return;

	srcpath = &srcobj->path;
	destpath = &destobj->path;

	if (destpath->count > 0)
		destobj->update |= U_REDRAW;

	if (srcpath->count > 0)
		destobj->update |= U_ALL;

	/*
	 * Grow the destination path if necessary to hold the source path.
	 */
	if (srcpath->count > destpath->maxcount) {
		if (!AllocatePath(destpath, srcpath->count))
			Error("No memory to copy path");
	}

	/*
	 * Now copy the path coordinates.
	 */
	destpath->count = srcpath->count;
	destpath->begin = srcpath->begin;
	destpath->oldcount = srcpath->oldcount;

	if (srcpath->count > 0) {
		memcpy((char *) destpath->locs, (char *) srcpath->locs,
			(srcpath->count + 1) * sizeof(LOC));
	}
}


/*
 * Clear the current path from an object.
 * This resets it to the empty state.
 * The number of points the path had is saved for later restoring.
 */
void
ClearPath(obj)
	OBJECT *obj;
{
	PATH *	path;

	path = &obj->path;

	if (path->count > 0)
		obj->update |= U_REDRAW;

	path->oldcount = path->count;
	path->count = 0;
	path->begin = 0;
}


/*
 * Add a new coordinate to the path for an object.
 * Returns FALSE if the coordinate cannot be added.
 * This appends to the latest path loop, or starts a new one.
 * This clears the old path count so that no restoration of the old
 * path is possible anymore.
 */
BOOL
AddPointToPath(obj, row, col)
	OBJECT *obj;
	COORD	row;
	COORD	col;
{
	PATH *	path;
	int	count;

	path = &obj->path;
	count = path->count;

	path->oldcount = 0;

	if ((count > 0) && (path->locs[count - 1].row == row) &&
		(path->locs[count - 1].col == col))
	{
		return TRUE;
	}

	/*
	 * Make sure there is enough room for this new point.
	 */
	if (!AllocatePath(path, count + 1))
		return FALSE;

	/*
	 * Set the bounds if this is the first point in the path.
	 */
	if (count == 0) {
		path->minrow = row;
		path->maxrow = row;
		path->mincol = col;
		path->maxcol = col;
	}

	/*
	 * Increase the bounds to include this point.
	 */
	if (path->minrow > row)
		path->minrow = row;

	if (path->maxrow < row)
		path->maxrow = row;

	if (path->mincol > col)
		path->mincol = col;

	if (path->maxcol < col)
		path->maxcol = col;

	/*
	 * Append this new point to the end of the path.
	 */
	path->locs[count].row = row;
	path->locs[count].col = col;
	path->locs[count].flag = FALSE;
	path->count++;

	/*
	 * Duplicate the beginning coordinate of this loop at the end to
	 * implicitly close it.  The user can explicitly close it later if
	 * desired in order to start another disjoint piece of the path.
	 */
	path->locs[path->count] = path->locs[path->begin];

	obj->update |= U_ALL;

	return TRUE;
}


/*
 * Close the current path so as to let another disjoint piece be added.
 * This connects the current end of the path back to the starting point.
 * This is always possible because the append to point routine leaves
 * one extra point available for this purpose.
 */
void
ClosePath(obj)
	OBJECT *obj;
{
	PATH *	path;
	int	count;

	path = &obj->path;
	count = path->count;

	/*
	 * If there is no path or if the path is already closed then
	 * do nothing.
	 */
	if ((count <= 0) || path->locs[count - 1].flag)
		return;

	/*
	 * Copy the beginning point of the latest loop to the end to close it.
	 * Remember the beginning of the next piece of the loop to come.
	 */
	path->locs[count] = path->locs[path->begin];
	path->locs[count].flag = TRUE;
	path->count++;

	path->begin = path->count;

	obj->update |= U_ALL;
}


/*
 * Shorten the current path by the specified number of segments.
 */
void
ShortenPath(obj, count)
	OBJECT *obj;
	COUNT	count;
{
	PATH *	path;

	path = &obj->path;

	if (count <= 0)
		return;

	if (count >= path->count) {
		ClearPath(obj);

		return;
	}

	path->count -= count;
	path->oldcount = 0;

	FindBeginLoop(path);

	obj->update |= U_REDRAW;
}


/*
 * Restore the path that was just cleared.
 * This cannot be done once a new path has been started.
 * Returns TRUE if there was an old path that was restored.
 */
BOOL
RestorePath(obj)
	OBJECT *obj;
{
	PATH *	path;

	path = &obj->path;

	if (path->oldcount == 0)
		return FALSE;

	path->count = path->oldcount;
	path->oldcount = 0;

	FindBeginLoop(path);

	obj->update |= U_ALL;

	return TRUE;
}


/*
 * Recalculate the beginning of the last loop in a path.
 */
static void
FindBeginLoop(path)
	PATH *	path;
{
	int	i;

	path->begin = 0;

	for (i = 0; i < path->count; i++) {
		if (path->locs[i].flag)
			path->begin = i + 1;
	}

	/*
	 * Duplicate the beginning of the loop at the end to implicitly
	 * close the path.
	 */
	path->locs[path->count] = path->locs[path->begin];
}


/*
 * Mark all cells which are contained within the path for the object.
 * Returns FALSE if there are no cells within the path.
 */
BOOL
MarkPath(obj, mark)
	OBJECT *obj;
	MARK	mark;
{
	PATH *	path;
	ROW *	rp;
	CELL *	cp;
	BOOL	status;

	path = &obj->path;

	if (path->count <= 0)
		return FALSE;

	mark |= MARK_ANY;

	status = FALSE;

	for (rp = obj->firstrow; rp != termrow; rp = rp->next) {
		if (rp->row < path->minrow)
			continue;

		if (rp->row > path->maxrow)
			return status;

		for (cp = rp->firstcell; cp != termcell; cp = cp->next) {
			if (cp->col < path->mincol)
				continue;

			if (cp->col > path->maxcol)
				break;

			if (PointIsInsidePath(obj, rp->row, cp->col)) {
				cp->marks |= mark;
				status = TRUE;
			}
		}
	}

	return status;
}


/*
 * Test to see whether a point is within the path defined for an object.
 * Returns TRUE if the point is either on the boundary of the path, or is
 * inside the closed path.
 */
static BOOL
PointIsInsidePath(obj, row, col)
	OBJECT *obj;
	COORD	row;
	COORD	col;
{
	PATH *	path;
	int	i;
	int	count;

	path = &obj->path;

	if (path->count <= 0)
		return FALSE;

	if ((row < path->minrow) || (row > path->maxrow))
		return FALSE;

	if ((col < path->mincol) || (col > path->maxcol))
		return FALSE;

	if (PointIsOnPath(obj, row, col))
		return TRUE;

	/*
	 * Count the number of intersections of the line segment from the
	 * left edge of the path's boundaries to the coordinate of interest.
	 * The point is inside the path if the total number of intersections
	 * is odd.  For the purpose of this call we translate all the points
	 * by the leftmost edge's coordinate.
	 */
	count = 0;

	for (i = 0; i < path->count; i++) {
		if (path->locs[i].flag)
			continue;

		if (Intersects((path->locs[i].row - row) * 15 + 1,
			path->locs[i].col - path->mincol,
			(path->locs[i + 1].row - row) * 15 + 1,
			path->locs[i + 1].col - path->mincol,
			col - path->mincol))
		{
			count++;
		}
	}

	return ((count & 0x01) != 0);
}


/*
 * Determine if the horizontal line segment whose endpoints are at (0,0) and
 * (ix, 0) intersects the line segment with endpoints (x1, y1) and (x2, y2).
 * Returns TRUE if so.  This routine knows that the input coordinates are
 * somewhat constrained (e.g., x1 and x2 are non-negative).
 */
static BOOL
Intersects(y1, x1, y2, x2, ix)
	COORD	y1;
	COORD	x1;
	COORD	y2;
	COORD	x2;
	COORD	ix;
{
	COORD	tmp;

	if ((y1 < 0) && (y2 < 0))
		return FALSE;

	if ((y1 > 0) && (y2 > 0))
		return FALSE;

	if ((x1 > ix) && (x2 > ix))
		return FALSE;

	if ((x1 <= ix) && (x2 <= ix))
		return TRUE;

	if (y1 < y2) {
		tmp = x1; x1 = x2; x2 = tmp;
		tmp = y1; y1 = y2; y2 = tmp;
	}

	return (ix - x1) * (y2 - y1) <= y1 * (x1 - x2);
}


/*
 * Test to see whether a point is exactly on the path defined for an object.
 * Returns TRUE if the point is exactly on any of the path's line segments.
 */
static BOOL
PointIsOnPath(obj, row, col)
	OBJECT *obj;
	COORD	row;
	COORD	col;
{
	PATH *	path;
	COORD	x1;
	COORD	x2;
	COORD	y1;
	COORD	y2;
	COORD	tx;
	COORD	ty;
	int	i;

	path = &obj->path;

	if ((row < path->minrow) || (row > path->maxrow))
		return FALSE;

	if ((col < path->mincol) || (col > path->maxcol))
		return FALSE;

	for (i = 0; i < path->count; i++)
	{
		if (path->locs[i].flag)
			continue;

		/*
		 * Get the coordinates of the endpoints of the next segment,
		 * and of the point to be tested.
		 */
		x1 = path->locs[i].col;
		y1 = path->locs[i].row;
		x2 = path->locs[i+1].col;
		y2 = path->locs[i+1].row;

		tx = col;
		ty = row;

		/*
		 * Check if the test point is outside of the rectangular
		 * region whose opposite corners are defined by the
		 * endpoints of the segment.  If so, then the point can't
		 * be on the segment.
		 */
		if ((tx < x1) && (tx < x2))
			continue;

		if ((tx > x1) && (tx > x2))
			continue;

		if ((ty < y1) && (ty < y2))
			continue;

		if ((ty > y1) && (ty > y2))
			continue;

		/*
		 * Now check for horizontal or vertical segments.
		 * We easily know the answer for either of these cases.
		 */
		if (x1 == x2) {
			if (tx == x1)
				return TRUE;

			continue;
		}

		if (y1 == y2) {
			if (ty == y1)
				return TRUE;
			continue;
		}

		/*
		 * The segment is sloping and the test point is near it.
		 * Shift the first point down to the origin to make the
		 * calculations easier, and also to help prevent overflows
		 * of the multiplication.
		 */
		tx -= x1;
		x2 -= x1;
		ty -= y1;
		y2 -= y1;
		x1 = 0;
		y1 = 0;

		/*
		 * Now we can compare slopes easily to get the answer:
		 *	y2/x2 = ty/tx
		 */
		if (tx * y2 == ty * x2)
			return TRUE;
	}

	return FALSE;
}


/*
 * Allocate or grow a path array to hold the specified number of elements.
 * This allocates a few extra elements for extra growth later, and also
 * reserves one more point for the implicit closure of the path.
 * Returns TRUE if successful.
 */
static BOOL
AllocatePath(path, count)
	PATH *	path;
	int	count;
{
	LOC *	newlocs;
	int	newlen;

	if (count <= path->maxcount)
		return TRUE;

	newlen = (count + PATHALLOC + 1) * sizeof(LOC);

	if (path->maxcount > 0)
		newlocs = (LOC *) realloc((char *) path->locs, newlen);
	else
		newlocs = (LOC *) malloc(newlen);

	if (newlocs == NULL)
		return FALSE;

	path->locs = newlocs;
	path->maxcount = count + PATHALLOC;

	return TRUE;
}

/* END CODE */
