/*
 * Copyright (c) 1995 David I. Bell
 * Permission is granted to use, distribute, or modify this source,
 * provided that this copyright notice remains intact.
 */

#include "life.h"


/*
 * Row deltas to walk a cell
 */
static	COUNT	rowwalk[24] = {
	-1, 0, 1, 1, 0, 0, -1, -1,
	-1, 0, 0, 0, 1, 1, 1, 1,
	0, 0, 0, 0, -1, -1, -1, -1
};


/*
 * Col deltas to walk a cell
 */
static	COUNT	colwalk[24] = {
	0, 1, 0, 0, -1, -1, 0, 0,
	0, 1, 1, 1, 0, 0, 0, 0,
	-1, -1, -1, -1, 0, 0, 0, 0
};


static	void	MarkLoop PROTO((OBJECT *, COORD, COORD, VALUE, MARK));


/*
 * Mark the object at a given location as specified.  Returns FALSE if
 * no object exists there.  An object for this purpose is considered as a
 * king-wise connected set of live cells, or a set of cells separated by
 * gaps of only one squares.
 */
BOOL
MarkObject(obj, row, col, gap, mark)
	OBJECT *obj;
	COORD	row;
	COORD	col;
	VALUE	gap;
	MARK	mark;
{
	CELL *	cp;

	cp = FindCell(obj, row, col);

	if (cp == NULL)
		return FALSE;

	cp->marks |= mark;

	MarkLoop(obj, row, col, gap, mark);

	return TRUE;
}


/*
 * Recursive subroutine called from MarkObject.
 */
static void
MarkLoop(obj, row, col, gap, mark)
	OBJECT *obj;		/* object begin marked */
	COORD	row;		/* current row */
	COORD	col;		/* current column */
	VALUE	gap;		/* gap for marking (0 or 1) */
	MARK	mark;		/* marking value */
{
	CELL *	cp;		/* current cell */
	LOC *	lp;		/* pointer into list table */
	int	i;		/* to iterate over directions */
	int	steps;		/* number of steps */
	LOC	rclist[24];	/* row and column list */

	if (gap)
		steps = 24;
	else
		steps = 8;

	while (TRUE) {
		if (stop)
			return;

		lp = rclist;

		for (i = 0; i < steps; i++) {	/* find neighbors */
			row += rowwalk[i];
			col += colwalk[i];

			cp = FindCell(obj, row, col);

			if (cp == NULL)
				continue;

			if (cp->marks & mark)
				continue;

			cp->marks |= mark;
			lp->row = row;
			lp->col = col;
			lp++;
		}

		if (--lp != rclist) {		/* recurse if more than one */
			for (; lp >= rclist; lp--) {
				MarkLoop(obj, lp->row, lp->col, gap, mark);
			}

			return;
		}

		row = lp->row;		/* else follow single cell */
		col = lp->col;
	}
}


/*
 * Mark a whole object as specified.
 */
void
SetMarks(obj, mark)
	OBJECT *obj;
	MARK	mark;
{
	ROW *	rp;
	CELL *	cp;

	mark |= MARK_ANY;

	for (rp = obj->firstrow; rp != termrow; rp = rp->next) {
		for (cp = rp->firstcell; cp != termcell; cp = cp->next) {
			cp->marks |= mark;
		}
	}
}


/*
 * Copy the marks from one type to another for an object.
 * Returns FALSE if there were no cells with the given mark.
 */
BOOL
CopyMarks(obj, srcmark, destmark)
	OBJECT *obj;
	MARK	srcmark;
	MARK	destmark;
{
	ROW *	rp;
	CELL *	cp;
	BOOL	status;

	status = FALSE;

	for (rp = obj->firstrow; rp != termrow; rp = rp->next) {
		for (cp = rp->firstcell; cp != termcell; cp = cp->next) {
			if ((cp->marks & srcmark) == 0)
				continue;

			cp->marks |= destmark;
			status = TRUE;
		}
	}

	return status;
}


/*
 * Invert the marks for a whole object.
 */
void
InvertMarks(obj, mark)
	OBJECT *obj;
	MARK	mark;
{
	ROW *	rp;
	CELL *	cp;

	for (rp = obj->firstrow; rp != termrow; rp = rp->next) {
		for (cp = rp->firstcell; cp != termcell; cp = cp->next) {
			cp->marks ^= mark;
		}
	}
}


/*
 * Clear marks for a whole object as specified.
 */
void
ClearMarks(obj, mark)
	OBJECT *obj;
	MARK	mark;
{
	ROW *	rp;
	CELL *	cp;

	mark = ~mark;
	mark |= MARK_ANY;

	for (rp = obj->firstrow; rp != termrow; rp = rp->next) {
		for (cp = rp->firstcell; cp != termcell; cp = cp->next) {
			cp->marks &= mark;
		}
	}
}


/*
 * Mark the cells in a specified rectangular region as desired.
 * Returns the number of cells which were marked.
 */
COUNT
MarkRegion(obj, mark, minrow, maxrow, mincol, maxcol)
	OBJECT *obj;
	MARK	mark;
	COORD	minrow;
	COORD	maxrow;
	COORD	mincol;
	COORD	maxcol;
{
	ROW *	rp;
	CELL *	cp;
	COUNT	count;

	count = 0;

	for (rp = obj->firstrow; rp != termrow; rp = rp->next) {
		if (rp->row < minrow)
			continue;

		if (rp->row > maxrow)
			break;

		for (cp = rp->firstcell; cp != termcell; cp = cp->next) {
			if (cp->col < mincol)
				continue;

			if (cp->col > maxcol)
				break;

			cp->marks |= mark;
			count++;
		}
	}

	return count;
}


/*
 * Find the range of all marked cells for an object.  Returns FALSE if
 * no cells were marked.
 */
BOOL
MarkMinMax(obj, mark, minrow, maxrow, mincol, maxcol)
	OBJECT *obj;
	MARK	mark;
	COORD *	minrow;
	COORD *	maxrow;
	COORD *	mincol;
	COORD *	maxcol;
{
	ROW *	rp;
	CELL *	cp;
	COORD	row;
	COORD	minr;
	COORD	maxr;
	COORD	minc;
	COORD	maxc;

	minr = INFINITY;
	maxr = -INFINITY;
	minc = INFINITY;
	maxc = -INFINITY;

	for (rp = obj->firstrow; rp != termrow; rp = rp->next) {
		row = rp->row;

		for (cp = rp->firstcell; cp != termcell; cp = cp->next) {
			if ((cp->marks & mark) == 0)
				continue;

			if (row < minr)
				minr = row;

			if (row > maxr)
				maxr = row;

			if (cp->col < minc)
				minc = cp->col;

			if (cp->col > maxc)
				maxc = cp->col;
		}
	}

	if (minr > maxr)
		return FALSE;

	*minrow = minr;
	*maxrow = maxr;
	*mincol = minc;
	*maxcol = maxc;

	return TRUE;
}


/*
 * Count the number of marked cells in an object
 */
COUNT
CountMarks(obj, mark)
	OBJECT *obj;
	MARK	mark;
{
	ROW *	rp;
	CELL *	cp;
	COUNT	count;

	count = 0;

	for (rp = obj->firstrow; rp != termrow; rp = rp->next) {
		for (cp = rp->firstcell; cp != termcell; cp = cp->next) {
			if (cp->marks & mark)
				count++;
		}
	}

	return count;
}


/*
 * Move marked cells to another object.  If the destination object is NULL
 * the cells are just deleted.  The previous cells of the destination object
 * are deleted.
 */
void
MoveMarkedObject(sobj, dobj, mark)
	OBJECT *sobj;
	OBJECT *dobj;
	MARK	mark;
{
	ROW *	rp;
	CELL *	cp;		/* current cell pointer */
	CELL *	pcp;		/* previous cell pointer */
	CELL *	ncp;		/* next cell pointer */

	if (sobj == dobj)
		Error("Moving object to itself");

	if (dobj) {
		ZeroObject(dobj);

		dobj->currow = sobj->currow;
		dobj->curcol = sobj->curcol;
	}

	for (rp = sobj->firstrow; rp != termrow; rp = rp->next) {
		pcp = NULL;
		cp = rp->firstcell;

		while (cp != termcell) {
			if ((cp->marks & mark) == 0) {
				pcp = cp;
				cp = cp->next;

				continue;
			}

			if (dobj)
				AddCell(dobj, rp->row, cp->col);

			ncp = cp->next;

			if (pcp == NULL)
				rp->firstcell = ncp;
			else
				pcp->next = ncp;

			if (ncp == termcell)
				rp->lastcell = pcp;

			cp->next = freecells;
			freecells = cp;
			cp = ncp;
			rp->count--;
			sobj->count--;
		}
	}
}


/*
 * Copy marked cells to another object. The previous cells of the destination
 * object are deleted.  The source object is unaffected.
 */
void
CopyMarkedObject(sobj, dobj, mark)
	OBJECT *sobj;
	OBJECT *dobj;
	MARK	mark;
{
	ROW *	rp;
	CELL *	cp;

	if (sobj == dobj)
		Error("Copying object to itself");

	ZeroObject(dobj);

	dobj->currow = sobj->currow;
	dobj->curcol = sobj->curcol;

	for (rp = sobj->firstrow; rp != termrow; rp = rp->next) {
		for (cp = rp->firstcell; cp != termcell; cp = cp->next) {
			if (cp->marks & mark)
				AddCell(dobj, rp->row, cp->col);
		}
	}
}


/*
 * Rotate the marked cells in the given object around the current cursor
 * location.  This deletes the marked cells, and reinserts them after
 * their position has been rotated by 90 degrees clockwise.
 */
void
RotateMarkedObject(obj, mark)
	OBJECT *obj;
	MARK	mark;
{
	ROW *	rp;
	CELL *	cp;
	COORD	row;
	COORD	col;

	if (obj == tempobject)
		Error("Using temp object");

	MoveMarkedObject(obj, tempobject, mark);

	for (rp = tempobject->firstrow; rp != termrow; rp = rp->next) {
		row = rp->row - obj->currow;

		for (cp = rp->firstcell; cp != termcell; cp = cp->next) {
			col = cp->col - obj->curcol;
			AddCell(obj, obj->currow + col, obj->curcol - row);
		}
	}
}


/*
 * Flip the marked cells in the given object around the column of the current
 * cursor location.  This deletes the marked cells, and reinserts them after
 * their position has been flipped around the vertical axis.
 */
void
FlipColumnMarkedObject(obj, mark)
	OBJECT *obj;
	MARK	mark;
{
	ROW *	rp;
	CELL *	cp;
	COORD	row;
	COORD	col;

	if (obj == tempobject)
		Error("Using temp object");

	MoveMarkedObject(obj, tempobject, mark);

	for (rp = tempobject->firstrow; rp != termrow; rp = rp->next) {
		row = rp->row;

		for (cp = rp->firstcell; cp != termcell; cp = cp->next) {
			col = cp->col - obj->curcol;
			AddCell(obj, row, obj->curcol - col);
		}
	}
}


/*
 * Flip the marked cells in the given object around the row of the current
 * cursor location.  This deletes the marked cells, and reinserts them after
 * their position has been flipped around the horizontal axis.
 */
void
FlipRowMarkedObject(obj, mark)
	OBJECT *obj;
	MARK	mark;
{
	ROW *	rp;
	CELL *	cp;
	COORD	row;

	if (obj == tempobject)
		Error("Using temp object");

	MoveMarkedObject(obj, tempobject, mark);

	for (rp = tempobject->firstrow; rp != termrow; rp = rp->next) {
		row = 2 * obj->currow - rp->row;

		for (cp = rp->firstcell; cp != termcell; cp = cp->next)
			AddCell(obj, row, cp->col);
	}
}

/* END CODE */
