/*
 * Copyright (c) 1995 David I. Bell
 * Permission is granted to use, distribute, or modify this source,
 * provided that this copyright notice remains intact.
 *
 * Program to play Conway's game of LIFE on an infinite board.
 */

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <setjmp.h>
#include <signal.h>
#include <string.h>


#ifdef	__STDC__
#define	PROTO(a)	a
#else
#define	PROTO(a)	()
#endif


/*
 * Special characters.
 */
#define	ESC	'\033'
#define	FF	'\014'
#define	DEL	'\177'


#define	isblank(ch)	(((ch) == ' ') || ((ch) == '\t'))
#define	isdigit(ch)	(((ch) >= '0') && ((ch) <= '9'))
#define	islower(ch)	(((ch) >= 'a') && ((ch) <= 'z'))
#define	isupper(ch)	(((ch) >= 'A') && ((ch) <= 'Z'))
#define	isletter(ch)	(islower(ch) || isupper(ch))


#ifndef	LIFELIB
#define	LIFELIB	"/usr/games/lib/life"	/* directory for general life library */
#endif

#define	LIFEOPTS	"LIFEOPTS"	/* environment name for options */
#define	LIFEEXT		".l"		/* extension for life files */
#define	MAXPATH		1024		/* maximum size of path names */
#define	MAXFILE		18		/* maximum size of file name for list */
#define	MAXLIST		5		/* maximum depth for listing files */
#define	MAXLIBS		10		/* maximum number of user libraries */
#define	ALLOCOBJ	10		/* how many new objects to allocate */
#define	ALLOCROW	100		/* how many new rows to allocate */
#define	ALLOCCELL	1000		/* how many new cells to allocate */
#define	LISTARGSIZE	200		/* incremental arg size for lists */
#define	LISTBUFSIZE	2000		/* incremental buffer size for lists */
#define	STRINGSIZE	(1024*4)	/* temp string buffer size */
#define	LOOPSIZE	5000		/* characters in command loops */
#define	SCAN_SIZE	200		/* maximum command length */
#define	MAXARGS		10		/* maximum command line arguments */
#define	MAXINPUT	20		/* maximum nesting of command inputs */
#define	MAXNAME		32		/* maximum object name size */
#define	MAXSCALE	1000		/* maximum scale factor */
#define	WRITEROWS	100		/* default maximum rows for writing */
#define	WRITECOLS	79		/* default maximum cols for writing */
#define	INFINITY	0x7fffffff	/* infinite value */
#define	LIFE		9		/* value for live cell */
#define	RULESIZE	18		/* size of rules array */
#define	DEFAULTRULE	"3,23"		/* default rule string */

#define	STDIN		0		/* standard input */
#define	STDOUT		1		/* standard output */
#define	STDERR		2		/* standard error */

#define	RELATIVE	0		/* do object additions relatively */
#define	ABSOLUTE	1		/* do object additions absolutely */

#define	M_MOVE		0		/* movement mode */
#define	M_INSERT	1		/* insertion mode */
#define	M_DELETE	2		/* deletion mode */


/*
 * Mark bit values.
 */
#define	MARK_ANY	((MARK) 0x1)	/* mark always set */
#define	MARK_USR	((MARK) 0x2)	/* marked due to user specification */
#define	MARK_CMD	((MARK) 0x4)	/* marked only for current command */
#define	MARK_SRC	((MARK) 0x8)	/* marked by searching */
#define	MARK_SEE	MARK_USR	/* marks seen by user */
#define	MARK_ALL	((MARK) (MARK_ANY|MARK_USR|MARK_CMD|MARK_SRC))


#define	INP_TTY		0		/* input is terminal */
#define	INP_FILE	1		/* input is file */
#define	INP_LOOP	2		/* input is command loop */
#define	INP_MACRO	3		/* input is command macro */

#define	SCAN_ABORT	1		/* setjmp value for aborted command */
#define	SCAN_EDIT	2		/* setjmp value for edited command */
#define	SCAN_EOF	3		/* setjmp value for no command data */

#define	NULL_CMD	0xff		/* null command character */


/*
 * Flags for updating of display.
 */
#define	U_POS		((FLAGS) 0x01)	/* update cursor position */
#define	U_STAT		((FLAGS) 0x02)	/* update status line */
#define	U_VIEW		((FLAGS) 0x04)	/* update view of cells */
#define	U_ALL		((FLAGS) (U_POS | U_STAT | U_VIEW))	/* most */
#define	U_CLEAR		((FLAGS) 0x08)	/* clear screen before drawing */
#define	U_REDRAW	((FLAGS) (U_ALL | U_CLEAR))	/* refresh too */


#ifndef	TRUE
#define	TRUE		((BOOL) 1)	/* booleans */
#define	FALSE		((BOOL) 0)
#endif


/*
 * Macro to detect reserved names.
 */
#define	BADNAME(s)	((s[0] == '.') && ((s[1] == '\0') || (s[1] == '.')))


/*
 * Basic typedefs
 */
typedef	long		COORD;		/* coordinate of a cell */
typedef	long		COUNT;		/* counts of things */
typedef	long		VALUE;		/* value of an expression */
typedef	unsigned int	MARK;		/* mark value */
typedef	int		BOOL;		/* boolean value */
typedef	int		SCALE;		/* scale factor */
typedef	unsigned char	UCHAR;		/* for character handling */
typedef	unsigned int	FLAGS;		/* flag values */


/*
 * Structure to hold row and column pairs and an extra boolean value.
 */
typedef struct	{
	COORD	row;
	COORD	col;
	BOOL	flag;
} LOC;


/*
 * Structure holding information about a path.
 * Paths are used to manipulate selected regious of an object.
 * A path can consist of a single loop, or else multiple disjoint loops.
 * An extra point is assumed at the end of the last loop to close it.
 */
typedef	struct	{
	int	count;			/* number of points in path */
	int	maxcount;		/* maximum points storable */
	int	oldcount;		/* number of points before clearing */
	int	begin;			/* beginning offset of latest path */
	COORD	minrow;			/* bounds of the whole path */
	COORD	maxrow;
	COORD	mincol;
	COORD	maxcol;
	LOC *	locs;			/* table of coordinates */
} PATH;


/*
 * Structure for each cell.
 */
typedef	struct	cell	CELL;
struct	cell	{
	CELL *	next;			/* link to next cell */
	COORD	col;			/* column number */
	MARK	marks;			/* marking values for cell */
};


/*
 * Structure for each row of cells.
 */
typedef	struct	row	ROW;
struct	row	{
	ROW *	next;			/* link to next row */
	CELL *	firstcell;		/* link to first cell */
	CELL *	lastcell;		/* link to last real cell */
	COORD	row;			/* row number */
	COUNT	count;			/* number of cells in this row */
};


/*
 * Structure for each object.
 */
typedef	struct	object	OBJECT;
struct	object	{
	OBJECT *next;			/* next object */
	ROW *	firstrow;		/* first row */
	ROW *	lastrow;		/* last row */
	COUNT	count;			/* number of live cells */
	COUNT	born;			/* number of cells born */
	COUNT	died;			/* number of cells died */
	COORD	currow;			/* current row */
	COORD	curcol;			/* current column */
	COORD	minrow;			/* minimum row seen in window */
	COORD	maxrow;			/* maximum row seen in window */
	COORD	mincol;			/* minimum column seen in window */
	COORD	maxcol;			/* maximum column seen in window */
	COUNT	viewrows;		/* how many rows can be seen */
	COUNT	viewcols;		/* how many columns can be seen */
	COORD	origrow;		/* row of origin */
	COORD	origcol;		/* column of origin */
	FLAGS	update;			/* update flags for displaying */
	MARK	mark;			/* mark value for new cells */
	COUNT	gen;			/* current generation */
	SCALE	scale;			/* current scaling factor for view */
	COUNT	frequency;		/* frequency of output for object */
	BOOL	autoscale;		/* doing autoscaling */
	BOOL	reserved;		/* reserved object */
	PATH	path;			/* path defined within object */
	LOC	wherelocs[26];		/* locations where remembered */
	char	name[MAXNAME + 1];	/* name of object */
};


/*
 * The following structure holds all data necessary for processing
 * characters from some source.
 */
typedef	struct	input	INPUT;
struct	input	{
	int	(*getchar) PROTO((INPUT *));	/* read next character */
	void	(*term) PROTO((INPUT *));	/* terminate reading */
	int	type;		/* type of input */

				/* following for file reading only */
	FILE *	file;		/* file handle */
	OBJECT *obj;		/* object to restore on reentry */
	COORD	row;		/* row to restore */
	COORD	col;		/* column to restore */
	COORD	origrow;	/* origin row to restore */
	COORD	origcol;	/* origin column to restore */

				/* following for loop or macro reading only */
	UCHAR *	begptr;		/* beginning of command data */
	UCHAR *	endptr;		/* end of command data */
	UCHAR *	curptr;		/* current character */
	COUNT	curval;		/* current iteration value */
	COUNT	endval;		/* ending iteration value */
	BOOL	first;		/* processing new chars */
	char	macro;		/* macro being defined */
};


/*
 * Structure to interface to various input/output devices.
 * This supports both normal terminals and graphics output.
 */
typedef	struct	DEV	DEV;

struct	DEV	{
	BOOL	(*open) PROTO((DEV *));		/* open device */
	void	(*close) PROTO((DEV *));	/* close device */
	void	(*update) PROTO((DEV *, BOOL));	/* make sure display is up to date */
	void	(*refresh) PROTO((DEV *));	/* redraw whole display from scratch */
	BOOL	(*inputready) PROTO((DEV *));	/* check whether input is ready */
	int	(*readchar) PROTO((DEV *, BOOL));	/* read character with optional wait */
	void	(*movecursor) PROTO((DEV *, SCALE, COORD, COORD));	/* move position of cursor */
	void	(*showview) PROTO((DEV *, OBJECT *));	/* show view of life cells */
	void	(*showstatus) PROTO((DEV *, char *));	/* show status line */
	void	(*addstatus) PROTO((DEV *, char *));	/* add to status line */
	void	(*showhelp) PROTO((DEV *));	/* show help information */
	void	(*addhelp) PROTO((DEV *, char *));	/* add information to help */
	BOOL	(*assign) PROTO((DEV *, VALUE, UCHAR));	/* assign button to macro */
	void	(*writeassign) PROTO((DEV *, FILE *));	/* write assignments */	

	COUNT	rows;			/* rows available for life cells */
	COUNT	cols;			/* columns available for life cells */
	COUNT	textrows;		/* rows of text available for help */
	COUNT	textcols;		/* columns of text available for help */
	SCALE	minscale;		/* minimum legal scale value */
	SCALE	maxscale;		/* maximum legal scale value */
	SCALE	defaultscale;		/* desirable default scale */
};


/*
 * Structure to hold a list of strings allocated from a common buffer.
 * This is used for collecting file names for listing.
 */
typedef	struct	{
	char **	argv;			/* pointer to list of pointers */
	char *	buf;			/* pointer to string storage */
	int	argc;			/* number of pointers */
	int	maxargc;		/* maximum number of pointers */
	int	used;			/* chars used in string storage */
	int	maxused;		/* maximum chars in string storage */
} LIST;


/*
 * Extern definitions for non-initialized data.
 * These are defined externally in all modules, except in main.
 */
#ifdef	DEFINE_GLOBALS
#define	EXTERN
#else
#define	EXTERN	extern
#endif


EXTERN	DEV *	dev;		/* device being used */
EXTERN	OBJECT *objects;	/* list of active objects */
EXTERN	OBJECT *curobj;		/* currently selected object */
EXTERN	OBJECT *prevobj;	/* previously selected object */
EXTERN	OBJECT *mainobject;	/* the main object */
EXTERN	OBJECT *deleteobject;	/* object last deleted */
EXTERN	OBJECT *backupobject;	/* backup object */
EXTERN	OBJECT *tempobject;	/* temporary object */
EXTERN	OBJECT *freeobjects;	/* list of free objects */
EXTERN	OBJECT *newobjects;	/* top of new object allocation */
EXTERN	OBJECT *endobjects;	/* end of new objects */

EXTERN	ROW *	freerows;	/* list of free row structures */
EXTERN	ROW *	newrows;	/* top of new row allocation */
EXTERN	ROW *	endrows;	/* end of new rows */
EXTERN	ROW *	termrow;	/* terminus row */
EXTERN	ROW	initrow;	/* row to initialize list */

EXTERN	CELL *	freecells;	/* list of free cell structures */
EXTERN	CELL *	newcells;	/* top of new cell allocation */
EXTERN	CELL *	endcells;	/* end of new cells */
EXTERN	CELL *	termcell;	/* terminus cell */
EXTERN	CELL	initcell;	/* cell to initialize list */

EXTERN	INPUT *	curinput;	/* current input being read from */
EXTERN	COUNT	seecount;	/* number of cells we can see */
EXTERN	COUNT	freqcount;	/* current count */
EXTERN	COUNT	genleft;	/* generations left before stopping */
EXTERN	BOOL	reserve;	/* reserved object names allowed */
EXTERN	BOOL	dowait;		/* must wait for input */
EXTERN	BOOL	interact;	/* still being interactive */
EXTERN	BOOL	stop;		/* user wants to stop current action */
EXTERN	BOOL	intjmpok;	/* ok to use interrupt jump buffer */
EXTERN	SCALE	defaultscale;	/* default scale value for new objects */
EXTERN	COUNT	defaultfrequency;	/* default frequency for new objects */
EXTERN	int	mode;		/* mode of movement */
EXTERN	char	gridchar;	/* character to use for grid */
EXTERN	char *	errorstring;	/* error string to type */
EXTERN	char *	userlibs[MAXLIBS];	/* user's life libraries if any */
EXTERN	int	userlibcount;	/* number of user libraries specified */
EXTERN	jmp_buf	ttyjmp;		/* jump buffer for terminal input */
EXTERN	jmp_buf	intjmp;		/* jump buffer for interrupts */
EXTERN	INPUT	inputs[MAXINPUT];	/* list of input environments */
EXTERN	char	stringbuf[STRINGSIZE];	/* characters for string value */
EXTERN	char	rulestring[20];	/* string describing rule */

EXTERN	char	vkill;		/* line kill character */
EXTERN	char	verase1;	/* first erase character */
EXTERN	char	verase2;	/* second erase character */
EXTERN	char	vwerase;	/* word erase character */
EXTERN	char	veol;		/* end of line character */
EXTERN	char	veof;		/* EOF character */
EXTERN	char	vlnext;		/* literal next character */


/*
 * Externs for data which is initialized.
 */
extern	char	rule[RULESIZE];	/* life rule */
extern	DEV	ttydev;		/* tty device */

#if	defined(X11)
extern	DEV	x11dev;		/* graphics device */
#endif


/*
 * Global procedures.
 */
extern	OBJECT *	AllocateObject PROTO((void));
extern	OBJECT *	FindObject PROTO((char *));
extern	OBJECT *	GetObject PROTO((char *));
extern	ROW *		AllocateRow PROTO((void));
extern	ROW *		FindRow PROTO((OBJECT *, COORD));
extern	ROW *		GetRow PROTO((OBJECT *, COORD));
extern	CELL *		AllocateCell PROTO((void));
extern	CELL *		FindCell PROTO((OBJECT *, COORD, COORD));
extern	BOOL		AddCell PROTO((OBJECT *, COORD, COORD));
extern	BOOL		DeleteCell PROTO((OBJECT *, COORD, COORD));
extern	void		CheckStartupFiles PROTO((void));
extern	char *		ReadString PROTO((char *));
extern	void		ListVariables PROTO((void));
extern	VALUE		GetVariable PROTO((char *));
extern	VALUE		GetVariable1 PROTO((int));
extern	void		SetVariable PROTO((char *, VALUE));
extern	void		SetVariable1 PROTO((int, VALUE));
extern	VALUE		GetExpression PROTO((char *));
extern	void		SetObject PROTO((OBJECT *));
extern	void		ZeroObject PROTO((OBJECT *));
extern	void		DestroyObject PROTO((OBJECT *));
extern	void		MoveObject PROTO((OBJECT *, OBJECT *));
extern	void		AddObject PROTO((OBJECT *, OBJECT *, COUNT, BOOL));
extern	void		CopyObject PROTO((OBJECT *, OBJECT *));
extern	void		ListObjects PROTO((BOOL));
extern	BOOL		MinMax PROTO((OBJECT *, COORD *, COORD *, COORD *,
				COORD *));
extern	BOOL		SearchObject PROTO((OBJECT *, COUNT, BOOL));
extern	BOOL		MarkObject PROTO((OBJECT *, COORD, COORD, VALUE, MARK));
extern	void		SetMarks PROTO((OBJECT *, MARK));
extern	BOOL		CopyMarks PROTO((OBJECT *, MARK, MARK));
extern	void		ClearMarks PROTO((OBJECT *, MARK));
extern	void		InvertMarks PROTO((OBJECT *, MARK));
extern	COUNT		MarkRegion PROTO((OBJECT *, MARK, COORD, COORD,
				COORD, COORD));
extern	BOOL		MarkPath PROTO((OBJECT *, MARK));
extern	BOOL		MarkMinMax PROTO((OBJECT *, MARK, COORD *, COORD *,
				COORD *, COORD *));
extern	COUNT		CountMarks PROTO((OBJECT *, MARK));
extern	void		MoveMarkedObject PROTO((OBJECT *, OBJECT *, MARK));
extern	void		CopyMarkedObject PROTO((OBJECT *, OBJECT *, MARK));
extern	void		RotateMarkedObject PROTO((OBJECT *, MARK));
extern	void		FlipRowMarkedObject PROTO((OBJECT *, MARK));
extern	void		FlipColumnMarkedObject PROTO((OBJECT *, MARK));
extern	BOOL		MacroIsValid PROTO((int));
extern	void		Error PROTO((char *));
extern	int		ReadChar PROTO((void));
extern	BOOL		IsAbbreviation PROTO((char *, char *));
extern	int		ReadLine PROTO((char *, char *, int, int));
extern	BOOL		ShowHelp PROTO((char *));
extern	void		EndHelp PROTO((void));
extern	void		ReadRle PROTO((FILE *, OBJECT *));
extern	void		ReadXlife PROTO((FILE *, OBJECT *));
extern	void		ReadPicture PROTO((FILE *, OBJECT *));
extern	void		WriteObject PROTO((OBJECT *, char *, char *, VALUE,
				VALUE));
extern	void		WriteMacros PROTO((FILE *));
extern	void		DoGeneration PROTO((OBJECT *));
extern	void		DoLineCommand PROTO((VALUE, VALUE, BOOL, BOOL));
extern	void		DoCommand PROTO((void));
extern	void		Backup PROTO((void));
extern	void		CheckRun PROTO((void));
extern	void		GetFiles PROTO((LIST *, char *, char *, int, int));
extern	void		ListFiles PROTO((LIST *));
extern	void		SetScale PROTO((OBJECT *, SCALE));
extern	SCALE		AutoScale PROTO((void));
extern	void		PositionView PROTO((COORD, COORD, COORD, COORD));
extern	void		UpdateView PROTO((void));
extern	void		SetRule PROTO((char *));
extern	void		ViewStatus PROTO((void));

extern	void		ScanInit PROTO((int (*)(), jmp_buf));
extern	int		ScanChar PROTO((void));
extern	void		ScanAbort PROTO((void));
extern	void		ScanEof PROTO((void));
extern	void		ScanReset PROTO((void));
extern	void		Beep PROTO((void));

extern	BOOL		TtyIsInput PROTO((void));
extern	BOOL		SetTty PROTO((void));
extern	BOOL		SetFile PROTO((char *));
extern	BOOL		SetLoop PROTO((VALUE, VALUE, int));
extern	void		EndLoop PROTO((void));
extern	BOOL		SetMacro PROTO((VALUE, VALUE, int));

extern	void		InitializePath PROTO((OBJECT *));
extern	void		FreePath PROTO((OBJECT *));
extern	void		CopyPath PROTO((OBJECT *, OBJECT*));
extern	void		ClearPath PROTO((OBJECT *));
extern	BOOL		RestorePath PROTO((OBJECT *));
extern	BOOL		AddPointToPath PROTO((OBJECT *, COORD, COORD));
extern	void		ClosePath PROTO((OBJECT *));
extern	void		ShortenPath PROTO((OBJECT *, COUNT));

/* END CODE */
