/*
 * Copyright (c) 1995 David I. Bell
 * Permission is granted to use, distribute, or modify this source,
 * provided that this copyright notice remains intact.
 */

#include "life.h"


/*
 * Life rule.
 */
char	rule[RULESIZE];


static	ROW *	ComputeRow PROTO((OBJECT *, ROW *, ROW *, ROW *));


/*
 * Set the rule based on the specified rule string.
 * The rule is of the form Bmmm/Snnn, where mmm are digits specifying
 * neighbors required for birth, and nnn are digits specifying neighbors
 * required for survival.  The B and S are optional, and the slash may
 * also be a comma.  If the rule string is bad, the real rule is not
 * changed.
 */
void
SetRule(cp)
	char *	cp;
{
	char *	rp;
	char	newrule[RULESIZE];

	memset(newrule, 0, RULESIZE);

	if ((*cp == 'B') || (*cp == 'b'))
		cp++;

	while ((*cp >= '0') && (*cp <= '8'))
		newrule[*cp++ - '0'] = 1;

	if ((*cp != ',') && (*cp != '/'))
		Error("Bad born string");

	cp++;

	if ((*cp == 'S') || (*cp == 's'))
		cp++;

	while ((*cp >= '0') && (*cp <= '8'))
		newrule[LIFE + *cp++ - '0'] = 1;

	if (*cp)
		Error("Bad survive string");

	/*
	 * The rule is ok, copy it into the real place, and
	 * generate the rule string in our standard format.
	 */
	memcpy(rule, newrule, RULESIZE);

	cp = rulestring;

	for (rp = rule; rp < &rule[LIFE]; rp++) {
		if (*rp)
			*cp++ = '0' + (rp - rule);
	}

	*cp++ = ',';

	for (rp = &rule[LIFE]; rp < &rule[RULESIZE]; rp++) {
		if (*rp)
			*cp++ = ('0' - LIFE) + (rp - rule);
	}

	*cp = '\0';
}


/*
 * Compute one full generation of the current configuration.
 * This is done by calling ComputeRow with each triple of rows which
 * contain any live cells, and remembering the result.  When all rows
 * are finished, we change the object.
 */
void
DoGeneration(obj)
	OBJECT *obj;
{
	ROW *	crp;		/* current row */
	ROW *	prp;		/* previous row */
	ROW *	nrp;		/* next row */
	ROW *	newrp;		/* current row of new list */
	ROW *	srp;		/* saved row pointer */
	COORD	row;		/* current row number */

	if (genleft <= 0)
		return;

	if ((--genleft == 0) || (--freqcount <= 0))
		obj->update |= U_ALL;

	obj->gen++;
	obj->born = 0;
	obj->died = 0;
	newrp = &initrow;
	prp = termrow;
	crp = termrow;
	nrp = obj->firstrow;
	srp = nrp->next;
	row = nrp->row - 1;

	/*
	 * Loop over each triple of rows.
	 */
	while (TRUE) {
		if (row >= (INFINITY-1))
			break;

		prp = ComputeRow(obj, prp, crp, nrp);

		if (prp) {		/* have something in current row */
			newrp->next = prp;
			newrp = prp;
			newrp->row = row;
			obj->count += prp->count;
		}

		row++;
		prp = crp;
		crp = nrp;
		nrp = srp;

		if (nrp->row == row + 1) {
			srp = nrp->next;
			continue;
		}

		nrp = termrow;

		if ((prp != termrow) || (crp != termrow))
			continue;

		nrp = srp;
		srp = nrp->next;
		row = nrp->row - 1;
	}

	ZeroObject(obj);

	if (newrp == &initrow) {		/* died off */
		genleft = 0;
		obj->update |= U_ALL;

		return;
	}

	obj->firstrow = initrow.next;
	newrp->next = termrow;
	obj->lastrow = newrp;

	if ((obj->born == 0) && (obj->died == 0)) {	/* no change */
		genleft = 0;
		obj->update |= U_ALL;
	}
}


/*
 * Compute the result of three adjacent rows, and return a row structure
 * containing the new middle row, or NULL if no live cells are produced.
 * When determining if a cell is dead or alive, each live neighbor counts
 * as a 1, but the current cell counts as 9 when alive.  Indexing the rule
 * table with the sum then automatically produces the correct result.
 */
static ROW *
ComputeRow(obj, prevrow, currow, nextrow)
	OBJECT *obj;
	ROW *	prevrow;
	ROW *	currow;
	ROW *	nextrow;
{
	CELL *	pcp;		/* head of previous row of cells */
	CELL *	ccp;		/* head of current row of cells */
	CELL *	ncp;		/* head of next row of cells */
	CELL *	tcp;		/* temporary cell */
	CELL *	newcp;		/* new row of cells */
	ROW *	rp;		/* new row pointer */
	COUNT	i;		/* sum of live cells and other uses */
	COORD	col;		/* current column being examined */
	COORD	colp1;		/* one more than column */
	COUNT	count;		/* live cells */

	pcp = prevrow->firstcell;
	ccp = currow->firstcell;
	ncp = nextrow->firstcell;
	newcp = &initcell;
	count = 0;
	col = -INFINITY;

	/*
	 * Loop over the cells of all three rows.
	 */
	while (TRUE) {
		/*
		 * Find next column where a cell exists on any row
		 */
		i = col - 1;

		while (i > pcp->col)
			pcp = pcp->next;

		while (i > ccp->col)
			ccp = ccp->next;

		while (i > ncp->col)
			ncp = ncp->next;

		i = ccp->col;

		if (pcp->col < i)
			i = pcp->col;

		if (ncp->col < i)
			i = ncp->col;

		if (i == INFINITY)
			break;

		i--;

		if (col < i)
			col = i;

		i = 0;
		colp1 = col + 1;

		if (pcp->col <= colp1) {	/* add cells in previous row */
			i++;
			tcp = pcp->next;
			i += (tcp->col <= colp1);
			tcp = tcp->next;
			i += (tcp->col <= colp1);
		}

		if (ccp->col <= colp1) {	/* add cells on our row */
			i++;
			tcp = ccp->next;

			if ((ccp->col == col) || (tcp->col == col)) {
				i += (LIFE - 1);
			}

			i += (tcp->col <= colp1);
			tcp = tcp->next;
			i += (tcp->col <= colp1);
		}

		if (ncp->col <= colp1) {	/* add cells in next row */
			i++;
			tcp = ncp->next;
			i += (tcp->col <= colp1);
			tcp = tcp->next;
			i += (tcp->col <= colp1);
		}

		if (rule[i]) {			/* cell is alive */
			obj->born += (i < LIFE);
			tcp = AllocateCell();
			tcp->col = col;
			newcp->next = tcp;
			newcp = tcp;
			count++;
		} else				/* cell is dead */
			obj->died += (i >= LIFE);

		col++;
	}

	if (newcp == &initcell)
		return NULL;

	newcp->next = termcell;

	rp = AllocateRow();
	rp->firstcell = initcell.next;
	rp->lastcell = newcp;
	rp->count = count;

	return rp;
}

/* END CODE */
