/* $Id: history.c,v 30000.26 1993/05/25 00:57:02 kkeys Exp $ */
/******************************************************************
 * Copyright 1993 by Ken Keys.
 * Permission is granted to obtain and redistribute this code freely.
 * All redistributions must contain this header.  You may modify this
 * software, but any redistributions of modified code must be clearly
 * marked as having been modified.
 ******************************************************************/


/****************************************************************
 * Fugue history and logging                                    *
 *                                                              *
 * Maintains the circular lists for input and output histories. *
 * Handles text queuing and file I/O for logs.                  *
 ****************************************************************/

#include <ctype.h>
#include "port.h"
#include "dstring.h"
#include "tf.h"
#include "util.h"
#include "history.h"
#include "world.h"
#include "socket.h"
#include "expand.h"
#include "output.h"
#include "macro.h"

#define torange(x, r) (((x) >= 0) ? ((x)%(r)) : ((x)%(r) + (r)))
#define empty(hist) (!(hist)->alines || !(hist)->size)

static void     FDECL(init_history,(History *hist, int maxsize));
static History *FDECL(parse_hist_opts,(char **argp));
static void     FDECL(saveline,(History *hist, Aline *aline));
static void     FDECL(save_to_hist,(History *hist, Aline *aline));
static void     FDECL(save_to_log,(TFILE *logfile, char *str));
static void     FDECL(hold_input,(char *str));
static void     FDECL(display_history,(History *hist, int start, int end,
                int numbers, char *pattern, TFILE *file, int attrs));
static int      FDECL(check_watchname,(History *hist));
static int      FDECL(check_watchdog,(History *hist, char *str));

void   NDECL(init_histories);
void   FDECL(free_history,(History *hist));
void   FDECL(record_world,(History *hist, Aline *aline));
void   FDECL(record_global,(Aline *aline));
void   FDECL(record_local,(Aline *aline));
void   FDECL(record_input,(char *str));
int    FDECL(recall_history,(char *args, TFILE *file));
void   FDECL(recall_input,(Stringp str, int dir));
int    FDECL(do_saveline,(char *args));
int    FDECL(is_suppressed,(History *hist, char *str));
void   FDECL(history_sub,(char *pattern));
int    FDECL(do_log,(char *args));
void   FDECL(listlog,(World *world));

static History input[1], global[1], local[1];
static Aline *blankline;
static int norecord = 0;         /* supress history (but not log) recording */
static int nolog = 0;            /* supress log (but not history) recording */

extern int log_on;

static void init_history(hist, maxsize)
    History *hist;
    int maxsize;
{
    hist->alines = (Aline **) MALLOC(maxsize * sizeof(Aline *));
    hist->maxsize = maxsize;
    hist->pos = hist->index = -1;
    hist->size = hist->num = 0;
    hist->logfile = NULL;
}

void init_histories()
{
    init_history(input, SAVEINPUT);
    init_history(global, SAVEGLOBAL);
    init_history(local, SAVELOCAL);
    (blankline = new_aline("", F_NEWLINE))->links = 1;
    save_to_hist(input, blankline);
    input->index = input->pos;
}

#ifdef DMALLOC
void free_histories()
{
    free_history(input);
    free_history(global);
    free_history(local);
    free_aline(blankline);
}
#endif

void free_history(hist)
    History *hist;
{
    int i;

    if (hist->alines) {
        for (i = 0; i < hist->size; i++) free_aline(hist->alines[i]);
        FREE(hist->alines);
        if (hist->logfile) {
            tfclose(hist->logfile);
            if (!--log_on) put_logging(FALSE);
        }
    }
}

static void save_to_hist(hist, aline)
    History *hist;
    Aline *aline;
{
    if (!hist->alines) init_history(hist, SAVEWORLD);
    hist->pos = torange(hist->pos + 1, hist->maxsize);
    if (hist->size < hist->maxsize) hist->size++;
    else free_aline(hist->alines[hist->pos]);
    (hist->alines[hist->pos] = aline)->links++;
    hist->num++;
}

static void save_to_log(logfile, str)
    TFILE *logfile;
    char *str;
{
    if (wraplog) {
        /* ugly, but some people want it */
        char *start = str;
        int i, first = TRUE;
        do {
            if (!first && wrapflag)
                for (i = wrapspace; i; --i) tfputc(' ', logfile);
            tfputs(wrap(&start, &first)->s, logfile);
            tfflush(logfile);
        } while (*start);
    } else {
        tfputs(str, logfile);
        tfflush(logfile);
    }
}

static void saveline(hist, aline)
    History *hist;
    Aline *aline;
{
    if (!(aline->attrs & F_NOHISTORY) && !norecord) save_to_hist(hist, aline);
    if (hist->logfile && !nolog) save_to_log(hist->logfile, aline->str);
}

void record_global(aline)
    Aline *aline;
{
    saveline(global, aline);
}

void record_local(aline)
    Aline *aline;
{
    saveline(local, aline);
}

void record_world(hist, aline)
    History *hist;
    Aline *aline;
{
    if (hist) saveline(hist, aline);
}

static void hold_input(str)
    char *str;
{
    /* if (strcmp(str, input->alines[input->pos]) == 0) return; */
    free_aline(input->alines[input->pos]);
    (input->alines[input->pos] = new_aline(str, F_NEWLINE))->links++;
}

void record_input(str)
    char *str;
{
    if (*str) {
        hold_input(str);
        save_to_hist(input, blankline);
        if (input->logfile && !nolog) save_to_log(input->logfile, str);
    }
    input->index = input->pos;
}

void recall_input(s, dir)
    Stringp s;
    int dir;
{
    if (input->size == 1) return;
    if (input->index == input->pos) hold_input(s->s);
    input->index = torange(input->index + dir, input->size);
    Stringcpy(s, input->alines[input->index]->str);
}

int recall_history(args, file)
    char *args;
    TFILE *file;
{
    int num, start, end, numbers = FALSE;
    short attrs = 0;
    char opt, *arg, *place, *pattern;
    World *world = xworld();
    History *hist = NULL;
    static Aline *startmsg = NULL, *endmsg = NULL;
    extern TFILE *tfscreen;

    if (!startmsg) {
        startmsg = new_aline("---- Recall start ----", F_NEWLINE | F_NOHISTORY);
        endmsg = new_aline("----- Recall end -----", F_NEWLINE | F_NOHISTORY);
        startmsg->links = endmsg->links = 1;
    }
    startopt(args, "a:f:w:lgi");
    while ((opt = nextopt(&arg, &num))) {
        switch (opt) {
        case 'w':
            if (!*arg || (world = find_world(arg)) == NULL) {
                tfprintf(tferr, "%% No world %s", arg);
                return 0;
            }
            hist = world->history;
            break;
        case 'l':
            hist = local;
            break;
        case 'g':
            hist = global;
            break;
        case 'i':
            hist = input;
            break;
        case 'a': case 'f':
            if ((num = parse_attrs(arg)) < 0) return 0;
            attrs |= num;
            break;
        default: return 0;
        }
    }
    if (!hist) hist = world ? world->history : global;
    if (empty(hist)) return 0;
    if (arg && (pattern = strchr(arg, ' ')) != NULL) {
        *pattern++ = '\0';
        if (!smatch_check(pattern)) return 0;
    } else pattern = NULL;
    if (arg && *arg == '#') {
        numbers = TRUE;
        arg++;
    }

    if (!arg || !*arg) {
        start = end = hist->num - 1;
    } else if (*arg == '-') {
        start = end = hist->num - atoi(++arg);
    } else if (isdigit(*arg)) {
        num = atoi(arg);
        if ((place = strchr(arg, '-'))) {
            start = num - 1;
            end = (isdigit(place[1]) ? atoi(place + 1) : num) - 1;
        } else {
            start = hist->num - num;
            end = hist->num - 1;
        }
    } else {
        tfputs("% Bad recall syntax.", tferr);
        return 0;
    }

    if (!file) file = tfout;
    if (file == tfscreen) {
        norecord++;                     /* don't save this output in history */
        screenout(startmsg);
    }

    if (start < hist->num - hist->size) start = hist->num - hist->size;
    if (end >= hist->num) end = hist->num - 1;
    if (start < hist->num && end >= hist->num - hist->size && start <= end) {
        start = torange(start, hist->size);
        end = torange(end, hist->size);
        display_history(hist, start, end, numbers, pattern, file, ~attrs);
    }

    if (file == tfscreen) {
        screenout(endmsg);
        norecord--;
    }
    return 1;
}

static void display_history(hist, start, end, numbers, pattern, file, attrs)
    History *hist;
    TFILE *file;
    int start, end, numbers;
    char *pattern;
    int attrs;
{
    int i, done = FALSE;
    char *str;
    Aline *aline;
    extern Stringp keybuf;
    STATIC_BUFFER(buffer)

    if (hist == input) hold_input(keybuf->s);
    attrs |= F_NORM;
    for (i = start; !done; i = torange(i + 1, hist->size)) {
        if (i == end) done = TRUE;
        if (gag && (hist->alines[i]->attrs & F_GAG & attrs)) continue;
        if (pattern && !equalstr(pattern, hist->alines[i]->str)) continue;
        if (numbers) {
            Sprintf(buffer, "%d: %s",
                hist->num - torange(hist->pos - i, hist->size),
                hist->alines[i]->str);
            str = buffer->s;
        } else str = hist->alines[i]->str;

        if (numbers || hist->alines[i]->attrs & ~attrs & F_ALL) {
            aline = new_aline(str, (hist->alines[i]->attrs & attrs) | F_NEWLINE);
        } else aline = hist->alines[i];            /* share aline if possible */
#if 0
        tfputs(ctime(&aline->time), file);
#endif
        tfputa(aline, file);
    }
}

static int check_watchname(hist)
    History *hist;
{
    extern int wnmatch, wnlines;
    int nmatches = 1, i, slines;
    char *line, *name, *end, c;
    STATIC_BUFFER(buffer)

    slines = (wnlines > hist->size) ? hist->size : wnlines;
    name = hist->alines[torange(hist->pos, hist->size)]->str;
    for (end = name; *end && !isspace(*end); ++end);
    for (i = 1; i < slines; i++) {
        line = hist->alines[torange(hist->pos - i, hist->size)]->str;
        if (!strncmp(line, name, end - name) && (++nmatches == wnmatch)) break;
    }
    if (nmatches < wnmatch) return 0;
    c = *end;
    *end = '\0';
    Sprintf(buffer, "{%s}*", name);
    *end = c;
    oprintf("%% Watchname: gagging \"%S\"", buffer);
    add_macro(new_macro("",buffer->s,"",0,"","",NULL,gpri,100,F_GAG,0,0));
    return 1;
}

static int check_watchdog(hist, str)
    History *hist;
    char *str;
{
    extern int wdmatch, wdlines;
    int nmatches = 0, i, slines;
    char *line;

    if (wdlines > hist->size) slines = hist->size;
    else slines = wdlines;
    for (i = 1; i < slines; i++) {
        line = hist->alines[torange(hist->pos - i, hist->size)]->str;
        if (!cstrcmp(line, str) && (nmatches++ == wdmatch)) return 1;
    }
    return 0;
}

int is_suppressed(hist, str)
    History *hist;
    char *str;
{
    if (empty(hist)) return 0;
    return ((watchname && check_watchname(hist)) ||
            (watchdog && check_watchdog(hist, str)));
}

void history_sub(pattern)
    char *pattern;
{
    int size = input->size, pos = input->pos, i;
    Aline **l = input->alines;
    char *replace, *loc = NULL;
    STATIC_BUFFER(buffer)

    if (empty(input) || !*pattern) return;
    if ((replace = strchr(pattern, '^')) == NULL) return;
    else *replace++ = '\0';
    for (i = 0; i < size; i++)
        if ((loc = STRSTR(l[torange(pos - i, size)]->str, pattern)) != NULL)
            break;
    replace[-1] = '^';
    if (i == size) return;
    i = torange(pos - i, size);
    Stringncpy(buffer, l[i]->str, loc - l[i]->str);
    Stringcat(buffer, replace);
    Stringcat(buffer, loc + (replace - pattern - 1));
    record_input(buffer->s);
    check_command(FALSE, buffer);
}

void listlog(world)
    World *world;
{
    if (world->history->logfile)
        oprintf("%% Logging world %s output to %s",
          world->name, world->history->logfile->name);
}

static History *parse_hist_opts(argp)
    char **argp;
{
    History *history = global;
    World *world;
    char c;

    startopt(*argp, "lgiw:");
    while ((c = nextopt(argp, NULL))) {
        switch (c) {
        case 'l':
            history = local;
            break;
        case 'i':
            history = input;
            break;
        case 'g':
            history = global;
            break;
        case 'w':
            if (!**argp) world = xworld();
            else world = find_world(*argp);
            if (!world) {
                tfprintf(tferr, "%% No world %s", *argp);
                history = NULL;
            } else history = world->history;
            break;
        }
    }
    return history;
}

int do_saveline(args)
    char *args;
{
    History *history;

    nolog++;
    if ((history = parse_hist_opts(&args))) {
        if (history == input) record_input(args);
        else saveline(history, new_aline(args, F_NEWLINE));
    }
    nolog--;
    return history ? 1 : 0;
}

int do_log(args)
    char *args;
{
#ifdef RESTRICT_FILE
    tfputs("% /log: restricted", tferr);
    return 0;
#else
    History *history;
    TFILE *logfile;

    if (!(history = parse_hist_opts(&args))) return 0;
    if (!*args) {
        if (log_on) {
            if (input->logfile)
                oprintf("%% Logging input to %s", input->logfile->name);
            if (local->logfile)
                oprintf("%% Logging local output to %s", local->logfile->name);
            if (global->logfile)
                oprintf("%% Logging global output to %s",global->logfile->name);
            mapsock(listlog);
        } else {
            oputs("% Logging disabled.");
        }
        return 1;
    } else if (cstrcmp(args, "OFF") == 0) {
        if (history->logfile) {
            tfclose(history->logfile);
            history->logfile = NULL;
            if (!--log_on) put_logging(FALSE);
        }
        return 1;
    } else if (cstrcmp(args, "ON") == 0) {
        logfile = tfopen(tfname(NULL, "LOGFILE"), "a");
    } else {
        logfile = tfopen(tfname(args, NULL), "a");
    }
    if (!logfile) {
        operror(args);
        return 0;
    }
    if (history->logfile) {
        tfclose(history->logfile);
        history->logfile = NULL;
        log_on--;
    }
    do_hook(H_LOG, "%% Logging to file %s", "%s", logfile->name);
    history->logfile = logfile;
    if (!log_on++) put_logging(TRUE);
    return 1;
#endif
}

