/* chat.c */
 
#include <ctype.h>
#include <string.h>
#include <math.h>
 
#include "config.h"
 
#if (CHAT_SYSTEM >=2)
 
#include "externs.h"
#include "db.h"
#include "interface.h"
#include "match.h"
#include "attrib.h"
#include "chat.h"

CHTAB chantab[32];

int nchan(channel)
     channel_type channel;
{
  /* given a flagwise channel, return its index, for chantab purposes */

#ifdef SUN_OS
  return ((int) log2((double) channel));
#else
  return ((int) (log((double) channel) / log((double)2)));
#endif				/* SUN_OS */
}
 
channel_type find_channel(p)
     char *p;
{
  /* given a channel name, return the flagwise channel */
 
  int b;
  channel_type c = 0;
 
  if (!*p)
    return c;
 
  for (b = 0; !c && (b < 32); b++) {
    if (NChanPrivs(b) != CHP_FORBID) {
      if ((NChanName(b) != NULL) && string_prefix(NChanName(b), p))
	c = 1 << b;
    }
  }
 
  return c;
}
 
channel_type find_channel_exact(name)
     char *name;
{
  /* exact-match a channel, scanning all channels */
  int b;
  channel_type c = 0;
 
  for (b = 0; !c && (b < 32); b++) {
    if ((NChanName(b) != NULL) && !strcasecmp(NChanName(b), name))
      c = 1 << b;
  }
  return c;
}
 
const char *channel_name(chan)
     channel_type chan;
{
  /* given a flagwise channel, return its name */
 
  const char *n;
 
  n = chantab[nchan(chan)].name;
 
  return n;
}
 
int convert_channel_privs(perms)
     char *perms;
{
  /* given a permission level name, return the permission mask */
 
  if (string_prefix(perms, "public"))
    return CHP_PUBLIC;
  else if (string_prefix(perms, "admin"))
    return CHP_ADMIN;
  else if (string_prefix(perms, "wizard"))
    return CHP_WIZARD;
  else
    return CHP_FORBID;
}

const char *convert_channel_privname(privs)
     int privs;
{
  /* given an integer permission, give permission name */

  switch (privs) {
  case CHP_PUBLIC:
    return "Public";
  case CHP_ADMIN:
    return "Admin";
  case CHP_WIZARD:
    return "Wizard";
  default:
    return "Forbidden";
  }
}
 
int add_channel(name, privs, pos)
     char *name;
     int privs;
     int pos;	
{
  /* add a channel, return 1 on success, 0 on failure */

  int b;
  int found = 0;

  /* find it */

  if (!pos) {
    /* pick first empty spot, leave first few empty */

    for (b = 4; (b < 32) && !found; b++) {
      if ((chantab[b].name == NULL) && (chantab[b].perms == CHP_FORBID))
	found = 1;
    }

    if (!found)
      return 0;

    b--;			/* we're one ahead of the empty spot */
  } else {
    /* specified a place to put it */
    b = nchan(pos);
  }

  /* add it */
  chantab[b].name = strdup(name);
  chantab[b].perms = privs;

  return 1;
}

void do_channel(player, name, com)
     dbref player;
     const char *name;
     const char *com;
{
    /* join, quit, wipe, or who a channel */

    channel_type chan;
    int i;
 
    if (Typeof(player) != TYPE_PLAYER) {
	notify(player, "Sorry, the chat system is only for players.");
	return;
    }
 
    if (!name && !*name) {
	notify(player, "You need to specify a channel.");
	return;
    }
    if (!com && !*com) {
	notify(player, "What do you want to do with that channel?");
	return;
    }
 
    chan = find_channel(name);
 
    if (!chan) {
	notify(player, "I don't recognize that channel.");
	return;
    }
 
    if (!ChannelPermit(player, chan)) {
	notify(player, "Sorry, you aren't authorized to do that.");
	return;
    }
 
    /* add, delete, wipe, or check who is on a channel */
    if (!strcasecmp("on", com)) {
	if (!OnChannel(player, chan)) {
	    Channels(player) |= chan;
	    notify(player, "Channel added.");
	    channel_broadcast(chan, "<%s> %s has joined this channel.", 
			      channel_name(chan), Name(player));
	} else {
	    notify(player, "You are already on that channel.");
	}
	return;
    } else if (!strcasecmp("off", com)) {
	if (OnChannel(player, chan)) {
	    Channels(player) &= ~chan;
	    notify(player, "Channel deleted.");
	    channel_broadcast(chan, "<%s> %s has left this channel.", 
			      channel_name(chan), Name(player));
	} else {
	    notify(player, "You are not on that channel.");
	}
	return;
    } else if (!strcasecmp("wipe", com)) {
	if (!Wizard(player)) {
	    notify(player, "Such power is not within you.");
	} else {
	    for (i = 0; i < db_top; i++)
		Channels(i) &= ~chan;
	    notify(player, "Channel wiped.");
	}
	return;
    } else if (!strcasecmp("who", com)) {
	do_channel_who(player, chan);
	return;
    } else {
	notify(player, "I don't understand what you want to do.");
	return;
    }
}
 
void do_chat(player, chan, arg1)
     dbref player;
     channel_type chan;
     const char *arg1;
{
  /* send a message to a channel */

  int key;
  const char *gap;
 
  if (!chan) {
    notify(player, "I don't recognize that channel.");
    return;
  }
 
  if (!ChannelPermit(player, chan)) {
    notify(player, "Sorry, you're not authorized to be on that channel.");
    return;
  }
 
  /* figure out what kind of message we have */
  gap = " ";
  switch (*arg1) {
  case SEMI_POSE_TOKEN:
    gap = "";
  case POSE_TOKEN:
    key = 1;
    arg1 = arg1 + 1;
    break;
  case NULL:
    key = 3;
    break;
  default:
    key = 2;
    break;
  }
 
  /* now send out the message. If the player isn't on that channel, tell
   * him what he said.
   */
  switch (key) {
  case 1:
    channel_broadcast(chan, "<%s> %s%s%s", channel_name(chan),
		      Name(player), gap, arg1);
    if (!OnChannel(player, chan))
      notify(player, tprintf("To channel %s: %s%s%s", channel_name(chan),
			     Name(player), gap, arg1));
    break;
  case 2:
    channel_broadcast(chan, "<%s> %s says, \"%s\"", channel_name(chan),
		      Name(player), arg1);
    if (!OnChannel(player, chan))
      notify(player, tprintf("To channel %s: %s says, \"%s\"",
			     channel_name(chan), Name(player), arg1));
    break;
  case 3:
    notify(player, "What do you want to say to that channel?");
    break;
  }
}
 
void do_chan_admin(player, name, perms, flag)
     dbref player;
     char *name;
     char *perms;
     int flag;
{
  /* wipe, add, remove, rename, or change the permissions of a channel */

  int chan, privs;
  int i;
 
  if (!Wizard(player)) {
    notify(player, "Only a wizard may modify channels.");
    return;
  }
 
  chan = privs = 0;
 
  switch (flag) {
  case 0:
    /* add a channel */
    if (*name == '\0')  {
      notify(player, "You must name the channel.");
      return;
    }
    if (*perms == '\0') {
      notify(player, "You must specify the restrictions on the channel.");
      return;
    }
    /* make sure the channel name is unique */
    if (find_channel(name) != 0) {
      notify(player, "The channel needs a more unique name.");
      return;
    }
    /* get the permissions. Invalid specifications default to "forbidden" */
    privs = convert_channel_privs(perms);
    if (privs == CHP_FORBID)
      notify(player, "Warning: channel will be created locked to all.");
    if (add_channel(name, privs, 0))
      notify(player, "Channel created.");
    else
      notify(player, "No more room in the channel table.");
    return;
    break;			/* NOT REACHED */
  case 1:
    /* remove a channel */
    if (*name == '\0') {
      notify(player, "You must specify a channel to remove.");
      return;
    }
    chan = find_channel(name);
    if (!chan) {
      notify(player, "That channel does not appear to exist.");
      return;
    }
    /* zap the channel */
    /* remove everyone from the channel */
    for (i = 0; i < db_top; i++)
      if (Typeof(i) == TYPE_PLAYER)
	db[i].channels &= ~chan;
    free(chantab[nchan(chan)].name);
    chantab[nchan(chan)].perms = CHP_FORBID;
    notify(player, "Channel removed.");
    return;
    break;			/* NOT REACHED */
  case 2:
    /* rename a channel */
    if (*name == '\0') {
      notify(player, "You must specify a channel to rename.");
      return;
    }
    if (*perms == '\0')  {
      notify(player, "What do you want to rename the channel to?");
      return;
    }
    /* make sure channel exists */
    chan = find_channel_exact(name);
    if (!chan) {
      notify(player, "That channel does not appear to exist.");
      return;
    }
    /* make sure the channel name is unique */
    if (find_channel(perms) != 0) {
      notify(player, "The channel needs a more unique name.");
      return;
    }
    free(chantab[nchan(chan)].name);
    chantab[nchan(chan)].name = strdup(perms);
    notify(player, "Channel renamed.");
    break;
  case 3:
    /* change the permissions on a channel */
    if (*name == '\0')  {
      notify(player, "You must specify a channel.");
      return;
    }
    chan = find_channel_exact(name);
    if (!chan) {
      notify(player, "That channel does not appear to exist.");
      return;
    }
    if (*perms == '\0') {
      notify(player, "You must specify the restrictions on the channel.");
      return;
    }
    privs = convert_channel_privs(perms);
    if (privs == CHP_FORBID)
      notify(player, "Warning: channel will be locked to all.");
    chantab[nchan(chan)].perms = privs;
    notify(player, "Permissions on channel changed.");
    return;
    break;			/* NOT REACHED */
  }
}

void do_channel_list(player)
     dbref player;
{
  int b;

  if (Wizard(player)) {
    notify(player, "Channel        Name         Privs");

    for (b = 0; b < 32; b++) {
      if ((chantab[b].name == NULL) && (chantab[b].perms == CHP_FORBID))
	continue;
      notify(player, tprintf("0x%-8x     %-10s   %s", 
			     (int) pow((double) 2, (double) b),
			     chantab[b].name,
			     convert_channel_privname(chantab[b].perms)));
    }
  } else {
    notify(player, "Name         Privs");

    for (b = 0; b < 32; b++) {
      if ((chantab[b].name == NULL) && (chantab[b].perms == CHP_FORBID))
	continue;
      notify(player, tprintf("%-10s   %s", chantab[b].name,
			     convert_channel_privname(chantab[b].perms)));
    }
  }
}

 
void init_chat()
{
  /* clear out the channel table and do initialization of defaults. */

  int b;

  for (b = 0; b < 32; b++) {
    chantab[b].name = NULL;
    chantab[b].perms = CHP_FORBID;
  }

  add_channel("Public", CHP_PUBLIC, 0x10);
  add_channel("Code", CHP_PUBLIC, 0x20);
  add_channel("Mush", CHP_PUBLIC, 0x40);
  add_channel("Admin", CHP_ADMIN, 0x10000);
  add_channel("Wizard", CHP_WIZARD, 0x1000000);
}
 
#endif				/* CHAT_SYSTEM */
