/*  interface.c,v 1.14 1993/03/07 14:56:13 jan Exp

    Part of XPCE
    Designed and implemented by Anjo Anjewierden and Jan Wielemaker
    E-mail: jan@swi.psy.uva.nl

    Copyright (C) 1992 University of Amsterdam. All rights reserved.
*/


		/********************************
		*        DECLARATIONS		*
		********************************/

#ifdef __GNUC__
#define TermVector(name, size)  Term name[size]
#else
#define TermVector(name, size) \
	Term *name = (Term *) alloca(size * sizeof(Term))
#endif

static PceObject	pceSelector P((Term));
static inline PceObject pceObjectName P((Term));
static PceObject	termToObject P((Term, char *, bool));
static bool		unifyRef P((Term, int, PceCValue));

static char 	       *DefaultModule;	/* module for message(@prolog, ...) */

#ifndef Show
#define Show(t)
#define ShowAll(argc, argv)
#endif

#ifndef ShowAll
#define ShowAll(ac, av) \
  { int _i; \
    for(_i=0; _i<(ac); _i++) \
      Show(av[_i]); \
  }
#endif

#ifndef Hide
#define Hide(t)
#define HideAll(argc, argv)
#endif

#ifndef HideAll
#define HideAll(ac, av) \
  { int _i; \
    for(_i=(ac)-1; _i>=0; _i--) \
      Hide(av[_i]); \
  }
#endif

#define TestAndHide(t, code) if ( !(code) ) { Hide(t); fail; }


		/********************************
		*        INITIALISATION         *
		*********************************/

#ifndef INSTALL_HIDDEN_PREDICATE
#define INSTALL_HIDDEN_PREDICATE(n, a, f) INSTALL_PREDICATE(n, a, f)
#endif
#ifndef INSTALL_META_PREDICATE
#define INSTALL_META_PREDICATE(n, a, f) INSTALL_PREDICATE(n, a, f)
#endif
#ifndef INSTALL_HIDDEN_META_PREDICATE
#define INSTALL_HIDDEN_META_PREDICATE(n, a, f) INSTALL_META_PREDICATE(n, a, f)
#endif

static void
registerPredicates()
{ INSTALL_META_PREDICATE("send",			2, pl_send0);
  INSTALL_META_PREDICATE("send",			3, pl_send1);
  INSTALL_META_PREDICATE("send",			4, pl_send2);
  INSTALL_META_PREDICATE("send",			5, pl_send3);
  INSTALL_HIDDEN_META_PREDICATE("$pce_send",		3, pl_sendN);
  INSTALL_META_PREDICATE("get",				3, pl_get0);
  INSTALL_META_PREDICATE("get",				4, pl_get1);
  INSTALL_META_PREDICATE("get",				5, pl_get2);
  INSTALL_META_PREDICATE("get",				6, pl_get3);
  INSTALL_HIDDEN_META_PREDICATE("$pce_get",	   	4, pl_getN);
  INSTALL_HIDDEN_META_PREDICATE("$pce_get_object", 	4, pl_get_objectN);
  INSTALL_PREDICATE("object",			   	1, pl_object1);
  INSTALL_PREDICATE("object",			   	2, pl_object2);
  INSTALL_META_PREDICATE("new",			   	2, pl_new);
}


/* - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
Code used constants are fetched from bot Prolog and PCE at boot time  to
simplify code and improve performance.
- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - */

static PceObject	NAME_functor;		/* "functor" */
static PceObject	NAME_Arity;		/* "Arity" */
static PceObject	NAME_Arg;		/* "Arg" */
static PceObject	NAME_obtain;		/* "?" */
static PceObject	NAME_message;		/* "message" */
static PceObject	PROLOG;			/* @prolog object */
static PceObject	PCEOBJ;			/* @pce object */
static PceObject	NIL;			/* @nil object */

static void
initPceConstants()
{ NAME_functor = cToPceName("functor");
  NAME_Arity   = cToPceName("_arity");
  NAME_Arg     = cToPceName("_arg");
  NAME_obtain  = cToPceName("?");
  NAME_message = cToPceName("message");

  PROLOG       = cToPceAssoc("host");
  PCEOBJ       = cToPceAssoc("pce");
  NIL	       = cToPceAssoc("nil");
}


static inline PceName
GetPceAtom(t)
Term t;
{ PceITFSymbol symbol;
  PceName name;
  hostHandle handle = AtomHandle(t);

  if ( (symbol = pceLookupHandle(0, handle)) && symbol->name )
    return symbol->name;

  name = cToPceName(GetAtom(t));
  pceRegisterName(0, handle, name);
  return name;
}


		/********************************
		*         PROLOG --> PCE	*
		********************************/

static inline PceObject
pceObjectName(ref)
Term ref;
{ PceObject obj;
  Term arg;

  assert(IsPceReference(ref));

  arg = GetArg(ref, 1);
  if ( IsInt(arg) )
  { if ( !(obj = cToPceReference(GetInteger(arg))) )
    { Warning("Bad PCE integer reference: @%d", GetInteger(arg));
      fail;
    }
  } else if ( IsAtom(arg) )
  { hostHandle handle = AtomHandle(arg);
    PceITFSymbol symbol = pceLookupHandle(0, handle);

    if ( symbol && symbol->object )
      obj = symbol->object;
    else if ( (obj = cToPceAssoc(GetAtom(arg))) ) /* might do a call-back */
      pceRegisterAssoc(0, handle, obj);
    else
    { Warning("Unknown PCE object: @%s", GetAtom(arg));
      fail;
    }
  } else
  { Warning("Illegal PCE object reference");
    fail;
  }

  return obj;
}


static PceName
pceSelector(sel)
Term sel;
{ if ( IsAtom(sel) )
    return GetPceAtom(sel);

  Warning("Illegal PCE selector");
  fail;
}


/* - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
Convert a Prolog term (as the 2nd  argument to new/2  or the arguments
to send/[2-12] or get/[3-13]) into a  PCE object.  The  term can be an
@/1 reference or a term, whose functor denotes the class and arguments
are the initialising arguments.  Note that an  atom is ambiguous as it
can both represent a  name or an  object which initialisation does not
require arguments (e.g. `chain' can represent a  the  PCE-name "chain"
or an empty chain.  For the principal  functor to new/2  we will treat
atoms as class names.  Otherwise they are treated  as PCE-names.  This
is  indicated by the  last argument.   Thus new(@ch, chain(foo))  will
create a chain with one member: the PCE-name "foo".
- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - */

static PceObject
termToObject(t, assoc, new)
Term t;
char *assoc;
bool new;
{ if ( IsPceReference(t) )
    return pceObjectName(t);

  if ( IsAtom(t) )
  { PceName name = GetPceAtom(t);
    return new ? pceNew(assoc, name, 0, NULL) : name;
  }
    
  if ( IsInt(t) )
    return cToPceInteger(GetInteger(t));

#ifdef PROLOG_STRING
  if ( IsString(t) )
    return cToPceString(assoc, GetString(t));
#endif

  if ( IsCompound(t) )
  { if ( IsNew1(t) )
      return termToObject(GetArg(t, 1), NULL, TRUE);
    if ( IsNew2(t) )
    { PceObject rval;
      int type;
      PceCValue value;
      Term r = GetArg(t, 1);
      Term a;

      Show(r);
      if ( !UnifyPceReference(r) )
      { Warning("send: new/2: Illegal object reference");
	Hide(r);
	fail;
      }

      a = GetArg(r, 1);
      if ( IsAtom(a) )
      { TestAndHide(r, rval = termToObject(GetArg(t, 2), GetAtom(a), TRUE));
      } else
      { TestAndHide(r, rval = termToObject(GetArg(t, 2), NULL, TRUE));
      }

      type = pceToC(rval, &value);
      TestAndHide(r, unifyRef(r, type, value));
      Hide(r);
      return rval;
    }

    if ( IsString1(t) )
    { char tmp[25];
      char *s = tmp;
      Term a;

      a = GetArg(t, 1);
      switch( GetType(a) )
      { case PROLOG_ATOM:
	  s = GetAtom(a);
	  break;
	case PROLOG_INTEGER:
	  sprintf(tmp, "%d", GetInteger(a));
	  break;
	case PROLOG_FLOAT:
	  sprintf(tmp, "%f", GetFloat(a));
	  break;
#ifdef PROLOG_STRING
	case PROLOG_STRING:
	  s = GetString(a);
	  break;
#endif
	case PROLOG_COMPOUND:
	  if ( (s = ListToCharp(a)) != NULL )
	    break;      
	default:
	  Warning("Argument to string/1 should be atomic");
	  fail;
      }
      return cToPceString(assoc, s);
    }

    { int arity;
      char *class_str;

      GetCompound(t, &class_str, &arity);
      { PceObject classname = cToPceName(class_str);
	int n;
	int done = 0;
	ArgVector(argv, arity);

	Show(t);
	TestAndHide(t, argv[0] = termToObject(GetArg(t, 1), NULL, FALSE));

	if ( arity >= 2 )
#if MODULE
	{ if ( argv[0] == PROLOG &&
	       (classname == NAME_message || classname == NAME_obtain) )
	  { Term a;

	    a = GetArg(t, 2);
	    if ( arity >= 3 && IsAtomCall(a) )
	    { a = GetArg(t, 3);
	      done = 3;			/* message(@prolog, call, ... */
	    } else
	      done = 2;			/* message(@prolog, ... */

	    { a = StripModuleTag(a, &DefaultModule);
	      if ( IsAtom(a) && DefaultModule != NULL )
	      { char *s = GetAtom(a);
		char tmp[LINESIZE];

		if ( strchr(s, ':') == NULL )
		{ sprintf(tmp, "%s:%s", DefaultModule, s);
		  s = tmp;
		}

		TestAndHide(t, argv[done-1] = cToPceName(s));
	      } else
	      { TestAndHide(t, argv[done-1] = termToObject(a, NULL, FALSE));
	      }
	    }
	  }
#endif /*MODULE*/
	}

	for(n=2; n <= arity; n++)
	{ if ( n != done )
	    TestAndHide(t, argv[n-1] = termToObject(GetArg(t,n), NULL, FALSE));
	}

	Hide(t);
	return pceNew(assoc, classname, arity, argv);
      }
    }
  }

  if ( IsFloat(t) )
    return cToPceReal(GetFloat(t));

  Warning("Illegal PCE object description");
  fail;
}

/* - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
unifyObject() unifies a Prolog Term with an  object  `description'.   An
object  description consists of a term whose functor is the class of the
object and whose arguments are the same argument than those that have to
be presented to `new' for initialisation of the object.

unifyObject() works recursively.  On the toplevel the object  is  always
expanded  to  a  description.   On the lower levels expansion only takes
place if the corresponding part of the Prolog term is instantiated to an
atom (class name) or a term, not equal to @/1. Thus

	?- new(@ch, chain(chain(0), chain(1, 2))), 
	   object(@ch, O), 
	   object(@ch, chain(chain(X), Y)).

	O = chain(@477532, @477600), X = 0, Y = @477600

Note that strings are treated special.
- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - */

static bool
unifyRef(t, type, value)
Term t;
int type;
PceCValue value;
{ Test(UnifyPceReference(t));
  t = GetArg(t, 1);

  if ( type == PCE_REFERENCE )
  { return UnifyInteger(t, value.integer);
  } else
  { PceITFSymbol symbol = value.itf_symbol;

    if ( symbol->handle[0] )
      return UnifyAtom(t, symbol->handle[0]);
    else
      return UnifyCharp(t, pceCharArrayToC(symbol->name));
  }
}



static bool
unifyObject(t, obj, top)
Term t;
PceObject obj;
bool top;
{ PceCValue value;
  int pltype, pcetype;
  char *s;

  switch( pcetype = pceToC(obj, &value) )
  { case PCE_INTEGER:
	return UnifyInteger(t, value.integer);
    case PCE_REAL:
	return UnifyFloat(t, value.real);
    case PCE_NAME:
      { PceITFSymbol symbol = value.itf_symbol;

	if ( symbol->handle[0] != NULL )
	  return UnifyAtom(t, symbol->handle[0]);
	else
	  return UnifyCharp(t, pceCharArrayToC(symbol->name));
      }
  }

  if ( (s = pceStringToC(obj)) )	/* string: handle special */
  { switch( GetType(t) )
    { case PROLOG_COMPOUND:
	if ( IsString1(t) )
	  return UnifyCharp(GetArg(t, 1), s);
	return unifyRef(t, pcetype, value);
      case PROLOG_VARIABLE:
	if ( top )
	{ UnifyCompound(t, "string", 1);
	  return UnifyCharp(GetArg(t, 1), s);
	}
	return unifyRef(t, pcetype, value);
      default:
	fail;
    }
  }

  if ( top ||
      (pltype = GetType(t)) == PROLOG_ATOM ||
      (pltype == PROLOG_COMPOUND && !IsPceReference(t)) )
  { PceObject got, arg;
    int arity, n;
    char *name;

    Test(got = pceGet(obj, NAME_functor, 0, NULL) );
    Test(pceToC(got, &value) == PCE_NAME);
    name = pceCharArrayToC(value.itf_symbol->name);

    Test(got = pceGet(obj, NAME_Arity, 0, NULL) );
    Test(pceToC(got, &value) == PCE_INTEGER);
    if ( (arity = value.integer) == 0 )
      return UnifyCharp(t, name);

    Test(UnifyCompound(t, name, arity));

    for(n=1; n <= arity; n++)
    { ArgVector(argv, 1);

      argv[0] = cToPceInteger(n);
      Test(arg = pceGet(obj, NAME_Arg, 1, argv) );
      Test(unifyObject(GetArg(t, n), arg, FALSE) );
    }
    succeed;
  }

  return unifyRef(t, pcetype, value);
}


/* - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
pl_get() and  pl_pce_get_object()  are called  directly   from Prologs
virtual  machine   interpreter.  See  the  called functions   and  the
description  of Prolog calling  foreign  predicates above for details.
In Prolog these predicates are named $pce_get/4 and $pce_get_object/4.
- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - */

static bool
pl_get(rec, sel, argc, argv, val, ref)
Term rec, sel, val;
int argc;
Term *argv;
bool ref;
{ PceObject receiver, selector, value;
  AnswerMark mark;
  PceObject rewind;
  int rval;

  DefaultModule = NULL;
  sel = StripModuleTag(sel, &DefaultModule);

  Show(rec);
  Show(sel);
  Show(val);
  ShowAll(argc, argv);
  
  markAnswerStack(mark);
  if ( !(receiver = termToObject(rec, NULL, FALSE)) ||
       !(selector = pceSelector(sel)) )
    value = PCE_FAIL;
  else if ( argc == 0 )
    value = pceGet(receiver, selector, 0, NULL);
  else
  { ArgVector(av, argc);
    int n;

    for(n=0; n<argc; n++)
      if ( !(av[n] = termToObject(argv[n], NULL, FALSE)) )
      { rval = FALSE;
	rewind = NIL;
	goto out;
      }

    value = pceGet(receiver, selector, argc, av);
  }

  if ( value != PCE_FAIL &&
       unifyObject(val, value, ref ? FALSE : TRUE) == TRUE )
  { rewind = (ref ? value : NIL);
    rval = TRUE;
  } else
  { rval = FALSE;
    rewind = NIL;
  }

out:
  HideAll(argc, argv);
  Hide(val);
  Hide(sel);
  Hide(rec);
  
  rewindAnswerStack(mark, rewind);
  return rval;
}


foreign_t
pl_get0(rec, sel, val)
Term rec, sel, val;
{ return pl_get(rec, sel, 0, NULL, val, TRUE);
}


foreign_t
pl_get1(rec, sel, arg, val)
Term rec, sel, arg, val;
{ return pl_get(rec, sel, 1, &arg, val, TRUE);
} 


foreign_t
pl_get2(rec, sel, arg1, arg2, val)
Term rec, sel, arg1, arg2, val;
{ Term argv[2];

  argv[0] = arg1;
  argv[1] = arg2;

  return pl_get(rec, sel, 2, &argv, val, TRUE);
} 


foreign_t
pl_get3(rec, sel, arg1, arg2, arg3, val)
Term rec, sel, arg1, arg2, arg3, val;
{ Term argv[3];

  argv[0] = arg1;
  argv[1] = arg2;
  argv[2] = arg3;

  return pl_get(rec, sel, 3, argv, val, TRUE);
} 


foreign_t
pl_getN(rec, sel, args, val)
Term rec, sel, args, val;
{ if ( IsAtom(args) )
  { return pl_get(rec, sel, 0, NULL, val, TRUE);
  } else
  { int n, argc = GetArity(args);
    TermVector(argv, argc);

    for(n=0; n<argc; n++)
      argv[n] = GetArg(args, n+1);

    return pl_get(rec, sel, argc, argv, val, TRUE);
  }
}


foreign_t
pl_get_objectN(rec, sel, args, val)
Term rec, sel, args, val;
{ if ( IsAtom(args) )
  { return pl_get(rec, sel, 0, NULL, val, FALSE);
  } else
  { int n, argc = GetArity(args);
    TermVector(argv, argc);

    for(n=0; n<argc; n++)
      argv[n] = GetArg(args, n+1);

    return pl_get(rec, sel, argc, argv, val, FALSE);
  }
}


/* - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
Implementation of $pce_send/3.
- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - */

static inline foreign_t
pl_send(rec, sel, argc, argv)
Term rec, sel, *argv;
int argc;
{ foreign_t rval;
  AnswerMark mark;
  PceObject receiver;
  PceName selector;

  DefaultModule = NULL;
  sel = StripModuleTag(sel, &DefaultModule);

  Show(rec);
  Show(sel);
  ShowAll(argc, argv);

  markAnswerStack(mark);
  if ( !(receiver = termToObject(rec, NULL, FALSE)) ||
       !(selector = pceSelector(sel)) )
    rval = FALSE;
  else if ( argc == 0 )
    rval = pceSend(receiver, selector, 0, NULL);
  else
  { ArgVector(av, argc);
    int n;

    for(n=0; n<argc; n++)
      if ( !(av[n] = termToObject(argv[n], NULL, FALSE)) )
      { rval = FALSE;
	goto out;
      }
    rval = pceSend(receiver, selector, argc, av);
  }
out:
  rewindAnswerStack(mark, NIL);

  HideAll(argc, argv);
  Hide(sel);
  Hide(rec);

  return rval;
}


foreign_t
pl_send0(rec, sel)
Term rec, sel;
{ return pl_send(rec, sel, 0, NULL);
} 


foreign_t
pl_send1(rec, sel, arg)
Term rec, sel, arg;
{ return pl_send(rec, sel, 1, &arg);
} 


foreign_t
pl_send2(rec, sel, arg1, arg2)
Term rec, sel, arg1, arg2;
{ Term argv[2];

  argv[0] = arg1;
  argv[1] = arg2;

  return pl_send(rec, sel, 2, &argv);
} 


foreign_t
pl_send3(rec, sel, arg1, arg2, arg3)
Term rec, sel, arg1, arg2, arg3;
{ Term argv[3];

  argv[0] = arg1;
  argv[1] = arg2;
  argv[2] = arg3;

  return pl_send(rec, sel, 3, argv);
} 


foreign_t
pl_sendN(rec, sel, args)
Term rec, sel, args;
{ if ( IsAtom(args) )
  { return pl_send(rec, sel, 0, NULL);
  } else
  { int n, argc = GetArity(args);
    TermVector(argv, argc);

    for(n=0; n<argc; n++)
      argv[n] = GetArg(args, n+1);

    return pl_send(rec, sel, argc, argv);
  }
}


/* - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
Implementation  of new/2  and new_global/2.  First  the PCE  object is
created  from the description.  Then the   first arguments is  matched
against  or instantiated with the @/1  operator.   If  the argument to
this operator turns out  to be a variable it  is instantiated with the
the object reference.
- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - */

foreign_t
pl_new(ref, descr)
Term ref, descr;
{ PceObject obj;
  Term r;
  AnswerMark mark;
  bool rval;

  DefaultModule = NULL;
  descr = StripModuleTag(descr, &DefaultModule);

  if ( !UnifyPceReference(ref) )
    return Warning("new/2: Illegal object reference");

  Show(ref);
  Show(descr);
  markAnswerStack(mark);

  r = GetArg(ref, 1);
  if ( IsAtom(r) )
    obj = termToObject(descr, GetAtom(r), TRUE);
  else
    obj = termToObject(descr, NULL, TRUE);

  if ( obj )
  { int type;
    PceCValue value;

    type = pceToCReference(obj, &value);
    rval = unifyRef(ref, type, value);
  } else
    rval = FALSE;

  rewindAnswerStack(mark, obj);
  Hide(descr);
  Hide(ref);

  return rval;
}


/* - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
Implement  object/[1-2]: checking for  the existence of an  object and
transforming object references into object descriptions.
- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - */

foreign_t
pl_object1(ref)
Term ref;
{ if ( IsPceReference(ref) )
  { Term arg = GetArg(ref, 1);
    
    switch( GetType(arg) )
    { case PROLOG_INTEGER:
	return pceExistsReference(GetInteger(arg));
      case PROLOG_ATOM:
	return pceExistsAssoc(GetAtom(arg));
    }
  }

  fail;
}


foreign_t
pl_object2(ref, description)
Term ref, description;
{ PceObject obj;

  Test(obj = termToObject(ref, NULL, FALSE));

  return unifyObject(description, obj, TRUE);
}


/* - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - 
hostAction() is called by PCE to manipulate or query the  hosts'  status
or call goals from the PCE tracer.
- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - */

static int
_hostAction(action, args)
int action;
va_list args;
{ switch(action)
  { case HOST_TRACE:
      PROLOG_TRACE();
      succeed;
    case HOST_BACKTRACE:
    { int frames = va_arg(args, int);
      PROLOG_BACKTRACE(frames);
      succeed;
    }
    case HOST_HALT:
      PROLOG_HALT();
      fail;
    case HOST_BREAK:
      PROLOG_BREAK();
      succeed;
    case HOST_ABORT:
      PROLOG_ABORT();
      succeed;
    case HOST_SIGNAL:
    { int sig = va_arg(args, int);
      VoidFunc func = va_arg(args, VoidFunc);
      PROLOG_SIGNAL(sig, func);
      succeed;
    }
    case HOST_RECOVER_FROM_FATAL_ERROR:
      PROLOG_ABORT();
      fail;			/* could not abort: failure */
    case HOST_WRITE:
    { char *s = va_arg(args, char *);

      PROLOG_WRITE(s);
      succeed;
    }
    case HOST_FLUSH:
      PROLOG_FLUSH();
      succeed;
    case HOST_ONEXIT:
    { OnExitFunction func = va_arg(args, OnExitFunction);
      char *s = va_arg(args, char *);

      PROLOG_ONEXIT(func, s);
      succeed;
    }
    default:
      Warning("Unknown action request from PCE: %d", action);
      fail;
  }
}


#ifdef __STDC__
int
hostAction(int action, ...)
{ va_list args;
  int rval;

  va_start(args, action);
  rval = _hostAction(action, args);
  va_end(args);
  return rval;
}

#else

int
hostAction(va_alist)
va_dcl
{ va_list args;
  int action, rval;

  va_start(args);
  action = va_arg(args, int);
  rval = _hostAction(action, args);
  va_end(args);
  return rval;
}

#endif

/* - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
TIMEOUT defines  the number  of milliseconds  to  wait for  something to
happen in PCE.  If this value  is 0,  PCE  will wait indefinitely for an
event or input.

For linux this value is currently set to 250 because linux's select call
appears  not  to  be   broken  when   a   signal  (notably  SIGCHLD  in 
unx-process.c) arrives.  This way pce will anyway  see the signal ...  A
better solution for signal handling is to be searched for (also avoiding
the possibility of reentrance at moments this is not allowed in PCE ...
- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - */

#ifndef TIMEOUT
#if linux
#define TIMEOUT 250
#else
#define TIMEOUT 0
#endif
#endif

static int
pce_dispatch()
{ if ( pceDispatch(0, TIMEOUT) == PCE_DISPATCH_INPUT )
    return PROLOG_DISPATCH_INPUT;

  return PROLOG_DISPATCH_TIMEOUT;
}


/* - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
prolog_pce_init() is called by  Prolog  before  it  performs  any  other
actions.   It  passes the program's Argc and Argv.  After initialisation
pceDoDispatch() is called to allow event  dispatching  while  Prolog  is
reading from the users' input channal.
- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - */

int
prolog_pce_init()
{ int argc;
  char **argv;
  static bool initialised = FALSE;
  
#if DLD
{ extern char *(*getFunctionNameFromAddress)();

  getFunctionNameFromAddress = dld_find_function;
}
#endif

  PROLOG_INSTALL_REINIT_FUNCTION(prolog_pce_init);
  argc = PROLOG_ARGC();
  argv = PROLOG_ARGV();

  if ( !initialised )
  { if ( !pceInitialise(1, argc, argv) )
      FatalError("Could not initialise PCE");

    initPceConstants();			/* get code used PCE constants */  
    InitPrologConstants();		/* Public prolog constants */
    registerPredicates();		/* make the interface known */

    PROLOG_INSTALL_DISPATCH_FUNCTION(pce_dispatch);
    PROLOG_INSTALL_RESET_FUNCTION(pceReset);

    initialised = TRUE;
  } else
  { if ( !pceReInitialise(argc, argv) )
      FatalError("Could not initialise PCE");
  }

  succeed;
}


		/********************************
		*       QUERY THE HOST          *
		*********************************/

int
hostQuery(what, value)
int what;
PceCValue *value;
{ switch(what)
  { case HOST_SYMBOLFILE:
	if ( (value->string = PROLOG_SYMBOLFILE()) )
	  succeed;
	fail;
    case HOST_GETC:
	value->character = PROLOG_GETC();
	succeed;
    default:
	Warning("Unknown query from PCE: %d", what);
	fail;
  }
}
