/*
 * This file is part of the portable Forth environment written in ANSI C.
 * Copyright (C) 1995  Dirk Uwe Zoller
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free
 * Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * This file is version 0.9.13 of 17-July-95
 * Check for the latest version of this package via anonymous ftp at
 *	roxi.rz.fht-mannheim.de:/pub/languages/forth/pfe-VERSION.tar.gz
 * or	sunsite.unc.edu:/pub/languages/forth/pfe-VERSION.tar.gz
 * or	ftp.cygnus.com:/pub/forth/pfe-VERSION.tar.gz
 *
 * Please direct any comments via internet to
 *	duz@roxi.rz.fht-mannheim.de.
 * Thank You.
 */
/*
 * 4ed.c --- simple FORTH-screenfile editor
 * (duz 28May93)
 */

#include "forth.h"
#include "support.h"
#include "term.h"
#include "lined.h"

#include <stdlib.h>
#include <stdarg.h>
#include <string.h>
#include <ctype.h>
#include <time.h>
#include <signal.h>

#include "nonansi.h"
#include "missing.h"


typedef char line[64];
typedef line blck[16];		/* block buffer */

static line			/* block buffers used for editing */
  *buf, *blk,			/* and stacking source */
  *linestk, *linetop, *linesp;
static blck
  *blkstk, *blktop, *blksp;

static int row, col;		/* cursor position */
struct position { int row, col, scr; };
static struct position mark = { 0 };

static char			/* mode flags: */
  overtype = 0,			/*   overwrite mode */
  caps = 1,			/*   simulate CAPS (for stupid DIN keyboard) */
  stamp_changed = 1,		/*   shall changed screens be stamped? */
  was_replacing = 0,		/*   action of ^L, repeat search or replace? */
  readonly;			/*   file cannot be written */

static char log_name[16];	/* for stamp to screen */

static char search_str[32] = "";
static char search_history[512];
static struct lined search_lined =
{
  search_str, sizeof search_str,
  search_history, sizeof search_history,
  complete_dictionary, NULL
};

static char replace_str[32] = "";
static char replace_history[512];
static struct lined replace_lined =
{
  replace_str, sizeof replace_str,
  replace_history, sizeof replace_history,
  complete_dictionary, NULL
};

#define setcursor(R,C)	c_gotoxy ((C) + 16, (R))


/*
 * file-i/o
 */

#define NOBLK UCELL_MAX

static int
scr_changed (void)
{
  blk = (line *)block (BLOCK_FILE, SCR);
  return memcmp (blk, buf, sizeof (blck)) != 0;
}

static int
block_empty (char *p)
{
  int n;

  for (n = 64; n < BPBUF; n++)
    if (p[n] != ' ' && printable (p[n]))
      return 0;
  return 1;
}

static int
scr_empty (int n)
{
  return block_empty (block (BLOCK_FILE, n));
}

static void
scr_copy (int dst, int src)
{
  block (BLOCK_FILE, src);
  BLOCK_FILE->n = dst;
  update (BLOCK_FILE);
  save_buffers (BLOCK_FILE);
}

static void
truncate_file (void)
{
  int n;

  for (n = BLOCK_FILE->size; n; n--)
    if (!scr_empty (n - 1))
      break;
  resize_file (BLOCK_FILE, n * BPBUF);
}

static void
stamp_screen (void)
{
  time_t t;
  struct tm *tm;
  char stamp[65];
  int n;

  time (&t);
  tm = localtime (&t);
  n = 64 - 3 - strlen (log_name) - 6 - 9;
  sprintf (stamp, "\\ %.*s %s %02d:%02d %02d/%02d/%02d",
	   n, &buf[0][2], log_name, tm->tm_hour, tm->tm_min,
	   tm->tm_mon + 1, tm->tm_mday, tm->tm_year);
  COPY (buf[0], stamp);
}

static void
writebuf (void)
{
  int dummy;

  if (SCR != NOBLK && scr_changed ())
    {
      if (stamp_changed)
	stamp_screen ();
      blk = (line *)buffer (BLOCK_FILE, SCR, &dummy);
      memcpy (blk, buf, sizeof (blck));
      update (BLOCK_FILE);
      save_buffers (BLOCK_FILE);
    }
}

static void
readbuf (int n)
{
  blk = (line *)block (BLOCK_FILE, n);
  memcpy (buf, blk, sizeof (blck));
  SCR = n;
}

static void
changescr (int n)
{
  writebuf ();
  readbuf (n);
}


/*
 * screen-i/o
 */

static int
getckey (void)
{
  char c = getwskey ();
  return c < ' '
    ? c + '@'
    : toupper (c);
}

static void
c_printf (char *fmt,...)
{
  char buf[0x200];
  va_list p;

  va_start (p, fmt);
  vsprintf (buf, fmt, p);
  va_end (p);
  c_puts (buf);
}

static void
type_line (char *p, int n)
{
  int i;

  for (i = 0; i < n; i++)
    if (!printable (p[i]))
      break;
  if (i < n)
    while (n--)
      c_putc_printable (*p++);
  else
    type (p, n);
}


/*
 * show help
 */

struct helpline
{
  char row, col;
  char *str;
};

static struct helpline *displayed_help = NULL;
static struct helpline primary_help[] =
{
  {0, 0, "CURSOR MOVE"},
  {1, 1, "^E up"},
  {2, 1, "^X down"},
  {3, 1, "^S left"},
  {4, 1, "^D right"},
  {5, 1, "^A word left"},
  {6, 1, "^F word right"},

  {0, 20, "INSERT"},
  {1, 21, "^W char"},
  {2, 21, "^N line"},
  {3, 21, "^P quotes ctl"},
  {4, 21, "^V toggle overtype"},

  {0, 40, "DELETE"},
  {1, 41, "^H char left"},
  {2, 41, "^G char right"},
  {3, 41, "^T word right"},
  {4, 41, "^Y line"},

  {0, 60, "SCREEN CHANGE"},
  {1, 61, "^R previous"},
  {2, 61, "^C next"},

  {3, 60, "OTHER"},
  {4, 61, "^L find string"},
  {5, 61, "^Q... more"},
  {6, 61, "^K... more"},
  {7, 61, "^U exit editor"},
};
static struct helpline ctl_k_help[] =
{
  {0, 0, "LINE STACK"},
  {1, 1, "X push line"},
  {4, 1, "Y push&delete"},
  {2, 1, "E pop line"},
  {3, 1, "W pop&insert"},
  {5, 1, "N push eol"},
  {6, 1, "T pop eol"},

  {0, 20, "EVALUATE"},
  {1, 21, "L this line"},
  {2, 21, "B this block"},
  {3, 21, "F block# 1"},

  {0, 40, "INSERT"},
  {1, 41, "V insert block"},
  {2, 40, "DELETE"},
  {3, 41, "G delete block"},

  {0, 60, "OTHER"},
  {1, 61, "M set mark"},
  {2, 61, "R restore block"},
};
static struct helpline ctl_o_help[] =
{
  {0, 0, "OPTIONS"},
  {1, 1, "C caps lock"},
  {2, 1, "S stamp on/off"},
};
static struct helpline ctl_q_help[] =
{
  {0, 0, "CURSOR MOVE"},
  {1, 1, "E top of block"},
  {2, 1, "X bottom"},
  {3, 1, "S begin line"},
  {4, 1, "D end of line"},

  {0, 20, "FIND&REPLACE"},
  {1, 21, "F find string"},
  {2, 21, "A find and replace"},

  {0, 40, "INSERT"},
  {1, 41, "N split line"},
  {2, 40, "DELETE"},
  {3, 41, "Y end of line"},

  {0, 60, "SCREEN CHANGE"},
  {1, 61, "R first"},
  {2, 61, "C last"},
  {3, 60, "OTHER"},
  {4, 61, "M goto mark"},
  {5, 61, "L redraw screen"},
};

static void
show_help (int row, int col, int n, struct helpline *h)
{
  for (; n > 0; n--, h++)
    {
      setcursor (row + h->row, col + h->col);
      c_puts (h->str);
    }
}

static void
show_bottom_help (int n, struct helpline *h)
{
  if (displayed_help == h)
    return;
  setcursor (17, -16);
  c_clrdown ();
  show_help (17, -16, n, h);
  displayed_help = h;
}


/*
 * display blocks and lines on screen
 */

static int
coleol (int row)
{
  char *p = buf[row];
  int col;

  col = 63;
  if (p[col] == ' ')
    while (col > 0 && p[col - 1] == ' ')
      col--;
  return col;
}

static char *
ptreol (int row)
{
  char *p, *q = buf[row];

  for (p = q + 64; p > q && p[-1] == ' '; p--);
  return p;
}

static void
show_line (int i, int col)
{
  char *p;
  int n;

  setcursor (i, col);
  p = &buf[i][col];
  n = ptreol (i) - p;
  if (n > 0)
    type_line (p, n);
  if (col + n < 64)
    c_clreol ();
}

static void
show_all_lines (int from)
{
  int i;

  for (i = from; i < 16; i++)
    show_line (i, 0);
}

static void
show_screen (void)
{
  setcursor (0, -11);
  c_printf ("%4u", (unsigned) SCR);
  show_all_lines (0);
}

static void
show_status (void)
{
  setcursor (4, -16);
  c_printf ("%3d  %3d", (int) row, (int) col);
  setcursor (5, -12);
  c_printf ("%02X", (unsigned char) buf[row][col]);
  if (readonly)
    {
      if (scr_changed ())
	{
	  memcpy (buf, blk, sizeof (blck));
	  c_bell ();
	  show_all_lines (0);
	}
    }
  else
    {
      setcursor (0, -4);
      c_putc (scr_changed () ? '*' : ' ');
    }
}

static void
show_snr (void)
{
  c_underline_on ();
  setcursor ( 8, -15), c_printf ("%-12.12s", search_str);
  setcursor (10, -15), c_printf ("%-12.12s", replace_str);
  c_underline_off ();
}

static void
show_options (void)
{
  setcursor (12, -15);
  c_printf ("%c %c %c %c",
	    caps ? 'C' : ' ',
	    overtype ? 'O' : 'I',
	    was_replacing ? 'R' : 'F',
	    stamp_changed ? 'S' : ' ');
}

static void
show_line_stack (void)
{
  char buf[65];

  if (linesp == linetop)
    memset (buf, '-', 64);
  else
    memcpy (buf, *linesp, 64);
  buf[64] = '\0';
  setcursor (16, -16);
  c_reverse ();
  c_printf ("line stack:  %2d %s", linetop - linesp, buf);
  c_normal ();
}

static void
show_frame (void)
{
  int i;

  setcursor ( 0, -16), c_puts ("blk #");
  setcursor ( 1, -16), c_printf ("%-10.10s", BLOCK_FILE->name);
  setcursor ( 3, -16), c_puts ("row  col");
  setcursor ( 5, -16), c_puts ("hex");
  setcursor ( 7, -16), c_puts ("find:");
  setcursor ( 9, -16), c_puts ("replace:");
  setcursor (11, -16), c_puts ("options:");
  if (readonly)
    {
      setcursor (0, -4);
      c_putc ('%');
    }
  c_reverse ();
  for (i = 0; i < 16; i++)
    {
      setcursor (i, -3);
      c_printf ("%2d", i);
    }
  c_normal ();
}

static void
show_all (void)
{
  c_normal ();
  c_clrscr ();
  show_frame ();
  show_snr ();
  show_options ();
  show_screen ();
  show_line_stack ();
  displayed_help = NULL;
  show_bottom_help (DIM (primary_help), primary_help);
}

static void
show_ctl (char c)
{
  setcursor (15, -7);
  if (c)
    c_printf ("^%c", c);
  else
    c_puts ("  ");
}

static int
prompt_for (char *prompt, struct lined *l, char *dflt)
{
  setcursor (16, -16);
  c_reverse ();
  c_printf ("%15s[%*s]%*s", prompt,
	   l->max_length, "",
	   80 - 17 - l->max_length, "");
  setcursor (16, 0);
  lined (l, dflt);
  c_normal ();
  show_line_stack ();
  return l->length;
}

static int
yesno (char *prompt)
{
  int c;

  setcursor (16, -16);
  c_reverse ();
  c_printf ("%15s?%*s", prompt, 64, "");
  setcursor (16, 0);
  do
    c = toupper (c_getkey ());
  while (c != 'N' && c != 'Y');
  show_line_stack ();
  return c == 'Y';
}

static void
word_from_cursor (char *p, int n)
{
  char *q = &buf[row][col];
  while (q < buf[16] && q[0] == ' ')
    q++;
  while (q > buf[0] && q[-1] != ' ')
    q--;
  while (q < buf[16] && q[0] != ' ')
    {
      *p++ = *q++;
      if (--n == 0)
	break;
    }
  *p = '\0';
}


/*
 * insert / delete character, word, line
 */

static void
insertc (char c)
{
  char *p = &buf[row][col], *q;

  for (q = &buf[row][coleol (row)] + 1; --q > p;)
    q[0] = q[-1];
  *p = c;
}

static void
deletec (void)
{
  char *p, *q = ptreol (row) - 1;

  for (p = &buf[row][col]; p < q; p++)
    p[0] = p[1];
  *p = ' ';
}

static void
insertl (int row)
{
  int i;

  for (i = 15; i > row; i--)
    COPY (buf[i], buf[i - 1]);
  memset (buf + i, ' ', sizeof buf[15]);
}

static void
deletel (int row)
{
  int i;

  for (i = row; i < 15; i++)
    COPY (buf[i], buf[i + 1]);
  memset (buf + 15, ' ', sizeof buf[15]);
}

static void
clear_endl (void)
{
  char *p = &buf[row][col], *q = ptreol (row);

  if (q > p)
    memset (p, ' ', q - p);
}

static void
strip_blanks (char **p, int *n)
{
  while (*n && (*p)[*n - 1] == ' ')
    --*n;
  while (*n && **p == ' ')
    --*n, ++*p;
}

static int
append_line (char *ln)
{
  char *q = ptreol (row);
  int n = 64, j = q - buf[row];

  strip_blanks (&ln, &n);
  if (j)
    q++, j++;
  if (64 - j < n)
    return 0;
  memcpy (q, ln, n);
  return 1;
}

static void
split_line (void)
{
  if (row == 15)
    clear_endl ();
  else
    {
      insertl (row);
      memcpy (buf[row], buf[row + 1], col);
      memset (buf[row + 1], ' ', col);
    }
}

static void
join_line (void)
{
  if (row < 15 && append_line (buf[row + 1]))
    deletel (row + 1);
  else
    c_bell ();
}

static void
deletew (void)
{
  char *p = &buf[row][col];
  int n = ptreol (row) - p;

  if (n <= 0)
    {
      join_line ();
      show_all_lines (row);
    }
  else
    {
      while (n && *p != ' ')
	deletec (), n--;
      while (n && *p == ' ')
	deletec (), n--;
    }
}

static void
inserts (void)
{
  int n;

  writebuf ();
  for (n = BLOCK_FILE->size; n > SCR; n--)
    if (!scr_empty (n - 1))
      break;
  for (; n > SCR; n--)
    scr_copy (n, n - 1);
  memset (buf, ' ', sizeof (blck));
  stamp_screen ();
  writebuf ();
  show_screen ();
}

static int
deletes (void)
{
  int n;

  if ((!scr_empty (SCR) || !block_empty (buf[0]))
      && !yesno ("delete screen"))
    return 0;
  writebuf ();
  for (n = SCR + 1; n < BLOCK_FILE->size; n++)
    scr_copy (n - 1, n);
  memset (buffer (BLOCK_FILE, BLOCK_FILE->size - 1, &n), ' ', BPBUF);
  update_ ();
  readbuf (SCR);
  show_screen ();
  return 1;
}


/*
 * move cursor
 */

static void
fwd_word (void)
{
  char *p = &buf[row][col];
  int n;

  while (p < buf[16] - 1)
    if (*p != ' ')
      p++;
    else
      break;
  while (p < buf[16] - 1)
    if (*p == ' ')
      p++;
    else
      break;
  n = p - buf[0];
  row = n / 64;
  col = n % 64;
}

static void
back_word (void)
{
  char *p = &buf[row][col];
  int n;

  while (buf[0] < p)
    if (p[-1] == ' ')
      p--;
    else
      break;
  while (buf[0] < p)
    if (p[-1] != ' ')
      p--;
    else
      break;
  n = p - buf[0];
  row = n / 64;
  col = n % 64;
}


/*
 * line- and block-stack
 */

static int
push_to_linestk (char *p, int n)
{
  if (linesp == linestk)
    {
      c_bell ();
      return 0;
    }
  linesp--;
  memcpy (*linesp, p, n);
  memset (*linesp + n, ' ', 64 - n);
  show_line_stack ();
  return 1;
}

static int
pushln (int row)
{
  return push_to_linestk (buf[row], 64);
}

static int
popln (char *to)
{
  if (linesp == linetop)
    {
      c_bell ();
      return 0;
    }
  memcpy (to, *linesp++, 64);
  show_line_stack ();
  return 1;
}

void
push_del_line (void)
{
  if (pushln (row))
    {
      deletel (row);
      show_all_lines (row);
    }
}

void
push_line (void)
{
  if (pushln (row) && row < 15)
    row++;
}

void
pop_spread_line (void)
{
  if (linesp < linetop)
    {
      insertl (row);
      popln (buf[row]);
      show_all_lines (row);
    }
  else
    c_bell ();
}

void
pop_line (void)
{
  if (popln (buf[row]))
    {
      show_line (row, 0);
      if (row > 0)
	row--;
    }
}

void
push_line_end (void)
{
  if (push_to_linestk (&buf[row][col], 64 - col))
    {
      clear_endl ();
      show_line (row, col);
    }
}

void
pop_line_end (void)
{
  int c = coleol (row);

  if (c >= 63)
    {
      c_bell ();
      return;
    }
  col = c ? c + 1 : 0;
  if (linesp < linetop && append_line (*linesp))
    {
      linesp++;
      show_line_stack ();
      show_line (row, col);
    }
  else
    c_bell ();
}


/*
 * find and replace
 */

static int
search_string (int prompt)
{
  int i, n, l;
  char *b, *p;

  l = strlen (search_str);
  if (prompt || l == 0)
    {
      char buf[65];
      word_from_cursor (buf, sizeof buf);
      search_lined.overtype = overtype;
      l = prompt_for ("Search: ", &search_lined, buf);
      show_snr ();
    }
  if (l == 0)
    return 0;
  b = buf[0];
  n = &buf[row][col] + 1 - b;
  p = search (b + n, BPBUF - n, search_str, l);
  if (!p)
    for (i = SCR + 1; i < BLOCK_FILE->size; i++)
      {
	b = block (BLOCK_FILE, i);
	p = search (b, BPBUF, search_str, l);
	if (p)
	  {
	    changescr (i);
	    show_screen ();
	    break;
	  }
      }
  if (!p)
    return 0;
  n = p - b;
  row = n / 64;
  col = n % 64;
  return 1;
}

static int
replace_string (int prompt)
{
  int i, lr, ls;

  if (!search_string (prompt))
    return 0;
  ls = strlen (search_str);
  lr = strlen (replace_str);
  if (prompt || lr == 0)
    {
      replace_lined.overtype = overtype;
      lr = prompt_for ("Replace: ", &replace_lined, NULL);
      show_snr ();
    }
  if (lr == 0)
    return 0;
  for (i = 0; i < ls; i++)
    deletec ();
  for (i = lr; --i >= 0;)
    insertc (replace_str[i]);
  show_line (row, col);
  return 1;
}


/*
 * keyboard input and dispatch
 */

static struct helpline *sub_help = NULL;
static int sub_help_len;

static void
show_sub_help (int sig)
{
#if defined EMX || defined WC_OS2V2
  signal (sig, SIG_ACK);
#elif defined SIGALRM
  signal (SIGALRM, SIG_IGN);
#endif
  show_bottom_help (sub_help_len, sub_help);
  setcursor (row, col);
}

static void
submenu (char key, int n, struct helpline *h)
{
  show_ctl (key);
  if (key)
    {
      sub_help_len = n;
      sub_help = h;
#if defined SIGALRM
      signal (SIGALRM, show_sub_help);
      alarm (1);
#else
      show_sub_help (0);
#endif
    }
  else
    {
#if defined SIGALRM
      signal (SIGALRM, SIG_IGN);
#endif
      show_bottom_help (DIM (primary_help), primary_help);
    }
  setcursor (row, col);
}

static void
do_ctlK (void)
{
  int c;

  submenu ('K', DIM (ctl_k_help), ctl_k_help);
  c = getckey ();
  submenu (0, 0, NULL);
  switch (c)
    {
    default:
      c_bell ();
    case ' ':
    case '\033':
      break;
    case 'Y':
    case 'Z':
      push_del_line ();
      break;
    case 'X':
      push_line ();
      break;
    case 'W':
      pop_spread_line ();
      break;
    case 'E':
      pop_line ();
      break;
    case 'N':
      push_line_end ();
      break;
    case 'T':
      pop_line_end ();
      break;
    case 'D':
      stamp_screen ();
      show_line (0, 0);
      row = 0;
      col = 2;
      overtype = 1;
    case 'L':
      show_bottom_help (0, NULL);
      writebuf ();
      evaluate (buf[row], 64);
      readbuf (SCR);
      show_all ();
      break;
    case 'B':
      show_bottom_help (0, NULL);
      writebuf ();
      load (BLOCK_FILE, SCR);
      readbuf (SCR);
      show_all ();
      break;
    case 'F':
      show_bottom_help (0, NULL);
      writebuf ();
      truncate_file ();
      load (BLOCK_FILE, 1);
      readbuf (SCR);
      show_all ();
      break;
    case 'M':
      mark.row = row;
      mark.col = col;
      mark.scr = SCR;
      break;
    case 'R':
      memcpy (buf, blk, sizeof (blck));
      show_screen ();
      break;
    case 'V':
      inserts ();
      break;
    case 'G':
      deletes ();
      break;
    }
}

static void
do_ctlO (void)
{
  int c;

  submenu ('O', DIM (ctl_o_help), ctl_o_help);
  c = getckey ();
  submenu (0, 0, NULL);
  switch (c)
    {
    default:
      c_bell ();
    case ' ':
    case '\033':
      break;
    case 'C':
      caps ^= 1;
      show_options ();
      break;
    case 'S':
      stamp_changed ^= 1;
      show_options ();
      break;
    }
}

static void
do_ctlQ (void)
{
  int c;

  submenu ('Q', DIM (ctl_q_help), ctl_q_help);
  c = getckey ();
  submenu (0, 0, NULL);
  switch (c)
    {
    default:
      c_bell ();
    case ' ':
    case '[':
      break;
    case 'E':
      row = 0;
      break;
    case 'X':
      row = 15;
      break;
    case 'S':
      col = 0;
      break;
    case 'D':
      col = coleol (row);
      break;
    case 'I':
      --col;
      col -= col % ED_TABW;
      break;
    case 'N':
      split_line ();
      show_all_lines (row);
      break;
    case 'Y':
      clear_endl ();
      show_line (row, col);
      break;
    case 'L':
      show_all ();
      break;
    case 'R':
      changescr (0);
      show_screen ();
      break;
    case 'C':
      changescr (BLOCK_FILE->size - 1);
      show_screen ();
      break;
    case 'M':
      changescr (mark.scr);
      row = mark.row;
      col = mark.col;
      show_screen ();
      break;
    case 'F':
      if (!search_string (1))
	c_bell ();
      was_replacing = 0;
      show_options ();
      break;
    case 'A':
      if (!replace_string (1))
	c_bell ();
      was_replacing = 1;
      show_options ();
      break;
    }
}

static int
do_key (char c)
/* interpretiert Zeichen erster Stufe */
{
  switch (c)
    {
    case '\033':		/* map Esc to ^Q, on Linux: Alt-F == ^Q-F */
    case 'Q' - '@':
      do_ctlQ ();
      break;
    case 'K' - '@':
      do_ctlK ();
      break;
    case 'O' - '@':
      do_ctlO ();
      break;
    case 'P' - '@':
      c = c_getkey ();
    default:
      if (overtype)
	{
	  buf[row][col] = c;
	  c_putc (c);
	}
      else
	{
	  insertc (c);
	  show_line (row, col);
	}
    case 'D' - '@':
      if (++col >= 64)
	{
	  col -= 64;
    case 'X' - '@':
	  if (row < 15)
	    row++;
	}
      break;
    case 'S' - '@':
      if (--col < 0)
	{
	  col += 64;
    case 'E' - '@':
	  if (row > 0)
	    row--;
	}
      break;
    case '\x7F':
    case 'H' - '@':
      if (col == 0)
	break;
      col--;
    case 'G' - '@':
      deletec ();
      show_line (row, col);
      break;
    case 'I' - '@':
      col += ED_TABW - col % ED_TABW;
      break;
    case 'W' - '@':
      insertc (' ');
      break;
    case 'A' - '@':
      back_word ();
      break;
    case 'F' - '@':
      fwd_word ();
      break;
    case 'T' - '@':
      deletew ();
      show_line (row, col);
      break;
    case 'M' - '@':
      col = 0;
      if (row < 15)
	row++;
      break;
    case 'N' - '@':
      insertl (row);
      show_all_lines (row);
      break;
    case 'Y' - '@':
      deletel (row);
      show_all_lines (row);
      break;
    case 'Z' - '@':
      memset (buf, ' ', sizeof (blck));
      show_screen ();
      break;
    case 'V' - '@':
      overtype ^= 1;
      show_options ();
      break;
    case 'L' - '@':
      if (was_replacing
	  ? replace_string (0)
	  : search_string (0))
	c_bell ();
      break;
    case 'R' - '@':
      if (SCR <= 0)
	{
	  c_bell ();
	  break;
	}
      changescr (SCR - 1);
      show_screen ();
      break;
    case 'C' - '@':
      if (SCR == BLOCK_FILE->size && !scr_changed ())
	{
	  c_bell ();
	  break;
	}
      changescr (SCR + 1);
      show_screen ();
      break;
    case 'U' - '@':
      writebuf ();
      truncate_file ();
      show_bottom_help (0, NULL);
      return 1;
    }
  return 0;
}

static void
free_bufs (void)
{
  if (buf)
    free (buf);
  if (linestk)
    free (linestk);
  if (blkstk)
    free (blkstk);
}

static int
alloc_bufs (int ls, int bs)
{
  buf = (line *) malloc (sizeof (blck));
  linestk = (line *) malloc (sizeof *linestk * ls);
  blkstk = (blck *) malloc (sizeof *blkstk * bs);
  if (!buf || !linestk || !blkstk)
    {
      free_bufs ();
      return 0;
    }
  linesp = linetop = linestk + ls;
  blksp = blktop = blkstk + bs;
  return 1;
}


/*
 * register additional actions for certain events:
 */

static void (*saved_on_stop) (void);
static void (*saved_on_continue) (void);
static void (*saved_on_winchg) (void);
jmp_buf after_stop;

static void
ed_on_stop (void)
{
  show_bottom_help (0, NULL);
  saved_on_stop ();
}

static void
ed_on_continue (void)
{
  saved_on_continue ();
  tty_interrupt_key (0);
  longjmp (after_stop, 1);
}

static void
ed_on_winchg (void)
{
  saved_on_winchg ();
  show_all ();
}

void
edit (int n, int r, int c)
{
  char *logn;
  volatile char intkey = tty_interrupt_key (0);

  logn = getenv ("LOGNAME");
  strncpy (log_name, logn ? logn : LOGNAME, sizeof log_name);
  switch (BLOCK_FILE->mode)
    {
    case FMODE_RO:
    case FMODE_ROB:
      readonly = 1;
      break;
    default:
      readonly = 0;
      break;
    }
  if (!alloc_bufs (32, 10))
    tHrow (THROW_OUT_OF_MEMORY);
  readbuf (n);
  row = r;
  col = c;

  saved_on_stop = on_stop;
  saved_on_continue = on_continue;
  saved_on_winchg = on_winchg;
  on_stop = ed_on_stop;
  on_continue = ed_on_continue;
  on_winchg = ed_on_winchg;
  setjmp (after_stop);

  displayed_help = NULL;
  show_all ();
  for (;;)
    {
      int key;

      setcursor (row, col);
      key = getwskey ();
      if (caps)
	key = change_case (key);
      if (do_key (key))
	break;
      show_status ();
    }
  free_bufs ();
  tty_interrupt_key (intkey);

  on_stop = saved_on_stop;
  on_continue = saved_on_continue;
  on_winchg = saved_on_winchg;
}
