/*
    brs_f_list: Browsing an Analysis Tool for C-source code;
	        Print a list of all functions in a project

    Copyright (C) 1994  Eckehard Stolz

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation (version 2 of the License).

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */




#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/stat.h>
#include <unistd.h>
#include <errno.h>
#include <malloc.h>
#include <time.h>

#include "hash.h"
#include "c-bat.h"
#include "clients.h"


/* Types for Hash-Entrys
 */
#define TYPE_FUNCTION      0x1
#define TYPE_FILE          0x2

#define MAX_LINE           256  /* maximum of line lenght */

/* Flags used in Hash-Table-Entry
 */
#define FLAG_PRINTED        0x1  /* Print Identifier                  */
#define FLAG_SORTED         0x2  /* Identifier has been sorted        */
#define FLAG_FILE_PRINTED   0x4  /* used for printing files just once */

#define max(x,y)   (((x) > (y)) ? (x) : (y))


/* Additional data, stored in Hash-Table. 
 */
typedef struct hash_add_data    t_hash_add;

struct hash_add_data
{ long           file;   /* file, where function is defined             */
  long           line;   /* line, where function is defined             */
  long           length; /* length of the function                      */
  short          stat;   /* static function                             */
  short          flags;  /* flags used for printing and sorting         */
  t_hash_add     *next;  /* next entry, if there are functions with the */
                         /* same name like an other (static) function   */
 };


/* formatstring used to print out the header of every function
   Can be overwritten by the -ft option !
 */
static char *template = 
"\n================================================================================\n"
"Function   : %s\n"
"File       : %s    Line %ld\n"
"Length     : %ld Lines\n"
"Declaration:\n%s"
"--------------------------------------------------------------------------------\n"
"%s\n";

/* seperator string between two list items
 */
static char *seperator = "   ";

static char output_line[MAX_LINE];         /* used for creating output */

static long line_length  = 80;   /* default for the lenght of a line (may be increased */
                                 /* if necessary )                                     */

static long list_indent  = 4;    /* indention for list of functions ... */


/* FLAGS
 */


/* if set to 1, we should print the leading comment of the actual function
 */
static short  flag_comments = 0;

/* if set to 1, we should print the functions called by actual function
 */
static short  flag_funct_called = 0;

/* if set to 1, we should print the files, where the functions called by 
   actual function are defined
 */
static short  flag_files_called = 0;

/* if set to 1, we should print the functions calling the actual function
 */
static short  flag_funct_calling = 0;

/* if set to 1, we should print the files, where the functions calling
   the actual function are defined
 */
static short  flag_files_calling = 0;

/* if set to 1, we sort the functions by their files, not by their 
   identifiers
 */
static short  flag_file_sort = 0;
static short  tmp_file_sort  = 0;

/* if set to 1, we should print the variables used by the actual function
 */
static short  flag_vars_used = 0;

/* if set to 1, we should print the files, where variables used by
   the actual function are defined
 */
static short  flag_files_used = 0;


/* set to 1 if we should NOT print information to stderr while processing
   functions, symbol-files, ...
 */
static short  quiet_flag = 0;


/* All symbols, which have to be printed, are stored in the hashtable
   and aditionally stored in this array (by their hash-indizes).
   So we don't have to check the whole hash-table for nonnull-entries
 */
static long *hash_array = NULL;
static long hash_array_cnt = 0;
static long hash_array_size = 0;

/* used for locally collecting sons of a function-node
 */
static struct son_e *son_hash_array = NULL;
static long  son_hash_array_cnt = 0;
static long  son_hash_array_size = 0;


/* prototypes
 */
extern void reset_hashtable(void);
extern long Add_ident_to_hash(char *ident, short flag, void *add_data );
int compare_hash_indizes( const void * p1, const void *p2);
int compare_hash_data( const void * p1, const void *p2);
void Add_to_Hash_Index_Array(long index);
void read_template( char *fname );
void output_list(char *title, long start, long end);
void print_funct_head( char *funct, t_hash_add *p_h_add );
void process_funct_called( char *funct_in, char *file_in, 
                      char *funct, char *file, char *module);
void process_funct_calling( char *funct_in, char *file_in, 
                      char *funct, char *file, char *module);


/* This function add's a symbol to the symbol-table. If the flag for
   printing is set, the index is added to the hash-index-array which
   causes the symbol to be printed after processing

   return-value is the index in the hash-table
 */
long Add_ident_to_hash(char *ident, short flag, void *add_data )
{ long index ;
  t_hash_data  h_entry;

  if( ident[0] == 0 )
    return; /* nothing to do with an empty item */

  strncpy( h_entry.ident, ident, SZE_HASH_IDENT - 1 );
  h_entry.ident[SZE_HASH_IDENT - 1] = 0;
  h_entry.flags = flag;
   
  h_entry.type = 1; /* used as counter */
  
  h_entry.data  = add_data;
  index = insert_token( &h_entry );

  return( index );
}


/* add index to hash array index to easyly find function-hash-indezes
   again
 */
void Add_to_Hash_Index_Array(long index)
{

  hash_array[hash_array_cnt] = index;
  hash_array_cnt++;

  if( hash_array_cnt == hash_array_size)
    { hash_array_size += HASH_ARRAY_INIT_SIZE;
      hash_array = realloc( hash_array,
		       sizeof( long ) * hash_array_size );

      if( !hash_array )
	{ fprintf(stderr, "Error ! Cannot allocate %ld Bytes of Memory\n",
		   sizeof( long ) * hash_array_size );
	  exit(1);
	 }
     }
 }

/* reads the template-string from the given file. The memory needed for
   this template-string is allocated via malloc
 */
void read_template( char *fname )
{ struct stat  stat_buffer;
  FILE  *fp; 
  long  size;
  char  *p;
  size_t   cnt_read;

  if(  stat( fname, &stat_buffer )  )
     return;  /* file does not exist */
   
  size = stat_buffer.st_size;
   
  fp = fopen( fname, "r" );
   
  template = malloc( size + 1 );
  if( !template )
   { fprintf(stderr, "error: malloc failed, not enough memory left !\n");
     exit(1);
    }
   
  p = template;
   
  while( size )
   {  
     cnt_read = fread( p, 1, stat_buffer.st_blksize, fp );
      
     size -= cnt_read;
     
     p += cnt_read;
    }
   
  /* terminating '\0' character
   */
  *p = 0;
   
  fclose( fp );
}


/* compare function for qsort. compares the two hash-idents with
   the indizes, p1 and p2 points to
 */
int compare_hash_indizes( const void * p1, const void *p2)
{ int ret;
  t_hash_data  data1, data2;
  t_hash_add   *add1, *add2;

   if( tmp_file_sort )
     { get_token( &data1,  *((long *)p1) );
       get_token( &data2,  *((long *)p2) );
       add1 = (t_hash_add *)data1.data;
       add2 = (t_hash_add *)data2.data;
      
       ret = strcmp( get_symbol( add1->file ), get_symbol( add2->file ) );
      }
   else
     ret = strcmp( get_symbol( *((long *)p1) ), get_symbol( *((long *)p2) ) );
   
   return( ret );
 }


/* compare function for qsort. compares the two hash-data-entries 
   p1 and p2 points to regarding to the filenames
 */
int compare_hash_data( const void * p1, const void *p2)
{ int ret;
   
   ret = strcmp( get_symbol( ((t_hash_add *)p1)->file ), 
                 get_symbol( ((t_hash_add *)p2)->file) );
   if( !ret )
     ret = ((t_hash_add *)p1)->line -  ((t_hash_add *)p2)->line ;
   return( ret );
 }

/* prints a list of hash-items between hash_array[start] and
   hash_array[end] (exclusive)
 */
void output_list(char *title, long start, long end)
{ long i, len;
  t_hash_data  h_data;
  t_hash_add   *p_h_add;
   
   
  printf("%s\n", title );
   
  for( i = 0; i < list_indent; i++ )
   { output_line[i] = ' ';
    }
  output_line[list_indent] = 0;
    
  for( i = start; i < end; i++ )
   { 
     get_token ( &h_data, hash_array[i] );   
            
     if( i != start )
       strcat( output_line, seperator);
      
     len = strlen( output_line ) + strlen(h_data.ident) + strlen(seperator);
     if( len  > line_length )
        { /* line too long, split
           */
          printf( "%s\n", output_line );
          output_line[list_indent] = 0;
         }
      
     strcat( output_line, h_data.ident );
    }
  printf( "%s\n", output_line );
}

/* get all functions called by funct_in ! Build a list of these functions,
   sort it and send this list to stdout

   file, and module are just the local string arrays defined in main which
   are given to this function to not have to allocate them again 
   (reducing stack size)
 */
void process_funct_called( char *funct_in, char *file_in, 
                           char *funct, char *file, char *module)
{ long  line, line_end, index_s, i, last_index = TOKEN_NOT_FOUND;
  short static_flag,
        ret = 0;
  long  hash_array_level = hash_array_cnt;
  t_hash_data   h_data;
  
   
  while( !ret )
   { /* get all called function of this function
      */
     ret = brs_get_funct_calls( funct_in, file_in, 1,
                     funct, file, &line, module, &static_flag);

     if( !ret )
       { /* we found a function 
            check if function allready in hash-table
          */
         if( !quiet_flag )
           fprintf( stderr, "." );
         
         if( flag_files_called )
           { /* get the definition position of all the functions
            
                Warning ! There might be a bug, if there is a static and a global
                function with the same name !!!!
              */
             ret = brs_get_funct_pos( funct, "", file, &line, &line_end, module);
             switch( ret )
               { case 0:
                 case RET_END:      break;
                 case RET_TIMEOUT:  fprintf(stderr, "No connection !\n");
                                    exit(1);
                 case RET_ERROR:    /* ignore this function (might be a unprototyped
                                       library function) if an error occurs
                                     */
                                    continue;
                 default:           fprintf(stderr, "Error pos 2: %d (%s %s) \n", ret,
                                                     file, module);
                                    exit(1);
                }
           }
            
         if( flag_funct_called )
          { /* look for this function in hash-table
             */
            index_s = search_token(funct, NULL);
            if( index_s == TOKEN_NOT_FOUND )
              { index_s = Add_ident_to_hash( funct, 0, NULL);
               }
           }
         else /* we look just for the files; test if we have allready
                 printed this filename
               */
          { index_s = search_token(file, &h_data);
            if( index_s == TOKEN_NOT_FOUND )
              { index_s = Add_ident_to_hash( file, FLAG_FILE_PRINTED, NULL);
               }
            else
              { if( h_data.flags & FLAG_FILE_PRINTED )
                  { continue; /* ignore this file, it has been listed allready */
                   }
                else
                  set_token_flag( index_s, h_data.flags | FLAG_FILE_PRINTED );
               }
           }
         
         /* add the - now definitly valid - index to the hash-array
            for later sorting and printing
          */
         Add_to_Hash_Index_Array(index_s);
            
         } /* endif !ret */
     else
      { switch( ret )
         { case 0: case RET_END:
                              break;
               
           case RET_TIMEOUT:  fprintf(stderr, "No connection !\n");
                              exit(1);
               
           case RET_ERROR:    /* This function is not defined in the project:
                                 continue as it has no sons !
                               */
                              break;
               
           default:           fprintf(stderr, "Error Pos 1: %d (%s %s) \n", ret,
                                               funct, file);
                              exit(1);
          }
       }
    } /* endwhile */
   
  /* now sort the array between hash_array[hash_array_level] and
     hash_array[hash_array_cnt-1]
   */
   
  qsort( &(hash_array[hash_array_level]),   /* base         */
         hash_array_cnt - hash_array_level, /* cnt elements */
         sizeof(long),                      /* size element */
         compare_hash_indizes );            /* cmp-function */
   
  /* now print the list of the functions in the hash-index array
   */ 
  if( flag_funct_called )
     output_list( "Functions called:", hash_array_level, hash_array_cnt );
  else
     output_list( "Functions called from files:", hash_array_level, hash_array_cnt );
   
  if( flag_files_called )
   { /* reset the FLAG_FILE_PRINTED-flags
      */
     for( i = hash_array_level; i < hash_array_cnt; i++ )
      { h_data.flags = get_token_flag( hash_array[i] );
        set_token_flag( hash_array[i], h_data.flags & (0xffff ^ FLAG_FILE_PRINTED) );
       }
    }
 
  /* reset hash array
   */
  hash_array_cnt = hash_array_level;
}

/* get all functions calling funct_in ! Build a list of these functions,
   sort it and send this list to stdout

   file, and module are just the local string arrays defined in main which
   are given to this function to not have to allocate them again 
   (reducing stack size)
 */
void process_funct_calling( char *funct_in, char *file_in, 
                            char *funct, char *file, char *module)
{ long  line, i, index_s;
  short static_flag,
        ret = 0;
  long  hash_array_level = hash_array_cnt;
  t_hash_data   h_data;
  
   
  while( !ret )
   { /* get all called function of this function
      */
     ret = brs_get_funct_called_by( funct_in, file_in, 1,
                     funct, file, &line, module, &static_flag);

     if( !ret )
       { /* we found a function 
            check if function allready in hash-table
          */
         if( !quiet_flag )
           fprintf( stderr, "." );
            
         if( flag_funct_called )
          { /* look for this function in hash-table
             */
            index_s = search_token(funct, NULL);
            if( index_s == TOKEN_NOT_FOUND )
              { index_s = Add_ident_to_hash( funct, 0, NULL);
               }
           }
         else /* we look just for the files; test if we have allready
                 printed this filename
               */
          { index_s = search_token(file, &h_data);
            if( index_s == TOKEN_NOT_FOUND )
              { index_s = Add_ident_to_hash( file, FLAG_FILE_PRINTED, NULL);
               }
            else
              { if( h_data.flags & FLAG_FILE_PRINTED )
                  { continue; /* ignore this file, it has been listed allready */
                   }
                else
                  set_token_flag( index_s, h_data.flags | FLAG_FILE_PRINTED );
               }
           }
         
         /* add the - now definitly valid - index to the hash-array
            for later sorting and printing
          */
         Add_to_Hash_Index_Array(index_s);
            
         } /* endif !ret */
     else
      { switch( ret )
         { case 0: case RET_END:
                              break;
               
           case RET_TIMEOUT:  fprintf(stderr, "No connection !\n");
                              exit(1);
               
           case RET_ERROR:    /* This function is not defined in the project:
                                 continue as it has no sons !
                               */
                              break;
               
           default:           fprintf(stderr, "Error Pos 1: %d (%s %s) \n", ret,
                                               funct, file);
                              exit(1);
          }
       }
    } /* endwhile */
   
  /* now sort the array between hash_array[hash_array_level] and
     hash_array[hash_array_cnt-1]
   */
   
  qsort( &(hash_array[hash_array_level]),   /* base         */
         hash_array_cnt - hash_array_level, /* cnt elements */
         sizeof(long),                      /* size element */
         compare_hash_indizes );            /* cmp-function */
   
  /* now print the list of the functions in the hash-index array
   */ 
  if( flag_funct_called )
     output_list( "Function called by:", hash_array_level, hash_array_cnt );
  else
     output_list( "Function is called from files:", hash_array_level, hash_array_cnt );
   
  if( flag_files_calling )
   { /* reset the FLAG_FILE_PRINTED-flags
      */
     for( i = hash_array_level; i < hash_array_cnt; i++ )
      { h_data.flags = get_token_flag( hash_array[i] );
        set_token_flag( hash_array[i], h_data.flags & (0xffff ^ FLAG_FILE_PRINTED) );
       }
    }
 
  /* reset hash array
   */
  hash_array_cnt = hash_array_level;
}


/* sort all functions by fcunction name (or function file )
   first sort the multiple occurences, then all functions
 */
void sort_functions()
{ long i;
  t_hash_data  h_data;
  t_hash_add   *p_h_add;
  t_hash_add   **p_list, **p1, *p2;
   
  /* first all multiple occurences
   */
  for( i = 0; i < hash_array_cnt; i++ )
   { get_token ( &h_data, hash_array[i] );
      
     p_h_add = h_data.data;
     if( p_h_add->next )
      { /* there is a list of function definition positions
           build a list of these positions for sorting
         */
        p_list = (t_hash_add **)malloc( (h_data.type + 1) * sizeof( t_hash_add *) );
        if( !p_list )
         { fprintf(stderr, "malloc failed !\n");
           exit(1);
          }
         
        p1 = p_list;
        /* We have a linked list of entries to sort. We first store them
           in an array, sort it by qsort and rebuild the linked list
           from the sorted array
         */
        *p1 = (t_hash_add *)h_data.data;
        while( (*p1)->next )
         { p2 = (*p1)->next;
           p1++;
           *p1 = p2;
          }
         
        /* terminating NULL entry, so we can find the end of 
           the list when relinking after sorting
         */
        p1++;
        *p1 = NULL;
         
        qsort( p_list,                  /* base         */
               h_data.type,             /* cnt elements */
               sizeof(t_hash_add *),    /* size element */
               compare_hash_data );     /* cmp-function */
         
        p1 = p_list;
        p2 = *(p1 + 1);
         
        h_data.data = *p1;
         
        /* link the list together again. Now this list is sorted
           by filenames (automatically NULL-terminated by the
           last NULL-entry appended above)
         */
        while( *p1 )
          { (*p1)->next = p2;
            p1++;
            if( *p1 )
              p2 = *(p1 + 1);
           }
         
        free( p_list );
        
       } /* endif */
    } /* next i */
   
 /* now we sort all the function-entries in the hash_array
  */
 if( flag_file_sort )
    tmp_file_sort = 1;
   
 qsort( hash_array,              /* base         */
        hash_array_cnt,          /* cnt elements */
        sizeof(long),            /* size element */
        compare_hash_indizes );  /* cmp-function */

 tmp_file_sort = 0;
   
}


static char master_dir[MAX_FILENAME] = "";
static char new_dir[MAX_FILENAME] = "";
   
/* print the header of the function to stdout
 */ 
void print_funct_head( char *funct, t_hash_add *p_h_add )
{ char *comment, *definition;
  short ret;
   

  if( brs_get_file_dir( get_symbol( p_h_add->file ), new_dir) )
   { fprintf(stderr, "Could not get working dir. of file '%s'\n",
         get_symbol( p_h_add->file ));
     exit(1);
    }
  
  if( !getcwd( master_dir, MAX_FILENAME) )
   { fprintf(stderr, "Could not get current working directory\n" );
     exit(1);
    }
   
  if( chdir( new_dir ) )
   { fprintf(stderr, "Could not change to directory '%s'\n", new_dir );
     exit(1);
    }
   
  ret = brs_cut( 0, get_symbol( p_h_add->file ), funct, 
              p_h_add->line, &definition, &comment, NULL);
   
  if( chdir( master_dir ) )
    { /* error: cannot set back old working directory
         ignore, because cutting from the file suceeded !
       */
      return;
     } 
   
  if( ret )
   { fprintf(stderr, "Function '%s' file '%s' line %ld not found !\n",
         funct, get_symbol( p_h_add->file ), p_h_add->line );
     exit(1);
    }
   
  if( flag_comments && comment )
   { printf( template, funct, get_symbol( p_h_add->file ), 
            p_h_add->line, p_h_add->length, definition, comment );
    }
  else
   { printf( template, funct, get_symbol( p_h_add->file ), 
            p_h_add->line, p_h_add->length, definition, "" );
    }
   
  if( definition )
    free( definition );
   
  if( comment )
    free( comment );
}


main( int argc, char *argv[] )
{ short  flag;
  short  ret,
         ret2,
         code,
         ignore_this_symbol,
         static_flag;
   
  int    server;
   
  FILE   *fp;
   
  char   funct_in[STRINGSIZE];
  char   funct[STRINGSIZE];
  char   file[2*STRINGSIZE];
  char   file_in[2*STRINGSIZE];
  char   module[2*STRINGSIZE];

  t_hash_add   *p_h_add,
               *p_h_add_s;

  t_hash_data  h_data,
               h_data_f,
               h_data_s;

  long         i,
               j,
               index,
               line,
               line_end,
               cnt_hash_items,
               line2,
               index_s;

   
   if( argv[1][1] == '?' )
     { fprintf(stderr, "Usage: brs_flist [Options] \n" );
       fprintf(stderr, "Prints a list of all functions defined in a project\n");
       fprintf(stderr, "This program needs the browser-server run in background !\n");
       fprintf(stderr, "\nOptions:\n");
       fprintf(stderr, "-b <browser-file>    Connect to server processing this browser file\n");
       fprintf(stderr, "-ft <template-file>  Use this file as template for the function-list\n");
       fprintf(stderr, "                     It will be used for a printf-statement with the\n");
       fprintf(stderr, "                     following arguments: \n");
       fprintf(stderr, "                     function (char *), file (char *), line (long)\n");
       fprintf(stderr, "                     number of lines (long)\n");
       fprintf(stderr, "                     \n");
       fprintf(stderr, "-s                   Sort functions by file, not by name\n");
       fprintf(stderr, "-q                   Quiet: Do not print information to stderr\n");
       fprintf(stderr, "-C                   Print heading comments\n");
       fprintf(stderr, "-c                   Print functions called by actual function\n");
       fprintf(stderr, "-c-                  Print only filenames of functions called\n");
       fprintf(stderr, "-g                   Print functions calling actual function\n");
       fprintf(stderr, "-g-                  Print only filenames of functions calling\n");
      
#if 0      
       fprintf(stderr, "-v                   Print variables used by actual function\n");
       fprintf(stderr, "-v-                  Print only filenames of variables used\n");
#endif

       fprintf(stderr, "-l<char per line>    set maximum characters per output line\n");
       fprintf(stderr, "-i<indent>           indent function-lists by <indent> characters \n");
       exit(1);
      }

   /* we have to preprocess args for looking for the
      -q flag. We need to initialize the hash-table before
      fully parsing arguments
    */   
   for( i = 1; i < argc; i++ )
     { if( argv[i][0] == '-' )
         { switch( argv[i][1] )
             { case 'q' : /* quiet-flag: no Info to stderr */
	         quiet_flag = 1;
	         break;

               default:
                 break;
              }
           }
       }
   
   if( !quiet_flag )
     fprintf( stderr, "Initializing Hashtable ...\n");

   init_hashtable ();

   for( i = 1; i < argc; i++ )
     { /* ordinary option given: */
       switch( argv[i][1] )
	  { case 'b' : /* Browser-Filename */
	      if( argv[i][2] )
	        { /* Filename given without blank */
                  server = brs_get_server_id( argv[i] + 2 );
		 }
	      else
	        { /* Browser-filename is in the next argument
		   */
		  i++;
                  server = brs_get_server_id( argv[i] );
		 }
              if( server < 0 )
                { fprintf(stderr, "Server with browser file \"%s\" not found !\n",
                         argv[i] );
                  exit(1);
                 }
              if( (ret = brs_connect_server( server )) > 0  )
                { fprintf(stderr, "Server connection failed (Error: %hd) !\n",
                         ret );
                  exit(1);
                 }
	      break;

	    case 'f' : /* Stop-Filename */
            
	      if( argv[i][2] == 't' )
	        { /* template file */
                  if( argv[i][3] )
                     read_template( argv[i] + 3 );
		  else
	            { /* template-filename is in the next argument
		       */
		      i++;
		      read_template( argv[i] );
		     }
                 }
              else
	        { fprintf(stderr, "Unknown Option \"%s\" !\n", argv[i] );
	          fprintf(stderr, "Type \"brs_flist -?\" for help !\n" );
	          exit(1);
                 }
	      break;
         
	    case 'l' : /* line size */
              if( argv[i][2] )
                 line_length = atoi( argv[i] + 2 );
	      else
	        { /* next argument
		   */
		  i++;
		  line_length = atoi( argv[i] );
		 }
	      break;
   
	    case 's' : /* file sort instead of funct sort */
              flag_file_sort = 1;
	      break;
   

	    case 'i' : /* list indention */
              if( argv[i][2] )
                 list_indent = atoi( argv[i] + 2 );
	      else
	        { /* next argument
		   */
		  i++;
		  list_indent = atoi( argv[i] );
		 }
	      break;

	    case 'C' : /* print leading comments */

              flag_comments = 1;            
	      break;

            case 'c' : /* functions called by act. funct */
	      if( argv[i][2] == '-' )
               { flag_files_called = 1;
		 }
              else
               { flag_funct_called = 1;
		 }
	      break;
         
            case 'g' : /* functions called by act. funct */
	      if( argv[i][2] == '-' )
               { flag_files_calling = 1;
		 }
              else
               { flag_funct_calling = 1;
		 }
	      break;
         
            case 'v' : /* functions called by act. funct */
	      if( argv[i][2] == '-' )
               { flag_files_used = 1;
		 }
              else
               { flag_vars_used = 1;
		 }
	      break;
            
            case 'q' : /* quiet-flag: no Info to stderr */
	      quiet_flag = 1;
	      break;

	    default:
	      fprintf(stderr, "Unknown Option \"%s\" !\n", argv[i] );
	      fprintf(stderr, "Type \"brs_f_list -?\" for help !\n" );
	      exit(1);
	   } 
      } /* next */
   
   /* initialize hash-index-array used to hold the hash-indizes of the
      processed functions
    */
   hash_array      = malloc(sizeof(long) * HASH_ARRAY_INIT_SIZE);

   if( !hash_array )
     { fprintf(stderr, "Error ! Cannot allocate %ld Bytes of Memory\n",
		       sizeof( long ) * HASH_ARRAY_INIT_SIZE );
       exit( 1 );
      }

   hash_array_size = HASH_ARRAY_INIT_SIZE;
   hash_array_cnt  = 0;

   if( !quiet_flag )
     fprintf( stderr, "Reading list of defined functions ...\n");

   /* Get all functions defined in the project and store them in the
      Hash-table
    */
   while( 1 )
    { /* get top-function-info
       */
      ret = brs_get_all_funct( funct, file, &line, module, &static_flag);

      switch( ret )
        { case 0:
          case RET_END:      break;
      
          case RET_TIMEOUT:  fprintf(stderr, "No connection !\n");
                             exit(1);
          case RET_ERROR:    fprintf(stderr, "Error pos 3: %s %s \n", file, module);
                             exit(1);
          default:           fprintf(stderr, "Error pos 4: %d (%s %s) \n", ret,
                                              file, module);
                             exit(1);
         }
   
      if( ret == RET_END )
        break;
   
      index = search_token( funct, &h_data);
   
      if( index != TOKEN_NOT_FOUND )
        { /* function is defined more than once ! Add this position to the list of
             this hash-entry ! Assuming thet brs_get_all_funct does NEVER return
             a function twice ! This should be assured by the server, everything
             else is a bug !
           */
          p_h_add = h_data.data;
      
          /* find end of list
           */
          while( p_h_add->next )
            { p_h_add = p_h_add->next;
             }
      
          p_h_add->next = malloc( sizeof( t_hash_add ) );
          if( !p_h_add->next )
            { fprintf(stderr, "Not enough physical memory available !\n");
              exit(1);
             }
      
          /* add this entry to hash index array too (multiple occurence
             of functions with the same identifier)  
           */
          Add_to_Hash_Index_Array(index);
      
          h_data.type++; /* used as counter */
          update_token( &h_data, index );
      
          /* continue with processing: hash-add is stored in p_h_add
           */
          p_h_add = p_h_add->next;
         }
      else
       { /* normal case: function is added in Hash-table the first time
          */
      
         p_h_add = malloc( sizeof( t_hash_add ) );
         if( !p_h_add )
           { fprintf(stderr, "Not enough physical memory available !\n");
             exit(1);
            }
               
         index = Add_ident_to_hash( funct, 0, p_h_add );

         Add_to_Hash_Index_Array(index);
        }
   
      /* store file and line in hash-add 
       */

      p_h_add->file    = search_token( file, NULL);
      if( p_h_add->file == TOKEN_NOT_FOUND )
        { p_h_add->file = Add_ident_to_hash( file, 0, NULL);
         }
      p_h_add->line    = line;
      p_h_add->stat    = static_flag;
      p_h_add->flags   = 0;
      p_h_add->next    = NULL;
   
     } /* endwhile(1) */


   if( !quiet_flag )
     fprintf( stderr, "processing function ...\n");

   /* freeze number of hash-entries !
    */
   cnt_hash_items = hash_array_cnt;

   /* sort all functions by name or by ident
    */
   sort_functions();

   /* walk through the hash-index-array. initially, there is only the top
      node in it, but it (and therefore hash_array_cnt) will grow as long
      as there are unprocessed functions
    */
   for( i = 0; i < cnt_hash_items; i++ )
     { get_token ( &h_data, hash_array[i] );

       ret = 0;

       if( !quiet_flag )
         fprintf( stderr, "%ld/%ld: %s ", i+1, hash_array_cnt, h_data.ident );
   
       p_h_add = NULL;
       if( h_data.data )
         p_h_add = (t_hash_add *)h_data.data;
       
       /* loop for every occurence of functions with the same identifier, that
          has allready been printed
        */
       while( p_h_add->flags & FLAG_PRINTED )
        { if( p_h_add->next )
            p_h_add = p_h_add->next;
          else
            break;
         }
      
       if( p_h_add->flags & FLAG_PRINTED )
        { /* bug: why are all entries marked printed ?
           */
          fprintf(stderr, "Error: no matching entry (%s) !\n", h_data.ident );
          exit(1);
         }
   
       p_h_add->flags |= FLAG_PRINTED;
   
       /* Now print the header of the function
        */
       strcpy( funct_in, h_data.ident );
                   
       get_token ( &h_data_f, p_h_add->file );      
       strcpy( file_in, h_data_f.ident );
   
       /* get function position to get the end-line of the function.
        */
       ret = brs_get_funct_pos( funct_in, file_in, 
                          file, &line, &line_end, module);
       switch( ret )
         { case 0:
           case RET_END:      break;

           case RET_TIMEOUT:  fprintf(stderr, "No connection !\n");
                              exit(1);
           case RET_ERROR:    fprintf(stderr, "Error pos 5: %s %s \n", file, module);
                              exit(1);
           default:           fprintf(stderr, "Error pos 6: %d (%s %s) \n", ret,
                                               file, module);
                              exit(1);
          }
      
       p_h_add->length = line_end - line;
      
       /* print the header of the function (with definition and comment)
        */
       print_funct_head( funct_in, p_h_add );
      
       /* if we should print the list of the called functions (or just their
          files), set ret to 0 so the while-condition is true
        */
       if( flag_funct_called || flag_files_called )
          process_funct_called( funct_in, file_in, funct, file, module);
      
       if( flag_funct_calling || flag_files_calling )
          process_funct_calling( funct_in, file_in, funct, file, module);
   
       if( !quiet_flag )
         fprintf( stderr, "\n" );
   
      } 

   brs_disconnect_server();

 } /* end main */



