/* General purpose program for Checker
   Copyright 1994 Tristan Gingold
		  Written July 1994 by Tristan Gingold

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License as
published by the Free Software Foundation; either version 2 of the
License, or (at your option) any later version.

This library is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
General Public License for more details.

You should have received a copy of the GNU General Public License 
along with this program; see the file COPYING.  If not, write to
the Free Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

 The author may be reached (Email) at the address gingold@amoco.saclay.cea.fr,
 or (US/French mail) as Tristan Gingold 
   			  8 rue Parmentier
   			  F-91120 PALAISEAU
   			  FRANCE
*/
#include <stdio.h>
#include <malloc.h>
#include <getopt.h>
#include <unistd.h>

#define HAVE_BFD

#ifdef HAVE_BFD
#include <bfd.h>
#endif

extern char *version;

/* The maximum line length. */
#define MAXLEN 2048

/* Where gcc can be found. */
#ifndef GCC_PATH
#define GCC_PATH "/usr/bin/gcc"
#endif

/* Where g++ can be found. */
#ifndef GXX_PATH
#define GXX_PATH "/usr/bin/g++"
#endif

/* Where the libraries of Checker are. */
#ifndef CHECKER_PATH
#define CHECKER_PATH "/usr/local/lib/checker/"
#endif

/* The basename of argv[0] */
char *program_name;

/* Give some tips and exit. */
void
give_tips(void) /* __attribute__ ((noreturn)) */
{
 fprintf(stderr, "Try `%s --help' for more informations.\n", program_name);
 exit(1);
}

/* Display the current version. */
void
disp_version(int argc, char *argv[])
{
  printf("Checker version %s\n", version);
}

/* Display the help. */
void
help(int argc, char *argv[])
{
  puts("Usage: checker COMMAND [OPTIONS] [ARGUMENTS]");
  puts("Options and arguments can be set according to the commands:\n");
  puts("checker -v");
  puts("checker --version");
  puts("  output version information and exit. No options.\n");
  puts("checker -h");
  puts("checker --help");
  puts("  display help and exit. No options.\n");
  puts("checker gcc [GCC_ARGUMENTS]...");
  puts("checker -gcc [GCC_ARGUMENTS]...");
  puts("  compile with GNU CC for Checker.\n");
  puts("checker g++ [G++_ARGUMENTS]...");
  puts("checker -g++ [G++_ARGUMENTS]...");
  puts("  compile with GNU G++ for Checker.\n");
#ifdef HAVE_BFD 
  puts("checker -i [OPTION]... [program [file]]");
  puts("checker --interpret [OPTION]... [program [file]]");
  puts("  interpret the FILE (or stdin) produced by PROGRAM (or a.out).");
  puts("  -o, --output OUTFILE  Set the output file.");
  puts("  -r, --realpath        Give the real path.");
  puts("  -b, --basename        Give the basename.");
#endif
}

#ifdef HAVE_BFD
/* Interpret a file:
 * replace each line   "$       pc=0xAAAAAAAA" 
 * by		       "        pc=0xAAAAAAAA in function() at file:line"
 * The informations are taken from the binary file.
 */
void
interpret(int argc, char *argv[])
{
 FILE *in = stdin;
 FILE *out = stdout;
 bfd *abfd;
 asection *textsection;
 unsigned int storage_needed;
 asymbol **symbol_table;
 unsigned int number_of_symbols;
 char *filename;
 char *target = NULL;
 char buffer[MAXLEN];
 bfd_vma offset;
 int i;
 char *srcname;
 char *srcname1;
 char *functionname;
 unsigned int line;
 
 static int path_flag = 0;
 static struct option interpret_opts[]={
 { "output", 1, 0, 'o'},
 { "realpath", 0, &path_flag, 1},
 { "basename", 0, &path_flag, 2},
 { 0, 0, 0, 0}};

 /* Getopt will emit a message in case of error. */ 
 opterr = 1;
 
 /* Read the arguments. */
 while (1)
   {
     i = getopt_long(argc, argv, "o:rb", interpret_opts, &i);
     if (i == -1)
       break;
     switch (i)
       {
         case 0:	/* 'realpath' & 'basename' */
           break;
         case 'o':	/* -o: set the output file. */
           if (strcmp("-", optarg) == 0)
             out = stdout;
           else
             {
               out = fopen(optarg, "w");
               if (out == NULL)
                 {
                   fprintf(stderr, "%s: can't open %s", program_name, optarg);
                   perror("");
                   exit(2);
                 }
             }
           break;
         case 'r':	/* -r: give the realpath */
           path_flag = 1;
           break;
         case 'b':	/* -b: give the basename. */
           path_flag = 2;
           break;
         default:	/* error */
           exit(2);
       }
   }
   
 /* PROGRAM can follow. */ 
 if (optind < argc)
   filename = argv[optind++];
 else
   filename = "a.out";	/* by default */
   
 /* FILE can follow. */
 if (optind < argc)
   {
     if (strcmp("-", argv[optind]) == 0)
       in = stdin;
     else
       {
         in = fopen(argv[optind], "r");
         if (in == NULL)
           {
             fprintf(stderr, "%s: can't open file %s", program_name, argv[optind]);
             perror("");
             exit(2);
           }
       }
     optind++;
   }
 
 if (optind != argc)
   {
     fprintf(stderr, "Too many arguments\n");
     give_tips();
   }
 
 /* Open the PROGRAM file */    
 abfd = bfd_openr(filename, target);
 if (abfd == NULL)
   {
     fprintf (stderr, "%s: ", program_name);
     bfd_perror (filename);
     exit(2);
   }
 /* PROGRAM must be an executable. */
 if (bfd_check_format(abfd, bfd_object) == false)
   {
     fprintf (stderr, "%s: %s is not an object file\n", program_name, filename);
     exit(2);
   }
 
 /* Use the ".text" section */
 textsection = bfd_get_section_by_name (abfd, ".text");
 
 /* Read the symbol table. */
 storage_needed = bfd_get_symtab_upper_bound (abfd);
 if (storage_needed == 0)
   {
     fprintf (stderr, "%s: no symbols!\n", program_name);
     exit(2);
   }
 symbol_table = (asymbol**)malloc(storage_needed);
 if (symbol_table == (asymbol**)0)
   {
     fprintf(stderr, "%s: virtual memory exhausted\n", program_name);
     exit(3);
   }
 number_of_symbols = bfd_canonicalize_symtab(abfd, symbol_table);
 
 /* Read lines of the file*/
 while (fgets(buffer, MAXLEN -2, in))
   {
     if (buffer[0] == '$')
       {
         /* The format is: "$\tpc=0x%08x\n" */
         /* Read the pc */
         offset = 0;
         for (i = 7; buffer[i] != '\n'; i++) 
           {
             offset <<= 4;	/*  *16  */
             if (buffer[i] >= '0' && buffer[i] <= '9')
               offset += buffer[i] - '0';
             else if (buffer[i] >= 'A' && buffer[i] <= 'F')
               offset += buffer[i] - 'A' + 10;
             else if (buffer[i] >= 'a' && buffer[i] <= 'f')
               offset += buffer[i] - 'a' + 10;
             else break;
           }

	 /* Find the symbols for offset. */
         if (bfd_find_nearest_line(abfd, textsection, symbol_table, offset,
         			    &srcname, &functionname, &line) == false)
           {
             fprintf (stderr, "%s: ", program_name);
             bfd_perror (filename);
             exit(2);
           }
         /* The srcname can be processed. */
         switch (path_flag)
           {
             case 0:
               srcname1 = srcname;
               break;
             case 1:
               /* SRCNAME is processed only if it begins with a '/', since
                * it might not belong to the current directory. */
               if (srcname[0] == '/')
                 {
                   /* Replace by the real path: follow the links, simplify */
                   realpath(srcname, buffer);
                   srcname1 = buffer;
                 }
               else
                 srcname1 = srcname;
               break;
             case 2:
               /* Replace by the basename. */
               srcname1 = srcname + strlen(srcname);
               while (srcname1 != srcname && srcname1[-1] != '/')
                 srcname1--;
               break;
             default:
               srcname1 = srcname;
           }
         fprintf (out, "\tpc=0x%08x in %s() at %s:%d\n",
                       (unsigned int)offset, functionname, srcname1, line);
       }
     else
       {
         /* Nothin to do. */
         fputs(buffer, out);
       }
   }   
   
 /* Close the files. */
 bfd_close(abfd);
 if (in != stdin)
   fclose(in);
 if (out != stdout)
   fclose(out);
 return;
}
#endif /* HAVE_BFD */


#define NEWARGS 1

/* Compile with gcc */
void
gcc(int argc, char *argv[])
{
 char *newargv[argc + NEWARGS];
 int i;
 
 newargv[0] = "checkergcc";
 newargv[1] = "-B" CHECKER_PATH "/";
 
 for (i=1; i < argc; i++)
   newargv[NEWARGS + i] = argv[i];
   
 execv(GCC_PATH, newargv);
 execvp("gcc", newargv);
 perror("gcc not found!");
 exit(1);
}

/* Compile with g++ */
void
gplusplus(int argc, char *argv[])
{
 char *newargv[argc + NEWARGS];
 int i;
 
 newargv[0] = "checkerg++";
 newargv[1] = "-B" CHECKER_PATH "/";
 
 for (i=1; i < argc; i++)
   newargv[NEWARGS + i] = argv[i];
   
 execv(GXX_PATH, newargv);
 execvp("g++", newargv);
 perror("g++ not found!");
 exit(1);
}

/* Describe which function to call according to flags. */
struct flags
{
  char short_flag;
  char *long_flag;
  void (*function)(int, char **);
};

struct flags commands[] = {
{ 'h', "--help",	&help},
{ 'v', "--version",	&disp_version},
#ifdef HAVE_BFD
{ 'i', "--interpret",	&interpret},
#endif
{  0,  "gcc",		&gcc},
{  0,  "-gcc",		&gcc},
{  0,  "g++",		&gplusplus},
{  0,  "-g++",		&gplusplus},
{ 0,   (char*)0 }};

int
main(int argc, char *argv[])
{
 struct flags *aflag;
 
 /* Extract the basename of argv[0]. */
 program_name = argv[0] + strlen(argv[0]);
 while (program_name != argv[0] && program_name[-1] != '/')
   program_name--;
 
 if (strcmp("checkergcc", program_name) == 0)
   gcc(argc, argv);
 
 if (strcmp("checkerg++", program_name) == 0)
   gplusplus(argc, argv);
   
 if (argc < 2)
   {
     fprintf(stderr, "Usage: %s command [args]\n", program_name);
     give_tips();
   }
 
 if (argv[1][0] == '-' && argv[1][1] != '\0' && argv[1][2] == '\0')
   {
     /* A short flag. */
     for (aflag = commands; aflag->long_flag; aflag++)
       {
         if (argv[1][1] == aflag->short_flag)
           {
             (*aflag->function)(argc - 1, &argv[1]);
             exit(0);
           }
       }
   }
 else
   {
     /* Something else */
     for (aflag = commands; aflag->long_flag; aflag++)
       {
         if (strcmp(argv[1], aflag->long_flag) == 0)
           {
             (*aflag->function)(argc - 1, &argv[1]);
             exit(0);
           }
       }
   }
 fprintf(stderr, "%s: unknown command %s\n", argv[0], argv[1]);
 give_tips();
 return 1;
}
