/* Parse arguments for Checker
   Copyright 1993 Tristan Gingold
		  Written October 1993 by Tristan Gingold

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License as
published by the Free Software Foundation; either version 2 of the
License, or (at your option) any later version.

This library is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
General Public License for more details.

You should have received a copy of the GNU General Public License 
along with this program; see the file COPYING.  If not, write to
the Free Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

 The author may be reached (Email) at the address gingold@amoco.saclay.cea.fr,
 or (US/French mail) as Tristan Gingold 
   			  8 rue Parmentier
   			  F-91120 PALAISEAU
   			  FRANCE
*/

#include "checker.h"	
#include <fcntl.h>
#include <sys/param.h>
#include <signal.h>
#include <limits.h>
#include "errlist.h"
#include "message.h"

#ifndef DEFAULT_HOME
#define DEFAULT_HOME "/tmp"
#endif
#ifndef OPTIONS_FILE
#define OPTIONS_FILE "/.checker"	/* never forget the '/' */
#endif
#ifndef DEFAULT_OUT_FILE
#define DEFAULT_OUT_FILE "/dev/tty"
#endif

/* If true, the initial message is not displayed */
static int silent_flag = 0;

/* If true, the program is stopped just before main() with kill(2).  This is 
 * helpful for debugging. */
static int stop_flag;

/* A copy of argv[0] */
static char *full_progname;

/* A pointer to the basename of argv[0] in full_progname */
static char *progname;

/* The full path of the program */
char *chkr_prog_path;

/* Some infos of the output file.  These are used to check if the user has
 *  tried to handle the output file (ie messages file). */
static dev_t st_dev;
static ino_t st_ino;

/* The name of the output file. CHKR_OUT_FILE is made from CHKR_OUT_PROTO.
 *  Its opening mode is set by CHKR_OUT_APPEND. */
static char *chkr_out_file;

/* True if the report must be appended to CHKR_OUT_FILE. */
static int chkr_out_append;

/* The original name of the output file. Can contains %p or %n.
   This is used when CHKR_OUT_FILE must be remade because %p or %n has 
    changed. */
static char *chkr_out_proto;

/* True if the report must be appended to the file. */
static int append_flag;

/* True if the verbose flag was set */
static int verbose_flag = 0;

/* Version of Checker. See version.c */
extern char *version;

/* Saved pid in numeric (my_pid) and in characteres (my_pid_c[]). */
pid_t my_pid;
char my_pid_c[6];

/* Size of the NULL zone */
unsigned int null_pointer_zone = 0x04;

/* Info on the binaries files of this processus (usually only the binary
 * file */
struct object *objects = (struct object*)0;

#ifdef MDCHECKER
int chkr_ld_options = 0;
#else
/* The address of the symbol is set according to the options. */
extern void *___chkr_ld_opts__;

/* The options */
int chkr_ld_options = (int)&___chkr_ld_opts__;
#endif

/* True if must detect garbage at the end */
static int do_detector_at_end;

#ifdef CHKR_SAVESTACK
/* True if the user don't want to use the symbols table */
int nosymtab_flag;
#endif

#ifdef CHKR_PROFILE
/* True if profile infos are displayed at the end */
int profile_flag = 0;
#endif

/* True if warns each time the alignment arg for memalign() is not a 
 *  power of 2 */
int memalign_align2;

/* True if Checker has already been initialized */
extern int chkr_is_init;

/* This is for parse_args() */
#define isspace(c) ((c)==' ' || (c)=='\t')

extern void init_morecore(void);
static int parse_args(char *input, char *ouput);
static void parse_opt_line(char *line);
static void print_message(char **message);
static void read_config_file(char *file);
static int ctod(const char c);
static void store_output_spec(void);
extern void save_signals(void);
extern int parse_disable(const char *str);
extern void set_malloc0_behavior(const char *opt);
extern void __chkr_garbage_detector(void);
extern void chkr_disp_shlibs(void);
extern void init_main_object(int linked, char *argv0, int nlib, char **libs);
extern void chkr_init_machine(void);
void make_pid(void);
void open_output_file(void);
void chkr_do_end(void);

#include "parse-args.mes"
   
/* ___chkr_init_chkr is called before main() and aims at initializing 
 * Checker.  The arguments are those of main().
 * NOTE: this function must not call malloc().  Only sys_malloc is OK. */
void
___chkr_init_chkr(int linked, int nlibs, char **libs,
		  int argc, char *argv[], char *argp[])
{
 static int already_init = 0;
 char *args;
 char cffile[MAXPATHLEN];

 /* ARGC is nul only if this program was called by ldd(1). Display the
  *  message and exit. */ 
 if (!already_init && argc == 0)
   {
     print_message(init_message);
     _exit(0);
   }
 
 /* If __chkr_init_chkr() was already called, something strange has happend */
 if (already_init)
   {
     chkr_printf(M_ALREADY_INIT);
     chkr_abort();
   }
 else
   already_init = 1;

#if 0
 chkr_printf("argc: %d argv[0]: %s\n", argc, argv[0]);
#endif

 /* Save the pid: set my_pid and my_pid_c */
 make_pid();
   
 /* dup the current output file. */
 dup2(chkr_out, CHKR_OUT_FD);
 chkr_out = CHKR_OUT_FD;
   
 /* Find the full path of the program. The full path is needed to read the
  *  symbols table */
 chkr_prog_path = chkr_find_executable(argv[0]);
 
 /* Set the output file name */
 chkr_out_file = sys_malloc(sizeof(DEFAULT_OUT_FILE) + 1);
 strcpy(chkr_out_file, DEFAULT_OUT_FILE);
 chkr_out_proto = sys_malloc(sizeof(DEFAULT_OUT_FILE) + 1);
 strcpy(chkr_out_proto, DEFAULT_OUT_FILE);
 
 /* Save argv[0] */
 full_progname = sys_malloc(strlen(argv[0]) + 1);
 strcpy(full_progname, argv[0]);
 
 /* Find the basename of argv[0] */
 progname = full_progname + strlen(full_progname);
 while (progname != full_progname && progname[-1] != '/')
   progname--;
 
 /* Parse each option of the environemnt variable CHECKEROPTS */
 args = getenv("CHECKEROPTS");
 if (args != (char*)0)
   parse_opt_line(args);

 /* Read ~/.checker file and parse its options */   
 args = getenv("HOME");
 if (args != (char*)0)
   strncpy(cffile, args, MAXPATHLEN-10);
 else
   strcpy(cffile, DEFAULT_HOME);
 strcat(cffile, OPTIONS_FILE);
 read_config_file(cffile);

 /* Set objects */
 init_main_object(linked, argv[0], nlibs, libs);
 objects->name = chkr_prog_path ? chkr_prog_path : full_progname;
 
 /* Do machine dependant stuff */
 chkr_init_machine();
      
 /* Display the copyright and the initial message */
 if (!silent_flag)
   {
     chkr_printf(M_COPYRIGHT, version);
     print_message(init_message);
     chkr_header(M_MESSAGES_FOR, full_progname);
     chkr_printf("\n");
   }

#ifndef MDCHECKER
 /* If the program contains files not compiled with Checker, emit a warning.
  *  The linker ld know it and set ___chkr_ld_opts__ according to that. */
 if (!silent_flag && (chkr_ld_options & 1))
   print_message(link_with_other_message);
#endif
       
 /* Some trivial checks. ??_red_zone must be a multiple of 4 */
 if ( be_red_zone % sizeof(PTR) || af_red_zone % sizeof(PTR))
   chkr_perror(M_I_BRS_MA_ET);

#ifndef MDCHECKER   
 /* Initialize the bitmaps */
 known_stack_limit = &argc;
 init_morecore();
#endif 

 /* If the full path of the program was not found, display a warning.
  *  This test must be done after parsing the options, since an option can
  *  set chkr_prog_path */
 if (chkr_prog_path == (char*)0)
   {
     chkr_header(M_PRGNAME_NOT_FOUND);
     chkr_printf(M_ARGV0_IS, argv[0]);
   }
   
 /* Checker is now initialized */
 chkr_is_init = 1;
 
#if 0
 /* Disp libs if verbose flag */
 if (verbose_flag)
   __chkr_disp_shlibs();
#endif
   
 /* If the --stop option is set, stop now */
 if(stop_flag)
   {
     chkr_printf(M_STOPPING_MYSELF, SIGCONT, my_pid);
     kill(my_pid, SIGSTOP);
   }
   
 /* Save infos of the output file.  They will be used to check if the user
  *  has tried to handle the output file. */   
 store_output_spec();
 
#ifndef MDCHECKER
 /* Save the signals handler/behavior so that they will be correctly 
  * handled. */
 save_signals();
#else
 /* Be sure chkr_do_end() is called before exiting.  Checker traps exit(2),
  * so this is OK.  However with MD, we must use atexit. */
 atexit(chkr_do_end);
#endif
}

/* Save the pid and fill my_pid_c. */
void
make_pid(void)
{
  int i;
  pid_t pid;
  
  my_pid = pid = getpid();
  my_pid_c[5] = '\0';
  pid %= 100000;
  for (i = 4; i >= 0; i--)
    {
      my_pid_c[i] = '0' + (pid % 10);
      pid /= 10;
    }
}

/* Extract the first field of INPUT and put it to OUTPUT
 * The metacharacters '"\ are recognize
 * Return the offset to the next field.
 */
static int
parse_args(char *input, char *output)
{
  char *buf;
  char *arg;
  int offset = 0;
  int squote = 0;	/* means ' */
  int dquote = 0;	/* means " */
  int bquote = 0;	/* means \ */
  
  arg = buf = alloca(strlen(input) + 1);
  
  /* forget the first blanks */
  while(*input != '\0' && isspace(*input))
    {
      input++;
      offset++;
    }
      
  while(*input != '\0')
    {
      if (bquote)
        {
          *arg++ = *input;
          bquote = 0;
        }
      else if (*input == '\\')
          bquote = 1;
      else if (squote)
        {
          if (*input == '\'')
            squote = 0;
          else
            *arg++ = *input;
        }
      else if (dquote)
        {
          if (*input == '\"')
            dquote = 0;
          else
            *arg++ = *input;
        }
      else if (*input == '\'')
        squote = 1;
      else if (*input == '\"')
        dquote = 1;
      else if (isspace(*input))
        break;
      else *arg++ = *input;
      input++;
      offset++;
    }
  *arg = '\0';
  
  if (arg != buf)
    {
      strcpy(output, buf);
      return offset;
    }
  else
    return 0;
}
      
struct Opt
{
 int short_flag;	/* short flag */
 char *long_flag;	/* long flag. (char*)0 means end. */
 char arg;		/* Is there an arg ?*/
};

struct Opt options[]={
{'s', "silent", 0},		/* no message at the begin */
{'q', "quiet", 0},		/* idem */
{'h', "help", 0},		/* display help */
{'o', "output", 1},		/* set the output file */
{'i', "image", 1},		/* set the image file */
{'n', "nosymtab", 0},		/* do not use symbol table */
{'a', "abort", 0},		/* abort */
{'p', "profile", 0},		/* display profile infos */
{'N', "NULLzone", 1},		/* set the size of the NULL zone */
{'d', "disable", 1},		/* disable an address */
{'S', "stop", 0},		/* stop before main, use SIGCONT to restart */
{'D', "detector", 1},		/* detector */
{'m', "malloc0", 1},		/* malloc(0) behavior */
{'A', "memalign", 0},		/* memalign align is a power of 2 */
{'v', "verbose", 0},
{  0, (char*)0, 0}};

/* Errors: */
#define GETOPT_END	-1	/* No more options */
#define GETOPT_NOPTION	-2	/* Not an option (doesn't begin with a '-') */
#define GETOPT_LNEEDARG	-3	/* There is no argument */
#define GETOPT_SNEEDARG -4	/* There is no argument */
#define GETOPT_LARG	-5	/* There is an argument */
#define GETOPT_SARG	-6	/* There is an argument */
#define GETOPT_UNKNOWN  -7	/* Option unknown */

/* Where the argument is written. Enough space must be allocated. */
static char *optarg;

/* The option line. */
static char *optline;

/* The short_flag (used when an error is returned) */
static int optopt;

/* Parse option: returns the short flag and set OPTARG if necessary */
static int
my_getopt(void)
{
 struct Opt *tmp;
 
 /* Skip the spaces */
 while (*optline && isspace(*optline))
   optline++;
 if (*optline == '\0')
   return GETOPT_END;
 if (*optline != '-')
   {
     optline += parse_args(optline, optarg);
     return GETOPT_NOPTION;
   }
 /* Skip the dash */
 optline++;
 if(*optline == '-')
   {
     /* There are two dashes.  This is a long flag */
     optline++;
     for(tmp = options; tmp->long_flag != (char*)0; tmp++)
       {
         if(strncmp(optline, tmp->long_flag, strlen(tmp->long_flag))==0)
           {
             optline += strlen(tmp->long_flag);
             optopt = tmp->short_flag;
             if (tmp->arg)
               {
                 /* This option require an argument. */
                 if (*optline == '=')
                   {
                     optline++;
                     optline += parse_args(optline, optarg);
                   }
                 else
                   {
                     while (*optline && isspace(*optline))
                       optline++;
                     if (*optline == '\0' || *optline == '-')
                       return GETOPT_LNEEDARG;
                     optline += parse_args(optline, optarg);
                   }
                 return tmp->short_flag;
               }
             else
               {
                 /* This option doesn't allow an argument */
                 if (*optline && !isspace(*optline))
                   {
                     optline += parse_args(optline, optarg);
                     return GETOPT_LARG;
                   }
                 else
                   {
                     *optarg = '\0';
                     return tmp->short_flag;
                   }
               }
           }
       }
     return GETOPT_UNKNOWN;	/* Not found */
   }
 else
   {
     /* There is only one dash.  This is a short flag */
     for(tmp = options; tmp->long_flag != (char*)0; tmp++)
       {
         if (*optline == tmp->short_flag)
           {
             optline++;
             optopt = tmp->short_flag;
             if (tmp->arg)
               {
                 /* This option require an argument. */
                 if (*optline == '=')
                   optline += parse_args(++optline, optarg);
                 else
                   {
                     while (*optline && isspace(*optline))
                       optline++;
                     if (*optline == '\0' || *optline == '-')
                       return GETOPT_SNEEDARG;
                     optline += parse_args(optline, optarg);
                   }
                 return tmp->short_flag;
               }
             else
               {
                 /* This option doesn't allow an argument. */
                 if (*optline && !isspace(*optline))
                   {
                     optline += parse_args(optline, optarg);
                     return GETOPT_SARG;
                   }
                 return tmp->short_flag;
               }
           }
       }
   }
 return GETOPT_UNKNOWN;
}

/* Handle an option line. */
static void
parse_opt_line(char *line)
{
 char *buf;
 int opt;
 
 optline = line;
 buf = alloca(strlen(line) + 1);
 
 do
   {
     optarg = buf;
     opt = my_getopt();
     /* No more option ? */
     if (opt == GETOPT_END)
       break;
     /* Argument needed and not present ? */
     if (opt == GETOPT_LNEEDARG)
       {
         chkr_printf(M_LOPTION_NEED_ARG, options[optopt].long_flag);
         continue;
       }
     if (opt == GETOPT_SNEEDARG)
       {
         chkr_printf(M_SOPTION_NEED_ARG, options[optopt].short_flag);
         continue;
       }
     /* Argument not needed ? */
     if (opt == GETOPT_LARG)
       {
         chkr_printf(M_LOPTION_NALLOW_ARG, options[optopt].long_flag);
         opt = optopt;
       }
     if (opt == GETOPT_SARG)
       {
         chkr_printf(M_SOPTION_NALLOW_ARG, options[optopt].short_flag);
         opt = optopt;
       }
     /* Not an option ? */
     if (opt == GETOPT_NOPTION)
       {
         chkr_printf(M_OPTION_ILLEGAL, optarg);
         continue;
       }
     if (opt == GETOPT_UNKNOWN)
       {
         char *beg;
         char *end;
         char *name;
         beg = optline;
         while (*optline && (*optline != ' '&& *optline != '\t'))
           optline++;
         end = optline;
         name = alloca(end - beg);
         strncpy(name, beg, end - beg);
         name[end - beg] = '\0';
         chkr_printf(M_OPTION_ILLEGAL, name);
         continue;
       }
     switch(opt)
       {
       case 's':
       case 'q':	/* quiet & silent */
         silent_flag = 1;
         verbose_flag = 0;
         break;
       case 'h':  /* help */
         print_message(init_message);
         print_message(help_message);
         silent_flag |= 2;	/* Do not redisplay the initial message */
         break;
       case 'a':  /* abort */
         print_message(abort_message);
         chkr_abort();
         break;
       case 'n':  /* no symtab */
#ifdef CHKR_SAVESTACK
         nosymtab_flag = 1;
#endif
         break;
       case 'o': /* output: set the output file */
         {
           int append = append_flag;	/* Use the default mode */
         
           if (*optarg == '+')
             {
               /* If the FILE begins with a '+', the file must be appended. */
               append = 1;
               optarg++;		/* Skip the '+' */
             }
           else if (*optarg == '-')
             {
               /* If the FILE begins with a '-', the file must be overwritten. */
               append = 0;
               optarg++;
             }

	   /* If FILE is only '+' or '-', set the append mode and that's all */           
           if (*optarg == '\0')
             {
               append_flag = append;
               break;
             }
           
           sys_free(chkr_out_proto);
           chkr_out_proto = sys_malloc(strlen(optarg) + 1);
           strcpy(chkr_out_proto, optarg);
          
           chkr_out_append = append;
           open_output_file();
         
           break;
         }
       case 'p':  /* profile */
#ifdef CHKR_PROFILE
         profile_flag = 1;
#endif       
         break;
       case 'N':  /* NULL zone size */
         null_pointer_zone = atod(optarg);
         break;
       case 'd':  /* disable */
         if(!parse_disable(optarg))	/* see l-malloc/maccess.c */
           chkr_printf(M_UNKNOWN_DISABLE, optarg);
         break;
#if 1 /* FIXME : we must check */
       case 'i':	/* image: set the image file */
         chkr_prog_path = copy_of_exec_file(optarg);
         break;
#endif
       case 'S':  /* stop */
         stop_flag = 1;
         break;
       case 'D': /* detector */
         if (strcmp(optarg, "end") == 0)
           do_detector_at_end = 1;
         break;
       case 'm': /* malloc(0) behavior */
         set_malloc0_behavior(optarg);
         break;
       case 'A': /* memalign behavior */
         memalign_align2 = 1;
         break;
       case 'v':
         verbose_flag++;
         break;
       default:
         chkr_printf(M_UNKNOWN_OP, options[optopt].long_flag);
       }
   }
 while(1);
}

/* Parse a line of configuration. This is used while parsing ~/.checker */
static void
parse_line(char *line)
{
 char *out;
 
 /* skip comment lines */
 if (line[0] == '\0' || line[0] == '#')
   return;
   
 out = alloca(strlen(line)+1);
 line += parse_args(line, out);
 if ((chkr_prog_path && strcmp(out, chkr_prog_path) == 0)
     || strcmp(out, progname) == 0 
     || strcmp(out, full_progname) == 0
     || strcmp(out, "default") == 0)
   parse_opt_line(line);
}

/* Parse the config file */
static void
read_config_file(char *file)
{
 int fd;
 int n;
 int offset;
 char *rc;
 char *line;
 int len;
 char buffer[_POSIX2_LINE_MAX + 1];
 
 fd = open(file, O_RDONLY);
 if (fd == -1)
   return;
 offset = 0;
 do
   {
     n = read(fd, buffer + offset, _POSIX2_LINE_MAX - offset);
     line = buffer;
     n += offset;
     len = n;
     if (n == 0)
       break;     
     while ((rc = memchr(line, '\n', len)) != (char*)0)
       {
         *rc = '\0';
         parse_line(line);
         len -= rc - line + 1;
         line = rc + 1;
       }
     strncpy(buffer, line, len);
     offset = len;
     if (offset == n)
       offset = 0;	/* line too long */
   }
 while (n == _POSIX2_LINE_MAX);
 
 /* there is perhaps no return at the last line */
 if (len != 0)
   {
     line[len] = '\0';
     parse_line(line);
   }
   
 close (fd);
}

static void
print_message(char **message)
{
 while(*message != (char*)0 && **message != 0)
   {
     write(chkr_out, *message, strlen(*message));
     message++;
   }
}

/* Convert a string to an int.
   Format are:
   0nnnnn...	base 8,
   0xnnnn...	base 16,
   0bnnnn...	base 2,
   nnnnnn...	base 10
   Stop when a bad caracter is found.
 */
int
atod(const char *c)
{
  int val=0;
  int base=10;
  int digit;
  
  if(*c == '0')
    {
      if(c[1] == 'x' || c[1] == 'X')
        {
          base = 16;
          c++;
        }
      else if(c[1] == 'b' || c[1] == 'B')
        {
          base = 2;
          c++;
        }
      else
        base = 8;
      c++;
    }
  while(*c != '\0')
    {
      digit = ctod(*c);
      c++;
      if (digit == -1 || digit >= base)
        break;
      val *= base;
      val += digit;
    }
  return val;
}
  
/* Convert a char to an int.  Used by atod() */
static int
ctod(const char c)
{
  if (c >= '0' && c <= '9')
    return c - '0';
  if (c >= 'a' && c <= 'f')
    return c - 'a' + 10;
  if (c >= 'A' && c <= 'F')
    return c - 'A' + 10;
  return -1;
}

/* This function is called just before _exit() and does some cleanup */
void
chkr_do_end(void)
{
 if (do_detector_at_end)
   __chkr_garbage_detector();
 chkr_remove_symtabfile();
#ifdef CHKR_PROFILE
 if (profile_flag)
   display_profile();
#endif  
}

/* Build the name of the output file with chkr_out_proto.
 * It parse %p and %n.
 * The result is allocated by sys_malloc and must be freed by sys_free
 */
char *
build_name_output_file(void)
{
 int i, j, len;
 char *res;
 
 j = 0;
 
 len = strlen(chkr_out_proto);
 res = sys_malloc(len + 1);
 
 /* Parse '%p' and '%n': replace %p by the pid and %n by the basename.
  *  Of course, '%%' is replaced by '%' */
 for (i = 0; i <= len; i++)
   {
     if (chkr_out_proto[i] != '%')
       {
         res[j++] = chkr_out_proto[i];
         continue;
       }
     switch(chkr_out_proto[++i])
       {
         case '%':
           res[j++] = '%';
           break;
         case 'n': /* the basename */
           res = sys_realloc(res, len + strlen(progname) + 1);
           strcpy(&res[j], progname);
           j += strlen(progname);
           break;
         case 'p': /* the pid */
           res = sys_realloc(res, len + 5 + 1);
           strcpy(&res[j], my_pid_c);
           j += 5;
           break;
         default:
           break;
       }
   }
 return res;
}

/* Open the output file. */
void
open_output_file(void)
{
 sys_free(chkr_out_file);
 chkr_out_file = build_name_output_file(); 
 chkr_out = open(chkr_out_file, 
                 O_WRONLY|O_CREAT| (chkr_out_append ? O_APPEND : O_TRUNC),
                 S_IRUSR|S_IWUSR);
 if (chkr_out == -1)
   chkr_out = 2;
 dup2(chkr_out, CHKR_OUT_FD);
 if (chkr_out != 2)
   close(chkr_out);
 chkr_out = CHKR_OUT_FD;
}

/* Re-open the output file if the name has changed. */
void
update_output_file(void)
{
 char *new_file;
 new_file = build_name_output_file();
 if (strcmp(new_file, chkr_out_file) == 0)
   return;
 sys_free(chkr_out_file);
 chkr_out_file = new_file;
 open_output_file();
}

/* Save some infos about the output file */
static void
store_output_spec(void)
{
  struct stat outstat;
  
  if (fstat(chkr_out, &outstat) != 0)
      st_dev = st_ino = 0;
  else
    {
      st_dev = outstat.st_dev;
      st_ino = outstat.st_ino;
    }
} 

/* Check if the output file has been handled by the user. If yes, reopen it */
void
check_output_spec(void)
{
 struct stat outstat;
 
 if (st_dev == 0 && st_ino == 0)
   return;
 
 if (fstat(chkr_out, &outstat) != 0 
     || st_dev != outstat.st_dev || st_ino != outstat.st_ino)
   {
     /* Always use O_APPEND, since the file is RE-open. */
     chkr_out = open(chkr_out_file, O_APPEND|O_WRONLY);
     if (chkr_out == -1)
       chkr_out = 2;
     dup2(chkr_out, CHKR_OUT_FD);
     close(chkr_out);
     chkr_out = CHKR_OUT_FD;
     store_output_spec();
   }
 return;
}
