/* Check the internal state of Malloc and miscellanous
   Copyright 1993, 1994 Tristan Gingold
		  Written December 1993 by Tristan Gingold

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License as
published by the Free Software Foundation; either version 2 of the
License, or (at your option) any later version.

This library is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; see the file COPYING.  If not, write to
the Free Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

 The author may be reached (Email) at the address gingold@amoco.saclay.cea.fr,
 or (US/French mail) as Tristan Gingold
   			  8 rue Parmentier
   			  F91120 PALAISEAU
   			  FRANCE
*/

#define _MALLOC_INTERNAL
#include "malloc.h"
#include "message.h"
#include "errlist.h"

/* The sizes of the red zones.  Must be a multiple of sizeof(void*) */
unsigned int be_red_zone = 4;
unsigned int af_red_zone = 32;

/* The malloc(0) behavior. */
#define MALLOC0_IS_ANSI 0	/* malloc(0) returns 0 */
#define MALLOC0_MASK 2
#define MALLOC0_IS_MALLOC1 1	/* malloc(0) is replaced by malloc(1) */
#define MALLOC0_IS_WARNING 2	/* malloc(0) produce a warning */
static int malloc0_behavior = MALLOC0_IS_ANSI;

struct mdesc *
find_mdesc(const PTR ptr)
{
  struct mdesc *mdp;
  
  mdp = _firstmdesc;
  while (mdp != (struct mdesc*)0)
    {
      if (ptr >= (PTR)(mdp->_firstblock) 
          && ptr <= (PTR)((uint) mdp->_lastblock + mdp->_lastblock->size))
        break;
      mdp = mdp->info.inmem.next_mdesc;
    }
  return mdp;
}
  
/*
 * Blocks must be like that:
 * +--------+-------------+----------------------------+-------------+
 * | header | be_red_zone |           block            | af_red_zone |
 * |        |             |                            |             |
 * +--------+-------------+----------------------------+-------------+
 *                        |<-------- real_size ------->|
 *          |<------------------------ size ------------------------>|
 */
/* Search the header for an address in a block.
 * Return NULL_HEADER if not found
 * If SOFT is set, PTR must point inside the block area (and not inside
 *   a red_zone or a header).
 */
struct malloc_header *
find_header (struct mdesc *mdp, const PTR ptr, int soft)
{
  struct malloc_header *res;
  u_int addr;
  if (mdp == (struct mdesc *)0)
    {
      mdp = find_mdesc(ptr);
      if (mdp == (struct mdesc *)0)
        return NULL_HEADER;
    }
  if (mdp->_firstblock == NULL_HEADER)
    return NULL_HEADER;
  res = mdp->_firstblock;
  if (soft)
    {
      addr = (u_int) ptr;
      if (addr < (u_int) res)
	return NULL_HEADER;
      for (; res != NULL_HEADER; res = res->next)
	{
	  if (addr <= ((u_int) res + res->size))
	    return res;
	}
    }
  else
    {
      addr = (u_int) ptr - HEADER_SIZE - be_red_zone;
      for (; res != NULL_HEADER; res = res->next)
	{
	  if ((addr >= (u_int) (res)) && (addr <= ((u_int) res + res->size - af_red_zone)))
	    return res;
	}
    }
  return NULL_HEADER;
}

/* Get the age of block.
 * If no age, returns -1.
 */
int
get_age (struct mdesc *mdp, struct malloc_header *block)
{
#ifdef NO_AGE
  return -1;
#else
  struct malloc_header *tmp;
  int age = 0;

  if (mdp == (struct mdesc *)0)
    {
      mdp = find_mdesc(block);
      if (mdp == (struct mdesc*)0)
        return -1;
    }
    
  if (block->state != MDAGED || mdp->_youngerblock == NULL_HEADER)
    return -1;
  tmp = mdp->_youngerblock;
  do
    {
      if (tmp == block)
	return age;
      age++;
      tmp = tmp->info.aged.next;
    }
  while (tmp);
  return -1;
#endif
}

/* Disp the history. */
void
disp_block_history (PTR * ptr)
{
  for (; *ptr; ptr++)
    chkr_show_addr ((PTR) ptr);
}

/* Set the malloc(0) behavior.  Called by parse-args.c */
void
set_malloc0_behavior (const char *opt)
{
  if (opt[0] == '0' && opt[1] == '\0')
    {
      malloc0_behavior &= MALLOC0_MASK;
      malloc0_behavior |= MALLOC0_IS_ANSI;
    }
  else if (opt[0] == '1' && opt[1] == '\0')
    {
      malloc0_behavior &= MALLOC0_MASK;
      malloc0_behavior |= MALLOC0_IS_MALLOC1;
    }
  else if (strncmp (opt, "warning", strlen (opt)) == 0)
    {
      malloc0_behavior |= MALLOC0_IS_WARNING;
    }
  else if (strncmp (opt, "no-warning", strlen (opt)) == 0)
    {
      malloc0_behavior &= ~MALLOC0_IS_WARNING;
    }
  else
    {
      chkr_printf (M_UNKNOWN_MALLOC0, opt);
    }
}

/* Modify SIZE (if necessary) according to the behavior of malloc(0).
 * ID identify the caller
 * returns the new size. */
size_t
test_malloc0 (size_t size, int id)
{
  static char *id_func[] =
  {"malloc",  "realloc",  "calloc",  "valloc",  "memalign",
   "mmalloc", "mrealloc", "mcalloc", "mvalloc", "mmemalign"};

  if (size)
    return size;
  if (malloc0_behavior & MALLOC0_IS_WARNING)
    {
      chkr_header (M_MALLOC0_WARNING, id_func[id]);
#ifdef CHKR_SAVESTACK
      chkr_frames_to_forget = 2;
      chkr_show_frames ();
#endif
    }
  if (malloc0_behavior & MALLOC0_IS_MALLOC1)
    return 1;
  else
    return 0;
}


  
/* Check internal structure of an heap. If something bad is found, display it.
 * This is very useful when you debug malloc
 */
void
__chkr_check_mdesc (struct mdesc *mdp)
{
  struct malloc_header *tmp;
  struct malloc_header *tmp1;
  int i;

  chkr_report(M_C_ICK_CK_ET);
  
  if (mdp == NULL_MDESC)
    {
      mdp = __mmalloc_default_mdp;
      if (mdp == NULL_MDESC)
        {
          chkr_printf (M_C_NO_BLOCKS);
          return;
        }
    }
    
  /* Do nothing if there are no mdesc. */
  if (_firstmdesc == NULL_MDESC || _lastmdesc == NULL_MDESC)
    {
      if (_firstmdesc != _lastmdesc)
        chkr_printf (M_C_FM_OR_LM_BAD);
      else
        chkr_printf (M_C_NO_BLOCKS);
      return;
    }
    
  chkr_printf (M_C_CHECK_MDESC, mdp);
  
  /* Do nothing if 'mmalloc' was never called */
  if (mdp->_firstblock == NULL_HEADER ||
      mdp->_lastblock == NULL_HEADER)
    {
      if (mdp->_firstblock != mdp->_lastblock)
	chkr_printf (M_C_FB_OR_LB_BAD);
      else
	chkr_printf (M_C_NO_BLOCKS);
      return;
    }

  /* first check */
  if (mdp->_firstblock->prev != NULL_HEADER)
    chkr_printf (M_C_FBLOCK_BAD);
  if (mdp->_lastblock->next != NULL_HEADER)
    chkr_printf (M_C_LBLOCK_BAD);

  /* Check for linearity: each next block must be after the block.
  * bl->next and bl->prev are checked here.
  */
  tmp1 = NULL_HEADER;
  for (tmp = mdp->_firstblock; tmp; tmp = tmp->next)
    {
      if (tmp < tmp1)
	chkr_printf (M_C_BAD_LINEARITY, tmp, tmp1);
      if (tmp->prev != tmp1)
	chkr_printf (M_C_BAD_PLINK, tmp);
      if (tmp->next && ((char *) tmp->next - (char *) tmp != (tmp->size + HEADER_SIZE)))
	chkr_printf (M_C_BAD_BLOCK_SIZE, tmp);
      tmp1 = tmp;
    }
  if (tmp1 != mdp->_lastblock)
    {
      chkr_printf (M_C_BAD_LAST_BLOCK);
      chkr_printf (M_C_STOP_HERE);
      return;
    }

  /* Clean the MDCHECK flag */
  for (tmp = mdp->_firstblock; tmp; tmp = tmp->next)
    tmp->state &= ~MDCHECK;

#ifndef NO_AGE
  /* Check all aged blocks */
  i = 0;
  tmp = mdp->_youngerblock;
  if (mdp->_agedblock == 0)
    {
      if (tmp != NULL_HEADER)
	chkr_printf (M_C_U_AGED_BLOCK, tmp);
    }
  else
    {
      if (tmp->info.aged.prev != NULL_HEADER)
	chkr_printf (M_C_BAD_PL_YOUNGB, tmp);
      while (tmp != NULL_HEADER)
	{
	  if (tmp->state != MDAGED)
	    chkr_printf (M_C_NON_AGED_BLOCK, tmp);
	  tmp->state |= MDCHECK;
	  tmp = tmp->info.aged.next;
	  i++;
	}
    }
  if (i != mdp->_agedblock)
    chkr_printf (M_C_BAD_NUM_AGED_B, i, mdp->_agedblock);
  if (i > _block_age)
    chkr_printf (M_C_TOO_AGED_BLOCK, i, _block_age);
#endif /* NO_AGE */

  /* Check all free blocks */
  for (i = 0; i < HASH_SIZE; i++)
    {
      tmp = mdp->_heapinfo[i];
      if (tmp == NULL_HEADER)
	continue;
      if (tmp->info.free.prev != NULL_HEADER)
	chkr_printf (M_C_BAD_PL_F_FREEB, tmp);
      tmp1 = NULL_HEADER;
      while (tmp)
	{
	  if (tmp->state != MDFREE)
	    {
	      chkr_printf (M_C_FREEB_BAD_STAT, tmp);
	      break;
	    }
	  if (log_size (tmp->size) != i)
	    chkr_printf (M_C_BLOCK_BAD_LIST, tmp);
	  tmp->state |= MDCHECK;
	  if (tmp->info.free.prev != tmp1)
	    chkr_printf (M_C_BAD_PLIN_FREEB, tmp);
	  tmp1 = tmp;
	  tmp = tmp->info.free.next;
	}
#ifndef NO_HEAPINDEX
      if (mdp->_heapindex < i)
	chkr_printf (M_C_BAD_HEAPINDEX);
#endif /* NO_HEAPINDEX */
    }

  /* Are there other free blocks ? */
  for (tmp = mdp->_firstblock; tmp; tmp = tmp->next)
    {
      if (tmp->state == MDFREE)
	chkr_printf (M_C_UNKNOWN_FREEB, tmp);
      /* clean the flag */
      tmp->state &= ~MDCHECK;
    }

  /* Are there consecutive free blocks ? */
  for (tmp = mdp->_firstblock; tmp; tmp = tmp->next)
    {
      if (tmp->state == MDFREE && tmp->next != NULL_HEADER && tmp->next->state == MDFREE)
	chkr_printf (M_C_FREE_AND_FREE, tmp);
    }
  /* TODO : check for no coalision
           good order in free lists */
}

/* Check each heap. */
void
__chkr_check_intern()
{
  struct mdesc *tmp1;
  struct mdesc *tmp;
  
  /* Do nothing if there are no mdesc. */
  if (_firstmdesc == NULL_MDESC || _lastmdesc == NULL_MDESC)
    {
      if (_firstmdesc != _lastmdesc)
        chkr_printf (M_C_FM_OR_LM_BAD);
      else
        chkr_printf (M_C_NO_BLOCKS);
      return;
    }
    
  /* Check for links. */
  tmp1 = NULL_MDESC;
  for (tmp = _firstmdesc; tmp; tmp = tmp->info.inmem.next_mdesc)
    {
      if (tmp->info.inmem.prev_mdesc != tmp1)
	chkr_printf (M_C_BAD_MPLINK, tmp);
      tmp1 = tmp;
    }
  if (tmp1 != _lastmdesc)
    {
      chkr_printf (M_C_BAD_LAST_MDESC);
      chkr_printf (M_C_STOP_HERE);
      return;
    }
  
  /* Check each heap. */
  for (tmp = _firstmdesc; tmp; tmp = tmp->info.inmem.next_mdesc)
    __chkr_check_mdesc(tmp);
}

/* Display some infos of each heap */
void
__chkr_disp_heaps(void)
{
  struct mdesc *mdp;
  
  mdp = _firstmdesc;
  if (mdp == NULL_MDESC)
    chkr_printf (M_C_NO_BLOCKS);
  else
    {
      chkr_printf(" mdesc*    |       base-breakval   | fd\n");
      while (mdp)
        {
          chkr_printf("0x%08x : 0x%08x-0x%08x | ", mdp, mdp->base, mdp->breakval);
          if (mdp->flags & MMALLOC_DEVZERO)
            chkr_printf("/dev/zero\n");
          else if (mdp->flags & MMALLOC_SBRK_HEAP)
            chkr_printf("sbrk()\n");
          else
            chkr_printf("%d\n", mdp->fd);
          mdp = mdp->info.inmem.next_mdesc;
        }
    }
}
      
void
__chkr_dump_heap (struct mdesc *mdp)
{
  struct malloc_header *tmp;

  if (mdp == NULL_MDESC)
    {
      mdp = __mmalloc_default_mdp;
      if (mdp == NULL_MDESC)
        {
          chkr_printf(M_C_NO_BLOCKS);
          return;
        }
    }
    
  tmp = mdp->_firstblock;
  if (tmp == NULL_HEADER)
    {
      chkr_printf (M_C_NO_BUSY_BLOCK);
      return;
    }

  chkr_printf (M_C_BUSY_BLOCK_ARE);

#if 0
  while (tmp != NULL)
    {
      if (tmp->state == MDBUSY)
	chkr_printf (M_C_BLOC_N_IS_BUSY, tmp, tmp->size / 1024);
      else if (tmp->state == MDINTRN)
	chkr_printf ("MDINTRN: block at 0x%x, size %dkb\n", tmp, tmp->size / 1024);
      tmp = tmp->next;
    }
#else
  while (tmp != NULL)
    {
      switch (tmp->state)
	{
	case MDBUSY:
	  chkr_printf ("BUSY: ");
	  switch (tmp->garbage_t)
	    {
	    case POINT_NOT:
	      chkr_printf("(no) ");
	      break;
	    case POINT_SURE:
	      chkr_printf("(sure) ");
	      break;
	    case POINT_MAYBE:
	      chkr_printf("(maybe) ");
	      break;
	    case POINT_LEAK:
	      chkr_printf("(leak) ");
	      break;
	    case POINT_SEEN:
	      chkr_printf("(seen) ");
	      break;
	    default:
	      chkr_printf("(%d?) ", tmp->garbage_t);
	   }
	  break;
	case MDFREE:
	  chkr_printf ("FREE: ");
	  break;
	case MDINTRN:
	  chkr_printf ("INTERN: ");
	  break;
	case MDAGED:
	  chkr_printf ("AGED: ");
	  break;
	case MDBRK:
	  chkr_printf ("BRK: ");
	  break;
	}
      chkr_printf ("block at 0x%x, size %dkb (%d bytes)\n", tmp, tmp->size / 1024, tmp->size);
      tmp = tmp->next;
    }
#endif
}

/* Display the free blocks list */
void
__chkr_dump_free (struct mdesc *mdp)
{
  int i;
  struct malloc_header *tmp;
  
  if (mdp == NULL_MDESC)
    {
      mdp = __mmalloc_default_mdp;
      if (mdp == NULL_MDESC)
        {
          chkr_printf(M_C_NO_BLOCKS);
          return;
        }
    }

  for (i = 0; i < HASH_SIZE; i++)
    {
      tmp = mdp->_heapinfo[i];
      if (tmp == NULL_HEADER)
	continue;
      chkr_printf (M_C_BLOCK_SIZ_LESS, 1 << i);
      while (tmp != NULL_HEADER)
	{
	  chkr_printf (M_C_BLOCK_AT_SIZE, tmp, tmp->size / 1024);
	  tmp = tmp->info.free.next;
	}
    }
}

/* Display all informations about a block */
void
__chkr_dump_block (PTR block)
{
  struct mdesc *mdp;
  struct malloc_header *blk;
  PTR *ptr_on_block;

  mdp = find_mdesc(block);
  blk = find_header (mdp, block, 0);
  if (blk == NULL_HEADER)
    {
      chkr_printf ("Block 0x%x not found.", block);
      return;
    }
  chkr_printf ("Block 0x%x (in fact 0x%x):\n", block, blk);
  chkr_printf ("  prev = 0x%x\n", blk->prev);
  chkr_printf ("  next = 0x%x\n", blk->next);
  chkr_printf ("  size = 0x%x\n", blk->size);
  switch (blk->state)
    {
    case MDBUSY:
      chkr_printf ("  BUSY:\n");
      chkr_printf ("    real_size = 0x%x\n", blk->info.busy.real_size);
      break;
    case MDFREE:
      chkr_printf ("   FREE:\n");
      chkr_printf ("    next = 0x%x\n", blk->info.aged.next);
      chkr_printf ("    prev = 0x%x\n", blk->info.aged.prev);
      break;
    case MDINTRN:
      chkr_printf ("   INTERN:\n");
      break;
    case MDAGED:
      chkr_printf ("   AGED:\n");
      chkr_printf ("    next = 0x%x\n", blk->info.aged.next);
      chkr_printf ("    prev = 0x%x\n", blk->info.aged.prev);
      break;
    case MDBRK:
      chkr_printf ("   BRK:\n");
      break;
    }
#if CHKR_SAVESTACK
  ptr_on_block = (PTR *) ((int) blk + blk->size + HEADER_SIZE - af_red_zone);
  chkr_load_symtab ();
  chkr_printf (M_BLOCK_ALLO_FRM);
  disp_block_history (ptr_on_block);
  if (blk->state == MDAGED)
    {
      chkr_printf (M_N_FREE_CALLED, get_age (mdp, blk));
      ptr_on_block = (PTR *) ((int) blk + HEADER_SIZE);
      disp_block_history (ptr_on_block);
    }
  chkr_unload_symtab ();
#endif /* CHKR_SAVESTACK */
}

/* Disp info about shared libs */
void
__chkr_disp_shlibs(void)
{
 struct object *obj;
 chkr_printf("text_beg text_end data_beg data_end bss_end  name\n");
 for (obj = objects; obj; obj = obj->next)
   chkr_printf("%08x %08x %08x %08x %08x %s\n",
   	obj->text_beg, obj->text_end, obj->data_beg, obj->data_end,
   	obj->bss_end, obj->name);
}
